#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

# This Class is a plugin for the Shinken Broker. It is in charge
# to brok information of the service perfdata into the file
# var/service-perfdata
# So it just manage the service_check_return
# Maybe one day host data will be useful too
# It will need just a new file, and a new manager :)

import time
import shutil
import os
import datetime

from shinken.basemodule import BaseModule
from shinken.util import get_day
from shinken.log import logger

properties = {
    'daemons' : ['broker'],
    'type'    : 'simple_log',
    'external': False,
    'phases'  : ['running'],
}


# called by the plugin manager to get a broker
def get_instance(modconf):
    logger.debug("Get a Simple log broker for plugin %s" % modconf.get_name())
    
    # Catch errors
    path = getattr(modconf, 'path', '/var/log/shinken/shinken.log')  # here only for archive, will not be open
    nagios_export_path = getattr(modconf, 'nagios_export_path', '/var/log/shinken/nagios_export.log')
    archive_path = modconf.archive_path
    # Remove trailing slash from archive_path if need
    if archive_path[-1] in (os.sep, os.altsep):
        archive_path = archive_path[:-1]
    
    instance = Simple_log_broker(modconf, path, nagios_export_path, archive_path)
    return instance


# Class for the put broks log in unified file
# Get broks and puts them in log file
class Simple_log_broker(BaseModule):
    def __init__(self, modconf, path, nagios_export_path, archive_path):
        BaseModule.__init__(self, modconf)
        self.path = path
        self.nagios_export_path = nagios_export_path
        self.archive_path = archive_path
        self.nagios_export_file = None
        try:
            os.stat(archive_path)
        except:
            os.mkdir(archive_path)
    
    
    def check_nagios_export_and_do_archive(self):
        must_reopen_file = self.do_check_and_do_archive(self.nagios_export_path, self.nagios_export_file)
        if must_reopen_file:
            logger.debug("I open the log file %s" % self.nagios_export_path)
            self.nagios_export_file = open(self.nagios_export_path, 'a')
        return must_reopen_file
    
    
    # Check the path file age. If it's last day, we archive it.
    # Return True if the file has moved
    def do_check_and_do_archive(self, path=None, file=None):
        now = int(time.time())
        # first check if the file last mod (or creation) was not our day
        try:
            t_last_mod = int(float(str(os.path.getmtime(path))))
        except OSError:  # there should be no path from now, so no move :)
            return False
        
        t_last_mod_day = get_day(t_last_mod)
        today = get_day(now)
        # Will be saved with the date of yesterday because all elements are from yesterday
        yesterday = get_day(now - 3600)
        # print "Dates: t_last_mod: %d, t_last_mod_day: %d, today: %d" % (t_last_mod, t_last_mod_day, today)
        if t_last_mod_day != today:
            logger.info("We are archiving the old log file")
            
            # For the first pass, it's not already open
            if file:
                file.close()
            
            # Now we move it
            # f_name is like nagios.log
            f_name = os.path.basename(path)
            # remove the ext -> (nagios,.log)
            (f_base_name, ext) = os.path.splitext(f_name)
            # make the good looking day for archive name like -05-09-2010-00
            d = datetime.datetime.fromtimestamp(yesterday)
            s_day = d.strftime("-%m-%d-%Y-00")
            archive_name = f_base_name + s_day + ext
            file_archive_path = os.path.join(self.archive_path, archive_name)
            logger.info("Moving the old log file from %s to %s" % (path, file_archive_path))
            
            shutil.move(path, file_archive_path)
            return True
        return False
    
    
    # A service check have just arrived, we UPDATE data info with this
    def manage_log_monitoring_brok(self, b):
        self.check_nagios_export_and_do_archive()
        self.nagios_export_file.write(b.data['log'].encode('UTF-8'))
        self.nagios_export_file.flush()
    
    
    def init(self):
        # Be sure to archive the shinken.log if present, was not created since 02.06.03
        self.do_check_and_do_archive(path=self.path)
        
        moved = self.check_nagios_export_and_do_archive()
        if not moved:
            logger.debug("I open the log file %s" % self.nagios_export_path)
            self.nagios_export_file = open(self.nagios_export_path, 'a')
