#!/usr/bin/env python
# -*- coding: utf-8 -*-
# Copyright (C) 2009-2012:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#    Andreas Karfusehr, andreas@karfusehr.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

import copy
import math
import time
import datetime
import urllib
from shinken.log import logger
from shinken.misc.sorter import hst_srv_sort
from shinken.objects.proxyitem import proxyitemsgraph
from perfdata_guess import get_perfometer_table_values

try:
    import json
except ImportError:
    # For old Python version, load
    # simple json (it can be hard json?! It's 2 functions guy!)
    try:
        import simplejson as json
    except ImportError:
        print "Error: you need the json or simplejson module"
        raise

# APP Global, set by the module
app = None


class Helper(object):
    def __init__(self):
        pass
    
    
    def set_app(self, app_):
        global app
        app = app_
        self.app = app_
    
    
    def gogo(self):
        return 'HELLO'
    
    
    def get_elt_dict(self, elt, skip_properties=()):
        if isinstance(elt, list):
            l = []
            for e in elt:
                _d = self.get_elt_dict(e, skip_properties)
                if _d:
                    l.append(_d)
            return l
        cls = elt.__class__
        couples = [(k, v) for (k, v) in cls.properties.iteritems()] + [(k, v) for (k, v) in getattr(cls, 'running_properties', {}).iteritems()]
        d = {}
        for (k, v) in couples:
            try:
                if k in skip_properties:
                    continue
                else:
                    value = getattr(elt, k)
                json.dumps(value)
                d[k] = value
            except Exception, exp:  # if cannot json, pass
                if k in ('output', 'long_output'):
                    d[k] = app._('error.invalide_output')
                # We don't print the value because if this no jsonable it should broke the logger.
                logger.debug('Invalid properties in [%s]-[%s] [%s] exp:[%s]' % (cls, elt.get_full_name(), k, exp))
        
        if hasattr(elt, 'id') and isinstance(elt.id, int):
            d['id'] = elt.id
        
        return d
    
    
    def act_inactive(self, b):
        if b:
            return 'Active'
        else:
            return 'Inactive'
    
    
    def yes_no(self, b):
        if b:
            return 'Yes'
        else:
            return 'No'
    
    
    def print_float(self, f):
        return '%.2f' % f
    
    
    def ena_disa(self, b):
        if b:
            return 'Enabled'
        else:
            return 'Disabled'
    
    
    # For a unix time return something like
    # Tue Aug 16 13:56:08 2011
    def print_date(self, t):
        if t == 0 or t == None:
            return 'N/A'
        return time.asctime(time.localtime(t))
    
    
    # For a time, print something like
    # 10m 37s  (just duration = True)
    # N/A if got bogus number (like 1970 or None)
    # 1h 30m 22s ago (if t < now)
    # Now (if t == now)
    # in 1h 30m 22s
    # Or in 1h 30m (no sec, if we ask only_x_elements=2, 0 means all)
    def print_duration(self, t, just_duration=False, x_elts=0):
        if t == 0 or t == None:
            return 'N/A'
        # print "T", t
        # Get the difference between now and the time of the user
        seconds = int(time.time()) - int(t)
        
        # If it's now, say it :)
        if seconds == 0:
            return 'Now'
        
        in_future = False
        
        # Remember if it's in the future or not
        if seconds < 0:
            in_future = True
        
        # Now manage all case like in the past
        seconds = abs(seconds)
        # print "In future?", in_future
        
        # print "sec", seconds
        seconds = long(round(seconds))
        # print "Sec2", seconds
        minutes, seconds = divmod(seconds, 60)
        hours, minutes = divmod(minutes, 60)
        days, hours = divmod(hours, 24)
        weeks, days = divmod(days, 7)
        months, weeks = divmod(weeks, 4)
        years, months = divmod(months, 12)
        
        minutes = long(minutes)
        hours = long(hours)
        days = long(days)
        weeks = long(weeks)
        months = long(months)
        years = long(years)
        
        duration = []
        if years > 0:
            duration.append('%dy' % years)
        else:
            if months > 0:
                duration.append('%dM' % months)
            if weeks > 0:
                duration.append('%dw' % weeks)
            if days > 0:
                duration.append('%dd' % days)
            if hours > 0:
                duration.append('%dh' % hours)
            if minutes > 0:
                duration.append('%dm' % minutes)
            if seconds > 0:
                duration.append('%ds' % seconds)
        
        # print "Duration", duration
        # Now filter the number of printed elements if ask
        if x_elts >= 1:
            duration = duration[:x_elts]
        
        # Maybe the user just want the duration
        if just_duration:
            return ' '.join(duration)
        
        # Now manage the future or not print
        if in_future:
            return 'in ' + ' '.join(duration)
        else:  # past :)
            return ' '.join(duration) + ' ago'
    
    
    def get_all_tags(self, app):
        r = set()
        for h in app.datamgr.get_hosts():
            for t in h.tags:
                r.add(t.lower())
        l = list(r)
        l.sort()
        return l
    
    
    # Return something like:
    # {
    #                  "id": "localhost",
    #                  "name": "localhost",
    #                  "data": {"$color":"red", "$dim": 5*2, "some other key": "some other value"},
    #                  "adjacencies": [{
    #                          "nodeTo": "main router",
    #                          "data": {
    #                              "$type":"arrow",
    #                              "$color":"gray",
    #                              "weight": 3,
    #                              "$direction": ["localhost", "main router"],
    #                          }
    #                      }
    #                      ]
    #              }
    # But as a python dict
    
    def get_all_nodes_from_aggregation_node(self, tree):
        res = [{'path': tree['path'], 'services': tree['services'], 'state': tree['state'], 'full_path': tree['full_path']}]
        for s in tree['sons']:
            r = self.get_all_nodes_from_aggregation_node(s)
            for n in r:
                res.append(n)
        return res
    
    
    def get_unauthorized_item(self, item):
        result = {
            'is_unknown': False,
            'unsight'   : True,
            'uuid'      : item.get_instance_uuid(),
        }
        if item.my_type == "service":
            result['host_name'] = item.host.get_name()
            result['service_description'] = item.get_name()
        else:
            result['host_name'] = item.get_name()
        summary = self.get_summary(item)
        result['status'] = summary['status']
        result['context'] = summary['summary']
        return result
    
    
    # Return a button with text, image, id and class (if need)
    def get_button(self, text, img=None, id=None, cls=None):
        # s = '<div class="buttons">\n'
        s = '<div class="btn">\n'
        if cls and not id:
            s += '<div class="%s">\n' % cls
        elif id and not cls:
            s += '<div id="%s">\n' % id
        elif id and cls:
            s += '<div class="%s" id="%s">\n' % (cls, id)
        else:
            s += '<div>\n'
        if img:
            s += '<img src="%s" alt=""/>\n' % img
        s += '<span class="title">%s</span>' % text
        s += ''' </div>
            </div>\n'''
        
        return s
    
    
    # For and host, return the services sorted by business
    # impact, then state, then desc
    def get_host_services_sorted(self, host):
        t = copy.copy(host.services)
        t.sort(hst_srv_sort)
        return t
    
    
    def get_input_bool(self, b, id=None):
        id_s = ''
        if id:
            id_s = 'id="%s"' % id
        if b:
            return """<input type="checkbox" checked="checked" %s/>\n""" % id_s
        else:
            return """<input type="checkbox" %s />\n""" % id_s
    
    
    def print_business_rules_mobile(self, tree, level=0, source_problems=[]):
        # safe_print("Should print tree", tree)
        # safe_print('with source_problems', source_problems)
        node = tree['node']
        name = node.get_full_name()
        fathers = tree['fathers']
        fathers = sorted(fathers, key=lambda dict: dict['node'].get_full_name())
        s = ''
        # Maybe we are the root problem of this, and so we are printing it
        root_str = ''
        if node in source_problems:
            # print "I am a root problem"
            root_str = ' <span class="alert-small alert-critical"> Root problem</span>'
        # Do not print the node if it's the root one, we already know its state!
        if level != 0:
            s += "%s is %s since %s %s\n" % (self.get_link_mobile(node), node.state, self.print_duration(node.last_state_change, just_duration=True), root_str)
        
        # If we got no parents, no need to print the expand icon
        if len(fathers) > 0:
            # We look if the below tree is goodor not
            tree_is_good = (node.state_id == 0)
            
            # If the tree is good, we will use an expand image
            # and hide the tree
            if tree_is_good:
                display = 'none'
                img = 'expand.png'
            else:  # we will already show the tree, and use a reduce image
                display = 'block'
                img = 'reduce.png'
            
            s += """<ul id="business-parents-%s" style="display: %s; ">""" % (name, display)
            
            for n in fathers:
                sub_node = n['node']
                sub_s = self.print_business_rules_mobile(n, level=level + 1, source_problems=source_problems)
                s += '<li class="%s">%s</li>' % (self.get_small_icon_state(sub_node), sub_s)
            s += "</ul>"
        # safe_print("Returning s:", s)
        return s
    
    
    def print_business_rules(self, app, tree, level=0, source_problems=[], prefix=''):
        # safe_print("Should print tree", tree)
        # safe_print('with source_problems', source_problems)
        node = tree['node']
        name = node.get_full_name()
        fathers = tree['fathers']
        fathers = sorted(fathers, key=lambda dict: dict['node'].get_full_name())
        s = ''
        
        # Maybe we are the root problem of this, and so we are printing it
        root_str = ''
        if node in source_problems:
            # print "I am a root problem"
            root_str = ' <span class="alert-small alert-critical"> Root problem</span>'
        # Do not print the node if it's the root one, we already know its state!
        if level != 0:
            s += "%s %s %s %s %s %s\n" % (self.get_link(node), app._('dep_tree.is'), node.state, app._('dep_tree.since'), self.print_duration(node.last_state_change, just_duration=True), root_str)
        
        # If we got no parents, no need to print the expand icon
        if len(fathers) > 0:
            # We look if the below tree is goodor not
            tree_is_good = (node.state_id == 0)
            
            # If the tree is good, we will use an expand image
            # and hide the tree. Aways unfold first level
            if tree_is_good and not level == 0:
                display = 'none'
                img = 'expand.png'
            else:  # we will already show the tree, and use a reduce image
                display = 'block'
                img = 'reduce.png'
            
            # If we are the root, we already got this
            if level != 0:
                s += """<a id="togglelink-%s" href="javascript:toggleBusinessElt(''%s %s')"><img id="business-parents-img-%s" src="/static/%s/images/%s" alt="toggle"> </a> \n""" % (
                    prefix + name, prefix, name, prefix + name, str(app.http_start_time), img)
            
            s += """<ul id="business-parents-%s" style="display: %s; ">""" % (prefix + name, display)
            
            for n in fathers:
                sub_node = n['node']
                sub_s = self.print_business_rules(app, n, level=level + 1, source_problems=source_problems, prefix=prefix + name)
                s += '<li style="background-image: url(/static/%d/img/icons/state_%s.png" class="%s">%s</li>' % (app.http_start_time, sub_node.state.lower(), self.get_small_icon_state(sub_node), sub_s)
            s += "</ul>"
        # safe_print("Returning s:", s)
        return s
    
    
    def print_business_tree(self, tree, level=0, prefix=''):
        # safe_print("Should print tree", tree)
        node = tree['node']
        name = node.get_full_name()
        fathers = tree['fathers']
        fathers = sorted(fathers, key=lambda dict: dict['node'].get_full_name())
        s = ''
        # Do not print the node if it's the root one, we already know its state!
        if level != 0:
            s += "%s is %s since %s\n" % (self.get_link(node), node.state, self.print_duration(node.last_state_change, just_duration=True))
        
        # If we got no parents, no need to print the expand icon
        if len(fathers) > 0:
            # We look if the below tree is goodor not
            tree_is_good = (node.state_id == 0)
            
            # If the tree is good, we will use an expand image
            # and hide the tree
            if tree_is_good:
                display = 'none'
                img = 'expand.png'
            else:  # we will already show the tree, and use a reduce image
                display = 'block'
                img = 'reduce.png'
            
            # If we are the root, we already got this
            if level != 0:
                s += """<a id="togglelink-%s" href="javascript:toggleBusinessElt('%s %s')"><img id="business-parents-img-%s" src="/static/'+str(app.http_start_time)+'/images/%s" alt="toggle"> </a> \n""" % (
                    prefix + '-' + name, prefix, name, prefix + '-' + name, img)
            
            s += """<ul id="business-parents-%s" class="treeview" style="display: %s; ">""" % (prefix + '-' + name, display)
            
            for n in fathers:
                sub_node = n['node']
                sub_s = self.print_business_rules(n, level=level + 1, prefix=prefix)
                s += '<li style="background-image: url(/static/%d/img/icons/state_%s.png" class="%s">%s</li>' % (app.http_start_time, sub_node.state.lower(), self.get_small_icon_state(sub_node), sub_s)
            s += "</ul>"
        # safe_print("Returning s:", s)
        return s
    
    
    # Get the small state for host/service icons
    # and satellites ones
    def get_small_icon_state(self, obj):
        if obj.__class__.my_type in ['service', 'host']:
            # managed degraded cluster
            if obj.__class__.my_type == 'host' and obj.got_business_rule:
                if obj.bp_state == 1:
                    return 'warning'
                elif obj.bp_state == 3:
                    return 'unknown'
            if obj.state == 'PENDING':
                return 'unknown'
            if obj.state == 'OK':
                return 'ok'
            if obj.state == 'UP':
                return 'up'
            # Outch, not a good state...
            if obj.problem_has_been_acknowledged:
                return 'acknowledged'
            if obj.in_scheduled_downtime:
                return 'downtime'
            if obj.is_flapping:
                return 'flapping'
            # Ok, no excuse, it's a true error...
            return obj.state.lower()
        # Maybe it's a satellite
        if obj.__class__.my_type in ['scheduler', 'poller',
                                     'reactionner', 'broker',
                                     'receiver']:
            if not obj.alive:
                return 'critical'
            if not obj.reachable:
                return 'warning'
            return 'ok'
        return 'unknown'
    
    
    # For an object, give it's business impact as text
    # and stars if need
    def get_business_impact_text(self, obj):
        txts = {0: 'None', 1: 'Low', 2: 'Normal',
                3: 'High', 4: 'Very important', 5: 'Top for business'}
        nb_stars = max(0, obj.business_impact - 2)
        stars = '<img src="/static/' + str(app.http_start_time) + '/img/icons/star.png" alt="star">\n' * nb_stars
        
        res = "%s %s" % (txts.get(obj.business_impact, 'Unknown'), stars)
        return res
    
    
    # We will output as a ul/li list the impacts of this
    def got_impacts_list_as_li(self, obj):
        impacts = obj.impacts
        r = '<ul>\n'
        for i in impacts:
            r += '<li>%s</li>\n' % i.get_full_name()
        r += '</ul>\n'
        return r
    
    
    # Return the impacts as a business sorted list
    def get_impacts_sorted(self, obj):
        t = copy.copy(obj.impacts)
        t.sort(hst_srv_sort)
        return t
    
    
    def get_link(self, obj, short=False, mobile=False):
        if obj.__class__.my_type == 'service':
            if short:
                name = obj.get_name()
            else:
                name = obj.get_full_name()
        else:
            name = obj.get_name()
        
        return '<a target="_parent" href="%s"> %s </a>' % (self.get_link_dest(obj), name)
    
    
    def get_link_mobile(self, obj, short=False):
        if obj.__class__.my_type == 'service':
            if short:
                name = obj.get_name()
            else:
                name = obj.get_full_name()
            return '<a href="/mobile/service/%s" rel="external"> %s </a>' % (obj.get_full_name(), name)
        # if not service, host
        return '<a href="/mobile/host/%s" rel="external"> %s </a>' % (obj.get_full_name(), obj.get_full_name())
    
    
    # Give only the link to detail elements
    def get_link_dest(self, obj):
        check = ""
        detail = ""
        list_detail_filter = ""
        if obj.__class__.my_type == 'service':
            uuid = obj.host.uuid
            check = "checks"
            # open panel detail
            uuid_check = obj.uuid
            detail = "detail/%s-%s" % (uuid, uuid_check)
            # filter list check
            name_check = obj.get_name()
            filter = '[{"service_description": "%s"}]' % name_check.encode('utf-8')
            filter = urllib.quote(filter, safe='~@#$&()*!+=:;,.?/\'')
            list_detail_filter = "?listDetailFilter=%s" % filter
        else:
            uuid = obj.uuid
        
        return "/#/hosts/%s/%s/%s%s" % (uuid, check, detail, list_detail_filter)
    
    
    # For an host, give it's own link, for a services, give the link of its host
    def get_host_link(self, obj):
        if obj.__class__.my_type == 'service':
            return self.get_link(obj.host)
        return self.get_link(obj)
    
    
    def get_icon_state(self, state):
        if state == 'nothing':
            return '/static/' + str(app.http_start_time) + '/img/icons/state_%s.png' % state
        return '/static/' + str(app.http_start_time) + '/img/icons/%s.svg' % state
    
    
    # Get a simple tag icon that show the host or service if a tag is
    # available. If not, use a classic OK icon
    def get_tag_icon(self, app, obj, not_founded=''):
        tag = app.get_first_tag(obj.tags)
        if tag:
            return '/static/' + str(app.http_start_time) + '/images/sets/%s/tag.png' % tag
        # Ok it's not founded, so if not_founded provided, take this one
        if not_founded:
            return '/static/' + str(app.http_start_time) + '/images/sets/%s/tag.png' % not_founded
        state = obj.state.lower()
        if getattr(obj, 'icon_set', '') != '' and not obj.got_business_rule:
            return '/static/' + str(app.http_start_time) + '/images/sets/%s/state_%s.png' % (obj.icon_set, state)
        else:
            return '/static/' + str(app.http_start_time) + '/img/icons/state_%s.png' % state
    
    
    def get_item_icon(self, item):
        icon = app.get_first_tag(item.tags)
        if not icon:
            if item.__class__.my_type == 'service':
                return '/static/%s/depgraph/images/check.png' % app.http_start_time
            elif item.got_business_rule:
                icon = 'cluster'
            else:
                icon = 'hardware'
        return '/static/%s/images/sets/%s/tag.png' % (app.http_start_time, icon)
    
    
    # Get
    def get_navi(self, total, pos, step=30):
        step = float(step)
        nb_pages = math.ceil(total / step)
        current_page = int(pos / step)
        
        step = int(step)
        
        res = []
        
        if nb_pages == 0 or nb_pages == 1:
            return None
        
        if current_page >= 3:
            # Name, start, end, is_current
            res.append((u'1', 0, step, False))
            res.append(('...', None, None, False))
        
        # print "Range,", current_page - 1, current_page + 1
        for i in xrange(current_page - 2, current_page + 4):
            if i < 0:
                continue
            # print "Doing PAGE", i
            is_current = (i == current_page)
            start = int(i * step)
            # Maybe we are generating a page too high, bail out
            if start > total:
                continue
            
            end = int((i + 1) * step)
            res.append(('%d' % (i + 1), start, end, is_current))
        
        if current_page < nb_pages - 4:
            start = int((nb_pages - 1) * step)
            end = int(nb_pages * step)
            res.append(('...', None, None, False))
            res.append((u'%d' % nb_pages, start, end, False))
        
        # print "Total:", total, "pos", pos, "step", step
        # print "nb pages", nb_pages, "current_page", current_page
        
        # print "Res", res
        
        return res
    
    
    # Get a perfometer part for html printing
    def get_perfometer(self, elt):
        if elt.perf_data != '':
            r = get_perfometer_table_values(elt)
            # If the perfmeter are not good, bail out
            if r is None:
                return '\n'
            
            lnk = r['lnk']
            metrics = r['metrics']
            title = r['title']
            s = '<a href="%s">' % lnk
            s += '''<div class="graph">
                       <table>
                          <tbody>
                            <tr>\n'''
            
            for (color, pct) in metrics:
                s += '            <td style="background-color: %s; width: %s%%;"></td>\n' % (color, pct)
            
            s += '''        </tr>
                         </tbody>
                      </table>
                    </div>
                    <div class="text">%s</div>
                    <img class="glow" src="/static/'+str(app.http_start_time)+'/images/glow.png"/>
                 </a>\n''' % title
            return s
        return '\n'
    
    
    # TODO: Will look at the string s, and return a clean output without
    # danger for the browser
    def strip_html_output(self, s):
        return s
    
    
    # We want the html id of an hostor a service. It's basically
    # the full_name with / changed as -- (because in html, / is not valid :) )
    def get_html_id(self, elt):
        if elt.__class__.my_type == 'host':
            return self.strip_html_id(elt.get_full_name())
        else:  # for service, replace the / with a -- instead, but the / in a service description
            # should be _ instead
            return '%s--%s' % (self.strip_html_id(elt.host.get_full_name()), self.strip_html_id(elt.get_name()))
    
    
    def strip_html_id(self, s):
        return s.replace('/', '_').replace(' ', '_').replace('.', '_').replace(':', '_').replace('[', '_').replace(']', '_')
    
    
    # URI with spaces are BAD, must change them with %20
    def get_uri_name(self, elt):
        return elt.get_full_name().replace(' ', '%20')
    
    
    # say if this user can launch an action or not
    def can_action(self, user):
        return user.is_admin or user.can_submit_commands
    
    
    def get_aggregation_paths(self, p):
        p = p.strip()
        if p and not p.startswith('/'):
            p = '/' + p
        if p.endswith('/'):
            p = p[-1]
        return [s.strip() for s in p.split('/')]
    
    
    def compute_aggregation_tree_worse_state(self, tree):
        # First ask to our sons to compute their states
        for s in tree['sons']:
            self.compute_aggregation_tree_worse_state(s)
        # Ok now we can look at worse between our services
        # and our sons
        # get a list of all states
        states = [s['state'] for s in tree['sons']]
        for s in tree['services']:
            states.append(s.state.lower())
        # ok now look at what is worse here
        order = [('critical', 2), ('warning', 1), ('unknown', 3), ('ok', 0), ('pending', 3)]
        for (o, v) in order:
            if o in states:
                tree['state'] = o
                tree['status'] = v
                tree['context'] = 'NOTHING'
                return
        # Should be never call or we got a major problem...
        tree['state'] = 'unknown'
    
    
    def assume_and_get_path_in_tree(self, tree, paths):
        # print "Tree on start of", paths, tree
        current_full_path = ''
        for p in paths:
            # Don't care about void path, like for root
            if not p:
                continue
            current_full_path += '/' + p
            founded = False
            for s in tree['sons']:
                # Maybe we find the good son, if so go on this level
                if p == s['path']:
                    tree = s
                    founded = True
                    break
            # Did we find our son? If no, create it and jump into it
            if not founded:
                s = {'path': p, 'sons': [], 'services': [], 'status': 3, 'context': 'NOTHING', 'state': 'unknown', 'full_path': current_full_path}
                tree['sons'].append(s)
                tree = s
        return tree
    
    
    def get_host_service_aggregation_tree(self, h):
        tree = {'path': '/', 'sons': [], 'services': [], 'state': 'unknown', 'full_path': '/'}
        for s in h.services:
            p = '/'
            paths = self.get_aggregation_paths(p)
            # print "Service", s.get_name(), "with path", paths
            leaf = self.assume_and_get_path_in_tree(tree, paths)
            leaf['services'].append(s)
        self.compute_aggregation_tree_worse_state(tree)
        
        return tree
    
    
    def print_aggregation_tree(self, tree, html_id):
        path = tree['path']
        full_path = tree['full_path']
        sons = tree['sons']
        services = tree['services']
        state = tree['state']
        _id = '%s-%s' % (html_id, self.strip_html_id(full_path))
        s = ''
        
        display = 'block'
        img = 'reduce.png'
        
        if path != '/':
            # If our state is OK, hide our sons
            if state == 'ok':
                display = 'none'
                img = 'expand.png'
            
            s += """<span class="alert-small alert-%s"> %s </span>""" % (state, path)
            s += """<a id="togglelink-aggregation-%s" href="javascript:toggleAggregationElt('%s')"><img id="aggregation-toggle-img-%s" src="/static/%s/images/%s" alt="toggle"> </a> \n""" % (_id, _id, _id, app.http_start_time, img)
        
        s += """<ul id="aggregation-node-%s" style="display: %s; ">""" % (_id, display)
        # If we got no parents, no need to print the expand icon
        if len(sons) > 0:
            for son in sons:
                sub_s = self.print_aggregation_tree(son, html_id)
                s += '<li class="no_list_style">%s</li>' % sub_s
        
        s += '<li class="no_list_style">'
        if path == '/' and len(services) > 0:
            s += """<span class="alert-small"> Others </span>"""
        s += '<ul style="margin-left: 0px;">'
        # Sort our services before print them
        services.sort(hst_srv_sort)
        for svc in services:
            s += '<li class="%s">' % svc.state.lower()
            s += """<span class='alert-small alert-%s' style="">%s</span> for <span style="">%s since %s</span>""" % (
                svc.state.lower(), svc.state, self.get_link(svc, short=True), self.print_duration(svc.last_state_change, just_duration=True, x_elts=2))
            for i in range(0, svc.business_impact - 2):
                s += '<img alt="icon state" src="/static/' + str(app.http_start_time) + '/images/star.png">'
            s += '</li>'
        s += "</ul></li>"
        
        s += "</ul>"
        # safe_print("Returning s:", s)
        return s
    
    
    # The output can got some change and strip if need
    def print_output(self, app, output):
        if app.max_output_length != 0:
            if len(output) >= app.max_output_length - 3:
                return "%s ..." % output[:app.max_output_length - 3]
        return output
    
    
    def get_ui_status(self, state_label):
        if state_label in ['OK', 'UP']:
            status = 0
        elif state_label == 'WARNING':
            status = 1
        elif state_label in ['CRITICAL', 'DOWN']:
            status = 2
        else:
            status = 3
        
        return status
    
    
    def get_summary(self, elt):
        clsname = elt.__class__.my_type
        is_host = (clsname == 'host')
        is_service = (clsname == 'service')
        
        status = self.get_ui_status(elt.state)
        # Catch cluster degraded state
        if is_host and elt.got_business_rule:
            status = elt.bp_state
        
        # DISABLED is for hosts, service can't be like this because an host disabled got no service
        if is_host and 'disabled' in elt.tags:
            return {'status': status, 'summary': 'DISABLED'}
        
        if elt.downtimes and next((dt for dt in elt.downtimes if dt.ref == elt and dt.is_in_effect), None):
            return {'status': status, 'summary': 'DOWNTIME'}
        
        if elt.is_in_inherited_downtime():
            return {'status': status, 'summary': 'INHERITED-DOWNTIME'}
        
        if elt.in_partial_downtime:
            return {'status': status, 'summary': 'PARTIAL-DOWNTIME'}
        
        if elt.acknowledgement and not elt.acknowledgement.automatic:
            return {'status': status, 'summary': 'ACKNOWLEDGED'}
        
        if (elt.acknowledgement and elt.acknowledgement.automatic) or (is_service and elt.host.acknowledgement and not elt.host.acknowledgement.automatic):
            return {'status': status, 'summary': 'INHERITED-ACKNOWLEDGED'}
        
        if elt.is_partial_acknowledged:
            return {'status': status, 'summary': 'PARTIAL-ACKNOWLEDGED'}
        
        if elt.is_flapping:
            return {'status': status, 'summary': 'FLAPPING'}
        
        if elt.got_business_rule and elt.is_partial_flapping:
            return {'status': status, 'summary': 'PARTIAL-FLAPPING'}
        
        return {'status': status, 'summary': 'NOTHING'}
    
    
    def get_all_my_fathers(self, elt):
        r = set()
        # Get from parents dep for classic dep (parents, etc)
        for i in elt.parent_dependencies:
            r.add(i)
        # and for cluster, look elsewhere
        if elt.got_business_rule:
            fathers = proxyitemsgraph.son_to_fathers[elt.get_instance_uuid()]  # warning, not .get() because will be None
            for father_uuid in fathers:
                father = self.__get_elt_from_uuid(father_uuid)
                if father is not None:
                    r.add(father)
        return r
    
    
    def get_elt_from_uuid(self, i_uuid):
        return self.__get_elt_from_uuid(i_uuid)
    
    
    def __get_elt_from_uuid(self, i_uuid):
        # detect if host or check
        if '-' in i_uuid:
            elt = self.app.datamgr.get_service_by_uuid(i_uuid)
        else:
            elt = self.app.datamgr.get_host_by_uuid(i_uuid)
        return elt
    
    
    SLA_STATE_TO_ICON_STATE = {
        'ok'      : 'ok',
        'warn'    : 'warning',
        'crit'    : 'critical',
        'unknown' : 'unknown',
        'missing' : 'unknown',
        'inactive': 'unknown',
    }
    
    
    def print_date_locate(self, date):
        if not date:
            return 'N/A'
        if app.lang == 'fr':
            date_format = '%d/%m/%Y'
        else:
            date_format = '%m/%d/%Y'
        if isinstance(date, datetime.datetime):
            return date.strftime(date_format)
        return datetime.datetime.fromtimestamp(date).strftime(date_format)
    
    
    def get_sla_state_icon(self, state):
        return '<img class="shinken-status-icon" src="/static/%s/img/icons/state_%s_32x32.png">' % (app.http_start_time, self.SLA_STATE_TO_ICON_STATE.get(state, state))
    
    
    def get_value_label(self, state, value, with_icon_and_tooltip=True):
        icon = self.get_sla_state_icon(state) if with_icon_and_tooltip else ''
        tooltip = 'data-toggle="tooltip" title="%s"' % app._('reporting.%s' % state) if with_icon_and_tooltip else ''
        return '<span class="value-label %s " %s>%s%.2f%%</span>' % (state, tooltip, icon, 100 * value)
    
    
    def get_value_bar(self, state, mode, value):
        if mode == 'sla':
            state_text = app._('reporting.sla')
            if state == 'ok':
                state_text += '''<span class="sla-icon shinicon-arrow-up"></span>'''
            else:
                state_text += '''<span class="sla-icon shinicon-arrow-down"></span>'''
            
            tooltip = 'data-toggle="tooltip" title="%s"' % app._('reporting.%s' % state)
        else:
            state_text = app._('reporting.%s' % state)
            tooltip = ''
        
        value_bar_html = '''
        <tr class="status-bar">
            <td class="icon-cell" %(tooltip)s>
                <div>
                    %(icon)s
                </div>
            </td>
            <td class="status-label-cell">
                <div class="status-label">%(state_text)s</div>
            </td>
            <td class="progress-cell">
                <div class="progress">
                    <div class="progress-bar progress-bar-%(state)s" role="progressbar" aria-valuenow="0.00" aria-valuemin="0" aria-valuemax="100" aria-valuetext="0.00" value="0.00" type="success" style="width:%(value).2f%%;"></div>
                </div>
            </td>
            <td class="percentage-cell">
                <div class="percentage">%(value).2f%%</div>
            </td>
        </tr>
        ''' % {
            "icon"      : self.get_sla_state_icon(state),
            "state_text": state_text,
            "state"     : state,
            "value"     : value * 100,
            "tooltip"   : tooltip,
        }
        return value_bar_html
    
    
    def get_downtime_on_item(self, item, elt_dict):
        elt_dict['in_scheduled_downtime'] = item.is_in_downtime()
        elt_dict['in_inherited_downtime'] = item.is_in_inherited_downtime()
        elt_dict['downtimes'] = self.get_elt_dict(item.downtimes)
        
        if item.__class__.my_type == 'service':
            host_downtimes = app.helper.get_elt_dict(item.host.downtimes)
            elt_dict['downtimes_inherited'] = host_downtimes
    
    
    def get_ack_on_item(self, item, elt_dict):
        elt_dict['acknowledgement'] = {}
        elt_dict['partial_acknowledge'] = {}
        elt_dict['acknowledgement_inherited'] = {}
        elt_dict['problem_has_been_acknowledged'] = False
        elt_dict['problem_has_been_inherited_acknowledged'] = 0
        
        if item.__class__.my_type == 'service':
            if item.acknowledgement and not item.acknowledgement.automatic:
                elt_dict['acknowledgement'] = self.get_elt_dict(item.acknowledgement)
                elt_dict['problem_has_been_acknowledged'] = True
            if item.partial_acknowledge:
                elt_dict['partial_acknowledge'] = self.get_elt_dict(item.partial_acknowledge)
            if item.host.got_business_rule:
                if item.host.acknowledgement and not item.host.acknowledgement.automatic:
                    # ACK inherited from cluster if cluster has acknowledge
                    elt_dict['acknowledgement_inherited'] = self.get_elt_dict(item.host.acknowledgement)
                    elt_dict['problem_has_been_inherited_acknowledged'] = 1
                    if item.host.partial_acknowledge:
                        elt_dict['partial_acknowledge'] = self.get_elt_dict(item.host.partial_acknowledge)
            elif item.host.acknowledgement:
                # ACK inherited from host
                elt_dict['acknowledgement_inherited'] = self.get_elt_dict(item.host.acknowledgement)
                elt_dict['problem_has_been_inherited_acknowledged'] = 1
                if item.host.partial_acknowledge:
                    elt_dict['partial_acknowledge'] = self.get_elt_dict(item.host.partial_acknowledge)
        
        else:
            if item.acknowledgement and not item.acknowledgement.automatic:
                elt_dict['acknowledgement'] = self.get_elt_dict(item.acknowledgement)
                elt_dict['problem_has_been_acknowledged'] = True
            elif item.acknowledgement and item.acknowledgement.automatic:
                elt_dict['acknowledgement_inherited'] = self.get_elt_dict(item.acknowledgement)
                elt_dict['problem_has_been_inherited_acknowledged'] = 2
            if item.partial_acknowledge:
                elt_dict['partial_acknowledge'] = self.get_elt_dict(item.partial_acknowledge)


helper = Helper()
