#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

"""
This Class is a plugin for the Shinken Broker. It is in charge
to get brok and recreate real objects, and propose a Web interface :)
"""

import base64
import codecs
import hmac
import imp
import json
import math
import os
import re
import socket
import sys
import threading
import time
import traceback
import uuid

import shinken.webui.bottlewebui as bottle
from arch_export_handler import ArchitectureExportHandler
from helper import helper
from shinken.basemodule import BaseModule
from shinken.daemon import Daemon
from shinken.log import logger, LOG_CHAPTER_SIZE, PartLogger, LoggerFactory
from shinken.message import Message
from shinken.misc.datamanager import datamgr
from shinken.misc.regenerator import Regenerator
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.modulesctx import modulesctx
from shinken.modulesmanager import ModulesManager
from shinken.util import to_bool
from shinken.webui.bottlewebui import static_file, view, route, request, response, template, HTTPError
from shinken.webui.cherrypybackend import CherryPyServerHTTP
from shinkensolutions.external_resources.external_resources import external_resources
from shinkensolutions.lib_modules.configuration_reader import read_bool_in_configuration, read_int_in_configuration
from shinkensolutions.localinstall import get_context_hash, get_shinken_current_short_version, get_shinken_current_version
from shinkensolutions.locking.fair_lock_ordonnancer import FairLockGroupOrdonnancer
from widget_service import widget_service

if TYPE_CHECKING:
    from shinken.misc.type_hint import Optional, Union
    from shinken.objects.contact import Contact

JS_APP_VERSION_PATTERN = re.compile(r'''shinken-enterprise.([a-zA-Z0-9]*).js''')
SHINKEN_JS_VERSION = 'main file not load'
TEMPLATE_LOG_QUERY_PERF = u'[ %-20s ] [ user UUID= %-32s ] [ start= %s  end= %s  Total= %.3fs  { lock wait= %.3fs  running time= %.3fs } ]'

bottle.debug(True)

# Import bottle lib to make bottle happy
bottle_dir = os.path.abspath(os.path.dirname(bottle.__file__))
sys.path.insert(0, bottle_dir)

# Look at the webui module root dir too
WEBUI_MODULE_DIRECTORY = os.path.abspath(os.path.dirname(__file__))
htdocs_dir = os.path.join(WEBUI_MODULE_DIRECTORY, 'htdocs')

INDEX_HTML_LANG_PARAMETER_NAME = '__shinken_lang__'

properties = {
    'daemons' : ['broker', 'scheduler'],
    'type'    : 'webui',
    'phases'  : ['running'],
    'external': True,
}

CONTEXT_PATH = '/var/lib/shinken/context.json'
CURRENT_VERSION = '02.06.00'

if os.path.exists(CONTEXT_PATH):
    context = json.loads(open(CONTEXT_PATH, 'r').read())
    CURRENT_VERSION = context.get('current_version', CURRENT_VERSION)


# called by the plugin manager to get an instance
def get_instance(plugin):
    # Add template only if we ask for a webui
    bottle.TEMPLATE_PATH.append(os.path.join(WEBUI_MODULE_DIRECTORY, 'views'))
    bottle.TEMPLATE_PATH.append(WEBUI_MODULE_DIRECTORY)
    logger.debug("Get a WebUI instance for plugin %s" % plugin.get_name())
    
    instance = WebuiBroker(plugin)
    return instance


class EnableCors(object):
    name = 'enable_cors'
    api = 2
    
    
    def apply(self, fn, context):  # noqa
        def _enable_cors(*args, **kwargs):
            # set CORS headers
            response.headers['Access-Control-Allow-Origin'] = 'http://127.0.0.1:9000'
            response.headers['Access-Control-Allow-Methods'] = 'GET, POST, PUT, OPTIONS, DELETE, PATCH'
            response.headers['Access-Control-Allow-Headers'] = 'Origin, Accept, Content-Type, X-Requested-With, X-CSRF-Token, X-Shinken-Token'
            response.headers['Access-Control-Allow-Credentials'] = 'true'
            if bottle.request.method != 'OPTIONS':
                # actual request; reply with the actual response
                return fn(*args, **kwargs)
        
        
        return _enable_cors


MAX_INT32 = int('1' * 31, base=2)


class WebuiBroker(BaseModule, Daemon):
    def __init__(self, module_configuration):
        BaseModule.__init__(self, module_configuration)
        
        # IMPORTANT: we must have a static value based on the server installation state (install + patch)
        # because numerous webui can be start, and so process start time cannot be used here
        # NOTE: must be a int32 because of the usage in javascript and in the .tpl currently.
        self.http_start_time = int(get_context_hash(), 16) % MAX_INT32
        
        self.plugins = []
        
        self.serveropts = {}
        umask = getattr(module_configuration, 'umask', None)
        if umask is not None:
            self.serveropts['umask'] = int(umask)
        bind_address = getattr(module_configuration, 'bindAddress', None)
        if bind_address:
            self.serveropts['bindAddress'] = str(bind_address)
        self.serveropts['numthreads'] = 64
        
        self.port = int(getattr(module_configuration, 'port', '7767'))
        self.http_port = int(getattr(module_configuration, 'http_port', '7766'))
        self.host = getattr(module_configuration, 'host', '0.0.0.0')
        self.show_skonf = int(getattr(module_configuration, 'show_skonf', '1'))
        self.auth_secret = getattr(module_configuration, 'auth_secret').encode('utf8', 'replace')
        self.play_sound = to_bool(getattr(module_configuration, 'play_sound', '0'))
        self.http_backend = getattr(module_configuration, 'http_backend', 'auto')
        self.login_text = getattr(module_configuration, 'login_text', None)
        self.allow_html_output = to_bool(getattr(module_configuration, 'allow_html_output', '0'))
        self.max_output_length = int(getattr(module_configuration, 'max_output_length', '0'))
        self.manage_acl = to_bool(getattr(module_configuration, 'manage_acl', '1'))
        self.remote_user_enable = getattr(module_configuration, 'remote_user_enable', '0')
        self.remote_user_variable = getattr(module_configuration, 'remote_user_variable', 'X-REMOTE-USER')
        self.remote_user_case_sensitive = to_bool(getattr(module_configuration, 'remote_user_case_sensitive', '1'))
        
        self.use_ssl = getattr(module_configuration, 'use_ssl', '0') == '1'
        self.ssl_key = getattr(module_configuration, 'ssl_key', '')
        self.ssl_cert = getattr(module_configuration, 'ssl_cert', '')
        
        self.share_dir = getattr(module_configuration, 'share_dir', 'share')
        self.share_dir = os.path.abspath(self.share_dir)
        # Load the photo dir and make it a absolute path
        self.photo_dir = getattr(module_configuration, 'photo_dir', 'photos')
        self.photo_dir = os.path.abspath(self.photo_dir)
        
        self.show_trending = getattr(module_configuration, 'show_trending', '0') == '1'
        
        self.csv_export_limit = int(getattr(module_configuration, 'broker__module_webui__module_event_manager_reader__events_export__max_nb_events', '500'))
        
        # Look for an additional pages dir
        self.additional_plugins_dir = getattr(module_configuration, 'additional_plugins_dir', '')
        if self.additional_plugins_dir:
            self.additional_plugins_dir = os.path.abspath(self.additional_plugins_dir)
        
        # We will save all widgets
        self.widgets = {}
        self.rg = Regenerator(module_configuration, self.get_name())
        
        self.arch_export_handler = ArchitectureExportHandler()
        
        self.bottle = bottle
        # get a list of tag images available for display
        self.img_tags = []
        sets_dir = os.path.join(self.share_dir, 'images', 'sets')
        # Look at the share/images/sets/*/tag.png and save the *
        for p in os.listdir(sets_dir):
            fp = os.path.join(sets_dir, p)
            if os.path.isdir(fp) and os.path.exists(os.path.join(fp, 'tag.png')):
                self.img_tags.append(p)
        
        self.lang = getattr(module_configuration, u'lang', u'en').lower()
        self.langs_path = u'/var/lib/shinken/modules/webui/htdocs/js/traductions'
        self.langs = {u'en': None, u'fr': None}
        self._index_html_content = None  # We will load the index.html and change the lang inside it
        self.tiles_background = getattr(module_configuration, 'tiles_background', 'context')
        self.colors_graphics = getattr(module_configuration, 'colors_graphics', '0095DA,E02C2C')
        self.apply_filter_method = getattr(module_configuration, 'apply_filter_method', 'key_enter')
        
        self.history__nb_changes_displayed = getattr(module_configuration, 'history__nb_changes_displayed', '30')
        self.history__size_sla_pane = getattr(module_configuration, 'history__size_sla_pane', None)
        self.history__display_outputs = getattr(module_configuration, 'history__default_display_outputs', '1') != '0'
        self.history__collapse_outputs = getattr(module_configuration, 'history__default_collapse_outputs', '0') == '1'
        
        if self.history__size_sla_pane:
            try:
                self.history__size_sla_pane = int(self.history__size_sla_pane)
            except ValueError:
                self.history__size_sla_pane = None
        try:
            self.history__nb_changes_displayed = int(self.history__nb_changes_displayed)
            if self.history__nb_changes_displayed <= 0:
                logger.warning('The key \'history__nb_changes_displayed can\'t be negative or null\'')
                self.history__nb_changes_displayed = 30
        except ValueError:
            self.history__nb_changes_displayed = 30
        
        self.graphs_errors = {}
        
        self.logger_ui_management = None  # type: Optional[PartLogger]
        self.logger_ui_management_query_perf = None  # type: Optional[PartLogger]
        self.logger_ui_management_query_running_perf = None  # type: Optional[PartLogger]
        self.logger_ui_management_query_incoming_perf = None  # type: Optional[PartLogger]
        self.logger_auth = None  # type: Optional[PartLogger]
        
        self.data_thread = None  # type: Optional[threading.Thread]
        self.query_stats_lock = threading.RLock()
        self.query_stats_running_query = {}
        self.query_stats_incoming_query = {}
        self.last_log_incoming_query = 0
        self._fair_lock_ordonnancer = None  # type: Optional[FairLockGroupOrdonnancer]
        self.query_stats_thread_display_thread = None  # type: Optional[threading.Thread]
        
        log_init = self.logger.get_sub_part('INITIALISATION', len('INITIALISATION'))
        
        # Broks eater fine tuning
        self.writer_early_lock = read_bool_in_configuration(module_configuration, 'webui__broks_getter__include_deserialisation_and_catchup_in_lock', False, log_fct=log_init)
        self.writer_catchup_enabled = read_bool_in_configuration(module_configuration, 'webui__broks_getter__activate_late_set_catchup', True, log_fct=log_init)
        self.writer_late_sets_allowed = read_int_in_configuration(module_configuration, 'webui__broks_getter__nb_late_set_allowed_before_catchup', 10, log_fct=log_init)
        self.writer_late_broks_max = read_int_in_configuration(module_configuration, 'webui__broks_getter__catchup_broks_managed_by_module_in_a_catchup_loop', 200000, log_fct=log_init)
        self.writer_catchup_loops = read_bool_in_configuration(module_configuration, 'webui__broks_getter__catchup_run_endless_until_nb_late_set_allowed_reached', True, log_fct=log_init)
        
        self.hook_error()
        
        self.current_version = get_shinken_current_version()
        self.current_short_version = get_shinken_current_short_version()
    
    
    def error_handler(self, error):
        return_value = error.output
        if getattr(error, 'warning', False):
            logger.warning("http error catch [%s]" % return_value)
        else:
            logger.error("http error catch [%s]" % return_value)
        if error.traceback:
            first_line = True
            for line_stack in error.traceback.splitlines():
                if first_line:
                    logger.error("ERROR stack : %s" % line_stack)
                    first_line = False
                else:
                    logger.error("%s" % line_stack)
        if response and response.content_type == 'application/json':
            return_value = json.dumps(error.output)
        else:
            with_menu = False
            user_p = None
            try:
                user_p = self.get_user_auth()
            except:
                pass
            if not user_p:
                user_p = {}
            return_value = self.bottle.template("%include error globals()", e=error, app=self, user=user_p, with_menu=with_menu)
        return return_value
    
    
    def hook_error(self):
        @bottle.error(514)
        def custom514(error):
            return self.error_handler(error)
        
        
        @bottle.error(513)
        def custom513(error):
            return self.error_handler(error)
        
        
        @bottle.error(512)
        def custom512(error):
            return self.error_handler(error)
        
        
        @bottle.error(511)
        def custom511(error):
            return self.error_handler(error)
        
        
        @bottle.error(500)
        def custom500(error):
            return self.error_handler(error)
        
        
        @bottle.error(400)
        def custom500(error):
            return self.error_handler(error)
        
        
        @bottle.error(404)
        def custom404(error):
            return self.error_handler(error)
        
        
        @bottle.error(401)
        def custom401(error):
            if response.content_type != 'application/json':
                response.set_header('location', '/static/ui/index.html')
                response.status = 303
                return ''
            else:
                return self.error_handler(error)
        
        
        @bottle.error(403)
        def custom403(error):
            return self.error_handler(error)
        
        
        @bottle.error(409)
        def custom409(error):
            return self.error_handler(error)
        
        
        @bottle.error(410)
        def custom410(error):
            return self.error_handler(error)
    
    
    # We check if the photo directory exists. If not, try to create it
    def check_photo_dir(self):
        if not os.path.exists(self.photo_dir):
            try:
                os.mkdir(self.photo_dir)
            except Exception as exp:
                pass
    
    
    # Called by Broker so we can do init stuff
    # TODO: add conf param to get pass with init
    # Conf from arbiter!
    def init(self):
        logger.debug('Init of the UI %s' % self.name)
        self.rg.load_external_queue(self.from_module_to_main_daemon_queue)
    
    
    # This is called only when we are in a scheduler
    # and just before we are started. So we can gain time, and
    # just load all scheduler objects without fear :) (we
    # will be in another process, so we will be able to hack objects
    # if need)
    def hook_pre_scheduler_mod_start(self, sched):
        self.rg.load_from_scheduler(sched)
    
    
    # In a scheduler we will have a filter of what we really want as a brok
    def want_brok(self, b):
        return self.rg.want_brok(b) or self.arch_export_handler.want_brok(b)
    
    
    def main(self):
        logger.set_name(self.name)

        allowed_langs = self.langs.keys()
        if self.lang not in allowed_langs:
            self._die_with_strong_error(u'For the parameter "lang" the value "%s" is not allowed. Values can be : "%s"' % (self.lang, u'", "'.join(allowed_langs)), with_trace_back=False)
            
        self.logger_ui_management = LoggerFactory.get_logger('UI MANAGEMENT')
        self.logger_ui_management_query_running_perf = self.logger_ui_management.get_sub_part('QUERY', len('QUERY')).get_sub_part('RUNNING', len('QUERY')).get_sub_part('PERF', len('PERF'))
        self.logger_ui_management_query_incoming_perf = self.logger_ui_management.get_sub_part('QUERY', len('QUERY')).get_sub_part('INCOMING', len('INCOMING')).get_sub_part('PERF', len('PERF'))
        self.logger_ui_management_query_perf = self.logger_ui_management.get_sub_part('QUERY', len('QUERY')).get_sub_part('PERF', len('PERF'))
        self.logger_ui_management_query_perf.set_enable(False)
        self.logger_auth = LoggerFactory.get_logger(u'AUTHENTICATION')
        
        # Daemon like init
        self.debug_output = []
        self.modules_dir = modulesctx.get_modulesdir()
        self.modules_manager = ModulesManager('webui', self.find_modules_path(), [])
        self.modules_manager.set_modules(self.modules)
        # We can now output some previously silenced debug output
        self.do_load_modules()
        for inst in self.modules_manager.get_all_alive_instances():
            f = getattr(inst, 'load', None)
            if f and callable(f):
                f(self)
        
        for s in self.debug_output:
            logger.debug(s)
        del self.debug_output
        
        self.check_photo_dir()
        self.datamgr = datamgr
        datamgr.load(self.rg)
        self.helper = helper
        self.helper.set_app(self)
        
        self.widget_service = widget_service
        self.widget_service.set_app(self)
        
        self.request = request
        self.abort = self._abort
        self.response = response
        self.template_call = template
        
        # the regenerator is ready in the valid process, we can start it's freshness thread
        self.rg.launch_freshness_thread(lang=self.lang)
        try:
            # import cProfile
            # cProfile.runctx('''self.do_main()''', globals(), locals(),'/tmp/webui.profile')
            self.do_main()
        except Exception as exp:
            self._die_with_strong_error(str(exp), with_trace_back=True)
            return  # already exit
    
    
    def _abort(self, code=500, text='Unknown Error: Application stopped.', warning=False):
        logger.error('A %s HTTP error occured : %s' % (code, text))
        http_error = HTTPError(code, text)
        http_error.warning = warning
        raise http_error
    
    
    def wrap_shinken_js_version_header(self, f):
        def __wrap(*args, **kwargs):
            self.response.headers['X-shinken-js-version'] = SHINKEN_JS_VERSION
            return f(*args, **kwargs)
        
        
        return __wrap
    
    
    def wrap_auth(self, f):
        def __wrap(*args, **kwargs):
            # First we look for the user sid so we bail out if it's a false one.
            user = self.get_user_auth()
            if not user:
                logger.info("Invalid user")
                return self.abort(401, 'Invalid user')
            return f(*args, **kwargs)
        
        
        return __wrap
    
    
    def wrap_json(self, f):
        def __wrap(*args, **kwargs):
            self.response.content_type = 'application/json'
            return f(*args, **kwargs)
        
        
        return __wrap
    
    
    # A plugin send us en external command. We just put it
    # in the good queue
    def push_external_command(self, e):
        logger.debug("UI: got an external command: %s" % str(e.__dict__))
        self.from_module_to_main_daemon_queue.put(e)
    
    
    def _die_with_strong_error(self, error, with_trace_back=True):
        crash_logger = LoggerFactory.get_logger('CRASH - INSIDE MODULE PROCESS')
        crash_logger.error(error)
        traceback_message = traceback.format_exc() if with_trace_back else error
        msg = Message(id=0, type='ICrash', data={'name': self.get_name(), 'exception': error, 'trace': traceback_message})
        self.from_module_to_main_daemon_queue.put(msg)
        # wait 2 sec so we know that the broker got our message, and die
        time.sleep(2)
        os._exit(2)  # noqa : forced exit
    
    
    def _compute_index_html_with_lang(self):
        # Compute the index.html content:
        # * load the htdocs/ui/index.html file
        # * look at the block:
        # <script>
        # var __shinken_lang__ = 'en';
        # </script>
        # and change the lang with our parameter
        _err_suffix = 'there is a critical error with your installation. Please open a ticket to your support.'
        index_html_file_path = os.path.join(WEBUI_MODULE_DIRECTORY, 'htdocs/ui/index.html')
        if not os.path.exists(index_html_file_path):
            self._die_with_strong_error('The file %s is missing: %s' % (index_html_file_path, _err_suffix), with_trace_back=False)
            return  # did exit
        try:
            with codecs.open(index_html_file_path, 'r', 'utf8') as f:  # utf8: we are ok with future editions
                index_html_content = f.read()
        except IOError as exp:
            self._die_with_strong_error('Cannot open the file %s with the error "%s": %s' % (index_html_file_path, exp, _err_suffix), with_trace_back=False)
            return  # did exit
        
        # Change the line but beware: MUST change the line, if not: the dev did do something silly
        _did_find_lang = False
        new_lines = []
        for line in index_html_content.splitlines():
            if 'var %s' % INDEX_HTML_LANG_PARAMETER_NAME in line:
                new_lines.append('''var %s = '%s';''' % (INDEX_HTML_LANG_PARAMETER_NAME, self.lang))
                _did_find_lang = True
            else:
                new_lines.append(line)
        
        if not _did_find_lang:
            self._die_with_strong_error('The %s variable was not found in the file %s: %s' % (INDEX_HTML_LANG_PARAMETER_NAME, index_html_file_path, _err_suffix), with_trace_back=False)
        
        new_lines.append('<!-- Generated by "%s" at %s with lang=%s -->\n' % (self.get_name(), int(time.time()), self.lang))
        self._index_html_content = '\n'.join(new_lines)
    
    
    # Real main function
    def do_main(self):
        global SHINKEN_JS_VERSION
        
        # I register my exit function
        self.set_exit_handler()
        
        self._fair_lock_ordonnancer = FairLockGroupOrdonnancer(
            name='ITEMS ACCESS ORDONNANCER',
            consumer_name='HTTP requests',
            consumer_max_wish_switch_time=1,
            producer_name='Broks management',
            producer_max_wish_switch_time=15,
            logger=self.logger,
            error_log_time=30
        )
        
        self.data_thread = None
        
        # Step - Check if the view dir really exist
        if not os.path.exists(bottle.TEMPLATE_PATH[0]):
            logger.error("The view path do not exist at %s" % bottle.TEMPLATE_PATH)
            sys.exit(2)
        
        # Step - Find js app version
        core_plugin_dir = os.path.join(WEBUI_MODULE_DIRECTORY, 'plugins')
        js_app_dir = os.path.join(WEBUI_MODULE_DIRECTORY, 'htdocs/ui/scripts/')
        i = 0
        if os.path.isdir(js_app_dir):
            for file_name in os.listdir(js_app_dir):
                main_file = JS_APP_VERSION_PATTERN.match(os.path.basename(file_name))
                if main_file:
                    SHINKEN_JS_VERSION = main_file.group(1)
                    i += 1
                    logger.info('loading js app version [%s]' % SHINKEN_JS_VERSION)
        if i == 0:
            self._die_with_strong_error('no app js found in [%s]' % js_app_dir, with_trace_back=False)
            return  # was already done
        elif i > 1:
            self._die_with_strong_error('multi app js found in [%s]' % js_app_dir, with_trace_back=False)
            return  # was already done
        
        # Get the index.html and set lang inside it
        self._compute_index_html_with_lang()
        
        # Step - Load the additional plugins so they will have the lead on URI routes
        if self.additional_plugins_dir:
            self.load_plugins(self.additional_plugins_dir)
        
        # Step - Modules can also override some views if need
        for inst in self.modules_manager.get_all_alive_instances():
            f = getattr(inst, 'get_webui_plugins_path', None)
            if f and callable(f):
                mod_plugins_path = os.path.abspath(f(self))
                self.load_plugins(mod_plugins_path)
        
        # Step - Then look at the plugins in toe core and load all we can there
        self.load_plugins(core_plugin_dir)
        
        # We must be sure the check plugins styles css are compiled before
        try:
            self.compiled_css_path, self.compiled_css_hash = external_resources.load_and_combine_css()
        except Exception:  # there was a problem in the css compilation, already log
            raise
        # Step - Declare the whole app static files AFTER the plugin ones
        self.declare_common_static()
        
        # Step - Launch the data thread
        self.data_thread = threading.Thread(None, self.manage_brok_thread, 'datathread')
        self.data_thread.start()
        # TODO: look for alive and killing
        
        self.query_stats_thread_display_thread = threading.Thread(None, self.query_stats_thread_display, 'query_stats_thread_display')
        self.query_stats_thread_display_thread.start()
        
        # Ok, you want to know why we are using a data thread instead of just call for a select with q._reader, the underlying file handle of the Queue()?
        # That's just because under Windows, select only manage winsock (so network) file descriptor! What a shame!
        logger.debug("UI starting application")
        self.srv = bottle.run(host=self.host, port=self.port, server=CherryPyServerHTTP, use_ssl=self.use_ssl, ssl_key=self.ssl_key, ssl_cert=self.ssl_cert, **self.serveropts)
        
        # Maybe the port is already open by a spare that is going down, so wait a it before give up
        max_nb_try = 10
        nb_start_try = 0
        while True:  # always finishing
            nb_start_try += 1
            try:
                self.srv.start()
            except socket.error as e:
                if e.message == 'No socket could be created':
                    _error = '[TRY %d/%d] The webui named [%s] can not start because the address %s:%s is already in use' % (nb_start_try, max_nb_try, self.get_name(), self.host, self.port)
                    if nb_start_try < max_nb_try:  # we can retry
                        logger.warning(_error)
                        time.sleep(1)
                    else:
                        self._die_with_strong_error(_error, with_trace_back=False)
                        return  # already exit
                else:
                    raise e
        
        # ^ IMPORTANT ^
        # We are not managing the lock at this level because we got 2 types of requests:
        # static images/css/js: no need for lock
        # pages: need it. So it's managed at a function wrapper at loading pass
    
    
    def manage_brok_thread(self):
        try:
            self._manage_brok_thread()
        except:
            self.logger.print_stack()
    
    
    # It's the thread function that will get broks
    # and update data. Will lock the whole thing
    # while updating
    def _manage_brok_thread(self):
        have_lock = False
        while not self.interrupted:
            start = time.time()
            try:
                broks = self.to_q.get()
            except EOFError:
                if self.interrupted:
                    return
                else:
                    raise

            if self.writer_early_lock and not have_lock:
                self._fair_lock_ordonnancer.producer_acquire()
                have_lock = True
                

            get_late_broks_set_time = time.time()
            nb_late_broks_sets_taken = 0
            if self.writer_catchup_enabled and (self.to_q.qsize() > self.writer_late_sets_allowed >= 0):
                t1 = time.time()
                while self.to_q.qsize() > 0:
                    t0 = time.time()
                    try:
                        _late_broks_set = self.to_q.get()
                    except EOFError:
                        if self.interrupted:
                            return
                        else:
                            raise
                    _current_nb_late_broks = len(_late_broks_set)
                    broks.extend(_late_broks_set)
                    broks_to_process = len(broks)
                    logger.info('[ MANAGE BROKS ] [PERF] [LATE BROKS SETS]  Getting brok set with %s broks in %.3fs [time for read queue size=%.3fs]. Total broks to process= %s/max:%s. Broks sets in queue: %s.' %
                                (_current_nb_late_broks, time.time() - t0, t0 - t1, broks_to_process, self.writer_late_broks_max, self.to_q.qsize()))
                    nb_late_broks_sets_taken += 1
                    t1 = time.time()
                    
                    if broks_to_process > self.writer_late_broks_max:
                        logger.info('[ MANAGE BROKS ] [PERF] [LATE BROKS SETS] Late brok taken => limit reach : %s / limit: %s.' % (broks_to_process, self.writer_late_broks_max))
                        break

            after_get = time.time()
            # First unpickle broks, but outside the lock time
            for b in broks:
                b.prepare()
            after_prepare = time.time()

            # For updating, we cannot do it while answering queries, so wait for no readers
            if not self.writer_early_lock and not have_lock:
                self._fair_lock_ordonnancer.producer_acquire()
                have_lock = True
            
            after_wait_write_lock = time.time()
            broks_type_counter = {}
            for b in broks:
                try:
                    self.rg.manage_brok(b)
                    self.arch_export_handler.manage_brok(b)
                    broks_type_counter[b.type] = broks_type_counter.get(b.type, 0) + 1
                except Exception as exp:
                    self._die_with_strong_error(str(exp), with_trace_back=True)
                    return  # already exits
            
            end = time.time()
            logger.info('[ MANAGE BROKS ] [PERF]  [ %4d broks ] [ wait and get first set on queue=%.3fs ] [ get %s late sets on=%.3fs ] [ unserialize=%.3fs ] [ wait write lock=%.3fs ] [ manage broks=%.3fs ] [ total=%.3fs ]' % (
                len(broks), get_late_broks_set_time - start, nb_late_broks_sets_taken, after_get - get_late_broks_set_time, after_prepare - after_get, after_wait_write_lock - after_prepare, end - after_wait_write_lock, end - start))
            
            message = []
            for b_type, count in broks_type_counter.iteritems():
                message.append('[%s=%s]' % (b_type, count))
            logger.info('[ MANAGE BROKS ] [PERF]                  => manage broks types : %s' % ' '.join(message))
            
            # if we are late we don't release the lock and we continue to process broks
            if self.writer_catchup_enabled and self.writer_catchup_loops:
                qsize = self.to_q.qsize()
                if qsize > self.writer_late_sets_allowed >= 0:
                    logger.info('[ MANAGE BROKS ] [PERF] Broks sets in queue after manage broks is %s. We keep the lock and continue the brok managing.' % qsize)
                    continue
            
            if have_lock:
                # We can release the lock as a writer
                self._fair_lock_ordonnancer.producer_release()
                have_lock = False
            
            broks = []  # no more need these broks, and need to release memory asap
    
    
    def load_plugin(self, fdir, plugin_dir):
        logger.debug('UI Loading %s from %s' % (fdir, plugin_dir))
        try:
            # Put the full qualified path of the module we want to load
            # for example we will give  webui/plugins/eltdetail/
            mod_path = os.path.join(plugin_dir, fdir)
            # Then we load the eltdetail.py inside this directory
            m = imp.load_module('%s' % fdir, *imp.find_module(fdir, [mod_path]))
            m_dir = os.path.abspath(os.path.dirname(m.__file__))
            sys.path.append(m_dir)
            
            logger.debug('UI module %s is loaded from %s' % (m_dir, str(m)))
            pages = m.pages
            do_static = False
            for (f, entry) in pages.items():
                routes = entry.get('routes', None)
                v = entry.get('view', None)
                static = entry.get('static', False)
                wrappers = entry.get('wrappers', ['auth'])
                wrappers.append('shinken-js-version')
                widget_lst = entry.get('widget', [])
                widget_desc = entry.get('widget_desc', None)
                widget_display_name = entry.get('widget_display_name', 'Widget')
                widget_name = entry.get('widget_name', None)
                widget_picture = entry.get('widget_picture', None)
                widget_size = entry.get('widget_size', {'width': 1, 'height': 1})
                widget_options = entry.get('widget_options', [])
                for wo in widget_options:  # by default option are required
                    if 'required' not in wo:
                        wo['required'] = True
                widget_favoritable = entry.get('widget_favoritable', False)
                resizable = entry.get('resizable', False)
                
                old_style = entry.get('old_style', True)
                
                f_name = f.__name__
                f.display_name = f_name
                for wrapper in wrappers:
                    wrap = {'auth': self.wrap_auth, 'json': self.wrap_json, 'shinken-js-version': self.wrap_shinken_js_version_header}.get(wrapper, None)
                    if wrap:
                        f_name = f.display_name
                        f = wrap(f)
                        f.display_name = f_name + '-w(' + wrapper + ')'
                
                # IMPORTANT: apply VIEW BEFORE route!
                if v:
                    f_name = f.display_name
                    f = view(v)(f)
                    f.display_name = f_name + '-v(' + v + ')'
                
                # Maybe there is no route to link, so pass
                if routes:
                    for r in routes:
                        method = entry.get('method', 'GET')
                        logger.debug('UI linking function [%s] and route [%s] for method [%s]' % (getattr(f, 'display_name', str(f)), r, method))
                        
                        # Ok, we will just use the lock for all
                        # plugin page, but not for static objects
                        # so we set the lock at the function level.
                        f_name = f.display_name
                        lock_version = self.lockable_function(f)
                        lock_version.display_name = f_name
                        f = route(r, callback=lock_version, method=[method, 'OPTIONS'])
                
                # If the plugin declare a static entry, register it
                # and remember: really static! because there is no lock
                # for them!
                if static:
                    do_static = True
                
                # It's a valid widget entry if it got all data, and at least one route
                # ONLY the first route will be used for Add!
                # print "Should I load a widget?",widget_name, widget_desc, widget_lst!=[], routes
                if widget_name and widget_desc and widget_lst != [] and routes:
                    for place in widget_lst:
                        if place not in self.widgets:
                            self.widgets[place] = []
                        w = {
                            'name'       : widget_name,
                            'displayName': widget_display_name,
                            'description': widget_desc,
                            'baseURI'    : routes[0],
                            'picture'    : widget_picture,
                            'size'       : widget_size,
                            'options'    : widget_options,
                            'favoritable': widget_favoritable,
                            'oldStyle'   : old_style,
                            'resizable'  : resizable,
                        }
                        self.widgets[place].append(w)
            
            if do_static:
                self.add_static(fdir, m_dir)
            
            # And we add the views dir of this plugin in our TEMPLATE
            # PATH
            bottle.TEMPLATE_PATH.append(os.path.join(m_dir, 'views'))
            
            # And finally register me so the pages can get data and other
            # useful stuff
            m.app = self
        
        
        except Exception as exp:
            logger.warning("Loading plugins: %s" % exp)
            logger.print_stack()
    
    
    # Here we will load all plugins (pages) under the webui/plugins directory.
    # Each one can have a page, views and htdocs dir that we must route correctly
    def load_plugins(self, plugin_dir):
        # Load plugin directories
        if not os.path.exists(plugin_dir):
            return
        
        plugin_dirs = [fname for fname in os.listdir(plugin_dir) if os.path.isdir(os.path.join(plugin_dir, fname))]
        
        sys.path.append(plugin_dir)
        # We try to import them, but we keep only the one of our type
        for fdir in plugin_dirs:
            self.load_plugin(fdir, plugin_dir)
    
    
    def add_static(self, fdir, m_dir):
        static_route = '/static/%s/' % self.http_start_time + fdir + '/:path#.+#'
        
        
        def plugin_static(path):
            self._set_allow_cache_headers()
            return static_file(path, root=os.path.join(m_dir, 'htdocs'))
        
        
        logger.debug("Declaring static route: %s=> %s" % (static_route, plugin_static))
        route(static_route, callback=plugin_static)
    
    
    # set CORS headers for browsers to be happy
    def _set_cors_headers(self):
        response.headers['Access-Control-Allow-Origin'] = 'http://127.0.0.1:9000'
        response.headers['Access-Control-Allow-Methods'] = 'GET, POST, PUT, OPTIONS, DELETE, PATCH'
        response.headers['Access-Control-Allow-Headers'] = 'Origin, Accept, Content-Type, X-Requested-With, X-CSRF-Token, X-Shinken-Token'
        response.headers['Access-Control-Allow-Credentials'] = 'true'
    
    
    def query_stats_thread_display(self):
        self.last_log_incoming_query = 0
        while True:
            
            with self.query_stats_lock:
                query_stats_running_query = self.query_stats_running_query.copy()
                
                current_time = int(math.floor(time.time()))
                to_del_keys = []
                incoming_query_time_and_messages = []
                for time_incoming, incoming_query in self.query_stats_incoming_query.iteritems():
                    if time_incoming < self.last_log_incoming_query:
                        to_del_keys.append(time_incoming)
                    elif current_time != time_incoming:
                        incoming_query_global = incoming_query.get('GLOBAL', 0)
                        messages = ['At %s : Total query %s' % (PartLogger.format_time(time_incoming), incoming_query_global)]
                        messages.extend([' [%s : %s]' % (f_name.split('-')[0], count) for f_name, count in incoming_query.iteritems() if f_name != 'GLOBAL'])
                        
                        incoming_query_time_and_messages.append((time_incoming, ' '.join(messages)))
                        
                        if current_time > self.last_log_incoming_query:
                            self.last_log_incoming_query = current_time
                
                for to_del_key in to_del_keys:
                    del self.query_stats_incoming_query[to_del_key]
            
            query_stats_running_query_global = query_stats_running_query.get('GLOBAL', 0)
            if query_stats_running_query_global:
                messages = ['Total query %s' % query_stats_running_query_global]
                messages.extend([' [%s : %s]' % (f_name.split('-')[0], count) for f_name, count in query_stats_running_query.iteritems() if f_name != 'GLOBAL'])
                self.logger_ui_management_query_running_perf.info(''.join(messages))
            
            incoming_query_time_and_messages.sort()
            for _, message in incoming_query_time_and_messages:
                self.logger_ui_management_query_incoming_perf.info(message)
            
            time.sleep(1)
    
    
    # We want a lock manager version of the plugin functions
    def lockable_function(self, f):
        def lock_version(**args):
            f = lock_version.f
            function_name = getattr(f, 'display_name', str(f))
            start = time.time()
            
            with self.query_stats_lock:
                incoming_query_key = int(math.floor(start))
                self.query_stats_running_query['GLOBAL'] = self.query_stats_running_query.get('GLOBAL', 0) + 1
                self.query_stats_running_query[function_name] = self.query_stats_running_query.get(function_name, 0) + 1
                self.query_stats_incoming_query[incoming_query_key] = self.query_stats_incoming_query.get(incoming_query_key, {})
                self.query_stats_incoming_query[incoming_query_key]['GLOBAL'] = self.query_stats_incoming_query[incoming_query_key].get('GLOBAL', 0) + 1
                self.query_stats_incoming_query[incoming_query_key][function_name] = self.query_stats_incoming_query[incoming_query_key].get(function_name, 0) + 1
            
            lock_was_set = False
            # the browser need CORS headers to be set
            self._set_cors_headers()
            
            # OPTIONS calls are for CORS, and don't need more than void return
            if bottle.request.method == 'OPTIONS':
                with self.query_stats_lock:
                    self.query_stats_running_query['GLOBAL'] = self.query_stats_running_query.get('GLOBAL', 0) - 1
                    self.query_stats_running_query[function_name] = self.query_stats_running_query.get(function_name, 0) - 1
                return
            
            # disable cache on all backend calls
            self._set_no_cache_headers()
            after_no_cache_headers = time.time()
            
            # LOCK Ask a lock for reading
            self._fair_lock_ordonnancer.consumer_acquire()
            try:
                after_wait_no_writers = time.time()
                res = f(**args)
                end = time.time()
                if self.logger_ui_management_query_perf.is_enable():
                    log_function_name = function_name.split('-')[0]
                    try:
                        user_uuid = self.get_user_auth().uuid
                    except:
                        user_uuid = u'user_UUID_not_set'
                    self.logger_ui_management_query_perf.info(TEMPLATE_LOG_QUERY_PERF % (
                        log_function_name, user_uuid, PartLogger.format_hours(start), PartLogger.format_hours(end), end - start, after_wait_no_writers - after_no_cache_headers, end - after_wait_no_writers))
                
                return res
            finally:  # IN ALL CASES: release the readers
                # UNLOCK:
                self._fair_lock_ordonnancer.consumer_release()
                
                with self.query_stats_lock:
                    self.query_stats_running_query['GLOBAL'] = self.query_stats_running_query.get('GLOBAL', 0) - 1
                    self.query_stats_running_query[function_name] = self.query_stats_running_query.get(function_name, 0) - 1
        
        
        lock_version.f = f
        return lock_version
    
    
    # For this request, ask for no cache at all. for example static that have always the same address across versions
    def _set_no_cache_headers(self):
        # The Cache-Control is per the HTTP 1.1 spec for clients and proxies
        # (and implicitly required by some clients next to Expires).
        # The Pragma is per the HTTP 1.0 spec for prehistoric clients.
        # The Expires is per the HTTP 1.0 and 1.1 spec for clients and proxies.
        # In HTTP 1.1, the Cache-Control takes precedence over Expires, so it's after all for HTTP 1.0 proxies only.
        response.headers['Cache-Control'] = 'no-cache,no-store,must-revalidate'
        response.headers['Pragma'] = 'no-cache'
        response.headers['Expires'] = '0'
    
    
    def _set_allow_cache_headers(self):
        # Ask for no tuning from cache or in the middle proxy
        response.headers['Cache-Control'] = 'no-transform,public,max-age=86400,s-maxage=86400'
        response.headers['Expires'] = '86400'  # old http 1.0 way
    
    
    def declare_common_static(self):
        
        # Give the combined css file
        # IMPORTANT: defined BEFORE the static/*
        # And disallow cache, as the ui will F5 when the hash will change
        @self.bottle.route('/static/css/check_plugin_styles_css.css')
        def give_check_plugin_styles_css():
            self._set_no_cache_headers()
            return self.bottle.static_file(os.path.basename(self.compiled_css_path), root=os.path.dirname(self.compiled_css_path))
        
        
        @route('/static/photos/:path#.+#')
        def give_photo(path):
            # If the file really exist, give it. If not, give a dummy image.
            if os.path.exists(os.path.join(self.photo_dir, path + '.jpg')):
                return static_file(path + '.jpg', root=self.photo_dir)
            else:
                return static_file('images/user.png', root=htdocs_dir)
        
        
        @self.bottle.route('/static/ui/:path#.+#')
        def server_static_app(path):
            # Route static ui files (they have their own cache preventing system)
            # By default give from the root in bottle_dir/htdocs. If the file is missing, search in the share dir
            path = 'ui/' + path
            root = htdocs_dir
            
            is_index_html = path == 'ui/index.html'
            
            if path == is_index_html and self.logger_ui_management_query_perf.is_enable():
                try:
                    user_uuid = self.get_user_auth().uuid
                except:
                    user_uuid = u'user_UUID_not_set'
                start = PartLogger.format_hours(time.time())
                self.logger_ui_management_query_perf.info(TEMPLATE_LOG_QUERY_PERF % (path, user_uuid, start, start, 0, 0, 0))
            
            # logger.debug('Asking for static: %s' % path)
            
            # Ok now manage cache things:
            # * all the *.js and *.css things are automatically changed paths
            #   when new version, so we can cache them
            # * but all the others things like html and so on don't have changed paths
            #   so we must protect them
            if path.endswith('.js') or path.endswith('.css'):
                self._set_allow_cache_headers()
            else:  # no cache at all
                # Our dear Internet Explorer have a bug about https+nocache+woff files.
                if not path.endswith('.woff'):
                    self._set_no_cache_headers()
            
            # SEF-6722: index.html was already load and the lang was set inside it
            if is_index_html:
                return self._index_html_content
            return self.bottle.static_file(path, root=root)
        
        
        @route('/static/%s' % self.http_start_time + '/:path#.+#')
        def server_static(path):
            # Route static files css files
            # By default give from the root in bottle_dir/htdocs. If the file is missing,search in the share dir
            
            root = htdocs_dir
            p = os.path.join(root, path)
            if not os.path.exists(p):
                root = self.share_dir
            # They have path at each boot, so we can cache them without problems
            self._set_allow_cache_headers()
            return static_file(path, root=root)
        
        
        # And add the favicon ico too
        @route('/favicon.ico')
        def give_favicon():
            # response.headers['Cache-Control'] = 'public, max-age=3600'
            return static_file('favicon.ico', root=os.path.join(htdocs_dir, 'images'))
    
    
    def authenticate_user_by_modules(self, username, password, requester, authentication_phase_id=None):
        # type: (unicode,unicode,unicode,Union[unicode, None]) -> Union[bool, Contact]
        
        # Need to get an uuid to trace anonymously this request ( do not trace user -> RGPD Safe )
        if not authentication_phase_id:
            authentication_phase_id = u'id-%s' % uuid.uuid1().get_hex()

        authentication_logger = self.logger_auth.get_sub_part(requester)
        
        authentication_logger.info(u'Need to check a user authentication ( authentication phase %s )' % authentication_phase_id)
        
        if not username:
            authentication_logger.info(u'No username was received, can\'t continue')
            return False
        
        authentication_logger.debug(u'A user try to connect from address "%s" with username "%s" ( authentication phase %s )' % (self.request.environ.get(u'REMOTE_ADDR'), username, authentication_phase_id))
        authenticated_contact = None
        
        for mod in self.modules_manager.get_internal_instances():
            check_auth_function = getattr(mod, u'check_auth', None)
            if check_auth_function and callable(check_auth_function):
                authentication_logger.info(u'Use module "%s" to authenticate user ( authentication phase %s )' % (mod.get_name(), authentication_phase_id))
                try:
                    authenticated_contact = check_auth_function(username, password, requester, authentication_phase_id)
                except Exception as exp:
                    self.logger.warning(u'The mod "%s" raise an exception during authentication phase %s : %s, I\'m tagging it to restart later' % (mod.get_name(), authentication_phase_id, str(exp)))
                    self.logger.debug(u'Exception type: %s' % type(exp))
                    self.logger.debug(u'Back trace of this kill: %s' % (traceback.format_exc()))
                    self.modules_manager.did_crash(mod)
                
                if authenticated_contact:
                    contact_name = authenticated_contact.contact_name
                    authentication_logger.info(u'The user has been authenticated by the module "%s" ( authentication phase %s )' % (mod.get_name(), authentication_phase_id))
                    authentication_logger.debug(u'User "%s" is authenticated ( authentication phase %s )' % (contact_name, authentication_phase_id))
                    # No need for other modules
                    break
                else:
                    authentication_logger.info(u'Module "%s" didn\'t authenticate user ( authentication phase %s )' % (mod.get_name(), authentication_phase_id))
        
        if authenticated_contact is False:
            authentication_logger.info(u'We tried to authenticate the user with all active module but it failed. AUTHENTICATION FAILED ( authentication phase %s )' % authentication_phase_id)
            authentication_logger.debug(u'User "%s" can\'t be authenticate by modules ( authentication phase %s )' % (username, authentication_phase_id))
        
        return authenticated_contact
    
    
    def get_check_plugin_styles_css_hash(self):
        return self.compiled_css_hash
    
    
    # Some helpers for string/byte handling
    def tob(self, s, enc='utf8'):
        return s.encode(enc) if isinstance(s, unicode) else bytes(s)
    
    
    def _lscmp(self, a, b):
        # Compares two strings in a cryptographically safe way:
        # Runtime is not affected by length of common prefix.
        return not sum(0 if x == y else 1 for x, y in zip(a, b)) and len(a) == len(b)
    
    
    def token_encode(self, data):
        key = self.auth_secret
        # Encode and sign a pickle-able object. Return a (byte) string
        msg = base64.b64encode(json.dumps(data))
        sig = base64.b64encode(hmac.new(self.tob(key), msg).digest())
        return self.tob('!') + sig + self.tob('?') + msg
    
    
    def token_decode(self, data):
        key = self.auth_secret
        # Verify and decode an encoded string. Return an object or None.
        data = self.tob(data)
        if self.token_is_encoded(data):
            sig, msg = data.split(self.tob('?'), 1)
            if self._lscmp(sig[1:], base64.b64encode(hmac.new(self.tob(key), msg).digest())):
                try:
                    return json.loads(base64.b64decode(msg))
                except:  # maybe there was a previous valid token with pickle, so this will fail, invalidate it
                    return None
        return ''
    
    
    def token_is_encoded(self, data):
        ''' Return True if the argument looks like a encoded cookie.'''
        return bool(data.startswith(self.tob('!')) and self.tob('?') in data)
    
    
    def get_token(self, uname):
        return self.token_encode(uname)
    
    
    # Authentication:
    # * only the cookie is a valid authentication method (ui conf + pure backend page)
    # * only the token  is NOT valid as iframe won't be happy
    # * if both: must be ok on the value, if not means that the UI is not aware of a cookie change
    def get_user_auth(self):
        # First we look for the user sid so we bail out if it's a false one
        cookie_user_id = self.request.get_cookie('user', secret=self.auth_secret)
        
        case_sensitive = True
        
        # cookie is mandatory, no cookie, no gain unless we allow remote login
        if not cookie_user_id and self.remote_user_enable != '1':
            return self.abort(401, "Your cookie is not valid. Please re-authentify with a valid user.")
        
        # Now look at the token, if present, must be the same as the cookie
        token = self.request.headers.get('X-Shinken-Token', '')
        # Maybe token was not in header, maybe in the get parameters?
        if token == '':
            token64 = self.request.query.get('_token', '')
            try:
                token = base64.b64decode(token64)
            except Exception:
                logger.error("[ui] bad base64 token: %s" % token64)
        if token.startswith('"'):
            token = token[1:]
        if token.endswith('"'):
            token = token[:-1]
        token_user_id = ''
        if token:
            token_user_id = self.token_decode(token)
        
        new_user = None
        if cookie_user_id:
            new_user = self.datamgr.get_contact(cookie_user_id, by_id=True)
        elif self.remote_user_enable == '1':
            # Maybe the user did want to look at Header auth, so look for this case, but only if there is no cookie here
            logger.debug('[Auth] Searching for auth in header [%s]' % self.remote_user_variable)
            if self.remote_user_variable in self.request.headers:
                header_user_name = self.request.headers[self.remote_user_variable]
                case_sensitive = self.remote_user_case_sensitive
                logger.debug('[Auth] Header auth, did found user: [%s] - case_sensitive [%s]' % (header_user_name, case_sensitive))
                if case_sensitive:
                    new_user = self.datamgr.get_contact(header_user_name)
                else:
                    new_user = self.datamgr.get_contact_case_insensitive(header_user_name)
                if not new_user:
                    self.abort(401, 'UI Authentification by header fail')
                
                cookie_user_id = getattr(new_user, 'uuid')
        
        # Still not user name aven after a header lookup? Bail out.
        if not cookie_user_id:
            return self.abort(401, "Your cookie/auth is not valid. Please re-authentify with a valid user.")
        
        # Ok now look at both token and cookie.
        # If the token is present, it must be the same as cookie.
        # Only the front have token so this is a optional test.
        if token and cookie_user_id != token_user_id:
            if not new_user:
                self.abort(401, "Your cookie is not valid. Please re-authentify with a valid user.")
            
            new_user_name = getattr(new_user, 'contact_name', 'User with no name')
            # token to used for furhter request
            new_token = self.get_token(cookie_user_id)
            logger.debug("[Auth] User change detected [%s-%s]" % (new_token, new_user_name))
            
            response.content_type = 'application/json'
            output = {
                "purpose" : "userSwitched",
                "token"   : new_token,
                "username": new_user.contact_name
            }
            self.abort(409, output)
        
        user_auth = self.datamgr.get_contact(cookie_user_id, by_id=True)
        if user_auth:
            logger.debug('[Auth] UI Authentication found: [%s]' % cookie_user_id)
        else:
            logger.debug('[Auth] UI Authentication fail: [%s]' % cookie_user_id)
        return user_auth
    
    
    # Try to got for an element the graphs uris from modules
    # The source variable describes the source of the calling. Are we displaying
    # graphs for the element detail page (detail), or a widget in the dashboard (dashboard) ?
    def get_graph_uris(self, elt, graphstart, graphend, source='detail'):
        # safe_print("Checking graph uris ", elt.get_full_name())
        
        uris = []
        for mod in self.modules_manager.get_internal_instances():
            try:
                f = getattr(mod, 'get_graph_uris', None)
                # safe_print("Get graph uris ", f, "from", mod.get_name())
                if f and callable(f):
                    r = f(elt, graphstart, graphend, source)
                    uris.extend(r)
            except Exception as exp:
                logger.warning("[%s] The mod %s raise an exception: %s, I'm tagging it to restart later" % (self.name, mod.get_name(), str(exp)))
                logger.debug("[%s] Exception type: %s" % (self.name, type(exp)))
                logger.debug("Back trace of this kill: %s" % (traceback.format_exc()))
                self.modules_manager.did_crash(mod)
        
        # safe_print("Will return", uris)
        # Ok if we got a real contact, and if a module auth it
        return uris
    
    
    def get_common_preference(self, key, default=None):
        for mod in self.modules_manager.get_internal_instances():
            try:
                f = getattr(mod, 'get_ui_common_preference', None)
                if f and callable(f):
                    r = f(key)
                    return r
            except Exception as exp:
                logger.warning("[%s] The mod %s raise an exception: %s, I'm tagging it to restart later" % (self.name, mod.get_name(), str(exp)))
                logger.debug("[%s] Exception type: %s" % (self.name, type(exp)))
                logger.debug("Back trace of this kill: %s" % (traceback.format_exc()))
                self.modules_manager.did_crash(mod)
        return default
    
    
    # Maybe a page want to warn if there is no module that is able to give user preference?
    def has_user_preference_module(self):
        for mod in self.modules_manager.get_internal_instances():
            f = getattr(mod, 'get_ui_user_preference', None)
            if f and callable(f):
                return True
        return False
    
    
    # Try to got for an element the graphs uris from modules
    def get_user_preference(self, user, key, default=None):
        for mod in self.modules_manager.get_internal_instances():
            try:
                f = getattr(mod, 'get_ui_user_preference', None)
                if f and callable(f):
                    r = f(user, key)
                    return r
            except Exception as exp:
                logger.warning("[%s] The mod %s raise an exception: %s, I'm tagging it to restart later" % (self.name, mod.get_name(), str(exp)))
                logger.debug("[%s] Exception type: %s" % (self.name, type(exp)))
                logger.debug("Back trace of this kill: %s" % (traceback.format_exc()))
                self.modules_manager.did_crash(mod)
        return default
    
    
    # Try to got for an element the graphs uris from modules
    def set_user_preference(self, user, key, value):
        for mod in self.modules_manager.get_internal_instances():
            try:
                f = getattr(mod, 'set_ui_user_preference', None)
                if f and callable(f):
                    f(user, key, value)
            except Exception as exp:
                logger.warning("[%s] The mod %s raise an exception: %s, I'm tagging it to restart later" % (self.name, mod.get_name(), str(exp)))
                logger.debug("[%s] Exception type: %s" % (self.name, type(exp)))
                logger.debug("Back trace of this kill: %s" % (traceback.format_exc()))
                self.modules_manager.did_crash(mod)
    
    
    def set_common_preference(self, key, value):
        for mod in self.modules_manager.get_internal_instances():
            try:
                f = getattr(mod, 'set_ui_common_preference', None)
                if f and callable(f):
                    f(key, value)
            except Exception as exp:
                logger.warning("[%s] The mod %s raise an exception: %s, I'm tagging it to restart later" % (self.name, mod.get_name(), str(exp)))
                logger.debug("[%s] Exception type: %s" % (self.name, type(exp)))
                logger.debug("Back trace of this kill: %s" % (traceback.format_exc()))
                self.modules_manager.did_crash(mod)
                
                # end of all modules
    
    
    # For a specific place like dashboard we return widget lists
    def get_widgets_for(self, place):
        return self.widgets.get(place, [])
    
    
    # Will get all label/uri for external UI like PNP or NagVis
    def get_external_ui_link(self):
        lst = []
        for mod in self.modules_manager.get_internal_instances():
            try:
                f = getattr(mod, 'get_external_ui_link', None)
                if f and callable(f):
                    r = f()
                    lst.append(r)
            except Exception as exp:
                logger.warning("[%s] Warning: The mod %s raise an exception: %s, I'm tagging it to restart later" % (self.name, mod.get_name(), str(exp)))
                logger.debug("[%s] Exception type: %s" % (self.name, type(exp)))
                logger.debug("Back trace of this kill: %s" % (traceback.format_exc()))
                self.modules_manager.did_crash(mod)
        return lst
    
    
    def insert_template(self, tpl_name, d):
        try:
            r = template(tpl_name, d)
        except Exception as exp:
            pass
    
    
    def get_webui_port(self):
        port = self.port
        return port
    
    
    def get_skonf_port(self):
        port = self.http_port
        return port
    
    
    def get_skonf_active_state(self):
        state = self.show_skonf
        return state
    
    
    def get_first_tag(self, tags):
        for t in tags:
            if t in self.img_tags:
                return t
        return None
    
    
    @staticmethod
    def is_check_uuid(uuid):
        return '-' in uuid
    
    
    def get_name_from_uuid(self, uuid):
        r = None
        if '-' in uuid:
            check = datamgr.get_service_by_uuid(uuid)
            if check:
                r = [check.host.host_name, check.service_description]
        else:
            host = datamgr.get_host_by_uuid(uuid)
            if host:
                r = [host.host_name]
        
        if not r:
            return self.abort(404, 'unknow uuid [%s]' % uuid, True)
        return r
    
    
    # Traduction call
    # TODO: too much hard coded paths there, find a better way
    def _(self, s):
        # First find lang of the current call, and by default take the global lang parameter
        lang = self.request.query.get('lang', self.lang)
        d = self.langs.get(lang, None)
        # If missing, load the file
        if d is None:
            pth = os.path.join(self.langs_path, lang + '.js')
            if not os.path.exists(pth):
                logger.error('Cannot load the lang file %s: no such file' % pth)
                return ''
            f = open(pth, 'r')
            buf = f.read()
            f.close()
            lines = buf.splitlines()
            new_lines = []
            for line in lines:
                line = line.strip()
                if line.startswith('//'):
                    continue
                if line.startswith('var lang'):
                    line = '{'
                if line.startswith('};'):
                    line = '}'
                new_lines.append(line)
            buf = '\n'.join(new_lines)
            o = json.loads(buf)
            self.langs[lang] = o
        o = self.langs[lang]
        elts = [e.strip() for e in s.split('.') if e.strip()]
        for e in elts:
            o = o.get(e, None)
            if o is None:
                logger.error('Translation: cannot find %s in the lang %s' % (s, lang))
                return ''
        return o
    
    
    @staticmethod
    def get_contact_groups(contact):
        return contact.contactgroups
    
    
    def _get_mongo(self):
        for inst in self.modules_manager.get_all_alive_instances():
            if inst.properties['type'] == 'mongodb':
                return inst
        return None
    
    
    # May raise exceptions, must be try/excepted when used
    def save_versionned_object(self, col_name, uid_key, item_uuid, data, create_object=False):
        logger.debug("Saving %s object of uuid %s" % (col_name, item_uuid))
        prev_version = 0
        inst = self._get_mongo()
        db = inst.db
        col = getattr(db, col_name)
        if col:
            prev_data = col.find_one({uid_key: item_uuid})
            if prev_data:
                prev_version = int(prev_data.get('saveVersion', 0))
            elif not create_object:
                raise LookupError("%s object not found in database : %s" % (col_name, item_uuid))
        
        data_update = data.get('update', False)
        data.pop('update', None)
        if data_update and prev_version < int(data.get('saveVersion', 0)):
            col.update({uid_key: item_uuid}, data, upsert=True)
        elif not data_update and prev_version == int(data.get('saveVersion', 0)):
            col.update({uid_key: item_uuid}, data, upsert=True)
        else:
            logger.info("Save %s object of uuid %s aborted because submitted version is older. (current: %s. submitted : %s)" % (col_name, item_uuid, prev_version, data.get('saveVersion', 0)))
            raise ValueError(self._('error.save_versionned_object'))
    
    
    def get_screen_collection(self, screen_type):
        inst = self._get_mongo()
        if not inst:
            return None
        db = inst.db
        if not db:
            return None
        if screen_type not in ['hive', 'list', 'dashboard']:
            return None
        col = getattr(db, screen_type)
        return col
    
    
    def get_hive_collection(self):
        inst = self._get_mongo()
        if not inst:
            return None
        db = inst.db
        if not db:
            return None
        col = getattr(db, 'hive')
        return col
    
    
    def get_dashboard_collection(self):
        inst = self._get_mongo()
        if not inst:
            return None
        db = inst.db
        if not db:
            return None
        col = getattr(db, 'dashboard')
        return col
    
    
    def get_list_collection(self):
        inst = self._get_mongo()
        if not inst:
            return None
        db = inst.db
        if not db:
            return None
        col = getattr(db, 'list')
        return col
    
    
    def get_share_collection(self):
        inst = self._get_mongo()
        if not inst:
            return None
        db = inst.db
        if not db:
            return None
        col = getattr(db, 'share')
        return col
    
    
    def get_user_collection(self):
        inst = self._get_mongo()
        if not inst:
            return None
        db = inst.db
        if not db:
            return None
        col = getattr(db, 'user')
        return col
    
    
    def get_tiles_collection(self):
        inst = self._get_mongo()
        if not inst:
            return None
        db = inst.db
        if not db:
            return None
        col = getattr(db, 'tiles')
        return col
