#!/usr/bin/python

# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

import base64
import datetime
import json
import uuid

from shinken.log import LoggerFactory
from shinken.misc.type_hint import TYPE_CHECKING

if TYPE_CHECKING:
    from shinken.misc.type_hint import Optional
    from webui.module import WebuiBroker

app = None  # type: Optional[WebuiBroker]

logger_auth = LoggerFactory.get_logger(u'AUTHENTICATION')


def user_login_redirect():
    app.bottle.redirect('/')
    return {}


def user_logout_api():
    app.response.set_cookie('user', '', secret=app.auth_secret, path='/', httponly=True)
    return {}


def user_auth_api():
    login64 = app.request.forms.get(u'login', '')
    password64 = app.request.forms.get(u'password', '')
    
    try:
        password = base64.b64decode(password64).decode(u'utf8', u'ignore')
        login = base64.b64decode(login64).decode(u'utf8', u'ignore')
    except Exception, exp:  # bad encoding? get out!
        return app.abort(400, unicode(exp))
    
    contact = app.authenticate_user_by_modules(login, password, u'UI Visualization')
    
    if contact:
        contact_id = contact.uuid
        app.response.set_cookie('user', contact_id, secret=app.auth_secret, path='/', expires=datetime.datetime(2100, 1, 1), httponly=True)
        token = app.get_token(contact_id)
        app.response.headers['X-Shinken-Token'] = token
        return json.dumps(token)
    else:
        return app.abort(401, '')


def user_auth_nagvis_api():
    authentication_phase_id = u'id-%s' % uuid.uuid1().get_hex()
    restrict_to_shinken_admin64 = app.request.forms.get(u'restrict_to_shinken_admin', u'0')
    try:
        restrict_to_shinken_admin = base64.b64decode(restrict_to_shinken_admin64).decode(u'utf8', u'ignore')
        restrict_to_shinken_admin = bool(restrict_to_shinken_admin == u'1')
    except Exception, exp:  # bad encoding? get out!
        restrict_to_shinken_admin = False
    
    # If a http header is provided for SSO and it corresponds to a valid user, return username
    if app.remote_user_enable == u'1' and app.remote_user_variable in app.request.headers:
        logger_auth.info(u'Try to authenticate user for NagVis trough HTTP header ( authentication phase %s )' % authentication_phase_id)
        username_from_header = app.request.headers[app.remote_user_variable]
        
        shinken_contact = app.datamgr.get_contact(username_from_header) if app.remote_user_case_sensitive else app.datamgr.get_contact_case_insensitive(username_from_header)
        
        logger_auth.debug(u'Use HTTP variable "%s" with value "%s" to find Shinken user in database ( authentication phase %s )' % (app.remote_user_variable, username_from_header, authentication_phase_id))
        
        if shinken_contact.is_admin or not restrict_to_shinken_admin:
            logger_auth.info(u'User was authenticated by HTTP header to access NagVis ( authentication phase %s )' % authentication_phase_id)
            return shinken_contact.contact_name
        elif restrict_to_shinken_admin and not shinken_contact.is_admin:
            logger_auth.info(u'User was authenticated by HTTP header to access to NagVis but he is not an Shinken administrator. Cancel authentication ( authentication phase %s )' % authentication_phase_id)
            return app.abort(401, u'')
        else:
            if not shinken_contact:
                logger_auth.debug(u'There is no user in Shinken database which match contact_name = %s  ( authentication phase %s )' % (shinken_contact.contact_name, authentication_phase_id))
            logger_auth.info(u'User wasn\'t authenticated by HTTP header to access NagVis ( authentication phase %s )' % authentication_phase_id)
            return app.abort(401, u'')
    
    # If a cookie has been provided and it corresponds to a valid user session, return username
    logger_auth.info(u'Try to authenticate user for NagVis trough cookie ( authentication phase %s )' % authentication_phase_id)
    user_id = app.request.get_cookie('user', secret=app.auth_secret)
    if user_id:
        logger_auth.debug(u'cookie exists with user_id "%s" ( authentication phase %s )' % (user_id, authentication_phase_id))
        
        shinken_contact = app.datamgr.get_contact(user_id, by_id=True)
        
        if shinken_contact and (shinken_contact.is_admin or not restrict_to_shinken_admin):
            logger_auth.info(u'User was authenticated by cookie to access NagVis ( authentication phase %s )' % authentication_phase_id)
            return shinken_contact.contact_name
        elif shinken_contact and not shinken_contact.is_admin and restrict_to_shinken_admin:
            logger_auth.info(u'User was authenticated by cookie to access to NagVis but he is not an Shinken administrator. Cancel authentication ( authentication phase %s )' % authentication_phase_id)
            return app.abort(401, u'')
        else:
            if not shinken_contact:
                logger_auth.debug(u'There is no user in Shinken database which match _id = %s  ( authentication phase %s )' % (user_id, authentication_phase_id))
            logger_auth.info(u'User wasn\'t authenticated by cookie to access NagVis ( authentication phase %s )' % authentication_phase_id)
    else:
        logger_auth.info(u'There is no existing cookie to check authentication ( authentication phase %s )' % authentication_phase_id)
    
    # If no cookie provided in request, proceed to check auth credentials
    logger_auth.info(u'Try to authenticate user for NagVis by credentials ( authentication phase %s )' % authentication_phase_id)
    login64 = app.request.forms.get(u'login', u'')
    password64 = app.request.forms.get(u'password', u'')
    
    try:
        password = base64.b64decode(password64).decode(u'utf8', u'ignore')
        login = base64.b64decode(login64).decode(u'utf8', u'ignore')
    except Exception, exp:  # bad encoding? get out!
        return app.abort(400, str(exp))
    
    if not login and not password:
        logger_auth.info(u'There is no credentials provided  ( authentication phase %s )' % authentication_phase_id)
        return app.abort(401, u'')
    
    contact = app.authenticate_user_by_modules(login, password, u'NagVis', authentication_phase_id=authentication_phase_id)
    
    if contact:
        contact_id = contact.uuid
        
        if contact.is_admin or not restrict_to_shinken_admin:
            app.response.set_cookie('user', contact_id, secret=app.auth_secret, path='/', expires=datetime.datetime(2100, 1, 1), httponly=True)
            token = app.get_token(contact_id)
            app.response.headers['X-Shinken-Token'] = token
            return json.dumps(token)
        else:
            logger_auth.info(u'User was authenticated to access to NagVis but he is not an Shinken administrator. Cancel authentication ( authentication phase %s )' % authentication_phase_id)
            return app.abort(401, u'')
    else:
        return app.abort(401, u'')


# manage the /. If the user is known, go to problems page.
# Should be /dashboard in the future. If not, go login :)
def get_root():
    # user = app.request.get_cookie('user', secret=app.auth_secret)
    return app.bottle.redirect('/static/ui/index.html')


def get_placeholder():
    for mod in app.modules_manager.get_internal_instances():
        if getattr(mod, u'login_placeholder', u''):
            return {u'placeholder': mod.login_placeholder}
    return {}


def get_token():
    user = app.get_user_auth()
    token = app.get_token(user.uuid)
    app.response.headers['X-Shinken-Token'] = token
    return json.dumps(token)


pages = {
    user_login_redirect : {'routes': ['/login'], 'static': True, 'wrappers': []},
    get_placeholder     : {'routes': ['/login/placeholder'], 'wrappers': []},
    user_auth_api       : {'routes': ['/auth'], 'method': 'POST', 'wrappers': ['json']},
    user_auth_nagvis_api: {'routes': ['/auth_nagvis'], 'method': 'POST', 'wrappers': ['json']},
    user_logout_api     : {'routes': ['/apilogout'], 'method': 'POST', 'wrappers': ['json']},
    get_root            : {'routes': ['/'], 'static': True, 'wrappers': []},
    get_token           : {'routes': ['/get_token'], 'static': True, 'wrappers': ['auth', 'json']},
}
