import math
import inspect
import os
import signal
import sys
import time
import uuid
from threading import Thread, Lock
from Queue import Empty as EmptyQueue
from ctypes import c_bool
from multiprocessing import Process, Event, Queue, Value

from shinken.misc.type_hint import Optional, NoReturn, TYPE_CHECKING
from shinken.subprocess_helper.error_handler import ERROR_LEVEL, ErrorHandler
from .log import logger as _logger, LoggerFactory, PART_INITIALISATION, PartLogger
from .thread_helper import Thread
from .util import start_malloc_trim_thread, set_process_name

if TYPE_CHECKING:
    from .misc.type_hint import List, Any, Optional, Callable, Str, Number
    from .log import PartLogger

ARG_SEPARATOR = ':::'
_UNINTERRUPTABLE_SLEEP_TIME = 1.0
_old_process = {}

ON_LINUX = sys.platform.startswith("linux")


def get_old_process(class_name):
    my_pid = os.getpid()
    if my_pid in _old_process:
        return _old_process[my_pid].get(class_name, None)


def register_old_process(class_name, instance):
    my_pid = os.getpid()
    for pid in _old_process.keys():
        if pid != my_pid:
            del _old_process[pid]
    
    if my_pid not in _old_process:
        _old_process[my_pid] = {}
    
    _old_process[my_pid][class_name] = instance


class LookAtMyFatherThread(Thread):
    def __init__(self, father_pid, father_name, to_kill_name, loop_speed=1, logger=None):
        super(LookAtMyFatherThread, self).__init__(loop_speed=loop_speed, logger=logger)
        self.father_pid = father_pid
        self.father_name = father_name
        self.to_kill_name = to_kill_name
    
    
    def loop_turn(self):
        try:
            os.kill(self.father_pid, 0)
        except:
            self.logger.error('The father process [%s]-[%s] is dead, I kill [%s]-[%s].' % (self.father_pid, self.father_name, os.getpid(), self.to_kill_name))
            os._exit(0)
    
    
    def get_thread_name(self):
        return 'look-at-father'


class EventHandler(Thread):
    event = None  # type: Event
    event_name = 'event-handler'  # type:str
    
    
    def __init__(self, event_name='event-handler', error_handler=None, time_wait_for_started=0):
        # type: (str, ErrorHandler, float) -> None
        self._private_running = Value(c_bool, False)
        super(EventHandler, self).__init__(error_handler=error_handler)
        self.event = Event()
        self.event_name = event_name
        self.time_wait_for_started = time_wait_for_started
    
    
    @property
    def _running(self):
        return self._private_running.value
    
    
    @_running.setter
    def _running(self, _running_value):
        self._private_running.value = _running_value
    
    
    def get_thread_name(self):
        return self.event_name
    
    
    def loop_turn(self):
        self.event.wait()
        self.callback()
        self.event.clear()
    
    
    def send_event(self):
        time_wait = 0
        while not self._running:
            if not self.time_wait_for_started or time_wait > self.time_wait_for_started:
                raise Exception('You can\'t send an event before starting the event handler')
            time.sleep(0.1)
            time_wait += 0.1
        self.event.set()
    
    
    def callback(self):
        raise NotImplementedError()


class QueueHandler(Thread):
    _queue = None  # type: Queue
    _queue_name = 'queue-handler'  # type:str
    all_items = []
    
    
    def __init__(self, max_size, queue_name='queue-handler', error_handler=None):
        super(QueueHandler, self).__init__(error_handler=error_handler)
        self.owner_pid = os.getpid()
        self._queue = Queue()
        self._queue_name = queue_name
        self._max_size = max_size
    
    
    def stop(self):
        Thread.stop(self)
        if self._queue:
            self._queue.close()
            self._queue.join_thread()
    
    
    def reset(self):
        self.stop()
        self.owner_pid = os.getpid()
        self._queue = Queue()
        self.all_items = []
        if self.is_running() and self._thread:
            self.ask_stop()
            self._thread.join()
        self.start_thread()
        
        self.logger.info('Reset %s for owner_pid:%s' % (self._queue_name, self.owner_pid))
    
    
    def get_thread_name(self):
        return self._queue_name
    
    
    def loop_turn(self):
        try:
            while True:
                item = self._queue.get_nowait()
                self.all_items.append(item)
                self.on_add_item(item)
                if len(self.all_items) > self._max_size:
                    self.all_items.pop(0)
                self.interruptable_sleep(self.loop_speed)
        except EmptyQueue:
            # we will continue to loop until new item is available
            pass
        self.interruptable_sleep(self.loop_speed)
    
    
    def on_add_item(self, item):
        pass
    
    
    def put(self, item):
        self._queue.put(item)


class UnknownCommandException(Exception):
    pass


# Object that will be send by the daemon to the module with a specific call (and maybe args)
# Will be used to match respond and look if the answer have the same uuid than the request
class ToModuleCommandRequest(object):
    def __init__(self, command_name, args):
        # type: (Str, List) -> None
        str_send_command = command_name
        if args:
            str_send_command = [command_name]
            str_send_command.extend(args)
            str_send_command = ARG_SEPARATOR.join(str_send_command)
        self._command = str_send_command
        self._uuid = uuid.uuid4().hex
    
    
    def get_command(self):
        return self._command
    
    
    def get_uuid(self):
        # type: () -> Str
        return self._uuid
    
    
    def create_respond(self, result_payload):
        # type: (Callable) -> Any
        respond = FromModuleCommandRespond(self._uuid, result_payload)
        return respond
    
    
    def __str__(self):
        return u'ToModuleCommandRequest[%s-%s]' % (self._uuid, self._command)


# This will be create by a request (with the good uuid, etc) to give back a result from the module.
# Will be used by the original request to match if the respond match the request uuid (can be problem in queues)
class FromModuleCommandRespond(object):
    def __init__(self, request_uuid, result_payload):
        # type: (Str, Callable) -> None
        self._uuid = request_uuid
        self._payload = result_payload
    
    
    def do_match_request(self, request):
        # type: (ToModuleCommandRequest) -> Any
        request_uuid = request.get_uuid()
        return self._uuid == request_uuid
    
    
    def get_payload(self):
        return self._payload


class CommandQueueHandler(Thread):
    def __init__(self, name, parent_logger, commands_to_q, commands_from_q, main_process):
        # type: (Str, PartLogger, Callable, Callable, object) -> None
        super(CommandQueueHandler, self).__init__(loop_speed=-1, logger=parent_logger)
        self.daemon = True
        self.logger = parent_logger
        self.handler_name = name
        self.commands_to_q = commands_to_q
        self.commands_from_q = commands_from_q
        self.main_process = main_process
        self.name = name
        
        # Lock for commands from different threads racing for the result
        self.send_command_lock = Lock()


    def get_thread_name(self):
        return self.name

    
    def loop_turn(self):
        self.get_and_execute_command_from_master()
    
    
    def get_and_execute_command_from_master(self):
        cmd_and_arg = ''
        try:  # NOTE: this thread is not allowed to die
            
            # Will block so we don't hammer cpu
            try:
                request = self.commands_to_q.get(block=True, timeout=1)  # type: Optional[ToModuleCommandRequest]
            except:
                request = None
            # Nothing in the queue, just loop
            if request is None:
                return
            cmd_and_arg = request.get_command()
            _split = cmd_and_arg.split(ARG_SEPARATOR)
            cmd = _split[0]
            arg = _split[1:]
            f = getattr(self.main_process, cmd, None)
            if callable(f):
                self.logger.debug('[PID:%s] Executing command [%s] with param %s' % (os.getpid(), cmd, arg))
                arg_spec = inspect.getargspec(f)
                if arg and len(arg_spec.args) == len(arg):
                    result = f(*arg)
                else:
                    result = f()
                respond = request.create_respond(result)
                self.commands_from_q.put(respond)
            else:
                self.logger.warning('[PID:%s] Received unknown command [%s] from father process !' % (os.getpid(), cmd))
        except:
            self.logger.error('Our father process did send us the command [%s] that did fail because:' % cmd_and_arg)
            self.logger.print_stack()
            time.sleep(0.01)  # if we crash in loop, do not hammer CPU
    
    
    def send_command(self, command_name, args=None, timeout=2):
        # type: (str, List[str], Number) -> Any
        
        with self.send_command_lock:
            try:
                request = ToModuleCommandRequest(command_name, args)
                retry_count = 0
                before = time.time()
                self.commands_to_q.put(request)
                while True:  # will timeout at max 2s because the commands request are block and we are the only one asking for it
                    try:
                        response = self.commands_from_q.get(block=True, timeout=timeout)  # type: FromModuleCommandRespond
                    except EmptyQueue:
                        # Empty? Means a timeout
                        retry_count += 1
                        if retry_count == 1:
                            self.logger.warning('The command call [%s] for module %s was sent but the call did timeout (%ss). We will retry one time.' % (command_name, self.handler_name, timeout))
                            continue
                        # Ok still an error? need to give a real error about the timeout
                        raise
                    if not response.do_match_request(request):
                        self.logger.warning('The command call [%s] was called but another respond was received. Retrying.' % command_name)
                        continue
                    self.logger.debug('The command call [%s] was executed by the module %s in %.3fs' % (command_name, self.handler_name, time.time() - before))
                    payload = response.get_payload()
                    return payload
            
            # Empty queue means a timeout
            except EmptyQueue:
                _message = 'Fail to send command call [%s] for module %s because the module did timeout (%ss)' % (command_name, self.handler_name, timeout)
                self.logger.error(_message)
                raise Exception(_message)
            
            # Another exception? It's not good and must be shown as a stack
            except Exception as e:
                self.logger.print_stack()
                _message = 'Fail to send command call [%s] for module %s because of an unknown error %s' % (command_name, self.handler_name, e.message)
                self.logger.error(_message)
                raise Exception(_message)


class BaseSubProcess(object):
    def __init__(self, name, father_name='UNSET', loop_speed=1, stop_process_on_error=True, only_one_process_by_class=False, error_handler=None):
        # type: (str, str, int, bool, bool, Optional[ErrorHandler]) -> None
        my_class = type(self)
        old_process = get_old_process(my_class)
        if only_one_process_by_class and old_process:
            old_process.stop()
        
        # this part is launch in the father process
        self.name = name
        self.father_name = father_name
        self.father_pid = os.getpid()
        # set initial logger with name. we call _set_logger_name in _sub_process_common_warmup cause _set_logger_name must be call after all init.
        self.logger = LoggerFactory.get_logger(self.name)
        
        self._process = None
        self.interrupted = False
        self.loop_speed = loop_speed
        self._stop_process_on_error = stop_process_on_error
        self._error_handler = error_handler
        
        self.look_at_father_thread = LookAtMyFatherThread(self.father_pid, self.father_name, self.name, logger=self.logger)
        
        register_old_process(my_class, self)
    
    
    def update_father(self, father_name):
        # type: (str) -> NoReturn
        self.father_name = father_name
        self.father_pid = os.getpid()
        self.look_at_father_thread.father_name = self.father_name
        self.look_at_father_thread.father_pid = self.father_pid
    
    
    def manage_signal(self, sig, frame):
        if not ON_LINUX:
            return
        if sig == signal.SIGUSR1:  # if USR1, ask a memory dump
            if sys.version_info[1] == 6:  # python 2.6
                try:
                    from guppy import hpy
                    hp = hpy()
                    self.logger.error('(support-only) MEMORY DUMP (to be sent to the support):\n%s' % hp.heap())
                    return
                except ImportError:
                    self.logger.error('(support-only) MEMORY DUMP: FAIL check if guppy lib is installed')
            if sys.version_info[1] == 7:  # python 2.7
                try:
                    import meliae.scanner
                    import meliae.loader
                    _f = "/tmp/memorydump-%s.json" % self.name
                    meliae.scanner.dump_all_objects(_f)
                    self.logger.error('(support-only) Memory information dumped to file %s (to be sent to the support)' % _f)
                except ImportError:
                    self.logger.error('(support-only) MEMORY DUMP: FAIL check if meliae lib is installed')
        else:
            self.interrupted = True
    
    
    def set_signal_handler(self, sigs=None):
        if not ON_LINUX:
            return
        
        if sigs is None:
            sigs = (signal.SIGINT, signal.SIGTERM, signal.SIGUSR1)
        
        for sig in sigs:
            signal.signal(sig, self.manage_signal)
    
    
    set_exit_handler = set_signal_handler
    
    
    # In the sub process, we should start some threads, like the one that look at parent
    # etc etc. Sub Class can have other threads for special purpose
    def _start_sub_process_threads(self):
        start_malloc_trim_thread()
        self.look_at_father_thread.start_thread()
    
    
    def _set_logger_name(self):
        global logger
        logger_name = self.get_logger_name()
        if isinstance(logger_name, (list, tuple)):
            logger_names = [i.replace('[', '').replace(']', '') for i in logger_name]
            if len(logger_names) > 1:
                _logger_part_names = logger_names[1:]
                my_logger = self.logger
                my_logger.set_default_part(_logger_part_names[0])
                for part_name in _logger_part_names[1:]:
                    my_logger = my_logger.get_sub_part(part_name)
                self.logger = my_logger
                self.look_at_father_thread.logger = my_logger
            logger_name = logger_name[0]
        
        logger_name = logger_name.replace('[', '(').replace(']', ')')
        _logger.set_name(logger_name)
    
    
    def _sub_process_common_warmup(self):
        self._set_logger_name()
        self.logger.debug(PART_INITIALISATION, '_sub_process_common_warmup')
        set_process_name(self.get_process_name())
        
        self.logger.debug(PART_INITIALISATION, 'stop http')
        from http_daemon import daemon_inst
        if daemon_inst:
            # We are shutdown the http daemon because we should not answer to the http daemon calls.
            # NOTE: this is done in a thread
            daemon_inst.shutdown(quiet=True)
        
        self.logger.debug(PART_INITIALISATION, 'set signal handler')
        self.set_signal_handler()
        
        self.logger.debug(PART_INITIALISATION, 'start sub process')
        # Our sub process need some utility threads, like:
        # * malloc trimming
        # * looking at father process (if dead, we die)
        # * others set by the sub class
        self._start_sub_process_threads()
        self.logger.info(PART_INITIALISATION, 'Sub process ready to work.')
    
    
    def get_process_name(self):
        # type: () -> str
        return '%s [ %s ]' % (self.father_name, self.name)
    
    
    def get_logger_name(self):
        # type: () -> str
        return self.get_process_name()
    
    
    def main(self):
        try:
            self._sub_process_common_warmup()
            self.on_init()
        except Exception as e:
            self.interrupted = True
            if self._error_handler:
                try:
                    self._error_handler.handle_exception('Fatal error caused by : %s' % e, e, self.logger, level=ERROR_LEVEL.FATAL)
                except Exception:
                    self.logger.error('Sub-process %s have a fatal error : %s' % (self.name, e))
                    self.logger.print_stack()
            else:
                self.logger.error('Sub-process %s have a fatal error : %s' % (self.name, e))
                self.logger.print_stack()
        
        while not self.interrupted:
            try:
                self.loop_turn()
                self.interruptable_sleep(self.loop_speed)
            except Exception as e:
                if self._error_handler:
                    self._error_handler.handle_exception('Fatal error caused by : %s' % e, e, self.logger, level=ERROR_LEVEL.FATAL)
                else:
                    self.logger.error('Sub-process %s have a fatal error : %s' % (self.name, e))
                    self.logger.print_stack()
                if self._stop_process_on_error:
                    self.interrupted = True
        self.on_close()
    
    
    def interruptable_sleep(self, raw_duration):
        # type: (float) -> NoReturn
        if raw_duration < _UNINTERRUPTABLE_SLEEP_TIME:
            time.sleep(raw_duration)
        
        duration = int(math.floor(raw_duration / _UNINTERRUPTABLE_SLEEP_TIME))
        round_duration = raw_duration - duration
        for _ in xrange(duration):
            time.sleep(_UNINTERRUPTABLE_SLEEP_TIME)
            if self.interrupted:
                break
        
        time.sleep(round_duration)
    
    
    def loop_turn(self):
        # type: () -> NoReturn
        raise NotImplementedError()
    
    
    def on_init(self):
        # type: () -> NoReturn
        pass
    
    
    def on_close(self):
        # type: () -> NoReturn
        pass
    
    
    def join(self):
        # type: () -> NoReturn
        self._process.join()
    
    
    def start(self, father_name=''):
        # type: (str) -> NoReturn
        self._process = Process(target=self.main)
        self.interrupted = False
        if father_name:
            self.update_father(father_name)
        self._process.start()
    
    
    def stop(self):
        # type: () -> NoReturn
        self.logger.info('Stopping process %s' % self.get_process_name())
        if self._process and self._process.is_alive():
            self._process.terminate()
            self._process.join(timeout=1)
            if self._process.is_alive():
                self.logger.info('The [%s] sub-process is still alive, I help it to die' % self.get_process_name())
                self._kill()
    
    
    def _kill(self):
        if os.name == 'nt':
            self._process.terminate()
        else:
            # Ok, let him 1 second before really KILL IT
            os.kill(self._process.pid, signal.SIGTERM)
            time.sleep(1)
            # You do not let me another choice guy...
            try:
                if self._process.is_alive():
                    os.kill(self._process.pid, signal.SIGKILL)
            except AssertionError:  # zombie process
                try:
                    os.kill(self._process.pid, signal.SIGKILL)
                except OSError:
                    pass
