#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

import hashlib

from item import Item, Items
from shinken.log import logger
from shinken.property import StringProp, ListProp, IntegerProp
from shinken.util import strip_and_uniq


class Module(Item):
    id = 1  # zero is always special in database, so we do not take risk here
    my_type = 'module'
    
    properties = Item.properties.copy()
    properties.update({
        'module_name': StringProp(),
        'module_type': StringProp(),
        'modules'    : ListProp(default=''),
    })
    
    running_properties = Item.running_properties.copy()
    running_properties.update({
        'configuration_default_value'                       : StringProp(default={}),
        'minimal_time_before_an_element_become_missing_data': IntegerProp(default=0),
    })
    
    macros = {}
    
    
    def __init__(self, params={}, skip_useless_in_configuration=False):
        super(Module, self).__init__(params=params)
        self.configuration_properties = set(params.keys())
    
    
    def __compute_hash(self):
        hash_ = hashlib.md5()  # don't bother me about blabla md5 secutiry blabla. We don't care here baka
        keys = list(self.configuration_properties)
        keys.sort()
        for k in keys:
            hash_.update(k)
            value = getattr(self, k, '')
            if isinstance(value, basestring):
                hash_.update(value.encode('ascii', 'ignore'))
            else:  # maybe it's a list due to set the key multiple times.
                hash_.update(str(value))
        self.hash = hash_.hexdigest()
    
    
    def set_default_properties_values(self, default_properties_values):
        self.configuration_default_value = default_properties_values
        self.configuration_properties.add('configuration_default_value')
        self.__compute_hash()
    
    
    def set_minimal_time_before_an_element_become_missing_data(self, minimal_time_before_an_element_become_missing_data):
        try:
            self.minimal_time_before_an_element_become_missing_data = int(minimal_time_before_an_element_become_missing_data)
            self.configuration_properties.add('minimal_time_before_an_element_become_missing_data')
            self.__compute_hash()
        except ValueError:
            pass
    
    
    # For debugging purpose only (nice name)
    def get_name(self):
        return self.module_name
    
    
    def get_type(self):
        return getattr(self, 'module_type', '')
    
    
    # For debugging purpose only (nice name)
    def get_modules(self):
        return self.modules


class Modules(Items):
    name_property = "module_name"
    inner_class = Module
    
    
    def fill_running_properties(self, config):
        for module in self:
            module.set_default_properties_values(config.default_properties_values)
            module.set_minimal_time_before_an_element_become_missing_data(config.minimal_time_before_an_element_become_missing_data)
    
    
    def linkify(self):
        self.linkify_s_by_plug()
    
    
    def linkify_s_by_plug(self):
        for s in self:
            new_modules = []
            mods = s.modules.split(',')
            mods = strip_and_uniq(mods)
            for plug_name in mods:
                plug_name = plug_name.strip()
                
                # don't read void names
                if plug_name == '':
                    continue
                
                # We are the modules, we search them :)
                plug = self.find_by_name(plug_name)
                if plug is not None:
                    new_modules.append(plug)
                else:
                    err = "[module] unknown %s module from %s" % (plug_name, s.get_name())
                    logger.error(err)
                    s.configuration_errors.append(err)
            
            s.modules = new_modules
            # We need to update the module hash with the ones from the new modules, because if one
            # submodule change, we should change too
            modules_sorted = sorted(s.modules, key=lambda x: x.get_name())
            hash_compute = hashlib.md5(s.hash)  # start with the current hash as param and not value, but will do the trick
            for m in modules_sorted:
                hash_compute.update(m.hash)
            s.hash = hash_compute.hexdigest()
    
    
    def explode(self):
        pass
