#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#    Thibault Cohen, thibault.cohen@savoirfairelinux.com
#    Francois Mikus, fmikus@acktomic.com
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

""" This class is a common one for service/host. Here you
will find all scheduling related functions, like the schedule
or the consume_check. It's a very important class!
"""

import os
import random
import re
import time
import traceback

from item import InheritableItem
from shinken.acknowledge import Acknowledge
from shinken.check import Check, CHECK_CAUSE, NO_END_VALIDITY, CHECK_STATUS, get_for_retention_callback
from shinken.commandcall import UNSET_POLLER_REACTIONNER_TAG_VALUE
from shinken.dependencynode import DependencyNodeFactory, DependencyNode
from shinken.eventhandler import EventHandler
from shinken.log import logger, get_chapter_string
from shinken.macroresolver import MacroResolver
from shinken.notification import Notification
from shinken.objects.proxyitem import proxyitemsmgr, proxyitemsgraph, root_problems_to_uuids
from shinken.property import BoolProp, IntegerProp, FloatProp, StringProp, EpochProp, RawProp, VirtualProp, LinkToAnotherObjectProp, LinkToOthersObjectsProp
from shinken.rawdata import TYPE_RAW_DATA_CHECK_RESULT, TYPE_RAW_DATA_FULL
from shinken.util import get_active_downtime_uuids
from shinken.util import to_svc_hst_distinct_lists, get_acknowledge_id, to_list_of_names, to_name_if_possible, object_to_deepcopy
from shinken.checks_container import checks_container

# on system time change just reevaluate the following attributes:
on_time_change_update = ('last_notification', 'last_state_change', 'last_hard_state_change')

MONITORING_CHECK_CONSUME_DEBUG_FLAG = os.environ.get('MONITORING_CHECK_CONSUME_DEBUG_FLAG', '0') == '1'
ENABLE_ESCALATION_DEBUG = os.environ.get('SHINKEN_SUPPORT_ENABLE_ESCALATION_DEBUG', '0') == '1'

# #SEF-6735: when downtimes are activated twice
DOWNTIME_INCOHERENCY = get_chapter_string('DOWNTIME-INCOHERENCY')


class CHECK_LAUNCH_MODE(object):
    SCHEDULE = 'schedule'
    PASSIVE = 'passive'
    DEPENDENCY = 'dependency'
    FRESHNESS = 'freshness'


class SchedulingItem(InheritableItem):
    execution_timeout_display_type = ''
    custom_data_format = re.compile(r'[^0-9a-zA-Z_-]')
    
    properties = InheritableItem.properties.copy()
    properties.update({
        
        # Simple properties, can be set into C structure
        'active_checks_enabled'      : BoolProp(default='1', fill_brok=['full_status']),
        'passive_checks_enabled'     : BoolProp(default='1', fill_brok=['full_status']),
        'check_freshness'            : BoolProp(default='0', fill_brok=['full_status']),
        'freshness_threshold'        : IntegerProp(default='0', fill_brok=['full_status']),
        'event_handler_enabled'      : BoolProp(default='0', fill_brok=['full_status']),
        'process_perf_data'          : BoolProp(default='1', fill_brok=['full_status']),
        'notification_interval'      : IntegerProp(default='1440', fill_brok=['full_status']),
        'first_notification_delay'   : IntegerProp(default='0', fill_brok=['full_status']),
        'notifications_enabled'      : BoolProp(default='1', fill_brok=['full_status']),
        'time_to_orphanage'          : IntegerProp(default="300", fill_brok=['full_status']),
        'business_impact'            : IntegerProp(default='2', fill_brok=['full_status']),
        'check_running_timeout'      : IntegerProp(default='-1'),
        'warning_threshold_cpu_usage': IntegerProp(default='-1'),
        'is_cluster'                 : BoolProp(default='0'),
        
        # Configuration Structure, like links to others elements
        'check_period'               : LinkToAnotherObjectProp(default='24x7', brok_transformation=to_name_if_possible, fill_brok=['full_status']),
        'event_handler'              : StringProp(default='', fill_brok=['full_status']),
        'notification_period'        : LinkToAnotherObjectProp(default='24x7', brok_transformation=to_name_if_possible, fill_brok=['full_status']),
        
        'contacts'                   : LinkToOthersObjectsProp(default='', brok_transformation=to_list_of_names, fill_brok=['full_status'], merging='join', handle_additive_inheritance=True),
        'contact_groups'             : LinkToOthersObjectsProp(default='', fill_brok=['full_status'], merging='join', handle_additive_inheritance=True),
        'view_contacts'              : LinkToOthersObjectsProp(default='', brok_transformation=to_list_of_names, fill_brok=['full_status'], merging='join', handle_additive_inheritance=True),
        'view_contact_groups'        : LinkToOthersObjectsProp(default='', fill_brok=['full_status'], merging='join', handle_additive_inheritance=True),
        'notification_contacts'      : LinkToOthersObjectsProp(default='', brok_transformation=to_list_of_names, fill_brok=['full_status'], merging='join', handle_additive_inheritance=True),
        'notification_contact_groups': LinkToOthersObjectsProp(default='', fill_brok=['full_status'], merging='join', handle_additive_inheritance=True),
        'edition_contacts'           : LinkToOthersObjectsProp(default='', brok_transformation=to_list_of_names, fill_brok=['full_status'], merging='join', handle_additive_inheritance=True),
        'edition_contact_groups'     : LinkToOthersObjectsProp(default='', fill_brok=['full_status'], merging='join', handle_additive_inheritance=True),
        
        'notes_url'                  : StringProp(default='', fill_brok=['full_status']),
        'notes_multi_url'            : StringProp(default='', fill_brok=['full_status']),
        
        'poller_tag'                 : StringProp(default=UNSET_POLLER_REACTIONNER_TAG_VALUE),
        'reactionner_tag'            : StringProp(default=UNSET_POLLER_REACTIONNER_TAG_VALUE),
        'resultmodulations'          : LinkToOthersObjectsProp(default='', merging='ordered', handle_additive_inheritance=True),
        'business_impact_modulations': LinkToOthersObjectsProp(default='', merging='ordered', handle_additive_inheritance=True),
        'escalations'                : LinkToOthersObjectsProp(default='', fill_brok=['full_status'], merging='join', handle_additive_inheritance=True),
        'maintenance_period'         : LinkToAnotherObjectProp(default='', brok_transformation=to_name_if_possible, fill_brok=['full_status']),
        'checkmodulations'           : LinkToOthersObjectsProp(default='', fill_brok=['full_status'], merging='ordered'),
        'macromodulations'           : LinkToOthersObjectsProp(default='', merging='ordered', handle_additive_inheritance=True),
        
        'bp_rule'                    : StringProp(default=''),
        
        'sla_warning_threshold'      : FloatProp(default='99', fill_brok=['full_status']),
        'sla_critical_threshold'     : FloatProp(default='97', fill_brok=['full_status']),
    })
    
    # properties used in the running state
    running_properties = InheritableItem.running_properties.copy()
    running_properties.update({
        # Simple properties types
        'in_scheduled_downtime_during_last_check': BoolProp(default=False, retention=True, useless_in_configuration=True),
        'must_respread'                          : BoolProp(default=False, retention=True),
        'last_broker_data_update'                : IntegerProp(default=0),  # just used into the broker to know that the element state is not fresh anymore
        'monitoring_start_time'                  : EpochProp(default=-1L, fill_brok=['full_status']),
        'modified_attributes'                    : IntegerProp(default=0L, fill_brok=['full_status'], retention=True, useless_in_configuration=True),
        'last_chk'                               : EpochProp(default=0, fill_brok=['full_status', 'check_result'], fill_raw_data=[TYPE_RAW_DATA_CHECK_RESULT, TYPE_RAW_DATA_FULL], retention=True, useless_in_configuration=True),
        'last_normal_chk'                        : EpochProp(default=0, retention=True, useless_in_configuration=True),
        'next_chk'                               : EpochProp(default=0, fill_brok=['full_status', 'next_schedule'], retention=True, useless_in_configuration=True),
        'in_checking'                            : BoolProp(default=False, fill_brok=['full_status', 'check_result', 'next_schedule'], useless_in_configuration=True),
        'latency'                                : FloatProp(default=0, fill_brok=['full_status', 'check_result'], retention=True, useless_in_configuration=True),
        'notification_latency'                   : FloatProp(default=0, fill_brok=['full_status', 'check_result'], retention=True, useless_in_configuration=True),
        'eventhandler_latency'                   : FloatProp(default=0, fill_brok=['full_status', 'check_result'], retention=True, useless_in_configuration=True),
        'attempt'                                : IntegerProp(default=0, fill_brok=['full_status', 'check_result'], retention=True, useless_in_configuration=True),
        'state'                                  : StringProp(default='PENDING', fill_brok=['full_status', 'check_result'], retention=True, useless_in_configuration=True),
        'state_type'                             : StringProp(default='HARD', fill_brok=['full_status', 'check_result'], retention=True, useless_in_configuration=True),
        'state_type_id'                          : IntegerProp(default=0, fill_brok=['full_status', 'check_result'], fill_raw_data=[TYPE_RAW_DATA_CHECK_RESULT, TYPE_RAW_DATA_FULL], retention=True, useless_in_configuration=True),
        'context_id'                             : IntegerProp(default=0, fill_raw_data=[TYPE_RAW_DATA_CHECK_RESULT, TYPE_RAW_DATA_FULL], retention=True, useless_in_configuration=True),
        'state_validity_period'                  : IntegerProp(default=0, fill_brok=['full_status', 'check_result'], retention=True, useless_in_configuration=True),
        
        'last_state'                             : StringProp(default='PENDING', fill_brok=['full_status', 'check_result'], retention=True, useless_in_configuration=True),
        'last_state_id'                          : IntegerProp(default=3, fill_brok=['full_status', 'check_result'], retention=True, useless_in_configuration=True),
        
        'last_state_type'                        : StringProp(default='HARD', fill_brok=['full_status', 'check_result'], retention=True, useless_in_configuration=True),
        'last_state_change'                      : EpochProp(default=0, fill_brok=['full_status', 'check_result'], retention=True, useless_in_configuration=True),
        'last_hard_state_change'                 : EpochProp(default=0, fill_brok=['full_status', 'check_result'], retention=True, useless_in_configuration=True),
        'last_hard_state'                        : StringProp(default='PENDING', fill_brok=['full_status'], retention=True, useless_in_configuration=True),
        'last_hard_state_id'                     : IntegerProp(default=0, fill_brok=['full_status'], retention=True, useless_in_configuration=True),
        'last_state_update'                      : EpochProp(default=0, fill_brok=['full_status'], retention=True, useless_in_configuration=True),
        'last_state_as_string'                   : StringProp(default='PENDING', fill_brok=['full_status', 'check_result'], retention=True),
        
        'duration_sec'                           : IntegerProp(default=0, fill_brok=['full_status'], retention=True, useless_in_configuration=True),
        
        'output'                                 : StringProp(default='', fill_brok=['full_status', 'check_result'], retention=True, useless_in_configuration=True),
        'long_output'                            : StringProp(default='', fill_brok=['full_status', 'check_result'], retention=True, useless_in_configuration=True),
        'perf_data'                              : StringProp(default='', fill_brok=['full_status', 'check_result'], fill_raw_data=[TYPE_RAW_DATA_CHECK_RESULT, TYPE_RAW_DATA_FULL], retention=True, useless_in_configuration=True),
        'last_perf_data'                         : StringProp(default='', retention=True, useless_in_configuration=True),
        
        'is_flapping'                            : BoolProp(default=False, fill_brok=['full_status'], retention=True, useless_in_configuration=True),
        'is_partial_flapping'                    : BoolProp(default=False, fill_brok=['full_status', 'check_result'], retention=True, useless_in_configuration=True),
        'inherited_flapping'                     : BoolProp(default=False, fill_brok=['full_status', 'check_result'], retention=True, useless_in_configuration=True),
        'percent_state_change'                   : FloatProp(default=0.0, fill_brok=['full_status', 'check_result'], retention=True, useless_in_configuration=True),
        
        'is_partial_acknowledged'                : BoolProp(default=False, fill_brok=['full_status', 'check_result'], retention=True, useless_in_configuration=True),
        'acknowledgement_type'                   : IntegerProp(default=1, fill_brok=['full_status', 'check_result'], retention=True, useless_in_configuration=True),
        
        'last_notification'                      : EpochProp(default=0, fill_brok=['full_status'], retention=True, useless_in_configuration=True),
        'current_notification_number'            : IntegerProp(default=0, fill_brok=['full_status'], retention=True, useless_in_configuration=True),
        'last_event_handler'                     : EpochProp(default=0, fill_brok=['full_status'], retention=True, useless_in_configuration=True),
        
        'check_type'                             : IntegerProp(default=0, fill_brok=['full_status', 'check_result'], retention=True, useless_in_configuration=True),
        
        'execution_time'                         : FloatProp(default=0.0, fill_brok=['full_status', 'check_result'], retention=True, useless_in_configuration=True),
        'u_time_sum'                             : FloatProp(default=0.0, useless_in_configuration=True),
        's_time_sum'                             : FloatProp(default=0.0, useless_in_configuration=True),
        'nb_executions'                          : IntegerProp(default=0, useless_in_configuration=True),
        
        'scheduled_downtime_depth'               : IntegerProp(default=0, fill_brok=['full_status'], retention=True, useless_in_configuration=True),
        
        'is_problem'                             : BoolProp(default=False, fill_brok=['full_status'], useless_in_configuration=True),
        'is_impact'                              : BoolProp(default=False, fill_brok=['full_status'], useless_in_configuration=True),
        # list of the impact I'm the cause of
        'impacts'                                : RawProp(default=[], useless_in_configuration=True, fill_brok=['full_status'], brok_transformation=to_svc_hst_distinct_lists),
        # list of problems that make us an impact
        'source_problems'                        : RawProp(default=[], fill_brok=['full_status'], brok_transformation=root_problems_to_uuids, useless_in_configuration=True),
        
        # Can be modified to be simple
        'flapping_changes'                       : RawProp(default=[], fill_brok=['full_status'], retention=True, useless_in_configuration=True),
        
        # keep a trace of the old state before being an impact
        'state_before_impact'                    : StringProp(default='PENDING', useless_in_configuration=True),
        # keep a trace of the old state id before being an impact
        'state_id_before_impact'                 : IntegerProp(default=0, useless_in_configuration=True),
        # Same for the output
        'output_before_impact'                   : StringProp(default='', useless_in_configuration=True),
        # if the state change, we know so we do not revert it
        'state_changed_since_impact'             : BoolProp(default=False, useless_in_configuration=True),
        
        # Say if we are business based rule or not
        'got_business_rule'                      : BoolProp(default=False, fill_brok=['full_status', 'check_result'], fill_raw_data=[TYPE_RAW_DATA_FULL, TYPE_RAW_DATA_CHECK_RESULT]),
        
        # the save value of our business_impact for "problems"
        'my_own_business_impact'                 : IntegerProp(default=-1, fill_brok=['full_status'], retention=False, useless_in_configuration=True),
        
        # BPSTate hook
        'bp_state'                               : IntegerProp(default=3, fill_brok=['full_status', 'check_result'], retention=True, useless_in_configuration=True),
        
        # Manage the unknown/unreach during hard state
        # From now its not really used
        'in_hard_unknown_reach_phase'            : BoolProp(default=False, retention=True, useless_in_configuration=True),
        'was_in_hard_unknown_reach_phase'        : BoolProp(default=False, retention=True, useless_in_configuration=True),
        
        # Set if the element just change its father/son topology
        'topology_change'                        : BoolProp(default=False, fill_brok=['full_status'], useless_in_configuration=True),
        
        'consecutive_ok'                         : IntegerProp(default=0, fill_brok=['full_status'], useless_in_configuration=True),
        
        ###################################  Complex structures, like list of objects : Purely scheduler internal part
        
        # No broks for _depend_of because of to much links to hosts/services
        # dependencies for actions like notif of event handler, so AFTER check return
        'act_depend_of'                          : RawProp(default=[]),
        # dependencies for checks raise, so BEFORE checks
        'chk_depend_of'                          : RawProp(default=[]),
        # elements that depend of me, so the reverse than just upper
        'act_depend_of_me'                       : RawProp(default=[]),
        # elements that depend of me
        'chk_depend_of_me'                       : RawProp(default=[]),
        
        # No broks, it's just internal, and checks have too links
        'checks_in_progress'                     : RawProp(default=[], useless_in_configuration=True, retention=True, retention_preparation=get_for_retention_callback),
        'notifications_in_progress'              : RawProp(default={}, retention=True, useless_in_configuration=True, retention_preparation=object_to_deepcopy),
        
        # use for having all contacts we have notified
        # Warning: for the notified_contacts retention save, we save only the names of the contacts, and we should RELINK
        # them when we load it.
        'notified_contacts'                      : RawProp(default=set(), retention=True, retention_preparation=to_list_of_names, useless_in_configuration=True),
        
        # put here checks and notif raised
        'actions'                                : RawProp(default=[], useless_in_configuration=True),
        # and here broks raised
        'broks'                                  : RawProp(default=[], useless_in_configuration=True),
        
        'business_rule'                          : RawProp(default=None),  # Our Dependency node for the business rule
        
        ###################################  Complex structures, like list of objects : must be brok in a way
        'notification_list'                      : RawProp(default=[], fill_brok=['full_status'], retention=True, useless_in_configuration=True),
        'in_maintenance'                         : RawProp(default=None, fill_brok=['full_status'], retention=True, useless_in_configuration=True),  # WARNING: this is not a bool
        'downtimes'                              : RawProp(default=[], fill_brok=['full_status'], retention=True, useless_in_configuration=True),
        'comments'                               : RawProp(default=[], useless_in_configuration=True),
        'active_downtime_uuids'                  : RawProp(default=[], fill_brok=['full_status'], retention=True, useless_in_configuration=True, brok_transformation=get_active_downtime_uuids),
        
        'acknowledgement'                        : RawProp(default=None, fill_brok=['full_status', 'check_result'], retention=True, useless_in_configuration=True),
        'partial_acknowledge'                    : RawProp(default=None, fill_brok=['full_status', 'check_result'], retention=True, useless_in_configuration=True),
        
        # Virtual because it's just for broks
        'acknowledge_id'                         : VirtualProp(default=None, fill_brok=['full_status', 'check_result'], retention=True, brok_transformation=get_acknowledge_id, useless_in_configuration=True),
        
        # Here it's the elements we are depending on so our parents as network relation, or a host
        # we are depending in a hostdependency or even if we are business based.
        'parent_dependencies'                    : RawProp(brok_transformation=to_svc_hst_distinct_lists, default=set(), fill_brok=['full_status']),
        # Here it's the guys that depend on us. So it's the total opposite of the parent_dependencies
        'child_dependencies'                     : RawProp(brok_transformation=to_svc_hst_distinct_lists, default=set(), fill_brok=['full_status']),
        
        # Previously processed business rule (with macro expanded)
        'processed_business_rule'                : StringProp(default="", fill_brok=['full_status']),
        
        # 'timeout'                             : IntegerProp(default=0, fill_brok=['full_status', 'check_result'], retention=True, useless_in_configuration=True),
        # 'start_time'                          : IntegerProp(default=0, fill_brok=['full_status', 'check_result'], retention=True, useless_in_configuration=True),
        # 'end_time'                            : IntegerProp(default=0, fill_brok=['full_status', 'check_result'], retention=True, useless_in_configuration=True),
        # 'early_timeout'                       : IntegerProp(default=0, fill_brok=['full_status', 'check_result'], retention=True, useless_in_configuration=True),
        # 'return_code'                         : IntegerProp(default=0, fill_brok=['full_status', 'check_result'], retention=True, useless_in_configuration=True),
        
    })
    
    
    # Call by pickle to data-ify the host
    # we do a dict because list are too dangerous for
    # retention save and co :( even if it's more
    # extensive
    # The setstate function do the inverse
    def __getstate__(self):
        cls = self.__class__
        # id is not in *_properties
        res = {'id': self.id}
        for prop in cls.properties:
            if hasattr(self, prop):
                res[prop] = getattr(self, prop)
        for prop in cls.running_properties:
            if hasattr(self, prop):
                res[prop] = getattr(self, prop)
        return res
    
    
    # Inversed function of __getstate__
    def __setstate__(self, state):
        cls = self.__class__
        self.id = state['id']
        for prop in cls.properties:
            if prop in state:
                setattr(self, prop, state[prop])
        for prop in cls.running_properties:
            if prop in state:
                setattr(self, prop, state[prop])
    
    
    # Register the son in my child_dependencies, and
    # myself in its parent_dependencies
    def register_son_in_parent_child_dependencies(self, son):
        # So we register it in our list
        self.child_dependencies.add(son)
        
        # and us to its parents
        son.parent_dependencies.add(self)
    
    
    def change_default_values_into_finals(self):
        tag_properties = ('poller_tag', 'reactionner_tag')
        for prop in tag_properties:
            current_value = getattr(self, prop, None)
            if current_value == UNSET_POLLER_REACTIONNER_TAG_VALUE:
                setattr(self, prop, 'None')
    
    
    def check_flapping_change(self, state_id, prev_state_id):
        if self.my_type == 'host' and not self.got_business_rule:
            state_to_state_code = {0: 'o', 1: 'd', 2: 'd', 3: 'u'}
        else:
            state_to_state_code = {0: 'o', 1: 'w', 2: 'c', 3: 'u'}
        state_code = state_to_state_code.get(state_id, 'u')
        
        if state_code in self.flap_detection_options:
            self.add_flapping_change(state_id != prev_state_id)
    
    
    # Add a flapping change, but no more than 20 states
    # Then update the self.is_flapping bool by calling update_flapping
    def add_flapping_change(self, b):
        cls = self.__class__
        
        # If this element is not in flapping check, or
        # the flapping is globally disable, bailout
        if not self.flap_detection_enabled or not cls.enable_flap_detection:
            if self.is_flapping or self.flapping_changes:
                self.flapping_changes = []
                self.is_flapping = False
                b = self.get_update_status_brok()
                self.broks.append(b)
                # Also update the proxy objects that we did change
                proxyitemsmgr.update_flapping(self.get_instance_uuid(), self.is_flapping, inherited_flapping=False)
            return
        
        self.flapping_changes.append(b)
        
        # Keep just 20 changes (global flap_history value)
        flap_history = cls.flap_history
        
        if len(self.flapping_changes) > flap_history:
            self.flapping_changes.pop(0)
        
        # Now we add a value, we update the is_flapping prop
        self.update_flapping()
        # Cluster case: can inherit flapping from sons
        if self.got_business_rule and not self.is_flapping:
            is_flapping, is_partial_flapping = self.business_rule.get_flapping_state()
            b = False
            if self.is_flapping != is_flapping:
                b = True
                self.is_flapping = is_flapping
                inherited_flapping = True if is_flapping else False
                # Update proxy object too
                proxyitemsmgr.update_flapping(self.get_instance_uuid(), is_flapping, inherited_flapping=inherited_flapping)
            if self.is_partial_flapping != is_partial_flapping:
                b = True
                self.is_partial_flapping = is_partial_flapping
                proxyitemsmgr.update_partial_flapping(self.get_instance_uuid(), is_partial_flapping)
            if b:
                b = self.get_update_status_brok()
                self.broks.append(b)
                # if self.process_by_provider:
                #    self.raw_datas.append(b)
    
    
    # We update the is_flapping prop with value in self.flapping_states
    # Old values have less weight than new ones
    def update_flapping(self, automatic=False):
        flap_history = self.__class__.flap_history
        # We compute the flapping change in %
        r = 0.0
        i = 0
        for b in self.flapping_changes:
            if b:
                r += i * (1.2 - 0.8) / (flap_history - 1) + 0.8
            i += 1
        r = r / flap_history
        r *= 100
        
        # We can update our value
        self.percent_state_change = r
        
        # Now we get the low_flap_threshold and high_flap_threshold values
        # They can be from self, or class
        (low_flap_threshold, high_flap_threshold) = (self.low_flap_threshold, self.high_flap_threshold)
        if low_flap_threshold == -1:
            cls = self.__class__
            low_flap_threshold = cls.global_low_flap_threshold
        if high_flap_threshold == -1:
            cls = self.__class__
            high_flap_threshold = cls.global_high_flap_threshold
        
        # Now we check is flapping change, but only if we got enough
        # states to look at the value accuracy
        if self.is_flapping and r < low_flap_threshold:
            self.is_flapping = False
            # We also raise a log entry
            self.raise_flapping_stop_log_entry(r, low_flap_threshold)
            # and a notification
            self.create_notifications('FLAPPINGSTOP')
            # And update our status for modules
            self.broks.append(self.get_update_status_brok())
            # Also update the proxy objects that we did change
            proxyitemsmgr.update_flapping(self.get_instance_uuid(), self.is_flapping, inherited_flapping=self.inherited_flapping)
        
        if not self.is_flapping and r >= high_flap_threshold:
            self.is_flapping = True
            # We also raise a log entry
            self.inherited_flapping = False
            self.raise_flapping_start_log_entry(r, high_flap_threshold)
            # and a notification
            self.create_notifications('FLAPPINGSTART')
            # And update our status for modules
            self.broks.append(self.get_update_status_brok())
            # Also update the proxy objects that we did change
            proxyitemsmgr.update_flapping(self.get_instance_uuid(), self.is_flapping, inherited_flapping=self.inherited_flapping)
        
        if not self.is_flapping and automatic:
            self.is_flapping = True
            # We also raise a log entry
            self.inherited_flapping = True
            self.raise_flapping_start_log_entry(r, high_flap_threshold)
            # and a notification
            self.create_notifications('FLAPPINGSTART')
            # And update our status for modules
            self.broks.append(self.get_update_status_brok())
            # Also update the proxy objects that we did change
            proxyitemsmgr.update_flapping(self.get_instance_uuid(), self.is_flapping, inherited_flapping=self.inherited_flapping)
    
    
    # For a time, print something like
    # 10m 37s
    def _sprintf_duration(self, time_delta, x_elts=2):
        if time_delta is None:
            return 'N/A'
        # print "T", t
        # Get the difference between now and the time of the user
        seconds = time_delta
        
        # Now manage all case like in the past
        seconds = abs(seconds)
        # print "In future?", in_future
        
        # print "sec", seconds
        seconds = long(round(seconds))
        # print "Sec2", seconds
        minutes, seconds = divmod(seconds, 60)
        hours, minutes = divmod(minutes, 60)
        days, hours = divmod(hours, 24)
        weeks, days = divmod(days, 7)
        months, weeks = divmod(weeks, 4)
        years, months = divmod(months, 12)
        
        minutes = long(minutes)
        hours = long(hours)
        days = long(days)
        weeks = long(weeks)
        months = long(months)
        years = long(years)
        
        duration = []
        if years > 0:
            duration.append('%dy' % years)
        else:
            if months > 0:
                duration.append('%dM' % months)
            if weeks > 0:
                duration.append('%dw' % weeks)
            if days > 0:
                duration.append('%dd' % days)
            if hours > 0:
                duration.append('%dh' % hours)
            if minutes > 0:
                duration.append('%dm' % minutes)
            if seconds > 0:
                duration.append('%ds' % seconds)
        
        # print "Duration", duration
        # Now filter the number of printed elements if ask
        if x_elts >= 1:
            duration = duration[:x_elts]
        
        return ' '.join(duration)
    
    
    # IMPORTANT: this method should be used ONLY BY THE BROKER/regenerator so
    # do not send Broks or launch checks or whatever the scheduler is doing
    def look_for_fake_unknown_state_due_to_missing_data_in_broker(self, regenerator, lang, now):
        if self.state_validity_period == NO_END_VALIDITY:
            return
        
        data_age = now - self.last_broker_data_update
        if data_age < 0:  # last check in the future? skip this
            return
        
        # TODO set a new margin
        margin = 60
        freshness_threshold = self.state_validity_period
        if regenerator.minimal_time_before_an_element_become_missing_data:
            freshness_threshold = max(regenerator.minimal_time_before_an_element_become_missing_data, freshness_threshold + margin)
        
        # State is still fresh
        if data_age < freshness_threshold + margin:
            return
        
        now = int(time.time())
        # Ok so now fake the result
        # Save a change only if we really change state
        current_state_id = self.state_id if not self.got_business_rule else self.bp_state
        if current_state_id != 3 or self.state != 'UNKNOWN':
            self.last_state_as_string = self.state
            self.last_state_id = current_state_id
            self.last_state_change = now
            regenerator.item_last_unknown_date[self.instance_uuid] = now
        self.state_type = 'HARD'
        self.state_type_id = 1  # HARD
        self.state = 'UNKNOWN'
        if not self.got_business_rule:
            self.state_id = 3
        else:
            self.bp_state = 3
        time_delta_string = self._sprintf_duration(data_age)
        threshold_string = self._sprintf_duration(freshness_threshold)
        ERROR_MSG = {
            'en': u'<p>WEBUI [ %s ] :<br>The status is set to UNKNOWN because the last check by Shinken Enterprise is %s old (> tolerance threshold is %s).</p><p>It can be a Shinken Enterprise platform problem (like network connection lost).</p>',
            'fr': u'''<p>WEBUI [ %s ] :<br>Le statut est positionné à UNKNOWN car la dernière vérification faite par Shinken Enterprise sur cet élément date de %s ( > seuil de tolérance de %s ).</p><p>Ceci peut provenir d'un problème sur la plateforme Shinken Enterprise (comme une perte de connexion réseau).</p>'''
            
        }
        error_format = ERROR_MSG.get(lang, ERROR_MSG['en'])
        error_text = error_format % (regenerator.import_in, time_delta_string, threshold_string)
        self.output = error_text
        self.long_output = ''
        
        # Invalidation of list cache (see webui-enterprise/plugins/tabular/tabular.py test_remote_tabular)
        if hasattr(self, 'list_cache_var'):
            delattr(self, 'list_cache_var')
    
    
    # Add an attempt but cannot be more than max_check_attempts
    def add_attempt(self):
        self.set_attempt(self.attempt + 1)
    
    
    def set_attempt(self, attempt):
        if attempt == 0:
            raise Exception('BAD set attempt for an element! %s  state=%s  last_state=%s  state_type=%s  max_check_attempts=%s. Stopping daemon to avoid data corruption.' % (
                self.get_full_name(), self.state, self.last_state, self.state_type, self.max_check_attempts))
        new_attempt = min(attempt, self.max_check_attempts)
        if self.attempt != new_attempt:
            if MONITORING_CHECK_CONSUME_DEBUG_FLAG:
                logger.info('[monitoring] The element %30s attempt did change %s => %s (max=%s)' % (self.get_full_name(), self.attempt, new_attempt, self.max_check_attempts))
            self.attempt = new_attempt
    
    
    # Return True if attempt is at max
    def is_max_attempts(self):
        return self.attempt >= self.max_check_attempts
    
    
    # Do we have parents/service dependencies that we must check when we are in error before we send
    # a notification?
    def have_parents_like_dependencies(self):
        return len(self.act_depend_of) != 0
    
    
    # Call by scheduler to see if last state is older than
    # freshness_threshold if check_freshness, then raise a check
    # even if active check is disabled
    def do_check_freshness(self):
        now = time.time()
        # Before, check if class (host or service) have check_freshness OK
        # Then check if item want freshness, then check freshness
        cls = self.__class__
        if not self.in_checking and \
                cls.global_check_freshness and \
                self.check_freshness and \
                self.freshness_threshold != 0 and \
                self.last_state_update < now - (self.freshness_threshold + cls.additional_freshness_latency) and \
                self.passive_checks_enabled and \
                not self.active_checks_enabled:
            
            if self.check_period is None or self.check_period.is_time_valid(now):
                self.raise_freshness_log_entry(int(now - self.last_state_update), int(now - self.freshness_threshold))
                return self.launch_check(now, CHECK_LAUNCH_MODE.FRESHNESS)
            else:
                logger.debug("Should have checked freshness for passive only checked host:%s, but host is not in check period." % self.host_name)
        return None
    
    
    # Raise all impact from my error. I'm setting myself
    # as a problem, and I register myself as this in all
    # hosts/services that depend_on_me. So they are now my
    # impacts
    def set_myself_as_problem(self, enable_impact_state):
        now = time.time()
        
        # If I'm a bp_rule based, I cannot be a real problem, so don't set is_problem=True
        # but update business impact value if need (business impact modulation) and
        # warn fathers about our business imapct value
        if self.got_business_rule:
            # Update business impact value if need
            self.update_business_impact_value()
            return
        
        was_pb = self.is_problem
        self.is_problem = True
        
        # we should warn potentials impact of our problem and they should be cool to register them so I've got my impacts list
        impacts = list(self.impacts)
        for (impact, status, dep_type, tp, inh_par) in self.act_depend_of_me:
            # Check if the status is ok for impact
            for s in status:
                if self.is_state(s):
                    # now check if we should bailout because of a not good timeperiod for dep
                    if tp is None or tp.is_time_valid(now):
                        new_impacts = impact.register_a_problem(self, enable_impact_state)
                        impacts.extend(new_impacts)
        
        # Only update impacts and create new brok if impacts changed and you didn't change the is_problem bool
        s_impacts = set(impacts)
        if was_pb and s_impacts == set(self.impacts):
            return
        self.impacts = list(s_impacts)
        
        # We can update our business_impact value now
        self.update_business_impact_value()
        
        # And we register a new broks for update status
        self.broks.append(self.get_update_status_brok())
    
    
    # We update our 'business_impact' value with the max of
    # the impacts business_impact if we got impacts. And save our 'configuration'
    # business_impact if we do not have do it before
    # If we do not have impacts, we revert our value
    def update_business_impact_value(self):
        # We look at our crit modulations. If one apply, we take apply it and it's done
        in_modulation = False
        for cm in self.business_impact_modulations:
            now = time.time()
            period = cm.modulation_period
            if period is None or period.is_time_valid(now):
                self.business_impact = cm.business_impact
                in_modulation = True
                # We apply the first available, that's all
                break
        
        # Save our updated business_impact if not already done
        if self.my_own_business_impact == -1:
            self.my_own_business_impact = self.business_impact
        
        # If we truly have impacts, we get the max business_impact
        # if it's huge than ourselves
        # also look at the cluster I'm in
        my_clusters_uuids = proxyitemsgraph.father_to_sons.get(self.get_instance_uuid(), [])
        my_clusters_business_impacts = [proxyitemsmgr[c_uuid].business_impact for c_uuid in my_clusters_uuids]
        
        if len(self.impacts) != 0 or len(my_clusters_business_impacts) != 0:
            if len(my_clusters_business_impacts) != 0:
                self.business_impact = max(self.business_impact, max(my_clusters_business_impacts))
            if len(self.impacts) != 0:
                self.business_impact = max(self.business_impact, max([e.business_impact for e in self.impacts]))
            # Update proxy item
            proxyitemsmgr.update_business_impact(self.get_instance_uuid(), self.business_impact)
            return
        
        # If we are not a problem, we setup our own_crit if we are not in a modulation period
        if self.my_own_business_impact != -1 and not in_modulation:
            self.business_impact = self.my_own_business_impact
            # update proxy item too
            proxyitemsmgr.update_business_impact(self.get_instance_uuid(), self.business_impact)
    
    
    # Look for my impacts, and remove me from theirs problems list
    def no_more_a_problem(self):
        was_pb = self.is_problem
        
        if self.is_problem:
            self.is_problem = False
            
            # we warn impacts that we are no more a problem
            for impact in self.impacts:
                impact.deregister_a_problem(self)
            
            # we can just drop our impacts list
            self.impacts = []
        
        # We update our business_impact value, it's not a huge thing :)
        self.update_business_impact_value()
        
        # If we were a problem, we say to everyone
        # our new status, with good business_impact value
        if was_pb:
            # And we register a new broks for update status
            self.broks.append(self.get_update_status_brok())
    
    
    # Call recursively by potentials impacts so they
    # update their source_problems list. But do not
    # go below if the problem is not a real one for me
    # like If I've got multiple parents for examples
    def register_a_problem(self, pb, enable_impact_state):
        # Maybe we already have this problem? If so, bailout too
        if pb in self.source_problems:
            return []
        
        now = time.time()
        was_an_impact = self.is_impact
        # Our father already look of he impacts us.
        # So if we are here, it's that we really are impacted
        self.is_impact = True
        
        impacts = []
        # Ok, if we are impacted, we can add it in our  problem list
        
        # Maybe I was a problem myself, now I can say: not my fault!
        if self.is_problem:
            self.no_more_a_problem()
        
        # Ok, we are now an impact, we should take the good state but only when we just go in impact state
        # Cannot impact cluster, they really compute their states, they don't need a shortcut
        if enable_impact_state and not was_an_impact and not self.got_business_rule:
            self.set_impact_state()
            proxyitemsmgr.update_state(self.get_instance_uuid(), self.state_id)
        
        # Ok now we can be a simple impact
        impacts.append(self)
        if pb not in self.source_problems:
            self.source_problems.append(pb)
            # logger.debug('ROOT PROBLEM: %s got a new root problem: %s (%s)' % (self.get_name(), pb.get_name(), pb.get_instance_uuid()))
            proxyitemsmgr.update_root_problems(self.get_instance_uuid(), [src.get_instance_uuid() for src in self.source_problems])
        # we should send this problem to all potential impact that
        # depend on us
        for (impact, status, dep_type, tp, inh_par) in self.act_depend_of_me:
            # Check if the status is ok for impact
            for s in status:
                if self.is_state(s):
                    # now check if we should bailout because of a
                    # not good timeperiod for dep
                    if tp is None or tp.is_time_valid(now):
                        new_impacts = impact.register_a_problem(pb, enable_impact_state)
                        impacts.extend(new_impacts)
        
        # And we register a new broks for update status
        self.broks.append(self.get_update_status_brok())
        
        # now we return all impacts (can be void of course)
        return impacts
    
    
    # Just remove the problem from our problems list
    # and check if we are still 'impacted'. It's not recursif because problem
    # got the list of all its impacts
    def deregister_a_problem(self, pb):
        self.source_problems.remove(pb)
        proxyitemsmgr.update_root_problems(self.get_instance_uuid(), [src.get_instance_uuid() for src in self.source_problems])
        
        # For know if we are still an impact, maybe our dependencies
        # are not aware of the remove of the impact state because it's not ordered
        # so we can just look at if we still have some problem in our list
        if len(self.source_problems) == 0:
            self.is_impact = False
            # No more an impact, we can unset the impact state
            self.unset_impact_state()
        
        # And we register a new broks for update status
        self.broks.append(self.get_update_status_brok())
    
    
    # When all dep are resolved, this function say if action can be raise or not by viewing dep
    # status network_dep have to be all raise to be no action
    # logic_dep: just one is enough
    def is_root_problem(self):
        # Use to know if notification is raise or not
        parent_is_down = []
        
        # If i have not dependency it my fault
        if not self.have_parents_like_dependencies():
            return True
        
        # So if one logic is Raise, is dep is one network is no ok, is not dep at the end, raise no dep
        for (dep, status, type, tp, inh_par) in self.act_depend_of:
            # For logic_dep, only one state raise put no action
            if type == 'logic_dep':
                for s in status:
                    if dep.is_state(s):
                        return False
            # more complicated: if none of the states are match, the host is down so -> network_dep
            else:
                p_is_down = False
                dep_match = [dep.is_state(s) for s in status]
                # check if the parent match a case, so he is down
                if True in dep_match:
                    p_is_down = True
                parent_is_down.append(p_is_down)
        # if a parent is not down, no dep can explain the pb
        if False in parent_is_down:
            return True
        else:  # every parents are dead, so... It's not my fault :)
            return False
    
    
    # We check if we are no action just because of ours parents (or host for service)
    # TODO: factorize with previous check?
    def check_and_set_unreachability(self):
        # We must have all parents dead to be unreachable
        all_parents_are_down = self.are_all_parents_down()
        
        if all_parents_are_down and self.state != 'UNREACHABLE':
            self.set_unreachable()
    
    
    def are_all_parents_down(self):
        # We have no parents? They are not DOWN :)
        if not self.have_parents_like_dependencies():
            return False
        # We have at least one parent, so if one it not down, we have our answer
        for (dep, status, type, _, _) in self.act_depend_of:
            if type == 'network_dep':
                is_down = any((dep.is_state(s) for s in status))
                if not is_down:
                    return False
        return True
    
    
    # Use to know if I raise dependency for someone else (with status)
    # If I do not raise dep, maybe my dep raise me. If so, I raise dep.
    # So it's a recursive function
    def do_i_raise_dependency(self, status, inherit_parents):
        # Do I raise dep?
        for s in status:
            if self.is_state(s):
                return True
        
        # If we do not inherit parent, we have no reason to be blocking
        if not inherit_parents:
            return False
        
        # Ok, I do not raise dep, but my dep maybe raise me
        now = time.time()
        for (dep, status, type, tp, inh_parent) in self.chk_depend_of:
            if dep.do_i_raise_dependency(status, inh_parent):
                if tp is None or tp.is_time_valid(now):
                    return True
        
        # No, I really do not raise...
        return False
    
    
    # Use to know if my dep force me not to be checked
    # So check the chk_depend_of if they raise me
    def is_no_check_dependent(self):
        now = time.time()
        for (dep, status, type, tp, inh_parent) in self.chk_depend_of:
            if tp is None or tp.is_time_valid(now):
                if dep.do_i_raise_dependency(status, inh_parent):
                    return True
        return False
    
    
    # call by a bad consume check where item see that he have dep
    # and maybe he is not in real fault.
    # Returns: if there were dep checks created
    def raise_dependencies_check(self, my_bad_check):
        now = time.time()
        cls = self.__class__
        raised_dep_checks = []
        for (dep, status, type, tp, inh_par) in self.act_depend_of:
            # If the dep timeperiod is not valid, do not raise the dep,
            # None=everytime
            if tp is not None and not tp.is_time_valid(now):
                continue
            
            # if the update is 'fresh', do not raise dep,
            # cached_check_horizon = cached_service_check_horizon for service
            if dep.last_state_update >= now - cls.cached_check_horizon:
                continue
            
            # If the parent do not have active checks, then it's useless to ask for a command
            # launch
            if not dep.active_checks_enabled:
                continue
            
            # as for a dep check only if the dependence is UP/OK
            OK_UP = dep.__class__.ok_up
            if dep.state != OK_UP and dep.state_type == 'HARD':
                logger.debug("dependency check not launch because we already know that %s is in state %s and state_type %s" % (dep.get_full_name(), dep.state, dep.state_type))
            else:
                i = dep.launch_check(now, CHECK_LAUNCH_MODE.DEPENDENCY, depend_on_me=my_bad_check)
                # logger.debug("dependency check append for %s in state %s %s" % (dep.get_full_name(), dep.state, dep.state_type))
                if i is not None:
                    raised_dep_checks.append(i)
                    if MONITORING_CHECK_CONSUME_DEBUG_FLAG:
                        logger.info('[monitoring] [%s] will now wait for check from %s (check id=%s)' % (self.get_full_name(), dep.get_full_name(), i))
                # else:
                #     logger.warning("dependency check not lauch because we already know that %s is in state %s and state_type %s" % (dep.get_full_name(), dep.state, dep.state_type))
        
        if len(raised_dep_checks) == 0:
            return False  # cannot create dep checks, so our own check will be able to be fully consumed
        
        # Let the check know that we are waiting for parent checks
        my_bad_check.register_dependency_checks(raised_dep_checks)  # -> my bad check goes WAITDEP
        
        # When we are unreachable we need to wait until we get results for all dependent checks
        # before we send the brok ; if one is back up, the result should not be "UNREACHABLE" anymore
        if self.state != 'UNREACHABLE':
            self.broks.append(self.get_check_result_brok())
        if MONITORING_CHECK_CONSUME_DEBUG_FLAG:
            logger.info('[monitoring] The element %-20s check analys is postpone until its parents are checked:  state=%s  last_state=%s  state_type=%s attempt=%s max_check_attempts=%s  because of %s parents' % (
                self.get_full_name(), self.state, self.last_state, self.state_type, self.attempt, self.max_check_attempts, len(self.act_depend_of)))
        
        return True  # we did create dep checks so our check will have to wait
    
    
    # Main scheduling function
    # If a check is in progress, or active check are disabled, do not schedule a check.
    # The check interval change with HARD state or not:
    # * SOFT: retry_interval
    # * HARD: check_interval
    # The first scheduling is evenly distributed, so all checks are not launched at the same time.
    # IMPORTANT: we do all we can to keeep this initial spreading, to not synchonize checks because of
    #            forced checks (dep) or from uis
    def schedule(self, force=False, force_time=None):
        # next_chk il already set, do not change
        # unless we force the check or the time
        if self.in_checking and not (force or force_time):
            return None
        
        cls = self.__class__
        # if no active check and no force, no check
        if (not self.active_checks_enabled or not cls.execute_checks) and not force:
            return None
        
        now = time.time()
        
        # If check_interval is 0, we should not add it for a service but suppose a 5min sched for hosts
        if self.check_interval == 0 and not force:
            if cls.my_type == 'service':
                return None
            else:  # host
                self.check_interval = 300 / cls.interval_length
        
        # used to create the check with the 'retry' cause
        is_a_retry = False
        # Interval change is in a HARD state or not
        # If the retry is 0, take the normal value
        if self.state_type == 'HARD' or self.retry_interval == 0:
            interval = self.check_interval * cls.interval_length
        else:
            interval = self.retry_interval * cls.interval_length
            is_a_retry = True
        
        # Determine when a new check (randomize and distribute next check time) or recurring check should happen.
        if self.next_chk == 0:
            # At the start, we cannot have an interval more than cls.max_check_spread
            # But after the first schedule we must reschedule for spread check with interval > cls.max_check_spread * cls.interval_length
            # This check are identify with must_respread
            if interval > cls.max_check_spread * cls.interval_length:
                interval = cls.max_check_spread * cls.interval_length
                self.must_respread = True
            time_add = interval * random.uniform(0.0, 1.0)
        elif self.must_respread:
            time_add = interval * random.uniform(0.0, 1.0)
            self.must_respread = False
        else:
            time_add = interval
        
        ## Do the actual Scheduling now
        
        # If not force_time, try to schedule
        if force_time is None:
            # If we are hard, use the last hard time for computing, instead of the current self.next_chk that is the last retry one
            if self.state_type == 'HARD':
                self.next_chk = self.last_normal_chk
            
            # Do not calculate next_chk based on current time, but based on the last check execution time.
            # Important for consistency of data for trending.
            if self.next_chk == 0 or self.next_chk is None:
                self.next_chk = now
            
            # If the neck_chk is already in the future, do not touch it.
            next_check_in_valid_check_period = self.check_period and self.check_period.is_time_valid(self.next_chk)
            if self.next_chk <= now or not next_check_in_valid_check_period:
                # maybe we do not have a check_period, if so, take always good (24x7)
                if self.check_period:
                    self.next_chk = self.check_period.get_next_valid_time_from_t(self.next_chk + time_add)
                else:
                    self.next_chk = int(self.next_chk + time_add)
            
            # Maybe we load next_chk from retention and the value of the next_chk is still the past even after add an interval
            # Other case: we load a next_chk that is in distant future because it leaped a time period.
            # Then, evaluate check_period again, it may have changed to a closer one.
            if not (now < self.next_chk < now + interval):
                interval = min(interval, cls.max_check_spread * cls.interval_length)
                time_add = interval * random.uniform(0.0, 1.0)
                
                # if we got a check period, use it, if not, use now
                if self.check_period:
                    self.next_chk = self.check_period.get_next_valid_time_from_t(now + time_add)
                else:
                    self.next_chk = int(now + time_add)
                    # else: keep the self.next_chk value in the future
                # When next_chk is update here (after get a new configuration) the current state validity of the item change without launching check so we update state_validity_period here
                # self.next_chk can be None if self.check_period don't have a valide time
                if self.next_chk is None:
                    self.state_validity_period = NO_END_VALIDITY
                else:
                    self.state_validity_period = self.next_chk - now
        else:
            self.next_chk = int(force_time)
        
        # If next time is None, do not go
        if self.next_chk is None:
            # Nagios do not raise it, I'm wondering if we should
            return None
        
        # if we are hard, save the previous scheduling, but not for forced check
        if self.state_type == 'HARD' and not force and force_time is None:
            self.last_normal_chk = int(self.next_chk)
        
        # Get the command to launch, and put it in queue
        self.launch_check(self.next_chk, CHECK_LAUNCH_MODE.SCHEDULE, force=force, retry=is_a_retry)
    
    
    # If we've got a system time change, we need to compensate it
    # If modify all past value. For active one like next_chk, it's the current
    # checks that will give us the new value
    def compensate_system_time_change(self, difference):
        # We only need to change some value
        for p in on_time_change_update:
            val = getattr(self, p)  # current value
            # Do not go below 1970 :)
            val = max(0, val + difference)  # diff may be negative
            setattr(self, p, val)
    
    
    # For disabling active checks, we need to set active_checks_enabled
    # to false, but also make a dummy current checks attempts so the
    # effect is immediate.
    def disable_active_checks(self):
        self.active_checks_enabled = False
        for c in self.checks_in_progress:
            c.status = CHECK_STATUS.WAITCONSUME
            c.exit_status = self.state_id
            c.output = self.output
            c.check_time = time.time()
            c.execution_time = 0
            c.perf_data = self.perf_data
    
    
    def remove_in_progress_check(self, c):
        # The check is consumed, update the in_checking properties
        if c in self.checks_in_progress:
            self.checks_in_progress.remove(c)
        self.update_in_checking()
    
    
    # Is in checking if and only if there are still checks not consumed
    def update_in_checking(self):
        self.in_checking = (len(self.checks_in_progress) != 0)
    
    
    # Del just a notification that is returned
    def remove_in_progress_notification(self, notification):
        self.last_notification = notification.check_time
        self.notification_latency = max(0, notification.check_time - notification.original_t_to_go)
        notification_id = notification.id
        notification.status = 'zombie'
        if notification_id in self.notifications_in_progress:
            del self.notifications_in_progress[notification_id]
    
    
    # We zombify problems and recovery because a fresher state notification is available
    def remove_in_progress_status_notifications(self):
        for n in self.notifications_in_progress.values():
            if n.type in ('PROBLEM', 'RECOVERY'):
                self.remove_in_progress_notification(n)
    
    
    # Get a event handler if item got an event handler
    # command. It must be enabled locally and globally
    def get_event_handlers(self, externalcmd=False):
        cls = self.__class__
        
        # The external command always pass
        # if not, only if we enable them (auto launch)
        if self.event_handler is None or ((not self.event_handler_enabled or not cls.enable_event_handlers) and not externalcmd):
            return
        
        # If we do not force and we are in downtime, bailout
        # if the no_event_handlers_during_downtimes is 1 in conf
        if cls.no_event_handlers_during_downtimes and not externalcmd and self.is_in_downtime():
            return
        
        m = MacroResolver()
        data = self.get_data_for_event_handler()
        cmd = m.resolve_command(self.event_handler, data)
        rt = self.event_handler.reactionner_tag
        shell_execution = self.event_handler.shell_execution
        command_name = getattr(getattr(self.event_handler, 'command', {}), 'command_name', 'MISSING_NAME')
        
        e = EventHandler(cmd, timeout=cls.event_handler_timeout, ref=self, reactionner_tag=rt, shell_execution=shell_execution, command_name=command_name)
        # print "DBG: Event handler call created"
        # print "DBG: ",e.__dict__
        self.raise_event_handler_log_entry(self.event_handler)
        logger.debug('[EVENTHANDLER] a new event handler %d is created (on the object %s local queue currently)' % (e.id, self.get_full_name()))
        
        # ok we can put it in our temp action queue
        self.actions.append(e)
    
    
    def consume_event_result(self, event):
        self.last_event_handler = event.check_time
        self.eventhandler_latency = max(0, event.check_time - event.original_t_to_go)
    
    
    ##########################################################################################################
    #                                     DOWNTIME                                                           #
    ##########################################################################################################
    
    def _update_in_scheduled_downtime(self):
        if self.scheduled_downtime_depth >= 1:
            self.in_scheduled_downtime = True
        else:  # 0
            self.in_scheduled_downtime = False
    
    
    # A new downtime just enter() so w must start to be in downtime if need
    def increase_downtime_depth(self):
        if self.scheduled_downtime_depth == 0:
            self.raise_enter_downtime_log_entry()
            self.create_notifications('DOWNTIMESTART')
        
        self.scheduled_downtime_depth += 1
        self._update_in_scheduled_downtime()
        # Update our proxy to allow other elements (like custer) to change themselves
        proxyitemsmgr.update_downtime(self.get_instance_uuid(), self.in_scheduled_downtime)
        
        services = getattr(self, 'services', [])
        for s in services:
            proxyitemsmgr.update_in_inherited_downtime(s.get_instance_uuid(), s.is_in_inherited_downtime())
    
    
    # A downtime just exit, decrease the level of downtime, and maybe we are no more in downtime
    def decrease_downtime_depth(self, reason='END'):
        self.scheduled_downtime_depth -= 1
        
        # Manage sub-zero case because old versions did have a cancel bug that make the
        # -1 -2 values possible
        if self.scheduled_downtime_depth <= 0:
            self.scheduled_downtime_depth = 0
            self.raise_exit_downtime_log_entry()
            self.create_notifications('DOWNTIME%s' % reason)
            self._update_in_scheduled_downtime()
            # Update our proxy to allow other elements (like custer) to change themselves
            proxyitemsmgr.update_downtime(self.get_instance_uuid(), self.in_scheduled_downtime)
            
            # when a downtime ends and the host/service was critical a notification is sent with the next critical check
            # So we should set a flag here which signals consume_result to send a notification
            self.in_scheduled_downtime_during_last_check = True
            
            services = getattr(self, 'services', [])
            for s in services:
                proxyitemsmgr.update_in_inherited_downtime(s.get_instance_uuid(), s.is_in_inherited_downtime())
                # Let the service know it was in downtime-inherited and so it it need, it can send new notification now
                s.in_scheduled_downtime_during_last_check = True
    
    
    # Whenever a non-ok hard state is reached, we must check whether this
    # host/service has a flexible downtime waiting to be activated
    def check_for_flexible_downtime(self):
        status_updated = False
        for dt in self.downtimes:
            # activate flexible downtimes (do not activate triggered downtimes)
            if dt.fixed == False and dt.is_in_effect == False and dt.start_time <= self.last_chk and self.state_id != 0 and dt.trigger_id == 0:
                n = dt.enter()  # returns downtimestart notifications
                if n is not None:
                    self.actions.append(n)
                status_updated = True
        if status_updated == True:
            self.broks.append(self.get_update_status_brok())
    
    
    # #SEF-6735: we can have downtime incoherency where a downtime can be enter() twice (I think because of trigger)
    # but I'm not sure about this, so at retention reload, try to detect it and fix it
    def detect_and_repair_downtime_incoherency(self):
        # If no downtime depth, no problem detected
        if self.scheduled_downtime_depth == 0:
            return
        # Bad case is sure when the depth is 1 or more, the count is not
        # the same as the number of active downtime
        nb_active_downtimes = len([dt for dt in self.downtimes if dt.in_scheduled_downtime()])
        if nb_active_downtimes != self.scheduled_downtime_depth:
            logger.error('%s The %s %s has bad downtime values (saved number of downtime=%s, actual=%s). We are fixing the values. Please report it to the support.' % (
                DOWNTIME_INCOHERENCY, self.my_type, self.get_full_name(), self.scheduled_downtime_depth, nb_active_downtimes))
            self.scheduled_downtime_depth = nb_active_downtimes
            self._update_in_scheduled_downtime()  # recompute if we are in downtime or not
    
    
    # UNKNOWN during a HARD state are not so important, and they should
    # not raise notif about it
    def update_hard_unknown_phase_state(self):
        self.was_in_hard_unknown_reach_phase = self.in_hard_unknown_reach_phase
        
        # We do not care about SOFT state at all
        # and we are sure we are no more in such a phase
        if self.state_type != 'HARD' or self.last_state_type != 'HARD':
            self.in_hard_unknown_reach_phase = False
        
        # So if we are not in already in such a phase, we check for
        # a start or not. So here we are sure to be in a HARD/HARD following
        # state
        if not self.in_hard_unknown_reach_phase:
            if self.state == 'UNKNOWN' and self.last_state != 'UNKNOWN' \
                    or self.state == 'UNREACHABLE' and self.last_state != 'UNREACHABLE':
                self.in_hard_unknown_reach_phase = True
                # We also backup with which state we was before enter this phase
                self.state_before_hard_unknown_reach_phase = self.last_state
                return
        else:
            # if we were already in such a phase, look for its end
            if self.state != 'UNKNOWN' and self.state != 'UNREACHABLE':
                self.in_hard_unknown_reach_phase = False
        
        # If we just exit the phase, look if we exit with a different state
        # than we enter or not. If so, lie and say we were not in such phase
        # because we need so to raise a new notif
        if not self.in_hard_unknown_reach_phase and self.was_in_hard_unknown_reach_phase:
            if self.state != self.state_before_hard_unknown_reach_phase:
                self.was_in_hard_unknown_reach_phase = False
    
    
    # consume a check return and send action in return
    # main function of reaction of checks like raise notifications
    # Special case:
    # is_flapping: immediate notif when problem
    # is_in_scheduled_downtime: no notification
    # is_volatile: notif immediately (service only)
    def consume_result(self, check_to_consume):
        if MONITORING_CHECK_CONSUME_DEBUG_FLAG:
            logger.info('[monitoring] We are consuming the check (%s) from %-20s exit_code=%s check.status=%s => element state=%s  last_state=%s  state_type=%s attempt=%s max_check_attempts=%s' % (
                check_to_consume.get_printable_name(), self.get_full_name(), check_to_consume.exit_status, check_to_consume.status, self.state, self.last_state, self.state_type, self.attempt, self.max_check_attempts))
        cls = self.__class__
        OK_UP = cls.ok_up  # OK for service, UP for host
        
        # Just be sure we are not here with an invalid check object
        if not check_to_consume.can_be_consume():
            raise Exception('ERROR: a check execution was trying to be consumed but without a valid state (%s). In order to protect against data integrity damage, the daemon is going to do an emergengy stop' % check_to_consume.status)
        
        # if self.attempt == 0 and self.state != 'PENDING' and check_to_consume.status != CHECK_STATUS.ALL_DEP_ARE_FINISH:
        #     raise Exception('consume_result:: was bad since the begining! %-40s  state=%s  last_state=%s  state_type=%s  max_check_attempts=%s' % (self.get_dbg_name(), self.state, self.last_state, self.state_type, self.max_check_attempts))
        
        # We can make multiple consume_result if the check need to wait a dep (see waitdep state)
        first_consume_result = check_to_consume.status == CHECK_STATUS.WAITCONSUME
        if first_consume_result:
            self._consume_result_data_from_check(check_to_consume)
        
        # if check_to_consume.status == CHECK_STATUS.ALL_DEP_ARE_FINISH:
        #     logger.info('DEPDEPDEP are genious: %s is having a re-consume' % self.get_dbg_name())
        
        # If we got a bad result on a normal check, and we have dep, we raise dep checks
        # put the actual check in waitdep and we return all new checks
        if self.have_parents_like_dependencies() and check_to_consume.need_to_launch_dependency_checks():
            # Make sure the check know about his dep
            # C is my check, and he wants dependencies
            dep_checks_are_created = self.raise_dependencies_check(check_to_consume)
            if dep_checks_are_created:  # our check is now WAITDEP so we need to wait for dep checks to be
                return  # finish before consume again
        
        # Here all our parents are up-to-date, we can continue to consume our check now :)
        # Use to know if notification/event handlers are raise or not
        is_root_problem = self.is_root_problem()
        # logger.info('MIDDLE CONSUME: %s is a root problem at this point? => %s' % (self.get_dbg_name(), is_root_problem))
        # we change the state, do whatever we are or not in an impact mode, we can put it
        self.state_changed_since_impact = True
        
        self.duration_sec = time.time() - self.last_state_change
        
        # We recheck just for network_dep. Maybe we are just unreachable and we need to override the state_id
        if check_to_consume.status == CHECK_STATUS.ALL_DEP_ARE_FINISH:
            self.check_and_set_unreachability()
        
        state_validity_period = check_to_consume.state_validity_period
        state_validity_period_when_go_in_soft = check_to_consume.state_validity_period_when_go_in_soft
        
        # Finish with this check object, take interesting data and clear it
        exit_status = check_to_consume.exit_status
        check_was_a_forced_check_by_a_son = check_to_consume.is_dependent()
        # We are done with this check state and dependencies, we will now only look at exit_status
        check_to_consume.set_fully_consumed()
        # The check is consumed, we do not need anymore in our host/service object
        self.remove_in_progress_check(check_to_consume)
        
        check_to_consume = None  # do not touch this check any more from here!
        
        if exit_status == 0:
            # OK following a previous OK. perfect if we were not in SOFT
            if self.last_state in (OK_UP, 'PENDING'):
                # print "Case 1 (OK following a previous OK): code:%s last_state:%s" % (c.exit_status, self.last_state)
                self.unacknowledge_problem()
                
                # If first run, then allow to launch the event_handler
                if self.last_state == 'PENDING':
                    self.get_event_handlers()
                
                # action in return can be notification or other checks (dependencies)
                if (self.state_type == 'SOFT') and self.last_state != 'PENDING':
                    if self.is_max_attempts():
                        self.state_type = 'HARD'
                else:
                    self.set_attempt(1)  # reset attempts
                    self.state_type = 'HARD'
            # OK following a NON-OK.
            else:
                self.unacknowledge_problem()
                # print "Case 2 (OK following a NON-OK): code:%s last_state:%s" % (c.exit_status, self.last_state)
                if self.state_type == 'SOFT':
                    # OK following a NON-OK still in SOFT state
                    if not check_was_a_forced_check_by_a_son:
                        self.add_attempt()
                    self.raise_alert_log_entry()
                    # Eventhandler gets OK;SOFT;++attempt, no notification needed
                    self.get_event_handlers()
                    # Internally it is a hard OK
                    self.state_type = 'HARD'
                    self.set_attempt(1)  # reset attempts
                elif self.state_type == 'HARD':
                    # OK following a HARD NON-OK
                    self.raise_alert_log_entry()
                    # Eventhandler and notifications get OK;HARD;maxattempts
                    # Ok, so current notifications are not needed, we 'zombie' them
                    self.remove_in_progress_status_notifications()
                    if is_root_problem:
                        self.create_notifications('RECOVERY')
                    self.get_event_handlers()
                    # Internally it is a hard OK
                    self.state_type = 'HARD'
                    self.set_attempt(1)  # reset attempts
                    
                    # self.update_hard_unknown_phase_state()
                    # I'm no more a problem if I was one
                    self.no_more_a_problem()
        # Bad state
        elif exit_status != 0:
            # Volatile part, Only for service, print "Case 3 (volatile only)"
            if getattr(self, 'is_volatile', False):
                # There are no repeated attempts, so the first non-ok results are in a hard state
                self.set_attempt(1)  # reset attempts
                self.state_type = 'HARD'
                # status != 0 so add a log entry (before actions that can also raise log
                # it is smarter to log error before notification)
                self.raise_alert_log_entry()
                self.check_for_flexible_downtime()
                self.remove_in_progress_status_notifications()
                if is_root_problem:
                    self.create_notifications('PROBLEM')
                # Ok, event handlers here too
                self.get_event_handlers()
                
                # PROBLEM/IMPACT
                # I'm a problem only if I'm the root problem
                if is_root_problem:
                    self.set_myself_as_problem(enable_impact_state=True)
            
            # NON-OK follows OK. Everything was fine, but now trouble is ahead
            elif self.last_state in (OK_UP, 'PENDING'):
                # print "Case 4: NON-OK follows OK: code:%s last_state:%s" % (c.exit_status, self.last_state)
                if self.is_max_attempts():
                    # if max_attempts == 1 we're already in deep trouble
                    self.state_type = 'HARD'
                    self.raise_alert_log_entry()
                    self.remove_in_progress_status_notifications()
                    self.check_for_flexible_downtime()
                    if is_root_problem:
                        self.create_notifications('PROBLEM')
                    # Oh? This is the typical go for a event handler :)
                    self.get_event_handlers()
                    
                    # PROBLEM/IMPACT
                    # I'm a problem only if I'm the root problem,
                    # so not no_action:
                    if is_root_problem:
                        self.set_myself_as_problem(enable_impact_state=True)
                
                else:
                    # This is the first NON-OK result. Initiate the SOFT-sequence
                    # Also launch the event handler, he might fix it.
                    self.set_attempt(1)  # reset attempts
                    self.state_type = 'SOFT'
                    self.raise_alert_log_entry()
                    self.get_event_handlers()
            
            # If no OK in a no OK: if hard, still hard, if soft, check at self.max_check_attempts
            # when we go in hard, we send notification
            elif self.last_state != OK_UP:
                # print "Case 5 (no OK in a no OK): code:%s last_state:%s state_type:%s" % (c.exit_status, self.last_state,self.state_type)
                if self.state_type == 'SOFT':
                    if not check_was_a_forced_check_by_a_son:
                        self.add_attempt()
                    if self.is_max_attempts():
                        # Ok here is when we just go to the hard state
                        self.state_type = 'HARD'
                        self.raise_alert_log_entry()
                        self.remove_in_progress_status_notifications()
                        # There is a request in the Nagios trac to enter downtimes
                        # on soft states which does make sense. If this becomes
                        # the default behavior, just move the following line
                        # into the else-branch below.
                        self.check_for_flexible_downtime()
                        if is_root_problem:
                            self.create_notifications('PROBLEM')
                        # So event handlers here too
                        self.get_event_handlers()
                        
                        # PROBLEM/IMPACT
                        # I'm a problem only if I'm the root problem,
                        # so not no_action:
                        if is_root_problem:
                            self.set_myself_as_problem(enable_impact_state=True)
                    
                    else:
                        self.raise_alert_log_entry()
                        # eventhandler is launched each time during the soft state
                        self.get_event_handlers()
                # Bad following a bad/HARD
                else:
                    # print "Case 6 (else):", self.get_full_name()
                    # Send notifications whenever the state has changed. (W -> C)
                    # but not if the current state is UNKNOWN (hard C-> hard U -> hard C should not restart notifications)
                    if self.state != self.last_state:
                        self.update_hard_unknown_phase_state()
                        # print self.last_state, self.last_state_type, self.state_type, self.state
                        if not self.in_hard_unknown_reach_phase and not self.was_in_hard_unknown_reach_phase:
                            self.unacknowledge_problem_if_not_sticky()
                            self.raise_alert_log_entry()
                            self.remove_in_progress_status_notifications()
                            if is_root_problem:
                                self.create_notifications('PROBLEM')
                    
                    elif self.in_scheduled_downtime_during_last_check == True:
                        # during the last check i was in a downtime.
                        # but now the status is still critical and notifications are possible again. send an alert immediately
                        self.remove_in_progress_status_notifications()
                        if is_root_problem:
                            self.create_notifications('PROBLEM')
                    
                    # PROBLEM/IMPACT
                    # Forces problem/impact registration even if no state change
                    # was detected as we may have a non OK state restored from
                    # retention data. This way, we rebuild problem/impact hierarchy.
                    # I'm a problem only if I'm the root problem,
                    # so not no_action:
                    if is_root_problem:
                        self.set_myself_as_problem(enable_impact_state=True)
        
        self.update_hard_unknown_phase_state()
        # Reset this flag. If it was true, actions were already taken
        self.in_scheduled_downtime_during_last_check = False
        
        if self.state_type == 'SOFT' and state_validity_period_when_go_in_soft is not None:
            state_validity_period = state_validity_period_when_go_in_soft
        
        # If the expiration_time is in a no-valid timeperiod, we have to set the state_validity_period to the next valid timeperiod + check_interval
        # Warning : the next_chk is the last_launch because it was not re-schedule (we are consuming the results)
        expiration_time = self.next_chk + state_validity_period
        if self.check_period is not None and not self.check_period.is_time_valid(expiration_time):
            next_valid_timeperiod = self.check_period.get_next_valid_time_from_t(expiration_time)
            if next_valid_timeperiod is None:
                state_validity_period = NO_END_VALIDITY
            else:
                state_validity_period = next_valid_timeperiod - self.next_chk + (self.check_interval * cls.interval_length)
        
        # logger.debug('[SLA] self.state_type:[%s] state_validity_period:[%s] in check HARD:[%s] SOFT[%s] ' % (self.state_type, state_validity_period, check_to_consume.state_validity_period, check_to_consume.state_validity_period_when_go_in_soft))
        self.state_validity_period = state_validity_period
        
        # now is the time to update state_type_id and our last_hard_state
        if self.state_type == 'HARD':
            self.state_type_id = 1
            self.last_hard_state = self.state
            self.last_hard_state_id = self.state_id
        else:
            self.state_type_id = 0
        
        # Fill last_hard_state_change to now
        # if we just change from SOFT->HARD or
        # in HARD we change of state (Warning->critical, or critical->ok, etc etc)
        if self.state_type == 'HARD' and (self.last_state_type == 'SOFT' or self.last_state != self.state):
            self.last_hard_state_change = int(time.time())
        
        self.broks.append(self.get_check_result_brok())
        
        if MONITORING_CHECK_CONSUME_DEBUG_FLAG:
            logger.info('[monitoring] Check return analysis is done. The element %-20s is now state=%s  last_state=%s  state_type=%s attempt=%s max_check_attempts=%s' % (
                self.get_full_name(), self.state, self.last_state, self.state_type, self.attempt, self.max_check_attempts))
    
    
    def _consume_result_data_from_check(self, check_to_consume):
        OK_UP = self.__class__.ok_up  # OK for service, UP for host
        
        # Protect against bad type output/longoutput and perfdata: if str, go in unicode
        check_to_consume.assert_unicode_strings()
        # Latency can be <0 is we get a check from the retention file so if <0, set 0
        try:
            self.latency = max(0, check_to_consume.check_time - check_to_consume.original_t_to_go)
        except TypeError:
            pass
        # Now get data from check
        self.last_chk = int(check_to_consume.check_time)
        # Get output and forgot bad UTF8 values for simple str ones (we can get already unicode with external commands)
        self.output = check_to_consume.output
        self.long_output = check_to_consume.long_output
        # Set the check result type also in the host/service
        # 0 = result came from an active check
        # 1 = result came from a passive check
        self.check_type = check_to_consume.check_type
        # Get the perf_data only if we want it in the configuration
        if self.__class__.process_performance_data and self.process_perf_data:
            self.last_perf_data = self.perf_data
            self.perf_data = check_to_consume.perf_data
        # Let the check modulate based on our own result modulations. Will be done only once
        check_to_consume.apply_result_modulations(self.resultmodulations)
        # remember how we was before this check
        self.last_state_type = self.state_type
        # keep a trace of the old state_id
        old_state_id = self.state_id
        old_state_as_string = self.state
        if self.got_business_rule:
            old_state_id = self.bp_state
        if self.state == 'PENDING':
            old_state_id = 3
        self.set_state_from_exit_status(check_to_consume.exit_status)
        state_id = self.state_id
        if self.got_business_rule:
            state_id = self.bp_state
        if state_id != old_state_id or self.last_state == 'PENDING' or self.state != old_state_as_string:
            self.last_state_change = self.last_state_update
            self.last_state_id = old_state_id
            self.last_state_as_string = old_state_as_string
        self.check_flapping_change(state_id, old_state_id)
        # logger.debug("[scheduler] [%s] phase:set_state_from_exit_status new item state [%s-%s] last_state_change[%s] " % (self.get_full_name(), self.state_id, self.state, self.last_state_change))
        self.consecutive_ok = self.consecutive_ok + 1 if self.state == OK_UP else 0
        # Now get real stats from our check, as we can really consume it once and only once here
        self.execution_time = check_to_consume.execution_time
        # Get the user/system execution ime for this
        # execution, and sum all
        self.u_time_sum += check_to_consume.get_cpu_time()
        self.s_time_sum += check_to_consume.s_time
        self.nb_executions += 1
    
    
    # Just update the notification command by resolving Macros
    # And because we are just launching the notification, we can say
    # that this contact have been notified
    def update_notification_command(self, n):
        cls = self.__class__
        m = MacroResolver()
        data = self.get_data_for_notifications(n.contact, n)
        n.command = m.resolve_command(n.command_call, data, allow_cache=False)
        if cls.enable_environment_macros or n.enable_environment_macros:
            n.env = m.get_env_macros(data)
    
    
    # See if an escalation is eligible at t and notif nb=n
    def is_escalable(self, n):
        cls = self.__class__
        
        # We search since when we are in notification for escalations
        # that are based on time
        in_notif_time = time.time() - n.creation_time
        
        # Check is an escalation match the current_notification_number
        for es in self.escalations:
            if es.is_eligible(n.t_to_go, self.state, n.notif_nb, in_notif_time, cls.interval_length):
                return True
        
        return False
    
    
    # Give for a notification the next notification time
    # by taking the standard notification_interval or ask for
    # our escalation if one of them need a smaller value to escalade
    def get_next_notification_time(self, notification, from_time=None):
        now = time.time()
        cls = self.__class__
        
        if from_time is None:
            from_time = notification.t_to_go
        
        creation_time = notification.creation_time
        
        # Look at the minimum notification interval
        notification_interval = self.notification_interval
        # and then look for currently active notifications, and take notification_interval
        # if filled and less than the self value
        in_notif_time = time.time() - creation_time
        for escalation in self.escalations:
            if escalation.is_eligible(from_time, self.state, notification.notif_nb, in_notif_time, cls.interval_length):
                if escalation.notification_interval != -1 and escalation.notification_interval != 0 and escalation.notification_interval < notification_interval:
                    if ENABLE_ESCALATION_DEBUG:
                        logger.debug('ESCALATION: [%s] The escalation %s is eligible and with a short notification interval, so we are using its interval: %s.' % (self.get_full_name(), escalation.get_name(), escalation.notification_interval))
                    notification_interval = escalation.notification_interval
        
        # So take the by default time
        std_time = from_time + notification_interval * cls.interval_length
        
        # Maybe the notification comes from retention data and next notification alert is in the past
        # if so let use the now value instead
        if std_time < now:
            std_time = now + notification_interval * cls.interval_length
        
        # standard time is a good one
        next_notification_time = std_time
        
        for escalation in self.escalations:
            # If the escalation was already raised, we do not look for a new "early start"
            if escalation.get_name() not in notification.already_start_escalations:
                r = escalation.get_next_notif_time(std_time, self.state, creation_time, cls.interval_length)
                # If we got a real result (time base escalation), and earlier that we previously have, take it
                if r is not None and now < r < next_notification_time:
                    if ENABLE_ESCALATION_DEBUG:
                        logger.debug('ESCALATION: [%s] The escalation %s give us a valid start time (%s). Looking if other escalations are giving us earlier time.' % (self.get_full_name(), escalation.get_name(), time.asctime(time.localtime(r))))
                    next_notification_time = r
            # if notification_interval of escalation is set to 0 we set next notification to infinite
            elif escalation.notification_interval == 0:
                if ENABLE_ESCALATION_DEBUG:
                    logger.debug('ESCALATION: [%s] The escalation %s have no notification interval, so disabling next notification.' % (self.get_full_name(), escalation.get_name()))
                next_notification_time = now + 525600 * 60 * 100
        
        # And we take the minimum of this result. Can be standard or escalation asked
        return next_notification_time
    
    
    # Get all contacts (uniq) from eligible escalations
    def get_escalable_contacts(self, n):
        cls = self.__class__
        
        # We search since when we are in notification for escalations
        # that are based on this time
        in_notif_time = time.time() - n.creation_time
        
        contacts = set()
        for es in self.escalations:
            if es.is_eligible(n.t_to_go, self.state, n.notif_nb, in_notif_time, cls.interval_length):
                contacts.update(es.contacts)
                # And we tag this escalations as started now
                n.already_start_escalations.add(es.get_name())
        
        return list(contacts)
    
    
    # Create a "master" notification here,
    # which will later (immediately before the reactionner gets it)
    # be split up in many "child" notifications, one for each contact.
    def create_notifications(self, type, t_wished=None, ack_data='', ack_author=''):
        cls = self.__class__
        # t_wished==None for the first notification launch after consume  here we must look at the self.notification_period
        
        if t_wished is None:
            now = time.time()
            t_wished = now
            # if first notification, we must add first_notification_delay
            if self.current_notification_number == 0 and type == 'PROBLEM':
                last_time_non_ok_or_up = self.last_time_non_ok_or_up()
                if last_time_non_ok_or_up == 0:
                    # this happens at initial
                    t_wished = now + self.first_notification_delay * cls.interval_length
                else:
                    t_wished = last_time_non_ok_or_up + self.first_notification_delay * cls.interval_length
            if self.notification_period is None:
                t = int(now)
            else:
                t = self.notification_period.get_next_valid_time_from_t(t_wished)
        else:
            # We follow our order
            t = t_wished
        
        if self.notification_is_blocked_by_item(type, t_wished) and self.first_notification_delay == 0 and self.notification_interval == 0:
            # If notifications are blocked on the host/service level somehow
            # and repeated notifications are not configured,
            # we can silently drop this one
            return
        # The current_notification_number of the item itself will only be change when this notification (or its children) have actually be sent.
        if type == 'PROBLEM':
            # Probleme notification incremented notification_number.
            next_notif_nb = self.current_notification_number + 1
        
        elif type == 'RECOVERY':
            # Recovery resets the notification counter to zero
            self.current_notification_number = 0
            next_notif_nb = self.current_notification_number
        else:
            # downtime/flap/etc do not change the notification number
            next_notif_nb = self.current_notification_number
        
        n = Notification(type=type,
                         status='scheduled',
                         command='VOID',
                         command_call=None,
                         ref=self,
                         contact=None,
                         t_to_go=t,
                         timeout=cls.notification_timeout,
                         notif_nb=next_notif_nb,
                         ack_author=ack_author,
                         ack_data=ack_data,
                         command_name="MASTER_NOTIFICATION")
        
        # Keep a trace in our notifications queue
        self.notifications_in_progress[n.id] = n
        # and put it in the temp queue for scheduler
        self.actions.append(n)
    
    
    # In create_notifications we created a notification "template" => master_notification.
    # When it's time to hand it over to the reactionner, this master notification needs
    # to be split in several child notifications, one for each contact
    # To be more exact, one for each contact who is willing to accept
    # notifications of this type and at this time
    def scatter_notification(self, master_notification):
        cls = self.__class__
        child_notifications = []
        
        if master_notification.contact:
            # only master notifications can be split up
            return []
        if master_notification.type == 'RECOVERY':
            if self.first_notification_delay != 0 and len(self.notified_contacts) == 0:
                # Recovered during first_notification_delay. No notifications
                # have been sent yet, so we keep quiet
                contacts = []
            else:
                # The old way.
                # Only send recover notifications to those contacts who also got problem notifications
                contacts = list(self.notified_contacts)
            self.notified_contacts.clear()
        
        else:
            # Check is an escalation match. If yes, get all contacts from escalations
            if self.is_escalable(master_notification):
                contacts = self.get_escalable_contacts(master_notification)
            # else take normal contacts
            else:
                contacts = self.contacts
        
        for contact in contacts:
            # We do not want to notify again a contact with notification interval == 0 that has been already notified.
            # Can happen when a service exit a dowtime and still in crit/warn (and not ack)
            if master_notification.type == "PROBLEM" and self.notification_interval == 0 and contact in self.notified_contacts:
                continue
            # Get the property name for notif commands, like service_notification_commands for service
            notif_commands = contact.get_notification_commands(cls.my_type)
            
            for cmd in notif_commands:
                rt = cmd.reactionner_tag
                shell_execution = cmd.shell_execution
                command_name = getattr(getattr(cmd, 'command', {}), 'command_name', 'MISSING_NAME')
                child_n = Notification(type=master_notification.type,
                                       status='scheduled',
                                       command='VOID',
                                       command_call=cmd,
                                       ref=self,
                                       contact=contact,
                                       t_to_go=master_notification.t_to_go,
                                       timeout=cls.notification_timeout,
                                       notif_nb=master_notification.notif_nb,
                                       reactionner_tag=rt,
                                       module_type=cmd.module_type,
                                       enable_environment_macros=cmd.enable_environment_macros,
                                       shell_execution=shell_execution,
                                       ack_data=master_notification.ack_data,
                                       ack_author=master_notification.ack_author,
                                       command_name=command_name)
                if not self.notification_is_blocked_by_contact(child_n, contact):
                    # Update the notification with fresh status information
                    # of the item. Example: during the notification_delay
                    # the status of a service may have changed from WARNING to CRITICAL
                    self.update_notification_command(child_n)
                    self.raise_notification_log_entry(child_n)
                    self.notifications_in_progress[child_n.id] = child_n
                    child_notifications.append(child_n)
                    
                    # New contact in displayed notification
                    if master_notification.type == 'PROBLEM':
                        # Remember the contacts. We might need them later in the
                        # recovery code some lines above
                        self.notified_contacts.add(contact)
        if child_notifications:
            # After a non-recovery notification, we need 20 checks to close the incident. Reset the counter if not a recovery.
            if master_notification.type != 'RECOVERY':
                self.consecutive_ok = 0
            # New incident bloc
            if not self.notification_list or 'bloc_end' in self.notification_list[-1]:
                self.notification_list.append({'bloc_start': time.time(), 'bloc_content': []})
            last_bloc = self.notification_list[-1]
            # New notification in bloc
            last_bloc['bloc_content'].append({'type': master_notification.type, 'time': time.time(), 'contacts': list(set(c.contact.contact_name for c in child_notifications)), 'status': self.state})
        return child_notifications
    
    
    # Filter to drop incidents that are older than a week
    # but keep not close one
    @staticmethod
    def _is_incident_ok_to_keep(incident):
        return ('bloc_end' not in incident) or ('bloc_end' in incident and incident['bloc_end'] >= time.time() - (7 * 86400))
    
    
    def clean_incidents(self):
        self.notification_list = filter(self._is_incident_ok_to_keep, self.notification_list)
        # End the incident with the newest notification time
        if self.consecutive_ok >= 20 and self.notification_list:
            self.notification_list[-1]['bloc_end'] = max([notif['time'] for notif in self.notification_list[-1]['bloc_content']])
    
    
    # someone want to hook in a existing check,
    def _find_in_progress_check_we_can_hot_hook(self):
        return next((check for check in self.checks_in_progress if check.can_be_hot_hook_for_dependency()))
    
    
    # Add a check to check the host/service and return id of the check
    def launch_check(self, t_to_go, check_launch_mode, depend_on_me=None, force=False, retry=False):
        check = None
        cls = self.__class__
        
        # Look if we are in check or not
        self.update_in_checking()
        
        now = int(time.time())
        
        # the check is being forced, so we just replace next_chk time by now
        if force and self.in_checking:
            check_in_progress = self.checks_in_progress[0]
            check_in_progress.force_time_change(now)
            # Force it to run now, so change both computed and current time to go
            checks_container.do_index_job_execution(check_in_progress)
            return check_in_progress.id
        
        # Dependency check, we have to create a new check that will be launched only once (now)
        # Otherwise it will delay the next real check. this can lead to an infinite SOFT state.
        if check_launch_mode == CHECK_LAUNCH_MODE.DEPENDENCY and not force and self.in_checking:
            hot_hookable_in_progress_check = self._find_in_progress_check_we_can_hot_hook()
            if hot_hookable_in_progress_check:
                was_rescheduled = hot_hookable_in_progress_check.hot_hook_for_dependency(depend_on_me)
                if was_rescheduled:
                    # Let the scheduler know we did change the parent check time, so it can fast index it
                    checks_container.do_index_job_execution(hot_hookable_in_progress_check)
                return hot_hookable_in_progress_check.id
            else:  # no current checks are hookable, so we must recreate them by copy it
                check_in_progress = self.checks_in_progress[0]  # 0 is OK because in_checking is True
                # c_in_progress has almost everything we need but we cant copy.deepcopy() it we need another c.id
                command_line = check_in_progress.command
                timeout = check_in_progress.timeout
                timeout_from = check_in_progress.timeout_from
                poller_tag = check_in_progress.poller_tag
                env = check_in_progress.env
                module_type = check_in_progress.module_type
                shell_execution = check_in_progress.shell_execution
                warning_threshold_cpu_usage = check_in_progress.warning_threshold_cpu_usage
                
                if hasattr(self, 'host') and hasattr(self, 'service_description') and self.host and hasattr(self.host, 'host_name'):
                    command_name = "%s-//-%s" % (self.host.host_name, self.service_description)
                else:
                    command_name = self.host_name
                command_name += "-//-%s" % check_in_progress.command_name
                check = Check('scheduled', command_line, self, t_to_go,
                              depend_on_me=depend_on_me,
                              timeout=timeout,
                              timeout_from=timeout_from,
                              poller_tag=poller_tag,
                              env=env,
                              module_type=module_type,
                              dependency_check=True,
                              shell_execution=shell_execution,
                              command_name=command_name,
                              cause=CHECK_CAUSE.DEPENDENCY,
                              warning_threshold_cpu_usage=warning_threshold_cpu_usage)
                
                self.checks_in_progress.append(check)
                self.actions.append(check)
                # A new check means the host/service changes its next_check need to be refreshed
                self.broks.append(self.get_next_schedule_brok())
                # print "Creating new check with new id : %d, old id : %d" % (c.id, c_in_progress.id)
                return check.id
        
        if not self.is_no_check_dependent() or force:
            if check_launch_mode == CHECK_LAUNCH_MODE.DEPENDENCY and self.my_type == 'host' and self.passive_checks_enabled and not self.active_checks_enabled:
                logger.debug("Host check is for an host that is only passively checked (%s), do not launch the check !" % self.host_name)
                return None
            
            # By default we will use our default check_command
            check_command = self.check_command
            # But if a checkmodulations is available, use this one instead.
            # Take the first available
            for checkmodulation in self.checkmodulations:
                command_checkmodulation = checkmodulation.get_check_command(t_to_go)
                if command_checkmodulation:
                    check_command = command_checkmodulation
                    break
            
            # Get the command to launch
            macro_resolver = MacroResolver()
            data = self.get_data_for_checks()
            
            # By default env is void
            env = {}
            # Maybe we are in a forced passive checks, then there can be no command here
            if check_command is not None:
                command_line = macro_resolver.resolve_command(check_command, data)
                # And get all environment variables only if needed
                if cls.enable_environment_macros or check_command.enable_environment_macros:
                    env = macro_resolver.get_env_macros(data)
            else:
                command_line = '(no command)'
            
            # By default we take the global timeout, but we use the command one if it define it (by default it's -1)
            (timeout, timeout_from) = self._get_command_timeout()
            warning_threshold_cpu_usage = self._get_command_warning_threshold_cpu_usage()
            
            if hasattr(self, 'host') and hasattr(self, 'service_description') and self.host and hasattr(self.host, 'host_name'):
                command_name = "%s-//-%s" % (self.host.host_name, self.service_description)
            else:
                command_name = self.host_name
            
            if self.check_command:
                command_name += "-//-%s" % self.check_command.command.command_name
            
            # the cause is a RETRY id retry is passed as param
            default_check_cause = CHECK_CAUSE.RETRY if retry else CHECK_CAUSE.SCHEDULE
            # if the check was force, set the cause to force, else use the default cause (schedule or retry)
            check_cause = CHECK_CAUSE.FORCE if force else default_check_cause
            
            # if force we don't need the computed t, we set the time.time() so the check will be done ASAP
            if force:
                t_to_go = now
            poller_tag = check_command.poller_tag if check_command else 'None'
            module_type = check_command.module_type if check_command else 'fork'
            shell_execution = check_command.shell_execution if check_command else False
            # Make the Check object and put the service in checking
            # Make the check inherit poller_tag from the command
            # And reactionner_tag too
            check = Check('scheduled', command_line, self, t_to_go,
                          depend_on_me=depend_on_me,
                          timeout=timeout,
                          timeout_from=timeout_from,
                          poller_tag=poller_tag,
                          env=env,
                          module_type=module_type,
                          shell_execution=shell_execution,
                          command_name=command_name,
                          cause=check_cause,
                          warning_threshold_cpu_usage=warning_threshold_cpu_usage)
            
            if check_launch_mode == CHECK_LAUNCH_MODE.PASSIVE:
                check.set_type_passive()
                have_freshness = cls.global_check_freshness and self.check_freshness and self.freshness_threshold != 0
                check.state_validity_period = self.freshness_threshold if have_freshness else NO_END_VALIDITY
            elif check_launch_mode in (CHECK_LAUNCH_MODE.SCHEDULE, CHECK_LAUNCH_MODE.DEPENDENCY):
                check.state_validity_period = self.check_interval * cls.interval_length
            elif check_launch_mode == CHECK_LAUNCH_MODE.FRESHNESS:
                check.state_validity_period = NO_END_VALIDITY
            
            # If the check we will launch change the state of our item, the next check will be done with the retry_interval if the check is schedule so we must update the state_validity_period
            if check_launch_mode in (CHECK_LAUNCH_MODE.SCHEDULE, CHECK_LAUNCH_MODE.DEPENDENCY):
                check.state_validity_period_when_go_in_soft = self.retry_interval * cls.interval_length
            else:
                check.state_validity_period_when_go_in_soft = None
            
            # We keep a trace of all checks in progress
            # to know if we are in checking_or not
            self.checks_in_progress.append(check)
        
        self.update_in_checking()
        
        # We need to put this new check in our actions queue
        # so scheduler can take it
        if check is not None:
            self.actions.append(check)
            # A new check means the host/service changes its next_check need to be refreshed
            self.broks.append(self.get_next_schedule_brok())
            return check.id
        # None mean I already take it into account
        return None
    
    
    # returns either 0 or a positive number
    # 0 == don't check for orphans
    # non-zero == number of secs that can pass before
    #             marking the check an orphan.
    def get_time_to_orphanage(self):
        # if disabled program-wide, disable it
        if not self.check_for_orphaned:
            return 0
        # otherwise, check what my local conf says
        if self.time_to_orphanage <= 0:
            return 0
        return self.time_to_orphanage
    
    
    # Get the perfdata command with macro resolved for this
    def get_perfdata_command(self):
        cls = self.__class__
        if not cls.process_performance_data or not self.process_perf_data:
            return
        
        if cls.perfdata_command is not None:
            m = MacroResolver()
            data = self.get_data_for_event_handler()
            cmd = m.resolve_command(cls.perfdata_command, data)
            reactionner_tag = cls.perfdata_command.reactionner_tag
            shell_execution = cls.perfdata_command.shell_execution
            command_name = getattr(getattr(self.perfdata_command, 'command', {}), 'command_name', 'MISSING_NAME')
            e = EventHandler(cmd, timeout=cls.perfdata_timeout,
                             ref=self, reactionner_tag=reactionner_tag,
                             shell_execution=shell_execution,
                             command_name=command_name)
            
            # ok we can put it in our temp action queue
            self.actions.append(e)
    
    
    # Create the whole business rule tree
    # if we need it
    def create_business_rules(self, proxy_items):
        cmdCall = getattr(self, 'check_command', None)
        
        # If we do not have a command, we bailout
        if cmdCall is None:
            return
        
        # we get our based command, like
        # check_tcp!80 -> check_tcp
        cmd = cmdCall.call
        elts = cmd.split('!')
        base_cmd = elts[0]
        
        # If it's bp_rule, we got a rule :)
        if base_cmd == 'bp_rule':
            self.got_business_rule = True
            # print "Got rule", elts, cmd
            rule = ''
            if len(elts) >= 2:
                rule = '!'.join(elts[1:])
                if len(rule) == 0:
                    self.business_rule = DependencyNode()
            # Only (re-)evaluate the business rule if it has never been
            # evaluated before, or it contains a macro.
            if rule != "" and (re.match(r"\$[\w\d_-]+\$", rule) or self.business_rule is None):
                data = self.get_data_for_checks()
                m = MacroResolver()
                rule = m.resolve_simple_macros_in_string(rule, data)
                prev = getattr(self, "processed_business_rule", None)
                
                if rule == prev:
                    # Business rule did not change (no macro was modulated)
                    return
                
                fact = DependencyNodeFactory(self)
                node = fact.eval_cor_pattern(rule, proxy_items, ref=self)
                self.processed_business_rule = rule
                if node == None:
                    self.business_rule = DependencyNode()
                else:
                    self.business_rule = node
    
    
    # Expands format string macros with item attributes
    def expand_business_rule_item_macros(self, template_string, item):
        output = template_string
        status = getattr(item, "state", "")
        output = re.sub(r"\$STATUS\$", status, output, flags=re.I)
        short_status = self.status_to_short_status(status)
        output = re.sub(r"\$SHORT_STATUS\$", short_status, output, flags=re.I)
        service_description = getattr(item, "service_description", "")
        output = re.sub(r"\$SERVICE_DESCRIPTION\$", service_description, output, flags=re.I)
        host_name = getattr(item, "host_name", "")
        output = re.sub(r"\$HOST_NAME\$", host_name, output, flags=re.I)
        full_name = item.get_full_name()
        output = re.sub(r"\$FULL_NAME\$", full_name, output, flags=re.I)
        return output
    
    
    # Returns status string shorten name.
    def status_to_short_status(self, status):
        mapping = {
            "OK"      : "O",
            "WARNING" : "W",
            "CRITICAL": "C",
            "UNKNOWN" : "U",
            "UP"      : "U",
            "DOWN"    : "D"
        }
        return mapping.get(status, status)
    
    
    # We ask us to manage our own internal check,
    # like a business based one
    def manage_internal_check(self, check):
        # print "DBG, ask me to manage a check!"
        if check.command.startswith('bp_'):
            try:
                # Re evaluate the business rule to take into account macro modulation.
                # Caution: We consider the that the macro modulation did not
                # change business rule dependency tree.
                # Only Xof: values should be modified by modulation.
                self.create_business_rules(proxyitemsmgr)
                state = self.business_rule.get_state(self)
                check.output = self.business_rule.output
                check.long_output = self.business_rule.long_output
            except Exception as e:
                # Notifies the error, and return an UNKNOWN state.
                check.output = "Error while re-evaluating business rule: %s" % e
                logger.debug("[%s] Error while re-evaluating business rule:\n%s" % (self.get_name(), traceback.format_exc()))
                state = 3
        # _internal_host_up is for putting host as UP
        elif check.command == '_internal_host_up':
            state = 0
            check.execution_time = 0
            check.output = 'Host assumed to be UP'
            check.long_output = ''  # c.output
        # Echo is just putting the same state again
        elif check.command == '_echo':
            state = self.state
            check.execution_time = 0
            check.output = self.output
            check.long_output = ''
        check.check_time = time.time()
        check.exit_status = state
        # logger.debug("manage_internal_check:: %s is getting return code: %s and output=%s" % (self.get_full_name(), state, check.output))
    
    
    # If I'm a business rule service/host, I register myself to the
    # elements I will depend on, so They will have ME as an impact
    def create_business_rules_dependencies(self, hosts, services):
        if self.got_business_rule and self.business_rule is not None:
            # print "DBG: ask me to register me in my dependencies", self.get_name()
            all_proxies = self.business_rule.list_all_elements()
            elts = []
            for p in all_proxies:
                if p.type == 'check':
                    e = services.find_srv_by_name_and_hostname(p.host_name, p.name)
                    elts.append(e)
                else:  # host
                    e = hosts.find_by_name(p.name)
                    elts.append(e)
            # I will register myself in this
            for e in elts:
                # print "I register to the element", e.get_name()
                # all states, every timeperiod, and inherit parents
                e.add_business_rule_act_dependency(self, ['d', 'u', 's', 'f', 'c', 'w'], None, True)
    
    
    # Rebuild the possible reference a schedulingitem can have
    def rebuild_ref(self):
        for o in self.downtimes:
            o.ref = self
    
    
    # Go launch all our triggers
    def eval_triggers(self):
        return
    
    
    def _get_command_warning_threshold_cpu_usage(self):
        # Return the "command" timeout first if available..
        if self.check_command and getattr(self.check_command, 'warning_threshold_cpu_usage', -1) != -1:
            return self.check_command.warning_threshold_cpu_usage
        
        # Else it is my value if present
        if self.warning_threshold_cpu_usage != -1:
            return self.warning_threshold_cpu_usage
        
        # If I am a service try to look at Host timeout,
        if self.my_type == 'service' and self.host.warning_threshold_cpu_usage != -1:
            return self.host.warning_threshold_cpu_usage
        
        # Else the default value (shinken.cfg)
        return self.__class__.warning_threshold_cpu_usage_default
    
    
    def _get_command_timeout(self):
        # Return the "command" timeout first if available..
        if self.check_command and self.check_command.timeout != -1:
            return self.check_command.timeout, 'command'
        
        # Else it is my value if present
        if self.check_running_timeout != -1:
            return self.check_running_timeout, 'check'
        
        # If I am a service try to look at Host timeout,
        if self.my_type == 'service':  # try host config
            if self.host.check_running_timeout != -1:
                return self.host.check_running_timeout, 'host'
        
        # Else the default value (shinken.cfg)
        return self.__class__.check_timeout, 'global'
    
    
    # We update the proxy objects with current values
    def update_proxy(self):
        my_uuid = self.get_instance_uuid()
        # state
        
        proxyitemsmgr.update_state(my_uuid, self.state_id)
        
        # flapping
        proxyitemsmgr.update_flapping(my_uuid, self.is_flapping, inherited_flapping=self.inherited_flapping)
        
        # ack
        author = ''
        comment = ''
        proxyitemsmgr.update_acknowledge(my_uuid, self.problem_has_been_acknowledged)
        if self.acknowledgement:
            proxyitemsmgr.update_acknowledge_author_and_comment_auto(my_uuid, self.acknowledgement.author, self.acknowledgement.comment, self.acknowledgement.automatic)
        
        if self.my_type == 'service' and self.is_in_inherited_acknowledged():  # try host config
            author = self.host.acknowledgement.author
            comment = self.host.acknowledgement.comment
        proxyitemsmgr.update_in_inherited_acknowledged(my_uuid, self.is_in_inherited_acknowledged(), author=author, comment=comment)
        
        proxyitemsmgr.update_partial_acknowledge(my_uuid, self.is_partial_acknowledged)
        if self.partial_acknowledge:
            proxyitemsmgr.update_partial_acknowledge_author_and_comment(my_uuid, self.partial_acknowledge.author, self.partial_acknowledge.comment)
        
        # downtime
        proxyitemsmgr.update_downtime(my_uuid, self.in_scheduled_downtime)
        proxyitemsmgr.update_in_inherited_downtime(my_uuid, self.is_in_inherited_downtime())
        
        # business impact
        proxyitemsmgr.update_business_impact(my_uuid, self.business_impact)
    
    
    ##########################################################################################################
    #                                     ACKNOWLEDGEMENT                                                    #
    ##########################################################################################################
    
    # Create a ack for this element
    def acknowledge_problem(self, sticky, notify, persistent, author, comment, end_time=0, automatic=False):
        if self.state != self.ok_up:
            if notify:
                self.create_notifications('ACKNOWLEDGEMENT', ack_data=comment, ack_author=author)
            self.problem_has_been_acknowledged = True
            if sticky == 2:
                sticky = True
            else:
                sticky = False
            self.acknowledgement = Acknowledge(self, sticky, notify, persistent, author, comment, end_time=end_time, automatic=automatic)
            if self.got_business_rule:
                try:
                    self.business_rule.get_state(self)
                    self.output = self.business_rule.output
                    self.long_output = self.business_rule.long_output
                except Exception as e:
                    # Notifies the error, and return an UNKNOWN state.
                    self.output = "Error while re-evaluating business rule: %s" % e
                    self.long_output = ""
                    logger.debug("[%s] Error while re-evaluating business rule:\n%s" % (self.get_name(), traceback.format_exc()))
            self.broks.append(self.get_update_status_brok())
            is_host = self.__class__.my_type == 'host'
            # for hosts, also update info about our services as they can be unack too
            if is_host:
                for s in self.services:
                    s.broks.append(s.get_update_status_brok())
            # also update proxy objects
            proxyitemsmgr.update_acknowledge(self.get_instance_uuid(), self.problem_has_been_acknowledged)
            proxyitemsmgr.update_acknowledge_author_and_comment_auto(self.get_instance_uuid(), author, comment, automatic)
            if is_host:
                for s in self.services:
                    proxyitemsmgr.update_in_inherited_acknowledged(s.get_instance_uuid(), s.is_in_inherited_acknowledged(), author=self.acknowledgement.author, comment=self.acknowledgement.comment)
    
    
    # Create a partial acknowledge for this element
    def partial_acknowledge_problem(self, sticky, persistent, author, comment, end_time=0):
        if self.state != self.ok_up:
            if sticky == 2:
                sticky = True
            else:
                sticky = False
            self.partial_acknowledge = Acknowledge(self, sticky, False, persistent, author, comment, end_time=end_time, automatic=True)
            self.broks.append(self.get_update_status_brok())
            # also update proxy objects
            proxyitemsmgr.update_partial_acknowledge(self.get_instance_uuid(), self.is_partial_acknowledged)
            proxyitemsmgr.update_partial_acknowledge_author_and_comment(self.get_instance_uuid(), author, comment)
    
    
    # Look if we got an ack that is too old with an expire date and should
    # be delete
    def check_for_expire_acknowledge(self):
        if self.acknowledgement and self.acknowledgement.end_time != 0 and self.acknowledgement.end_time < time.time():
            self.unacknowledge_problem()
    
    
    # Delete the acknowledgement object and reset the flag
    #  but do not remove the associated comment.
    def unacknowledge_problem(self):
        if self.problem_has_been_acknowledged:
            self.problem_has_been_acknowledged = False
            # Should not be deleted, a None is Good
            self.acknowledgement = None
            if self.got_business_rule:
                try:
                    self.business_rule.get_state(self)
                    self.output = self.business_rule.output
                    self.long_output = self.business_rule.long_output
                except Exception as e:
                    # Notifies the error, and return an UNKNOWN state.
                    self.output = "Error while re-evaluating business rule: %s" % e
                    self.long_output = ""
                    logger.debug("[%s] Error while re-evaluating business rule:\n%s" % (self.get_name(), traceback.format_exc()))
            self.broks.append(self.get_update_status_brok())
            is_host = self.__class__.my_type == 'host'
            # for hosts, also update info about our services as they can be unack too
            if is_host:
                for s in self.services:
                    s.broks.append(s.get_update_status_brok())
            
            # also update proxy objects
            proxyitemsmgr.update_acknowledge(self.get_instance_uuid(), self.problem_has_been_acknowledged)
            proxyitemsmgr.update_acknowledge_author_and_comment_auto(self.get_instance_uuid(), '', '', False)
            if is_host:
                for s in self.services:
                    proxyitemsmgr.update_in_inherited_acknowledged(s.get_instance_uuid(), s.is_in_inherited_acknowledged())
    
    
    # Check if we have an acknowledgement and if this is marked as sticky.
    # This is needed when a non-ok state changes
    def unacknowledge_problem_if_not_sticky(self):
        if hasattr(self, 'acknowledgement') and self.acknowledgement is not None:
            if not self.acknowledgement.sticky:
                self.unacknowledge_problem()
    
    
    # Get my direct fathers, so only level +1 one
    # * cluster: what are we computing
    # * hosts: parents
    def __get_my_direct_fathers(self):
        if self.got_business_rule:
            fathers = self.business_rule.list_all_elements()
        else:  # hosts
            # Get my parents, but as proxy elements
            fathers = [proxyitemsmgr[p.uuid] for p in self.parents]
        return fathers
    
    
    # return if all our fathers (source problems and direct cluster elements) are in a acknowledge context
    def __are_all_my_fathers_acked(self):
        # Now look at bp rule elements
        fathers = self.__get_my_direct_fathers()
        nb_acked = 0
        nb_partial_acked = 0
        nb_bad_state = 0
        # Expands child items format string macros.
        for father in fathers:
            if father.is_ok_up():
                continue
            nb_bad_state += 1
            if father.is_acknowledge():
                logger.debug('[%s] [are_all_my_fathers_acked] [%s] is ack ' % (self.get_full_name(), father.name))
                nb_acked += 1
                continue
            # Maybe our father is just in partial, so count it so we will be in partial too
            if father.is_partial_acknowledge():
                logger.debug('[%s] [are_all_my_fathers_acked] [%s] is partial ack ' % (self.get_full_name(), father.name))
                nb_partial_acked += 1
            
            # maybe it's a service and it's host is ack
            if father.type == 'service':
                father_host = proxyitemsmgr[father.host_uuid]
                if father_host.is_acknowledge():
                    logger.debug('[%s] [are_all_my_fathers_acked] [%s] host is ack ' % (self.get_full_name(), father.name))
                    nb_acked += 1
                    continue
                if father_host.is_partial_acknowledge():
                    logger.debug('[%s] [are_all_my_fathers_acked] [%s] host is partial ack ' % (self.get_full_name(), father.name))
                    nb_partial_acked += 1
        # partial can be because of other ack or een partial ack
        partial = (nb_acked != 0) or (nb_partial_acked != 0)
        # but for full we need to be with all in acks
        full_ack = (nb_bad_state != 0) and (nb_acked == nb_bad_state)
        # if we have both partial and full ack, then it means we are full ack only
        if partial and full_ack:
            partial = False
        return (partial, full_ack)
    
    
    def __are_all_my_fathers_flapping(self):
        # Now look at bp rule elements
        fathers = self.__get_my_direct_fathers()
        nb_fathers = len(fathers)
        nb_flapping = 0
        nb_partial_flapping = 0
        # Expands child items format string macros.
        
        for father in fathers:
            if father.in_flapping:
                logger.debug('[__are_all_my_fathers_flapping]:: %s is flap ' % father.name)
                nb_flapping += 1
                continue
            # Maybe our father is just in partial, so count it so we will be in partial too
            if father.in_partial_flapping:
                logger.debug('[__are_all_my_fathers_flapping]:: %s is partial flap ' % father.name)
                nb_partial_flapping += 1
            
            # maybe it's a service and it's host is ack
            if father.type == 'service':
                father_host = proxyitemsmgr[father.host_uuid]
                if father_host.in_flapping:
                    logger.debug('[__are_all_my_fathers_flapping]:: %s host is flap ' % father.name)
                    nb_flapping += 1
                    continue
                if father_host.in_partial_flapping:
                    logger.debug('[__are_all_my_fathers_flapping]:: %s host is partial flap ' % father.name)
                    nb_partial_flapping += 1
        # partial can be because of other ack or een partial ack
        partial = (nb_flapping != 0) or (nb_partial_flapping != 0)
        # but for full we need to be with all in acks
        full_flap = (nb_fathers != 0) and (nb_flapping == nb_fathers)
        # if we have both partial and full flapping, then it means we are full ack only
        if partial and full_flap:
            partial = False
        return (partial, full_flap)
    
    
    # update partial acknowledge, and if need, create a brok from it
    def __update_partial_ack(self, partial):
        was_partial = self.is_partial_acknowledged
        if partial == was_partial:
            return
        self.is_partial_acknowledged = partial
        if not partial:
            self.partial_acknowledge = None
        # just a check result is enough here
        self.broks.append(self.get_check_result_brok())
        
        # Update proxy object too
        proxyitemsmgr.update_partial_acknowledge(self.get_instance_uuid(), self.is_partial_acknowledged)
        
        if self.partial_acknowledge:
            proxyitemsmgr.update_partial_acknowledge_author_and_comment(self.get_instance_uuid(), self.partial_acknowledge.author, self.partial_acknowledge.comment)
        else:
            proxyitemsmgr.update_partial_acknowledge_author_and_comment(self.get_instance_uuid(), '', '')
    
    
    # update partial flapping, and if need, create a brok from it
    def __update_partial_flap(self, partial):
        was_partial = self.is_partial_flapping
        if partial == was_partial:
            return
        self.is_partial_flapping = partial
        # just a check result is enough here
        self.broks.append(self.get_check_result_brok())
        
        # Update proxy object too
        proxyitemsmgr.update_partial_flapping(self.get_instance_uuid(), self.is_partial_flapping)
    
    
    # update partial downtime, and if need, create a brok from it
    def update_partial_dt(self, partial):
        was_partial = self.in_partial_downtime
        if partial == was_partial:
            return
        self.in_partial_downtime = partial
        # just a check result is enough here
        self.broks.append(self.get_check_result_brok())
        
        # Update proxy object too
        proxyitemsmgr.update_partial_downtime(self.get_instance_uuid(), self.in_partial_downtime)
    
    
    # We look for possible automatic ack creation if ALL our sources are in acknowledge too.
    # * CREATION: all all our source problems are ack, we are going ack
    # * REMOVING: we remove our ack if one of our source pb is not ack,
    #             BUT only if the ack was an automatic one
    def compute_automatic_acknowledge(self):
        state_id = self.bp_state if self.got_business_rule else self.state_id
        
        # If the item is OK we remove the partial ack
        if state_id == 0:
            self.__update_partial_ack(False)
            return
        
        # We are an impact, so we must look if we already have a ack NOT automatic (because we won't remove it and exit)
        if self.problem_has_been_acknowledged:  # REMOVING
            if self.acknowledgement.automatic:  # but only for automatic
                partial_ack, full_ack = self.__are_all_my_fathers_acked()
                # first look if we need to update partial ack or not
                self.__update_partial_ack(partial_ack)
                if not full_ack:
                    self.unacknowledge_problem()
                    return
        else:  # CREATION
            # It may have some call to __update_partial_ack set to True so we have some item with is_partial_acknowledged to True without partial_acknowledged set
            # There partial_acknowledged will remove in next iteration
            # See SEF-5075 for a sample case
            author = ''
            comment = ''
            partial_ack, full_ack = self.__are_all_my_fathers_acked()
            # first look if we need to update partial ack or not
            self.__update_partial_ack(partial_ack)
            # no ack and my pbs are ack? go in automatic ack
            if full_ack or partial_ack:
                fathers = [p for p in self.__get_my_direct_fathers() if p.is_acknowledge() or p.is_partial_acknowledge()]
                # NOTE: beware, fathers are PROXYS!!!
                if len(fathers) == 1:
                    father = fathers[0]
                    # Note : beware father can be partial ack (father is a cluster
                    if father.ack_author and father.ack_comment:
                        author = father.ack_author
                        comment = father.ack_comment
                    elif father.partial_ack_authors and father.partial_ack_comments:
                        author = father.partial_ack_authors
                        comment = father.partial_ack_comments
                    elif father.inherited_ack_author and father.inherited_ack_comment:
                        author = father.inherited_ack_author
                        comment = father.inherited_ack_comment
                else:
                    authors = {}
                    for p in fathers:
                        if p.ack_author or p.in_inherited_acknowledged:
                            _author = p.ack_author if p.ack_author else p.inherited_ack_author
                            _comment = p.ack_comment if p.ack_author else p.inherited_ack_comment
                            if _author not in authors:
                                authors[_author] = []
                            authors[_author].append((p.get_full_name(), _comment))
                        
                        elif p.partial_ack_authors:
                            # In this case, author can be "by_multiple_authors but it can be found in comments"
                            _authors = p.partial_ack_comments.split('\n')
                            for _, _author, _host_name, _comment in [a.split('<->') for a in _authors if a.startswith('shinken_multiple')]:
                                if _author not in authors:
                                    authors[_author] = []
                                authors[_author].append((_host_name, _comment))
                            if p.partial_ack_authors not in authors:
                                authors[p.partial_ack_authors] = []
                            authors[p.partial_ack_authors].append((p.get_full_name(), p.partial_ack_comments))
                    
                    if len(authors) > 1:
                        author = 'by_multiple_authors'
                    elif len(authors) == 1:
                        author = authors.keys()[0]
                    else:  # Case in  which an old retention without author was loaded
                        author = ''
                    comments = []
                    for (_author, coms) in authors.iteritems():
                        for c in coms:
                            comments.append('shinken_multiple<->%s<->%s<->%s' % (_author, c[0], c[1]))
                    comment = '\n'.join(comments)
                
                if full_ack:
                    # create an automatic ack with notification
                    self.acknowledge_problem(2, True, None, author, comment, automatic=True)
                if partial_ack:
                    if self.partial_acknowledge and self.partial_acknowledge.author == author and self.partial_acknowledge.comment == comment:
                        return
                    if not author and not comment:
                        self.__update_partial_ack(False)
                    else:
                        # create an automatic ack with notification
                        self.partial_acknowledge_problem(2, None, author, comment)
    
    
    def compute_automatic_flapping(self):
        if self.is_flapping:
            if self.inherited_flapping:
                partial_flapping, full_flapping = self.__are_all_my_fathers_flapping()
                self.__update_partial_flap(partial_flapping)
                if not full_flapping:
                    self.update_flapping()
        else:  # CREATION
            partial_flapping, full_flapping = self.__are_all_my_fathers_flapping()
            # first look if we need to update partial ack or not
            self.__update_partial_flap(partial_flapping)
            # no flapping and my pbs are flapping? go in automatic flapping
            if full_flapping:
                # create an automatic flapping
                self.update_flapping(automatic=True)
    
    
    def get_active_downtime_uuids(self):
        return [dt.uuid for dt in self.downtimes if dt.is_in_effect]
