#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019:
# This file is part of Shinken Enterprise, all rights reserved.

from shinken.misc.fast_copy import fast_deepcopy
from shinken.misc.type_hint import TYPE_CHECKING, IN_TESTING, Optional
from .massive_change_returns import MASSIVE_CHANGE_RETURNS
from .massive_change_status import MASSIVE_CHANGE_STATUS
from ....dao.items.baseitem import BaseItem
from ....dao.def_items import STOP_INHERITANCE_VALUES
from ....front_end.helper_object import COUNTER
from ....front_end.mass_change_helper import MassChangeCounterSelect
from ....front_end.object.messages import ValidatorMessages
from ....dao.helpers import get_default_value

if TYPE_CHECKING:
    from .massive_change_property import MassiveChangeProperty


class MassiveChangeElement(object):
    
    def __init__(self, item):
        self.item = item
        if isinstance(item, BaseItem):
            self._id = item.get_uuid()
            self.name = item.get_name()
            self.item_type = item.get_type()
            
            self.raw_item = fast_deepcopy(item.get_raw_item(flatten_links=False))
        else:
            self._id = item.get('_id', '')
            self.name = item.get('name', '')
            self.item_type = item.get('item_type', '')
            
            self.raw_item = fast_deepcopy(item)
        
        self.properties = []
        self.modifications = 0
        self.overrides = []
        
        # Validation
        self.validation_messages = ValidatorMessages()
    
    
    def add_property(self, prop, modified):
        self.properties.append(prop)
        if modified:
            self.modifications += 1
    
    
    def add_overrides(self, override, modified):
        self.overrides.append(override)
        if modified:
            self.modifications += 1
    
    
    def get_overrides(self):
        return self.overrides
    
    
    def get_properties(self):
        return self.properties
    
    
    def get_property(self, name):
        # type: (str) -> Optional[MassiveChangeProperty]
        for prop in self.get_properties():
            if prop.name == name:
                return prop
        return None
    
    
    def get_uuid(self):
        return self._id
    
    
    def get_name(self):
        return self.name
    
    
    def get_name_to_html(self, app):
        return app.helper.helper_object.build_link_dom_element_depend_if_admin(self.item_type, self._id, self.get_name())
    
    
    def set_validation(self, validation):
        self.validation_messages = ValidatorMessages(validation)
        if self.validation_messages.has_critical():
            for _property in self.get_properties():
                _property._modified = MASSIVE_CHANGE_STATUS.UNMODIFIED
                _property.message = MASSIVE_CHANGE_RETURNS.ITEM_NOT_VALIDATE
    
    
    def get_counter_cell_content(self, app):
        if not self.validation_messages:
            return ""
        counters_values = {COUNTER.ERROR: self.validation_messages.get_nb_critical() + self.validation_messages.get_nb_error() or "", COUNTER.WARNING: self.validation_messages.get_nb_warning() or ""}
        _to_return = '<div class="shinken-counters-container shinken-user-select-none shinken-toggle-button-active">%s</div>'
        return _to_return % app.helper.helper_object.get_counters_html(counters=[COUNTER.ERROR, COUNTER.WARNING], counters_value=counters_values)
    
    
    def get_counter_cell_filter_value(self):
        _to_return = MassChangeCounterSelect.EMPTY
        if (self.validation_messages.has_critical() or self.validation_messages.has_error()) and self.validation_messages.has_warning():
            _to_return = "ERROR,WARNING"
        elif self.validation_messages.has_critical() or self.validation_messages.has_error():
            _to_return = "ERROR"
        elif self.validation_messages.has_warning():
            _to_return = "WARNING"
        return _to_return
    
    
    def get_flatten_prop(self, prop_name, app):
        if not self.get_property(prop_name):
            return self.item.flatten_prop(prop_name)
        _in_progress = self.raw_item.get(prop_name, None)
        if _in_progress is None:
            return get_default_value(self.item_type, prop_name, pythonize=False)
        _to_return = app.datamanagerV2.flatten_value(_in_progress, self.item_type, prop_name)
        if _to_return in STOP_INHERITANCE_VALUES:
            _to_return = None
        return _to_return
