#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2018:
# This file is part of Shinken Enterprise, all rights reserved.

import copy
import re
import time
import uuid

from shinken.log import logger
from shinken.misc.fast_copy import fast_deepcopy, NO_COPY
from shinken.misc.type_hint import TYPE_CHECKING
from shinkensolutions.date_helper import get_datetime_with_local_time_zone
from .items import ContactItem
from ..business.source.sourceinfo import SourceInfo
from ..dao import DataException
from ..dao.callbacks import PostLoader, PostSaver, PostDeleter, PreSaver, PreDeleter, PreMover, PostMover
from ..dao.callbacks.default import (callback_delete_change_info,
                                     callback_compute_diff_source_existing_item,
                                     callback_compute_diff_staging_production,
                                     callback_configuration_stats_cache_reset,
                                     callback_build_link,
                                     callback_compute_double_links,
                                     callback_on_rename,
                                     callback_save_deleted_item,
                                     callback_remove_from_source_cache)
from ..dao.callbacks.history import callback_history_info
from ..dao.dataprovider.dataprovider import DataProvider
from ..dao.def_items import DEF_ITEMS, HISTORY_ACTION, ITEM_STATE, ITEM_TYPE, METADATA, LINKIFY_MANAGE_STATES, STOP_INHERITANCE_VALUES, prop_is_linked
from ..dao.helpers import split_list_attr, get_name_from_type, safe_add_to_dict, safe_extend_to_dict, get_item_property, set_item_property
from ..dao.items import BaseItem
from ..dao.transactions.transactions import DBTransaction

if TYPE_CHECKING:
    from shinken.misc.type_hint import NoReturn, Dict, Union
    from .callbacks.history import HistoryEntry

FALLBACK_USER = {
    u'contact_name': u'shinken-core',
    u'is_admin'    : u'1',
    u'_id'         : u'-1',
}


def get_class_from_type(item_type):
    return DEF_ITEMS[item_type][u'table']


def get_type_item_from_class(class_item, item):
    obvious = [ITEM_TYPE.ESCALATIONS,
               ITEM_TYPE.NOTIFICATIONWAYS,
               ITEM_TYPE.MACROMODULATIONS,
               ITEM_TYPE.COMMANDS,
               ITEM_TYPE.BUSINESSIMPACTMODULATIONS,
               ITEM_TYPE.CONTACTGROUPS,
               ITEM_TYPE.HOSTGROUPS,
               ITEM_TYPE.TIMEPERIODS,
               ITEM_TYPE.RESULTMODULATIONS,
               ITEM_TYPE.REMOTE_SYNCHRONIZER]
    
    if class_item in obvious:
        return class_item
    
    is_template = (item.get('register', '1') == '0')
    has_name = item.get('name', '')
    
    if class_item == 'contacts':
        return ITEM_TYPE.CONTACTTPLS if is_template else ITEM_TYPE.CONTACTS
    if class_item == 'services':
        if is_template and has_name:
            return ITEM_TYPE.SERVICETPLS
        if item.get('apply_on_type') == ITEM_TYPE.CLUSTERS:
            return ITEM_TYPE.SERVICESCLUSTERS
        elif item.get('apply_on_type') == ITEM_TYPE.CLUSTERTPLS:
            return ITEM_TYPE.SERVICESCLUSTERTPLS
        elif item.get('apply_on_type') == ITEM_TYPE.HOSTS:
            return ITEM_TYPE.SERVICESHOSTS
        elif item.get('apply_on_type') == ITEM_TYPE.HOSTTPLS:
            return ITEM_TYPE.SERVICESHOSTTPLS
        else:  # HOST or HOST TPL
            return ITEM_TYPE.SERVICESHOSTTPLS if is_template else ITEM_TYPE.SERVICESHOSTS
    if class_item == 'hosts':
        is_cluster = False
        if item.get('bp_rule', None):
            is_cluster = True
        else:
            check_command = item.get('check_command', '')
            if isinstance(check_command, basestring) and check_command.startswith('bp_rule'):
                is_cluster = True
            elif isinstance(check_command, dict) and item.get('check_command', {}).get('raw_value', '').startswith('bp_rule'):
                is_cluster = True
            elif item.get('is_cluster', '0') == '1':
                is_cluster = True
        
        if is_template:
            return ITEM_TYPE.CLUSTERTPLS if is_cluster else ITEM_TYPE.HOSTTPLS
        else:
            return ITEM_TYPE.CLUSTERS if is_cluster else ITEM_TYPE.HOSTS
    
    raise DataException("[%s] is not a valid type or item is malformed." % class_item)


class DataManagerV2(object):
    def __init__(self, data_provider, synchronizer=None, compute_double_links=True, use_default_callbacks=True):
        self.data_provider = data_provider  # type: DataProvider
        self.synchronizer = synchronizer
        
        self.callback_before_save = []
        self.callback_after_save = []
        self.callback_after_load = []
        self.callback_before_delete = []
        self.callback_after_delete = []
        self.callback_before_move = []
        self.callback_after_move = []
        self.compute_double_links = compute_double_links
        if use_default_callbacks:
            self._set_default_callbacks()
    
    
    # Give callback_function that will be call after the datamanager save a item.
    # callback_function signature is
    # param item_id : current id of the item save
    # param item_type : the type of the item save
    # param item_state : the current state of the item save
    # param item : the item to save
    # param old_item : the previous item
    # param user : user use to save the item
    # param action : a HISTORY_ACTION
    # param data_manager : the instance of the DataManagerV2 use to save this item.
    def add_callback(self, callback_cls):
        
        if isinstance(callback_cls, PostLoader):
            self.callback_after_load.insert(0, callback_cls.called_after_load)
        
        if isinstance(callback_cls, PreSaver):
            self.callback_before_save.append(callback_cls.called_before_save)
        
        if isinstance(callback_cls, PostSaver):
            self.callback_after_save.append(callback_cls.called_after_save)
        
        if isinstance(callback_cls, PreDeleter):
            self.callback_before_delete.append(callback_cls.called_before_delete)
        
        if isinstance(callback_cls, PostDeleter):
            self.callback_after_delete.append(callback_cls.called_after_delete)
        
        if isinstance(callback_cls, PreMover):
            self.callback_before_move.insert(0, callback_cls.called_before_move)
        
        if isinstance(callback_cls, PostMover):
            self.callback_after_move.insert(0, callback_cls.called_after_move)
    
    
    def find_item_by_name(self, item_name, item_type='', item_state='', item_source='', lookup=None):
        item = self.data_provider.find_item_by_name(item_name, item_type, item_state, item_source, lookup=lookup)
        if item:
            self._call_callback_after_load(item['_id'], item_type, item_state, item, None, '', None)
        return item
    
    
    def find_item_by_id(self, item_id, item_type='', item_state='', item_source='', lookup=None):
        item = self.data_provider.find_item_by_id(item_id, item_type, item_state, item_source, lookup=lookup)
        if item:
            self._call_callback_after_load(item['_id'], item_type, item_state, item, None, '', None)
        return item
    
    
    def find_items(self, item_type, item_state='', item_source='', lookup=None, where=None):
        ret_val = []
        if item_type == ITEM_TYPE.ELEMENTS:
            types_to_search = DEF_ITEMS
        elif item_type == ITEM_TYPE.ALL_SERVICES:
            types_to_search = ITEM_TYPE.ALL_SERVICES
        else:
            types_to_search = [item_type]
        
        tmp_ret_val = self.data_provider.find_items(types_to_search, item_state, item_source, lookup=lookup, where=where)
        ret_val.extend(tmp_ret_val)
        for item in tmp_ret_val:
            self._call_callback_after_load(item['_id'], item_type, item_state, item, None, '', None)
        
        return ret_val
    
    
    def find_merge_state_items(self, item_type, item_states, item_source='', where=None, lookup=None):
        items = self.data_provider.find_merge_state_items(item_type, item_states, item_source, where=where, lookup=lookup)
        
        for item in items:
            item_type = METADATA.get_metadata(item, METADATA.ITEM_TYPE)
            item_state = METADATA.get_metadata(item, METADATA.STATE)
            self._call_callback_after_load(item['_id'], item_type, item_state, item, None, None, None)
        return items
    
    
    def save_item(self, item, user=None, item_type='', item_state='', item_source='', action=None, **kwargs):
        if not item:
            raise DataException('[%s] save_item : Please set item' % self.__class__.__name__)
        user = user or FALLBACK_USER
        
        if isinstance(item, BaseItem):
            item_type = item_type or item.get_type()
            item_state = item_state or item.get_state()
        
        # caller_name = inspect.stack()[2][3]
        # thread_id = threading.current_thread().ident
        # logger.debug('[LOCK-DATAMANAGER][%s] Ask SAVE_ITEM Lock [%s]' % (thread_id, caller_name))
        with DBTransaction(self, user):
            # logger.debug('[LOCK-DATAMANAGER][%s] Get SAVE_ITEM Lock [%s]' % (thread_id, caller_name))
            old_item = self.data_provider.find_item_by_id(item['_id'], item_type, item_state) if item_state != ITEM_STATE.CHANGES else None
            old_item_raw = self.get_raw_item(old_item, keep_metadata=True, flatten_links=False, item_type=item_type)
            
            action = action or (HISTORY_ACTION.ELEMENT_MODIFICATION if old_item else HISTORY_ACTION.ELEMENT_ADD)
            self._call_callback_before_save(item['_id'], item_type, item_state, item, old_item, user, action)
            
            item = self.data_provider.save_item(item, item_type, item_state, item_source, **kwargs)
            
            # FYI double link is made via a callback
            self._call_callback_after_save(item['_id'], item_type, item_state, item, old_item_raw, user, action)
        # logger.debug('[LOCK-DATAMANAGER][%s] Released SAVE_ITEM Lock [%s]' % (thread_id, caller_name))
        return item
    
    
    def delete_item(self, item, user=None, item_type='', item_state='', old_item=None, action=None):
        user = user or FALLBACK_USER
        action = action or HISTORY_ACTION.ELEMENT_DELETE
        
        if not item:
            raise DataException('[%s] delete_item : Please set item' % self.__class__.__name__)
        if not item_type:
            item_type = METADATA.get_metadata(item, METADATA.ITEM_TYPE)
            if not item_type:
                raise DataException('[%s] delete_item : Please set item_type' % self.__class__.__name__)
        if not item_state:
            item_state = METADATA.get_metadata(item, METADATA.STATE)
            if not item_state:
                raise DataException('[%s] delete_item : Please set item_state' % self.__class__.__name__)
        
        # caller_name = inspect.stack()[2][3]
        # thread_id = threading.current_thread().ident
        # logger.debug('[LOCK-DATAMANAGER][%s] Ask DELETE_ITEM Lock [%s]' % (thread_id, caller_name))
        with DBTransaction(self, user):
            # logger.debug('[LOCK-DATAMANAGER][%s] Get DELETE_ITEM Lock [%s]' % (thread_id, caller_name))
            if not item:
                logger.error("You try to delete null item.")
                return
            
            self._call_callback_before_delete(item['_id'], item_type, item_state, item, old_item, user, action)
            self.data_provider.delete_item(item, item_type, item_state)
            self._call_callback_after_delete(item['_id'], item_type, item_state, item, old_item, user, action)
        # logger.debug('[LOCK-DATAMANAGER][%s] Released DELETE_ITEM Lock [%s]' % (thread_id, caller_name))
    
    
    def reload_after_import(self):
        if hasattr(self.data_provider, 'reload_after_import'):
            return self.data_provider.reload_after_import()
        else:
            raise DataException('You must call reload_after_import with a metadata provider')
    
    
    def copy_state(self, from_state, to_state):
        self.data_provider.copy_state(from_state, to_state)
    
    
    def update_all_item(self, item_state, update):
        self.data_provider.update_all_item(item_state, update)
    
    
    def _make_double_link(self, item, item_type, user, old_item=None):
        if not item or not item_type:
            return False, {}, {}
        
        def_item = DEF_ITEMS[item_type]
        double_links = def_item.get('double_links', None)
        if not double_links:
            return False, {}, {}
        
        if hasattr(self.data_provider, '_make_double_link'):
            has_effective_done_double_link, target_added_to_me, modified_targets = self.data_provider._make_double_link(item, item_type, user, old_item=old_item)
            return has_effective_done_double_link, target_added_to_me, modified_targets
        
        start_time = time.time()
        item_name = get_name_from_type(item_type, item)
        has_effective_done_double_link = False
        modified_targets = {}
        target_added_to_me = {}
        nb_items = 0
        for double_link in double_links:
            item_attr = double_link['my_attr']
            target_type = double_link['of_type']
            target_attr = double_link['is_link_with_attr']
            
            target_names_string = item.get(item_attr, '')
            target_names = item.get(item_attr, '')
            old_target_names = '' if old_item is None else old_item.get(item_attr, '')
            old_item_name = '' if old_item is None else get_name_from_type(item_type, old_item)
            
            # If the user don't explicitly set the link we update our item with external link
            if target_names == old_target_names or old_item is None:
                target_added = self._update_my_link_with_external_data(item, item_name, item_type, item_attr, target_type, target_attr, user)
                if target_added:
                    safe_extend_to_dict(target_added_to_me, item_attr, target_added)
            target_names = split_list_attr(item, item_attr)
            
            # if the item was rename we delete all external references
            if old_item_name and old_item_name != item_name:
                self._remove_ref_item_in_target_attr(old_item_name, item_type, target_type, target_attr, [], user)
            self._remove_ref_item_in_target_attr(item_name, item_type, target_type, target_attr, target_names, user)
            
            for target_name in target_names:
                target, is_target_modified = self._add_my_item_name_in_target_type(item, item_name, item_attr, target_type, target_name, target_attr, user, ITEM_STATE.MERGE_SOURCES)
                
                if is_target_modified:
                    nb_items += 1
                    safe_add_to_dict(modified_targets, item_attr, target)
            
            if target_names_string != item.get(item_attr, ''):
                has_effective_done_double_link = True
        
        logger.log_perf(start_time, self, "make_double_link for [%s:%s] link with [%d] items" % (item_type, item_name, nb_items))
        return has_effective_done_double_link, target_added_to_me, modified_targets
    
    
    def _update_my_link_with_external_data(self, item, item_name, item_type, item_attr, target_type, target_attr, _user):
        start_time = time.time()
        
        use_double_link_index = next((True for double_link in DEF_ITEMS[target_type].get('double_links', []) if double_link['my_attr'] == target_attr), False)
        if use_double_link_index:
            items_that_target_me = list(self.data_provider.find_double_link_items(item_name, item_type, target_type, ITEM_STATE.MERGE_SOURCES))
        else:
            logger.warning('fail to use double link index on type[%s] for attr[%s]' % (item_type, target_attr))
            where = {target_attr: re.compile(re.escape(item_name), re.IGNORECASE)}
            items_that_target_me = list(self.data_provider.find_items(target_type, ITEM_STATE.MERGE_SOURCES, where=where))
        
        # new_targets_name must be the new list of target of item.
        # We build it with the items_that_target_me
        # But we will keep names that :
        # * The origin source is not syncui or auto-modification ( if the user set it in a external source like a cfg file )
        # * Target name that exist in stagging (it's not consider as a dead link)
        
        # Build set new_targets_name with the name of items_that_target_me
        old = self.get_raw_item(item, item_type=item_type)
        old_value = item.get(item_attr, '')
        old_targets_name = set(split_list_attr(old, item_attr))
        new_targets_name = set()
        added_target_item = []
        
        if old_value.replace('+', '') in STOP_INHERITANCE_VALUES:
            return added_target_item
        
        for in_item in items_that_target_me:
            if item_name in split_list_attr(in_item, target_attr):
                new_targets_name.add(get_name_from_type(target_type, in_item))
                if 'null' in old_targets_name:
                    double_link_error_msg = self.synchronizer._('source.invalid_link') % (METADATA.get_metadata(item, METADATA.FORMATED_NAME), item_attr, METADATA.get_metadata(in_item, METADATA.FORMATED_NAME), item_name, target_attr)
                    METADATA.update_metadata(item, METADATA.DOUBLE_LINKS_ERROR, double_link_error_msg)
                    METADATA.update_metadata(in_item, METADATA.DOUBLE_LINKS_ERROR, 'to_exclude')
                    added_target_item.append(in_item)
                elif get_name_from_type(target_type, in_item) not in old_targets_name:
                    added_target_item.append(in_item)
        
        removed_targets_names = old_targets_name - new_targets_name
        
        source_info = METADATA.get_metadata(item, METADATA.SOURCE_INFO, {})
        if source_info:
            source_info = SourceInfo.from_dict(source_info, item_type)
            # Target set by external source (not syncui or auto-modification) will be keep
            for removed_target_name in removed_targets_names:
                source_info_property = source_info.get_property(item_attr)
                # don't come from syncui or auto_modification and if we are during an import, don't remove him
                if not source_info_property or source_info_property.get_origin_of_value() not in ('syncui',):
                    new_targets_name.add(removed_target_name)
            
            removed_targets_names = old_targets_name - new_targets_name
        
        # Target that still exist will be keep
        for removed_target_name in removed_targets_names:
            removed_target = self.data_provider.find_item_by_name(removed_target_name, item_type=target_type, item_state=ITEM_STATE.MERGE_SOURCES)
            if removed_target and removed_target_name not in new_targets_name:
                new_targets_name.add(removed_target_name)
        
        if 'null' in new_targets_name and len(new_targets_name) > 1:
            new_targets_name.remove('null')
        
        if new_targets_name != old_targets_name:
            new_target_name = ','.join(new_targets_name)
            if new_target_name:
                item[item_attr] = new_target_name
            else:
                item.pop(item_attr, None)
            self.data_provider.save_item(item, item_type, ITEM_STATE.MERGE_SOURCES)
            logger.log_perf(start_time, self, "update [%s:%s] field[%s] with [%s] new value [%s]" % (item_type, item_name, item_attr, old_value, item.get(item_attr, '')))
            return added_target_item
        logger.log_perf(start_time, self, "no update of [%s:%s] field[%s] from external [%s]" % (item_type, item_name, item_attr, target_type))
        return added_target_item
    
    
    def _remove_ref_item_in_target_attr(self, item_name, item_type, target_type, target_attr, target_names, user):
        start_time = time.time()
        
        use_double_link_index = False
        double_links = DEF_ITEMS[target_type].get('double_links', [])
        for double_link in double_links:
            if double_link['my_attr'] == target_attr:
                use_double_link_index = True
                break
        
        _cmp = self._remove_ref_item_area(item_name, item_type, ITEM_STATE.MERGE_SOURCES, target_attr, target_names, target_type, use_double_link_index, user)
        
        if _cmp:
            logger.log_perf(start_time, self, 'remove ref to [%s:%s] in items [%d]-[%s]' % (item_type, item_name, _cmp, target_type))
        else:
            logger.log_perf(start_time, self, 'no ref to [%s:%s] in items [%s] to update' % (item_type, item_name, target_type))
    
    
    def _remove_ref_item_area(self, item_name, item_type, item_state, target_attr, target_names, target_type, use_double_link_index, _user):
        if use_double_link_index:
            targets_find = self.data_provider.find_double_link_items(item_name, item_type, target_type, item_state)
        else:
            where = {target_attr: re.compile(re.escape(item_name), re.IGNORECASE)}
            targets_find = list(self.data_provider.find_items(target_type, item_state, where=where))
        
        _cmp = 0
        for target_item in targets_find:
            target_name = get_name_from_type(target_type, target_item)
            
            if target_name in target_names:
                continue
            
            _cmp += 1
            old_value = target_item.get(target_attr)
            # clean the previous value from the entry, and relink it with ,
            new_value = ','.join([e for e in split_list_attr(target_item, target_attr) if e != item_name])
            # Do not keep void entries in table, better to have no property at all
            if not new_value:
                target_item.pop(target_attr, None)
            else:
                target_item[target_attr] = new_value
            
            if new_value != old_value:
                self.data_provider.save_item(target_item, target_type, item_state)
        return _cmp
    
    
    def _add_my_item_name_in_target_type(self, item, item_name, item_attr, target_type, target_name, target_attr, _user, item_state):
        if not target_name or target_name in STOP_INHERITANCE_VALUES:
            return [], False
        
        start_time = time.time()
        target = self.data_provider.find_item_by_name(target_name, target_type, item_state)
        
        if not target:
            logger.warning('Cannot find %s of type %s to make double link. Aborting double link.' % (target_name, target_type))
            return [], False
        
        target_attr_value = split_list_attr(target, target_attr)
        edited = False
        if target_attr_value and 'null' not in target_attr_value:
            if item_name not in target_attr_value:
                if hasattr(target, 'set_value'):
                    prev_value = target[target_attr]
                    new_value = "%s,%s" % (prev_value, item_name)
                    target.set_value(target_attr, new_value)
                else:
                    target[target_attr] += ',' + item_name
                edited = True
        else:
            if hasattr(target, 'set_value'):
                target.set_value(target_attr, item_name)
            else:
                target[target_attr] = item_name
            edited = True
            if 'null' in target_attr_value:
                error_msg = self.synchronizer._('source.invalid_link') % (METADATA.get_metadata(target, METADATA.FORMATED_NAME), target_attr, METADATA.get_metadata(item, METADATA.FORMATED_NAME), target_name, item_attr)
                METADATA.update_metadata(item, METADATA.DOUBLE_LINKS_ERROR, error_msg)
                METADATA.update_metadata(target, METADATA.DOUBLE_LINKS_ERROR, 'to_exclude')
        # logger.debug('_add_my_item_name_in_target_type: target [%s:%s] new value [%s]:[%s]' % (target_type, target_name, target_attr, target[target_attr]))
        if edited:
            self.data_provider.save_item(target, target_type, item_state)
        logger.log_perf(start_time, self, "add [%s] in [%s:%s:%s] prop [%s]" % (item_name, target_type, target_name, item_state, target_attr))
        return target, edited
    
    
    def count_items(self, item_type, item_state='', where=None):
        if item_type == ITEM_TYPE.ELEMENTS:
            count = 0
            for _item_type in DEF_ITEMS:
                count += self.data_provider.count_items(_item_type, item_state=item_state, where=where)
            return count
        else:
            return self.data_provider.count_items(item_type, item_state=item_state, where=where)
    
    
    def get_raw_item(self, item, keep_metadata=False, flatten_links=LINKIFY_MANAGE_STATES, item_type=None):
        if not item:
            return item
        if hasattr(item, 'get_raw_item'):
            return item.get_raw_item(keep_metadata=keep_metadata, flatten_links=flatten_links)
        
        if not item_type:
            raise Exception('The item is not a baseitem, need to set the item_type to continue')
        raw_item = fast_deepcopy(item)
        for prop in item:
            if prop.startswith('@'):
                if isinstance(keep_metadata, (list, tuple, set)) and prop == '@metadata':
                    for metadata_key in raw_item['@metadata']:
                        if metadata_key not in keep_metadata:
                            METADATA.remove_metadata(raw_item, metadata_key)
                elif not keep_metadata:
                    del raw_item[prop]
        
        if flatten_links:
            for prop_to_flatten in DEF_ITEMS[item_type]['props_links']:
                item_value = get_item_property(raw_item, prop_to_flatten)
                if item_value:
                    item_value_flatten = self.flatten_prop(item, item_type, prop_to_flatten, flatten_links)
                    set_item_property(raw_item, prop_to_flatten, item_value_flatten)
        return raw_item
    
    
    def flatten_prop(self, item, item_type, prop_to_flatten, flatten_states=LINKIFY_MANAGE_STATES):
        if not item.get(prop_to_flatten, None):
            return ''
        if hasattr(item, 'flatten_prop'):
            return item.flatten_prop(prop_to_flatten, flatten_states)
        
        item_value = get_item_property(item, prop_to_flatten)
        if not prop_is_linked(item_type, prop_to_flatten):
            return item_value
        return self.flatten_value(item_value, item_type, prop_to_flatten, flatten_states)
    
    
    def flatten_value(self, property_value, item_type, prop_to_flatten, flatten_states=LINKIFY_MANAGE_STATES):
        if isinstance(property_value, basestring):
            return property_value
        
        # avoid cycling import on baseitem
        from ..dao.items.mixins import CommandMixin, BPRuleMixin, HostNameMixin, HostgroupNameMixin, ServiceOverridesMixin, ServiceExcludesMixin
        
        flattener = DEF_ITEMS[item_type]['flatteners'].get(prop_to_flatten, None)
        if flattener == 'CommandMixin.flatten_not_linked_prop':
            return CommandMixin.flatten_not_linked_prop(property_value, prop_to_flatten, flatten_states, self)
        elif flattener == 'BPRuleMixin.flatten_not_linked_prop':
            return BPRuleMixin.flatten_not_linked_prop(property_value, prop_to_flatten, flatten_states, self)
        elif flattener == 'HostNameMixin.flatten_not_linked_prop':
            return HostNameMixin.flatten_not_linked_prop(property_value, prop_to_flatten, flatten_states, self)
        elif flattener == 'HostgroupNameMixin.flatten_not_linked_prop':
            return HostgroupNameMixin.flatten_not_linked_prop(property_value, prop_to_flatten, flatten_states, self)
        elif flattener == 'ServiceOverridesMixin.flatten_not_linked_prop':
            return ServiceOverridesMixin.flatten_not_linked_prop(property_value, prop_to_flatten, flatten_states, self)
        elif flattener == 'ServiceExcludesMixin.flatten_not_linked_prop':
            return ServiceExcludesMixin.flatten_not_linked_prop(property_value, prop_to_flatten, flatten_states, self)
        
        item_name_links = []
        for link in property_value['links']:
            if link['exists']:
                link_item_type = link['item_type']
                for link_item_state in flatten_states:
                    item = self.find_item_by_id(link['_id'], link_item_type, link_item_state)
                    if item:
                        item_name_links.append(get_name_from_type(link_item_type, item))
                        break
                    elif 'name' in link:
                        item_name_links.append(link['name'])
                        break
            else:
                item_name_links.append(link['name'])
        
        if len(item_name_links) > 1:
            item_name_links = [n for n in item_name_links if n != 'null']
        
        plus_as_string = '+' if property_value.get('has_plus', False) else ''
        return '%s%s' % (plus_as_string, ','.join(item_name_links))
    
    
    def flatten_overridden_property(self, check_type, override):
        if isinstance(override['value'], basestring):
            flattened_value = override['value']
        else:
            linked_item_type = DEF_ITEMS[check_type]['props_links'].get(override['key'], [])
            if len(linked_item_type) == 1:
                flattened_value = self.flatten_value(override['value'], linked_item_type[0], override['key'])
            elif len(linked_item_type) == 0:
                flattened_value = override['value']
            else:
                raise NotImplementedError
        return flattened_value
    
    
    def _set_default_callbacks(self):
        self.add_callback(callback_build_link)
        self.add_callback(callback_history_info)
        self.add_callback(callback_delete_change_info)
        self.add_callback(callback_compute_diff_staging_production)
        self.add_callback(callback_compute_diff_source_existing_item)
        self.add_callback(callback_on_rename)
        self.add_callback(callback_configuration_stats_cache_reset)
        self.add_callback(callback_save_deleted_item)
        self.add_callback(callback_remove_from_source_cache)
        # should not be needed anymore since 02.05.06 double links are manage as callbacks (_callbackComputeDoubleLinks)
        # self.callback_after_delete.append(self._callback_delete_remove_external_link)
        
        if self.compute_double_links:
            self.add_callback(callback_compute_double_links)
    
    
    def _call_callback_list(self, callback_list, item_id, item_type, item_state, item, old_item, user, action):
        for callback in callback_list:
            callback(item_id, item_type, item_state, item, old_item, user, action, self)
    
    
    def _call_callback_after_load(self, item_id, item_type, item_state, item, old_item, user, action):
        self._call_callback_list(self.callback_after_load, item_id, item_type, item_state, item, old_item, user, action)
    
    
    def _call_callback_after_save(self, item_id, item_type, item_state, item, old_item, user, action):
        self._call_callback_list(self.callback_after_save, item_id, item_type, item_state, item, old_item, user, action)
    
    
    def _call_callback_before_save(self, item_id, item_type, item_state, item, old_item, user, action):
        self._call_callback_list(self.callback_before_save, item_id, item_type, item_state, item, old_item, user, action)
    
    
    def _call_callback_after_delete(self, item_id, item_type, item_state, item, old_item, user, action):
        self._call_callback_list(self.callback_after_delete, item_id, item_type, item_state, item, old_item, user, action)
    
    
    def _call_callback_before_delete(self, item_id, item_type, item_state, item, old_item, user, action):
        self._call_callback_list(self.callback_before_delete, item_id, item_type, item_state, item, old_item, user, action)
    
    
    def _call_callback_before_move(self, item_id, item_type, item_state, item, old_item, user, action):
        self._call_callback_list(self.callback_before_move, item_id, item_type, item_state, item, old_item, user, action)
    
    
    def _call_callback_after_move(self, item_id, item_type, item_state, item, old_item, user, action):
        self._call_callback_list(self.callback_after_move, item_id, item_type, item_state, item, old_item, user, action)
    
    
    @staticmethod
    def _delete_extra_data(from_item, from_prop=None):
        val = from_item[from_prop]
        new_val = copy.copy(val)
        for prop in val.iterkeys():
            if prop.startswith('@'):
                del new_val[prop]
            elif isinstance(val[prop], dict):
                DataManagerV2._delete_extra_data(new_val, prop)
            elif isinstance(new_val[prop], (list, tuple)):
                new_val[prop] = copy.copy(new_val[prop])
                for index, obj in enumerate(new_val[prop]):
                    if isinstance(obj, dict) and '@link' in obj:
                        new_obj = copy.copy(obj)
                        del new_obj['@link']
                        new_val[prop][index] = new_obj
        
        from_item[from_prop] = new_val
    
    
    # Transaction stuff
    # forward the transaction information to the provider that implement it
    def start_transaction(self, transaction_uuid):
        if hasattr(self.data_provider, 'start_transaction'):
            self.data_provider.start_transaction(transaction_uuid)
    
    
    # forward the transaction information to the provider that implement it
    def commit_transaction(self, transaction_uuid):
        if hasattr(self.data_provider, 'commit_transaction'):
            return self.data_provider.commit_transaction(transaction_uuid)
    
    
    def save_history(self, history_entry, item_type, item_state, raw_item, user):
        # type: (HistoryEntry, unicode, unicode, Dict, Union[Dict, BaseItem]) -> NoReturn
        history_entry = fast_deepcopy(history_entry, additional_dispatcher={ContactItem: NO_COPY})
        item_id = raw_item.get(u'_id', None)
        
        if history_entry[u'action'] in [HISTORY_ACTION.ELEMENT_ADD, HISTORY_ACTION.ELEMENT_CLONED, HISTORY_ACTION.IMPORT]:
            for _prop, value in raw_item.iteritems():
                history_entry[u'change'].append({u'prop': _prop, u'old': u'', u'new': raw_item.get(_prop, u'')})
        
        history_entry[u'contact'][u'links'][0].pop(u'@link', None)
        history_entry[u'contact'][u'links'][0][u'name'] = user.get(u'contact_name')
        history_entry[u'_id'] = uuid.uuid4().hex
        history_entry[u'date'] = get_datetime_with_local_time_zone()
        history_entry[u'item_type'] = item_type
        history_entry[u'item_state'] = item_state
        if item_id:
            # HISTORY_ACTION.APPLY haven't an _id
            history_entry[u'item_uuid'] = item_id
        return self.data_provider.save_history(history_entry)
    
    
    def find_history_for_item(self, item_id, item_type):
        # type: (unicode, unicode) -> list
        return self.data_provider.find_history_for_item(item_id, item_type)
