#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2018:
# This file is part of Shinken Enterprise, all rights reserved.
import copy
import time

from dataprovider import DataProvider
from shinken.log import logger
from shinken.misc.fast_copy import fast_deepcopy
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.util import make_unicode
from shinkensolutions.readwritelock import ReadWriteLock, read_sync, write_sync
from .. import DataException
from ..def_items import (DEF_ITEMS,
                         ITEM_TYPE,
                         ITEM_STATE,
                         MAX_TEMPLATE_LOOKUP_LEVEL,
                         NO_HERITABLE_ATTR,
                         INTERNAL_ATTR,
                         WORKING_AREA_STATUS,
                         METADATA,
                         SOURCE_STATE,
                         LINKIFY_MANAGE_STATES,
                         SERVICE_OVERRIDE,
                         PROP_DEFAULT_VALUE,
                         VALUE_FORCE_DEFAULT,
                         HISTORY_ACTION)
from ..from_info import FromInfo, DEFAULT_FROM_INFO, ItemReversedLink
from ..helpers import get_name_from_type, safe_add_to_dict_but_uniq, get_property_separator, ShinkenDatabaseConsistencyError, set_item_property, get_origin, split_list_attr
from ..indexed_collection.collection import IndexedCollection
from ..item_saving_formatter import _build_prop_link_for_value
from ..items import get_item_instance, BaseItem
from ..transactions.transactions import get_transaction_object
from ..trash_manager.trash_manager import trash_manager
from ...business.source.sourceinfoproperty import SourceInfoProperty

if TYPE_CHECKING:
    from shinken.misc.type_hint import Set, Dict, Tuple, Union, TYPE_CHECKING, List, NoReturn, Optional
    from ..callbacks.history import HistoryEntry
    from ..items import ContactItem
    from ..items.mixins import UseMixin

TO_KEEP_METADATA = ((METADATA.REVERSE_LINKS_CHECKS, []), (METADATA.CHECKS, []), (METADATA.SONS, {}))


class ProxySetInexistingLink(dict):
    def __init__(self, trash_info):
        super(ProxySetInexistingLink, self).__init__()
        self.name = trash_info['name']
        self['_id'] = trash_info['item_id']
    
    
    def get_name(self):
        return self.name


class DataProviderMetadata(DataProvider):
    
    def __init__(self, data_provider, app=None):
        self.data_provider = data_provider  # type: DataProvider
        self.app = app
        self.inheritance_handle_states = (ITEM_STATE.WORKING_AREA, ITEM_STATE.STAGGING, ITEM_STATE.NEW)
        self.linkify_states = (ITEM_STATE.WORKING_AREA, ITEM_STATE.STAGGING, ITEM_STATE.NEW, ITEM_STATE.PREPROD, ITEM_STATE.PRODUCTION)
        self.handle_states = (ITEM_STATE.WORKING_AREA, ITEM_STATE.STAGGING, ITEM_STATE.NEW, ITEM_STATE.PREPROD, ITEM_STATE.PRODUCTION, ITEM_STATE.MERGE_SOURCES)
        
        self.linkify_rules = {
            ITEM_STATE.NEW         : {'check_exists': False, 'manage_non_existing': True, 'link_with': LINKIFY_MANAGE_STATES},
            ITEM_STATE.WORKING_AREA: {'check_exists': False, 'manage_non_existing': True, 'link_with': LINKIFY_MANAGE_STATES},
            ITEM_STATE.STAGGING    : {'check_exists': False, 'manage_non_existing': True, 'link_with': LINKIFY_MANAGE_STATES},
            ITEM_STATE.PRODUCTION  : {'check_exists': True, 'manage_non_existing': False, 'link_with': (ITEM_STATE.PRODUCTION,)},
            ITEM_STATE.PREPROD     : {'check_exists': True, 'manage_non_existing': False, 'link_with': (ITEM_STATE.PREPROD,)},
        }
        
        self.rw_lock = ReadWriteLock()
        
        # used to keep the link items that does not exists
        self.items_linked_with_name = IndexedCollection()
        self.items_linked_with_name.add_index('name')
        self.items_linked_with_name.add_index('type')
        
        # will contain all objects by types and by state
        self.objects = {}
        for item_state in self.handle_states:
            self.objects[item_state] = {}
            for item_type, def_item in DEF_ITEMS.iteritems():
                self.objects[item_state][item_type] = DataProviderMetadata._collection_factory(item_type, item_state)
    
    
    @write_sync
    def load_from_data_provider(self):
        logger.info('dataprovider metadata loading start')
        time_start_load = time.time()
        nb_items = 0
        for item_state in self.handle_states:
            for item_type, item_def in DEF_ITEMS.iteritems():
                load_time = time.time()
                list_item = self._load_type_from_provider(item_state, item_type)
                self.objects[item_state][item_type].extend(list_item)
                nb_items += len(list_item)
                logger.log_perf(load_time, self, 'load [%d] items [%s][%s] from base' % (len(list_item), item_state, item_type))
        logger.log_perf(time_start_load, self, 'All items loaded from base')
        
        # build item (inheritance + default value + metalink)
        
        time_start = time.time()
        self._generate_all_item()
        logger.log_perf(time_start, self, 'generate all [%d] items' % nb_items)
        logger.log_perf(time_start_load, self, 'load dataprovider metadata with [%d] items' % nb_items)
    
    
    @read_sync
    def find_item_by_name(self, item_name, item_type='', item_state='', item_source='', lookup=None):
        return self._find_item_by_name(item_name=item_name, item_type=item_type, item_state=item_state, item_source=item_source, lookup=lookup)
    
    
    # used internally because it doesn't grab the read lock
    def _find_item_by_name(self, item_name, item_type='', item_state='', item_source='', lookup=None):
        if not item_name:
            raise DataException('[%s] find_item_by_name : Please set item_name' % self.__class__.__name__)
        if not item_type:
            raise DataException('[%s] find_item_by_name : Please set item_type' % self.__class__.__name__)
        if not item_state:
            raise DataException('[%s] find_item_by_name : Please set item_state' % self.__class__.__name__)
        if item_state not in self.handle_states:
            logger.warning('[find_item_by_name] Trying to find item %s-%s-%s but state %s is not handled' % (item_state, item_type, item_name, item_state))
            return None
        
        return self.objects[item_state][item_type].find_one({DEF_ITEMS[item_type]['key_name']: item_name})
    
    
    @read_sync
    def find_item_by_id(self, item_id, item_type='', item_state='', item_source='', lookup=None):
        return self._find_item_by_id(item_id=item_id, item_type=item_type, item_state=item_state, item_source=item_source, lookup=lookup)
    
    
    # used internally because it doesn't grab the read lock
    def _find_item_by_id(self, item_id, item_type='', item_state='', item_source='', lookup=None):
        # type: (unicode, unicode, unicode, unicode, Dict) -> Optional[BaseItem]
        if not item_id:
            raise DataException('[%s] find_item_by_id : Please set item_id' % self.__class__.__name__)
        if not item_type:
            raise DataException('[%s] find_item_by_id : Please set item_type' % self.__class__.__name__)
        if not item_state:
            raise DataException('[%s] find_item_by_id : Please set item_state' % self.__class__.__name__)
        if item_state not in self.handle_states:
            logger.warning('[find_item_by_id] Trying to find item %s-%s-%s but state %s is not handled' % (item_state, item_type, item_id, item_state))
            return None
        
        return self.objects[item_state][item_type].find_one({'_id': item_id})
    
    
    @read_sync
    def find_items(self, item_type, item_state='', item_source='', where=None, lookup=None):
        return self._find_items(item_type=item_type, item_state=item_state, item_source=item_source, where=where, lookup=lookup)
    
    
    # used internally because it doesn't grab the read lock
    def _find_items(self, item_type, item_state='', item_source='', where=None, lookup=None):
        if not item_type:
            raise DataException('[%s] find_items : Please set item_type' % self.__class__.__name__)
        if not item_state:
            raise DataException('[%s] find_items : Please set item_state' % self.__class__.__name__)
        
        if item_type == ITEM_TYPE.ELEMENTS:
            item_type = DEF_ITEMS.keys()
        if isinstance(item_type, basestring):
            item_type = [item_type]
        if isinstance(item_state, basestring):
            item_state = [item_state]
        items = []
        
        for current_state in item_state:
            if current_state not in self.handle_states:
                logger.warning('[find_items] Trying to find item %s-%s but state %s is not handled' % (current_state, item_type, current_state))
                continue
            for current_type in item_type:
                items_for_type_and_state = self.objects[current_state][current_type].find(where)
                items.extend(items_for_type_and_state)
        return items
    
    
    @read_sync
    def find_merge_state_items(self, item_type, item_states, item_source='', where=None, lookup=None):
        return self._find_merge_state_items(item_type, item_states, item_source=item_source, where=where, lookup=lookup)
    
    
    # used internally because it doesn't grab the read lock
    def _find_merge_state_items(self, item_type, item_states, item_source='', where=None, lookup=None):
        _item_states = list(item_states)
        _item_states.reverse()
        items = {}
        for _item_state in _item_states:
            _items = self._find_items(item_type, _item_state, item_source, where, lookup)
            for _item in _items:
                METADATA.update_metadata(_item, METADATA.STATE, _item_state)
                items['%s-%s' % (_item.get_type(), _item['_id'])] = _item
        return items.values()
    
    
    @read_sync
    def find_double_link_items(self, item_link_name, item_type_link_to, item_type='', item_state='', item_source=''):
        return self._find_double_link_items(item_link_name=item_link_name, item_type_link_to=item_type_link_to, item_type=item_type, item_state=item_state, item_source=item_source)
    
    
    def _find_double_link_items(self, item_link_name, item_type_link_to, item_type='', item_state='', item_source=''):
        if not item_link_name:
            raise DataException('[%s] find_double_link_items : Please set item_link_name' % self.__class__.__name__)
        if not item_type_link_to:
            raise DataException('[%s] find_double_link_items : Please set item_type_link_to' % self.__class__.__name__)
        if not item_type:
            raise DataException('[%s] find_double_link_items : Please set item_type' % self.__class__.__name__)
        if not item_state:
            raise DataException('[%s] find_double_link_items : Please set item_state' % self.__class__.__name__)
        if item_state not in self.handle_states:
            logger.warning('[_find_double_link_item] Trying to find item %s-%s-%s but state %s is not handled' % (item_state, item_type, item_link_name, item_state))
            return []
        
        original_item = self._find_item_by_name(item_link_name, item_type_link_to, item_state)
        
        double_links = DEF_ITEMS[item_type_link_to].get('double_links', [])
        target_attr = next((double_link['is_link_with_attr'] for double_link in double_links if double_link['of_type'] == item_type), '')
        double_link_items = []
        for reversed_link in original_item.get_reverse_links().copy():
            if reversed_link.item_type == item_type and reversed_link.item_state == item_state and reversed_link.key == target_attr:
                linked_item = self._find_item_by_id(item_id=reversed_link.item_id, item_type=item_type, item_state=item_state)
                if original_item.check_reverse_link(reversed_link, linked_item):
                    double_link_items.append(linked_item)
        
        # TODO maybe we will need to find double link items link to us by name
        # items_linked_by_name = self.items_linked_with_name.find_one({'name': item_link_name, 'type': item_type_link_to})
        # if items_linked_by_name:
        #     pass
        return double_link_items
    
    
    @read_sync
    def save_item(self, item, item_type='', item_state='', item_source='', inheritance_cache=None):
        item = self._save_item(item, set([]), item_type=item_type, item_state=item_state, inheritance_cache=inheritance_cache)
        return item
    
    
    # used internally because it doesn't grab the write lock
    def _save_item(self, item, already_save, item_type='', item_state='', inheritance_cache=None):
        if not item:
            raise DataException('[%s] save_item : Please set item' % self.__class__.__name__)
        
        raw_item_to_save = None
        if not isinstance(item, BaseItem):
            raw_item_to_save = fast_deepcopy(item)
            item = get_item_instance(item_type, fast_deepcopy(item))
        
        if inheritance_cache is None:
            inheritance_cache = set()
        item_type = item_type or item.get_type()
        item_state = item_state or item.get_state()
        item_key = (item['_id'], item_type, item_state)
        
        if not item_type:
            raise DataException('[%s] save_item : [%s:%s:%s] Please set a type' % (self.__class__.__name__, item_type, item['_id'], item.get_name()))
        if not item_state:
            raise DataException('[%s] save_item : [%s:%s:%s] Please set a state' % (self.__class__.__name__, item_type, item['_id'], item.get_name()))
        
        item_source_hash = {}
        source_info = {}
        if item_state == ITEM_STATE.MERGE_SOURCES:
            item_source_hash = METADATA.get_metadata(item, METADATA.SOURCE_HASH)
            source_info = METADATA.get_metadata(item, METADATA.SOURCE_INFO)
        
        if not raw_item_to_save:
            raw_item_to_save = item.get_raw_item(flatten_links=False)
        
        if item_state == ITEM_STATE.MERGE_SOURCES:
            METADATA.update_metadata(raw_item_to_save, METADATA.SOURCE_HASH, item_source_hash)
            METADATA.update_metadata(raw_item_to_save, METADATA.ITEM_TYPE, item_type)
            METADATA.update_metadata(raw_item_to_save, METADATA.SOURCE_INFO, source_info)
        
        raw_item_for_mongo = copy.deepcopy(raw_item_to_save) if self.app and self.app.conf.protect_fields__activate_encryption else raw_item_to_save
        self.data_provider.save_item(raw_item_for_mongo, item_type, item_state)
        
        if item_key in already_save:
            logger.info('saving loop skipping item [%s]' % item)
            return item
        already_save.add(item_key)
        
        if item_state in self.handle_states:
            if item_state in self.linkify_states:
                item = self._regenerate_item(raw_item_to_save, item_type, item_state, already_save, inheritance_cache)
            else:
                self.objects[item_state][item_type].remove({'_id': raw_item_to_save['_id']})
                self.objects[item_state][item_type].append(item)
        
        return item
    
    
    @write_sync
    def delete_item(self, item_to_delete, item_type='', item_state='', item_source=''):
        return self._delete_item(item_to_delete, item_type, item_state)
    
    
    def _delete_item(self, item_to_delete, item_type, item_state):
        if not item_to_delete:
            raise DataException('[%s] delete_item : Please set item' % self.__class__.__name__)
        if not item_type:
            raise DataException('[%s] delete_item : Please set item_type' % self.__class__.__name__)
        if not item_state:
            raise DataException('[%s] delete_item : Please set item_state' % self.__class__.__name__)
        
        item_id = item_to_delete['_id']
        self.data_provider.delete_item(item_to_delete, item_type, item_state)
        
        # if we delete a change, we have to remove all metadatachange on the work_area and stagging items
        if item_state == ITEM_STATE.CHANGES:
            items_without_changes = []
            for item_state in (ITEM_STATE.WORKING_AREA, ITEM_STATE.STAGGING):
                item = self._find_item_by_id(item_id=item_id, item_type=item_type, item_state=item_state)
                if item:
                    items_without_changes.append(item)
            for item_without_changes in items_without_changes:
                # changes are in metadata, we don't need to save the item
                METADATA.remove_metadata(item_without_changes, METADATA.CHANGES)
                METADATA.update_metadata(item_without_changes, METADATA.SOURCE_STATE, SOURCE_STATE.IMPORT)
                self.objects[item_without_changes.get_state()][item_type].regenerate_index(item_without_changes)
            
            return
        
        old_item_to_delete = self._find_item_by_id(item_id, item_type=item_type, item_state=item_state)
        if not old_item_to_delete:
            return
        
        for parent in old_item_to_delete.get_all_state_link_items('use', only_exist=True):
            parent.remove_son(old_item_to_delete)
        self._unlinkify_host_check(old_item_to_delete, set([]), None)
        
        self.objects[item_state][item_type].remove({'_id': item_id})
        # Now remove all references items
        if item_state in LINKIFY_MANAGE_STATES:
            # List of items where delete item was remove in the property value
            # So we must save in db the item and do the work of callback :
            # * compute diff with staging/source
            # * compute diff with staging/production
            # * add history info,
            # like a call of datamanagerV2.save.
            # We cannot do a datamanagerV2.save because we change in place item so we lost old_item so
            # we don't have any chance to compute change done here in callback
            remove_link_edit_items = []
            # List of items where delete item was remove in @link and exist was pass to false
            # So we just need to update item in the provider and save in db
            edit_items = []
            for reverse_link in old_item_to_delete.get_reverse_links().copy():
                item_to_unlink = self._find_item_by_id(reverse_link.item_id, item_type=reverse_link.item_type, item_state=reverse_link.item_state)
                if old_item_to_delete.check_reverse_link(reverse_link, item_to_unlink):
                    effective = item_to_unlink.remove_item_link(reverse_link.key, old_item_to_delete)
                    if effective == 'must_save_with_callback':
                        safe_add_to_dict_but_uniq(item_to_unlink, '@properties_changes', reverse_link.key)
                    if effective == 'must_save_with_callback' and item_to_unlink not in remove_link_edit_items:
                        remove_link_edit_items.append(item_to_unlink)
                        try:
                            edit_items.remove(item_to_unlink)
                        except ValueError:
                            pass
                    elif effective and item_to_unlink not in edit_items and item_to_unlink not in remove_link_edit_items:
                        edit_items.append(item_to_unlink)
            
            for to_save_item in edit_items:
                self._save_item(to_save_item, set([]))
            
            # item in remove_link_edit_items have there properties change : item delete was remove in property
            for to_save_item in remove_link_edit_items:
                properties_changes = to_save_item.pop('@properties_changes', [])
                
                # We update SOURCE_INFO in merge_from_source_item to say the properties was change by a AUTO_MODIFICATION
                # So the current modification will be ignore for the next compute diff staging/source
                merge_from_source_item = self._find_item_by_id(to_save_item['_id'], to_save_item.get_type(), ITEM_STATE.MERGE_SOURCES)
                effective = False
                if merge_from_source_item:
                    source_info = METADATA.get_metadata(merge_from_source_item, METADATA.SOURCE_INFO)['_info']
                    for property_change in properties_changes:
                        if source_info.get(property_change, None) and not source_info[property_change]['property_modif_auto']:
                            source_info[property_change]['property_modif_auto'] = HISTORY_ACTION.AUTO_MODIFICATION
                            effective = True
                if effective:
                    self._save_item(merge_from_source_item, set([]))
                
                # We update last_modification in our item (job of add history info)
                # So the current modification will be ignore for the next compute diff staging/production
                last_modif = to_save_item.get(u'last_modification', {})
                last_modif[u'contact'] = last_modif.get(u'contact', {"links": [{"name": "shinken-core", "exists": False}], "has_plus": False})
                last_modif[u'action'] = HISTORY_ACTION.AUTO_MODIFICATION
                last_modif[u'date'] = last_modif.get(u'date', time.time())
                last_modif[u'change'] = []  # We cannot set change because we update item in place
                last_modif[u'show_diffs'] = last_modif.get(u'show_diffs', False)
                
                show_diffs_state = last_modif.get(u'show_diffs_state', {})
                for property_change in properties_changes:
                    show_diffs_state[property_change] = show_diffs_state.get(property_change, HISTORY_ACTION.AUTO_MODIFICATION)
                
                to_save_item.set_value(u'last_modification', last_modif)
                self._save_item(to_save_item, set([]))
    
    
    @read_sync
    def count_items(self, item_type, item_state='', item_source='', where=None):
        if not item_type:
            raise DataException('[%s] count_items : Please set item_type' % self.__class__.__name__)
        if not item_state:
            raise DataException('[%s] count_items : Please set item_state' % self.__class__.__name__)
        return self.objects[item_state][item_type].count(where)
    
    
    @read_sync
    def find_items_by_name(self, items_name, items_type, items_state):
        return self._find_items_by_name(items_name=items_name, items_type=items_type, items_state=items_state)
    
    
    # used internally because it doesn't grab the read lock
    def _find_items_by_name(self, items_name, items_type, items_state):
        ret = []
        for item_name in items_name:
            item = self._find_item_by_name(item_name, items_type, item_state=items_state)
            if item:
                ret.append(item)
        return ret
    
    
    @read_sync
    def find_items_by_id(self, items_id, items_type, items_state):
        return self._find_items_by_id(items_id=items_id, items_type=items_type, items_state=items_state)
    
    
    # used internally because it doesn't grab the read lock
    def _find_items_by_id(self, items_id, items_type, items_state):
        # type: (Set, unicode, unicode) -> List[Union[BaseItem, ContactItem]]
        ret = []
        for item_id in items_id:
            item = self._find_item_by_id(item_id, items_type, item_state=items_state)
            if item:
                ret.append(item)
        return ret
    
    
    @write_sync
    def update_all_item(self, item_state, update):
        self.data_provider.update_all_item(item_state, update)
        for i, item_type in enumerate(DEF_ITEMS.iterkeys()):
            logger.debug('update_all_item item_type [%s/%s] [%s]' % (i + 1, len(DEF_ITEMS), item_type))
            for item in self._find_items(item_type, item_state):
                for update_type, update_value in update.iteritems():
                    if update_type == '$set':
                        for item_prop, item_value in update_value.iteritems():
                            set_item_property(item, item_prop, item_value)
                    else:
                        raise DataException('[%s] update_all_item : unsupported update_type [%s]' % (self.__class__.__name__, update_type))
        logger.debug('update_all_item done')
    
    
    @write_sync
    def copy_state(self, from_state, to_state):
        if from_state not in self.handle_states:
            logger.warning('[copy_state] Trying to copy state from [%s]->[%s] to but state [%s] is not handled' % (from_state, to_state, from_state))
            return
        if to_state not in self.handle_states:
            logger.warning('[copy_state] Trying to copy state from [%s]->[%s] to but state [%s] is not handled' % (from_state, to_state, to_state))
            return
        
        self.data_provider.copy_state(from_state, to_state)
        for item_type, item_def in DEF_ITEMS.iteritems():
            list_item = self._load_type_from_provider(to_state, item_type)
            to_state_collection = DataProviderMetadata._collection_factory(item_type, to_state)
            to_state_collection.extend(list_item)
            self.objects[to_state][item_type] = to_state_collection
        
        updated_items = []
        for item_type, item_def in DEF_ITEMS.iteritems():
            for item in self._find_items(item_type, to_state):
                self._linkify_item(item, item_def, updated_items=updated_items)
        for item_type in DEF_ITEMS.iterkeys():
            for item in self._find_items(item_type, to_state):
                self._linkify_host_check(item)
        
        for updated_item in updated_items:
            raw_item_to_save = updated_item.get_raw_item(flatten_links=False)
            raw_item_to_save = fast_deepcopy(raw_item_to_save) if self.app and self.app.conf.protect_fields__activate_encryption else raw_item_to_save
            self.data_provider.save_item(raw_item_to_save, item_type=updated_item.get_type(), item_state=updated_item.get_state())
    
    
    @write_sync
    def reload_after_import(self):
        warning_message = []
        # --- Reload change
        for item_type, item_def in DEF_ITEMS.iteritems():
            raw_changes_list_items = self.data_provider.find_items(item_type, ITEM_STATE.CHANGES)
            changes_list_items = {}
            for raw_change in raw_changes_list_items:
                DataProviderMetadata._unserialize_change(item_type, raw_change)
                changes_list_items[raw_change['_id']] = raw_change['changes']
            
            for item_state in (ITEM_STATE.WORKING_AREA, ITEM_STATE.STAGGING):
                old_changes = self._find_items(item_type, item_state, where={'have_change': True})
                for old_change in old_changes:
                    METADATA.remove_metadata(old_change, METADATA.CHANGES)
                    METADATA.update_metadata(old_change, METADATA.SOURCE_STATE, SOURCE_STATE.IMPORT)
                items = self._find_items_by_id(changes_list_items.keys(), item_type, item_state)
                for item in items:
                    METADATA.update_metadata(item, METADATA.CHANGES, changes_list_items[item['_id']])
                    if item.have_manual_changes():
                        METADATA.update_metadata(item, METADATA.SOURCE_STATE, SOURCE_STATE.CHANGE)
                
                self.objects[item_state][item_type].add_index('have_change', getter=lambda x: bool(METADATA.get_metadata(x, METADATA.CHANGES)))
                self.objects[item_state][item_type].add_index('@metadata.%s' % METADATA.SOURCE_STATE)
        
        # --- Reload new
        for item_type, item_def in DEF_ITEMS.iteritems():
            # Put new item in provider
            list_item = self._load_type_from_provider(ITEM_STATE.NEW, item_type)
            for item in list_item[:]:
                _name = get_name_from_type(item_type, item)
                if item_type in ITEM_TYPE.ALL_DEDICATED_SERVICES:
                    if self._find_item_by_id(item['_id'], item_type, ITEM_STATE.STAGGING):
                        warning_message.append({
                            'msg'        : self.app._('element.duplicate_element_post_import') % (self.app._('type.%s' % item_type[:-1]), _name, self.app._('type.%s' % item_type[:-1])),
                            'item_id'    : ','.join(item.get('_SYNC_KEYS', item.get('_sync_keys', ['-1']))),
                            'source_name': item.get('sources', 'ALL_SOURCE')
                        })
                        list_item.remove(item)
                elif self._find_item_by_name(_name, item_type, ITEM_STATE.STAGGING) or (ITEM_TYPE.has_work_area(item_type) and self._find_item_by_name(_name, item_type, ITEM_STATE.WORKING_AREA)):
                    warning_message.append({
                        'msg'        : self.app._('element.duplicate_element_post_import') % (self.app._('type.%s' % item_type[:-1]), _name, self.app._('type.%s' % item_type[:-1])),
                        'item_id'    : ','.join(item.get('_SYNC_KEYS', item.get('_sync_keys', ['-1']))),
                        'source_name': item.get('sources', 'ALL_SOURCE')
                    })
                    list_item.remove(item)
            new_collection = DataProviderMetadata._collection_factory(item_type, ITEM_STATE.NEW)
            new_collection.extend(list_item)
            self.objects[ITEM_STATE.NEW][item_type] = new_collection
        
        # --- Clean old new in REVERSE_LINKS_CHECKS metadata of the checks
        for check in self._find_items(ITEM_TYPE.ALL_DEDICATED_SERVICES, LINKIFY_MANAGE_STATES):
            reverse_links_checks = METADATA.get_metadata(check, METADATA.REVERSE_LINKS_CHECKS)
            METADATA.update_metadata(check, METADATA.REVERSE_LINKS_CHECKS, [h for h in reverse_links_checks if h.get_state() != ITEM_STATE.NEW])
        
        # --- Relink all new with work area and staging items
        for item_type, item_def in DEF_ITEMS.iteritems():
            for item in self._find_items(item_type, ITEM_STATE.NEW):
                updated_items = []
                self._linkify_item(item, item_def, check_exists=True, updated_items=updated_items)
                for _item in updated_items:
                    self._save_item(_item, set([]))
        for item_type, item_def in DEF_ITEMS.iteritems():
            for item in self._find_items(item_type, ITEM_STATE.NEW):
                self._linkify_host_check(item)
        
        # --- Reload merge
        for item_type, item_def in DEF_ITEMS.iteritems():
            list_item = self._load_type_from_provider(ITEM_STATE.MERGE_SOURCES, item_type)
            new_collection = DataProviderMetadata._collection_factory(item_type, ITEM_STATE.MERGE_SOURCES)
            new_collection.extend(list_item)
            self.objects[ITEM_STATE.MERGE_SOURCES][item_type] = new_collection
        
        return warning_message
    
    
    @write_sync
    def _make_double_link(self, item, item_type, user, old_item=None):
        has_effective_done_double_link = False
        target_added_to_me = {}
        modified_targets = {}
        
        def_item = DEF_ITEMS[item_type]
        double_links = def_item.get('double_links', None)
        if not double_links:
            return has_effective_done_double_link, target_added_to_me, modified_targets
        
        for double_link in double_links:
            item_attr = double_link['my_attr']
            item_value = item.get(item_attr, {}) or {}
            target_type = double_link['of_type']
            target_attr = double_link['is_link_with_attr']
            
            # get the value originally set on us, remove the values coming from inheritance
            raw_item = METADATA.get_metadata(item, METADATA.RAW_ITEM, {})
            existing_ids_i_know = set([i['_id'] for i in raw_item.get(item_attr, {}).get('links', []) if i['exists']])
            
            existing_ids_old_item_know = existing_ids_i_know
            if old_item:
                raw_old_item = METADATA.get_metadata(old_item, METADATA.RAW_ITEM, {})
                if not raw_old_item:
                    raw_old_item = old_item
                existing_ids_old_item_know = set([i.get('_id') for i in raw_old_item.get(item_attr, {}).get('links', []) if i.get('exists', False)])
            
            # Keep double_links up to date in all states
            if DEF_ITEMS[item_type]['has_work_area']:
                self._update_double_links_in_other_states(item, item_type, target_type, target_attr)
            
            items_to_add_on_me = []
            items_with_bad_link_on_me = []
            # look for items to add on me only if the item doesn't modify this field explicitly
            if old_item is None or existing_ids_i_know != existing_ids_old_item_know:
                # look for item pointing in me before I exist
                linked_items = self._find_double_link_items(item.get_name(), item_type, target_type, ITEM_STATE.STAGGING)
                for linked_item in linked_items:
                    linked_item_id = linked_item['_id']
                    # if the linked item is in item field but not in the old item  ==> linked_item should be linked to us
                    if linked_item_id in existing_ids_i_know and linked_item_id not in existing_ids_old_item_know:
                        items_to_add_on_me.append(linked_item)
                        has_effective_done_double_link = True
                    # is the linked item not in item field but was here before ==> linked_item should be unlink
                    elif linked_item_id not in existing_ids_i_know and linked_item_id in existing_ids_old_item_know:
                        items_with_bad_link_on_me.append(linked_item)
                        has_effective_done_double_link = True
                    # else the item is in creation so I don't know the linked item and the old does not exists
                    else:
                        items_to_add_on_me.append(linked_item)
                        has_effective_done_double_link = True
                
                # now add the items_to_add_on_me on me and create the reverse link on them
                if items_to_add_on_me:
                    item.add_unique_items_links(item_attr, items_to_add_on_me)
                    target_added_to_me[item_attr] = items_to_add_on_me
            
            #
            #  Add (or_remove) myself on other items
            #
            items_modified = []
            for my_link_item in item_value.get('links', []):
                # my_link_obj['_id'] in existing_ids_i_know to remove link from inheritance
                if not (my_link_item['exists'] and my_link_item['_id'] in existing_ids_i_know):
                    continue
                
                my_link_obj = my_link_item.get(u'@link', {}).get(ITEM_STATE.STAGGING, None)
                if my_link_obj:
                    link_added = my_link_obj.add_unique_items_links(target_attr, [item])
                    if link_added:
                        self._remove_null_value_in_field(my_link_obj, target_attr)
                        items_modified.append(my_link_obj)
            
            if items_with_bad_link_on_me:
                for item_with_bad_link_on_me in items_with_bad_link_on_me:
                    item_with_bad_link_on_me.remove_item_link(target_attr, item)
                items_modified.extend(items_with_bad_link_on_me)
            
            if items_modified:
                modified_targets[item_attr] = items_modified
        
        # dont forget to save modified objects
        if has_effective_done_double_link:
            self._save_item_with_updated_links(item, set(), set())
        
        if modified_targets:
            for modified_attr, modified_items in modified_targets.iteritems():
                for modified_item in modified_items:
                    self._save_item_with_updated_links(modified_item, set(), set())
        
        return has_effective_done_double_link, target_added_to_me, modified_targets
    
    
    def _update_double_links_in_other_states(self, item, item_type, target_type, target_attr):
        # Here the item is in a new state (Ex. from work_area to staging) but the linked item don't know it !
        # So we take the reverse_links to add the item in the good state on the linked_item
        # See : SEF-5366
        reverse_links = [rl for rl in item.get_reverse_links() if rl.item_type == target_type and rl.key == target_attr]
        for reverse_link in reverse_links:
            _linked_item = self._find_item_by_id(reverse_link.item_id, reverse_link.item_type, reverse_link.item_state)
            if item.check_reverse_link(reverse_link, _linked_item):
                _link = next((i for i in _linked_item.get_links(reverse_link.key, item_type) if i['exists'] and i['_id'] == item['_id']), None)
                if _link and not _link[u'@link'].get(item.get_state(), None):
                    _link[u'@link'][item.get_state()] = item
    
    
    def _remove_null_value_in_field(self, my_link_obj, target_attr):
        # we search for PROP_DEFAULT_VALUE, VALUE_FORCE_DEFAULT to remove
        link_item_index = next((index for index, link in enumerate(my_link_obj.get(target_attr, {}).get('links', [])) if link.get('name', None) in (PROP_DEFAULT_VALUE, VALUE_FORCE_DEFAULT)), -1)
        if link_item_index != -1:
            del my_link_obj[target_attr]['links'][link_item_index]
            _raw_item = METADATA.get_metadata(my_link_obj, METADATA.RAW_ITEM, {})
            if link_item_index < len(_raw_item.get(target_attr, {}).get('links', [])):
                del _raw_item[target_attr]['links'][link_item_index]
    
    
    @staticmethod
    def _collection_factory(item_type, item_state):
        collection = IndexedCollection()
        collection.add_index('_id')
        collection.add_index(DEF_ITEMS[item_type]['key_name'])
        collection.add_index('%s_lower' % DEF_ITEMS[item_type]['key_name'], getter=lambda item: item.get(DEF_ITEMS[item_type]['key_name'], '').lower())
        collection.add_index('work_area_info.status', getter=WORKING_AREA_STATUS.get_work_area_status)
        collection.add_index('have_change', getter=lambda x: bool(METADATA.get_metadata(x, METADATA.CHANGES)))
        collection.add_index('@metadata.%s' % METADATA.SOURCE_STATE)
        collection.add_index('_SYNC_KEYS', on_list=True)
        collection.add_index('sources', on_list=True, getter=lambda x: split_list_attr(x, 'sources'))
        
        # specific indexes per item_type
        if item_type == ITEM_TYPE.HOSTS:
            collection.add_index('address')
        elif item_type == ITEM_TYPE.SERVICESHOSTTPLS and item_state != ITEM_STATE.MERGE_SOURCES:
            collection.add_index('host_name_links', on_list=True, getter=lambda x: [(link.get('_id', link.get('name')), link['exists']) for link in x._get_links_host_name('host_name', ITEM_TYPE.HOSTTPLS)])
        elif item_type == ITEM_TYPE.SERVICESCLUSTERTPLS and item_state != ITEM_STATE.MERGE_SOURCES:
            collection.add_index('host_name_links', on_list=True, getter=lambda x: [(link.get('_id', link.get('name')), link['exists']) for link in x._get_links_host_name('host_name', ITEM_TYPE.CLUSTERTPLS)])
        elif item_type in (ITEM_TYPE.SERVICESHOSTS, ITEM_TYPE.SERVICESCLUSTERS) and item_state != ITEM_STATE.MERGE_SOURCES:
            collection.add_index('host_name_links', on_list=True, getter=lambda x: [(link.get('_id', link.get('name')), link['exists']) for link in x.get_links('host_name')])
        
        if ITEM_TYPE.has_template(item_type):
            collection.add_index('use_id', on_list=True, getter=lambda x: x._get_use_ids())
        return collection
    
    
    def _load_type_from_provider(self, item_state, item_type):
        changes_list_items = {}
        # manage change only for working area and stagging
        manage_changes = item_state in (ITEM_STATE.WORKING_AREA, ITEM_STATE.STAGGING)
        
        if manage_changes:
            raw_changes_list_items = self.data_provider.find_items(item_type, ITEM_STATE.CHANGES)
            for raw_change in raw_changes_list_items:
                DataProviderMetadata._unserialize_change(item_type, raw_change)
                changes_list_items[raw_change['_id']] = raw_change['changes']
        list_items = self.data_provider.find_items(item_type, item_state)
        ret_list = []
        
        for item in list_items:
            self._init_item(item, item_type, item_state)
            item = get_item_instance(item_type, item)
            
            raw_change = changes_list_items.get(item['_id'], None)
            if raw_change:
                METADATA.update_metadata(item, METADATA.CHANGES, make_unicode(raw_change))
                if item.have_manual_changes():
                    METADATA.update_metadata(item, METADATA.SOURCE_STATE, SOURCE_STATE.CHANGE)
            ret_list.append(item)
        return ret_list
    
    
    def _init_item(self, raw_item, item_type, item_state):
        item = raw_item
        old_source_info = None
        old_source_hash = None
        if item_state == ITEM_STATE.MERGE_SOURCES:
            old_source_info = METADATA.get_metadata(item, METADATA.SOURCE_INFO)
            old_source_hash = METADATA.get_metadata(item, METADATA.SOURCE_HASH)
        
        item.pop('@metadata', None)
        raw_item = fast_deepcopy(item)
        item[u'@metadata'] = {
            METADATA.STATE       : item_state,
            METADATA.ITEM_TYPE   : make_unicode(item_type),
            METADATA.TABLE       : make_unicode(DEF_ITEMS[item_type]['table']),
            METADATA.NAME        : make_unicode(get_name_from_type(item_type, raw_item)),
            METADATA.FROM        : {},
            METADATA.SOURCE_STATE: SOURCE_STATE.NEW if item_state == ITEM_STATE.NEW else SOURCE_STATE.IMPORT,
            METADATA.RAW_ITEM    : raw_item,
            u'_id'               : make_unicode(item[u'_id'])
        }
        
        if item_state == ITEM_STATE.MERGE_SOURCES:
            METADATA.update_metadata(item, METADATA.SOURCE_INFO, old_source_info)
            METADATA.update_metadata(item, METADATA.SOURCE_HASH, old_source_hash)
        
        self._build_initial_from_info(item)
    
    
    def _build_initial_from_info(self, item):
        metadata_from = item['@metadata'][METADATA.FROM]
        item_state = item['@metadata'][METADATA.STATE]
        item_type = item['@metadata'][METADATA.ITEM_TYPE]
        if item_state in self.inheritance_handle_states:
            for item_property, item_value in item.iteritems():
                if item_property in INTERNAL_ATTR or not item_value or item_property.startswith('@'):
                    continue
                # for field with plus manage the from for each links
                if item_property in DEF_ITEMS[item_type].get('fields_with_plus', set()):
                    has_plus = item_value['has_plus']
                    item_ids_or_names = [i.get('_id', i.get('name')) for i in item_value['links']]
                    for item_link in item_ids_or_names:
                        safe_add_to_dict_but_uniq(metadata_from, item_property, FromInfo(item_link, item['_id'], item_type))
                    if has_plus:
                        safe_add_to_dict_but_uniq(metadata_from, item_property, FromInfo(u'+', item['_id'], item_type))
                else:
                    safe_add_to_dict_but_uniq(metadata_from, item_property, FromInfo(item_value, item['_id'], item_type))
    
    
    def _generate_all_item(self):
        time_start = time.time()
        nb_items = 0
        
        for item_state in self.linkify_states:
            for item_type, item_def in DEF_ITEMS.iteritems():
                list_items = self.objects[item_state][item_type]
                sub_start_time = time.time()
                for item in list_items:
                    self._linkify_item(item, item_def)
                logger.log_perf(sub_start_time, self, 'link [%d] [%s][%s] items' % (len(list_items), item_state, item_type))
        
        for item_state in self.linkify_states:
            for item_type, item_def in DEF_ITEMS.iteritems():
                list_items = self.objects[item_state][item_type]
                sub_start_time = time.time()
                for item in list_items:
                    self._linkify_host_check(item)
                
                nb_items += len(list_items)
                logger.log_perf(sub_start_time, self, 'link host check [%d] [%s][%s] items' % (len(list_items), item_state, item_type))
        
        logger.log_perf(time_start, self, 'link all [%d] items' % nb_items)
        
        # keep track of all already inherited objects, avoiding recomputing them again and again during the ascendant recursion
        # NOTE: We have a cache by template types
        time_start = time.time()
        for item_state in self.inheritance_handle_states:
            template_caches = {}
            template_types = ITEM_TYPE.TPLS_ITEM_TYPES
            not_template_types = ITEM_TYPE.ITEM_TYPES_WITH_TPLS - ITEM_TYPE.TPLS_ITEM_TYPES
            # First compute templates like elements
            for item_type in template_types:
                sub_start_time = time.time()
                list_items = self.objects[item_state][item_type]
                template_cache = set()
                template_caches[item_type] = template_cache
                for item in list_items:
                    # In this template we give it a void cache,
                    # but we want to have the final template item in this template type cache for
                    # the next compute: the final items ones
                    self._set_attr_with_inheritance(item, inheritance_cache=set())
                    template_cache.add(item.get_key())  # for use by the final types
                logger.log_perf(sub_start_time, self, 'build _set_attr_with_inheritance [%s][%s]' % (item_state, item_type))
            
            for item_type in not_template_types:
                item_type_template_type = ITEM_TYPE.get_template_type(item_type)
                sub_start_time = time.time()
                list_items = self.objects[item_state][item_type]
                
                # Get the cache from the templates
                template_cache = template_caches.get(item_type_template_type, set())
                for item in list_items:
                    self._set_attr_with_inheritance(item, inheritance_cache=template_cache)
                logger.log_perf(sub_start_time, self, 'build _set_attr_with_inheritance [%s][%s]' % (item_state, item_type))
            
            logger.log_perf(time_start, self, 'build set_attr_with_inheritance for all type')
            
            sub_start_time = time.time()
            for item_type, item_def in DEF_ITEMS.iteritems():
                sub_sub_time_start = time.time()
                list_items = self.objects[item_state][item_type]
                for item in list_items:
                    self._set_default_value(item, item_type)
                logger.log_perf(sub_sub_time_start, self, 'build _set_default_value [%s][%s]' % (item_state, item_type))
            logger.log_perf(sub_start_time, self, 'build set default value for all type')
        logger.log_perf(time_start, self, 'build all [%d] items' % nb_items)
    
    
    def _linkify_item(self, item, item_def, check_exists=False, updated_items=None):
        if item.get_state() not in self.linkify_states:
            return
        check_exists = check_exists or self.linkify_rules[item.get_state()]['check_exists']
        to_linkify_item_states = self.linkify_rules[item.get_state()]['link_with']
        for to_linkify_item_type, to_linkify_properties in item_def['item_links'].iteritems():
            for to_linkify_property in to_linkify_properties:
                self._linkify_value(item, to_linkify_property, to_linkify_item_type, to_linkify_item_states, check_exists, updated_items)
    
    
    def _unlinkify_host_check(self, item, already_save, inheritance_cache):
        specific_method = getattr(item, "_unlinkify_host_check", None)
        if specific_method and callable(specific_method):
            host_to_save = []
            specific_method(host_to_save)
            for host in host_to_save:
                self._save_item(host, already_save, inheritance_cache=inheritance_cache)
    
    
    def _linkify_host_check(self, item, on_update=False, already_save=None, inheritance_cache=None, old_item_in_rg_use_only_exist=None, modified_data_keys=None):
        # type: (BaseItem, bool, Set[Tuple[unicode, unicode, unicode]], Set[unicode], unicode, Set) -> None
        if item.get_state() in (ITEM_STATE.MERGE_SOURCES, ITEM_STATE.NEW):
            return
        if modified_data_keys is None:
            modified_data_keys = set()
        specific_method = getattr(item, u'_linkify_host_check', None)
        if specific_method and callable(specific_method):
            host_to_save = []
            specific_method(old_item_in_rg_use_only_exist, self._find_item_by_name, host_to_save, modified_data_keys)
            if on_update and host_to_save:
                for host in host_to_save:
                    self._save_item(host, already_save, inheritance_cache=inheritance_cache)
    
    
    def _linkify_value(self, item, to_linkify_property, to_linkify_item_type, to_linkify_item_states, check_exists, updated_items=None):
        link_items = item.get_links(to_linkify_property, to_linkify_item_type)
        if not link_items:
            return
        item_reversed_link = ItemReversedLink(item_type=item.get_type(), item_id=item['_id'], key=to_linkify_property, item_state=item.get_state())
        item_ids = set([i.get('_id') for i in link_items if i.get('exists', False)])
        not_existing_item_names = set((i.get('name') for i in link_items if not i.get('exists', False)))
        not_find_item_ids = item_ids.copy()
        for item_state in to_linkify_item_states:
            # Skip items if we are in work area and the type of items doesn't have one
            if not ITEM_TYPE.has_work_area(to_linkify_item_type) and item_state == ITEM_STATE.WORKING_AREA:
                continue
            
            items_to_links = self._find_items_by_id(item_ids, to_linkify_item_type, items_state=item_state)
            find_ids = set((i['_id'] for i in items_to_links))
            for _id in find_ids:
                not_find_item_ids.discard(_id)
            
            # create an index id -> link_item
            items_to_links_id_index = {}
            for item_to_link in items_to_links:
                if '_id' in item_to_link:
                    items_to_links_id_index[item_to_link['_id']] = item_to_link
            
            # now add the @links on every link_items
            for link_item in link_items:
                link_id = link_item.get('_id', None)
                if not (link_id and link_id in find_ids):
                    continue
                # find the item with the same id and add it to our @links
                item_to_link = items_to_links_id_index[link_id]
                link_item[u'@link'] = link_item.get(u'@link', {})
                link_item[u'@link'][item_state] = item_to_link
                
                if to_linkify_property == 'use':
                    item_to_link.add_son(item)
                
                # now add our reverse info inside the link_item
                item_to_link.add_reverse_link(item_reversed_link)
            
            # manage links that does not exists, add their names in the items_linked_with_name with our id, type and state to find us later
            if not self.linkify_rules[item_state]['manage_non_existing']:
                continue
            for non_existing_item_name in not_existing_item_names:
                non_existing_link_info = self.items_linked_with_name.find_one({'name': non_existing_item_name, 'type': to_linkify_item_type})
                if non_existing_link_info is None:
                    non_existing_link_info = {
                        'name'         : non_existing_item_name,
                        'type'         : to_linkify_item_type,
                        'reverse_links': set()
                    }
                    self.items_linked_with_name.append(non_existing_link_info)
                if item_reversed_link not in non_existing_link_info['reverse_links']:
                    non_existing_link_info['reverse_links'].add(item_reversed_link)
        
        # if check_exists and some ids are not find for any state, we will set the exists to False and try to find the item name
        if check_exists and not_find_item_ids:
            # try to find the item name in others states
            not_find_link_item = (link_item for link_item in link_items if link_item.get('_id', None) in not_find_item_ids)
            for link_item in not_find_link_item:
                found = None
                # If the item link was not found in prod we check it in staging / working area but
                # if we already check in LINKIFY_MANAGE_STATES we don't we recheck LINKIFY_MANAGE_STATES
                if to_linkify_item_states != LINKIFY_MANAGE_STATES:
                    find_items = self._find_merge_state_items(link_item['item_type'], item_states=LINKIFY_MANAGE_STATES, where={'_id': link_item['_id']})
                    if find_items:
                        found = find_items[0]
                
                if not found:
                    trash_info = trash_manager.find_deleted_items_by_id(link_item['_id'], link_item['item_type'])
                    if trash_info:
                        found = ProxySetInexistingLink(trash_info)
                
                if found:
                    item_was_update = item.set_inexisting_link(to_linkify_property, found)
                    if item_was_update and updated_items is not None:
                        updated_items.append(item)
                else:
                    raise ShinkenDatabaseConsistencyError(dict((k, v) for k, v in link_item.iteritems() if k != u'@link'), item[u'_id'], item.get_type(), item.get_state(), LINKIFY_MANAGE_STATES)
    
    
    def _set_attr_with_inheritance(self, initial_item, inheritance_cache):
        # First of all, check for loops and mark templates in loops to be ignored for inheritance
        templates_to_check = initial_item.get_link_items('use', states=[ITEM_STATE.STAGGING], only_exist=True)
        templates_cache = set([t.get_key() for t in templates_to_check])
        templates_to_ignore = set([])
        while templates_to_check:
            current_tpl = templates_to_check.pop(0)
            fathers = current_tpl.get_link_items('use', states=[ITEM_STATE.STAGGING], only_exist=True)
            for father in fathers:
                father_key = father.get_key()
                if father_key in templates_cache:
                    templates_to_ignore.add(father_key)
                else:
                    templates_cache.add(father_key)
                    templates_to_check.insert(0, father)
        
        working_item = self._internal_compute_property_inheritance(initial_item, deep=0, inheritance_cache=inheritance_cache, templates_to_ignore=templates_to_ignore)
        initial_item.update(working_item)
    
    
    def _internal_compute_property_inheritance(self, item, deep, inheritance_cache, templates_to_ignore):
        if deep > MAX_TEMPLATE_LOOKUP_LEVEL:
            return {}
        
        properties_with_plus = DEF_ITEMS[METADATA.get_metadata(item, METADATA.ITEM_TYPE)].get('fields_with_plus', set())
        resolve_item = item.copy()
        
        # keep track of the has_plus value for each link field
        has_plus_inherited = {}
        for item_property in properties_with_plus:
            if item_property in resolve_item:
                item_value = resolve_item[item_property]
                has_plus_inherited[item_property] = item_value.get('has_plus', False)
        # item key is used to identified already inherited items in inheritance_cache
        item_key = item.get_key()
        for template in item.get_link_items('use', states=[ITEM_STATE.STAGGING], only_exist=True):
            if not template.is_enabled():
                continue
            
            template_key = template.get_key()
            # go deeper in the recursion only of the item haven't already been inherited
            if template_key in inheritance_cache or template_key in templates_to_ignore:
                resolve_template = template
            else:
                template_type = template.get_type()
                raw_template = template.get_raw_item(keep_metadata=True, flatten_links=False, keep_links=True)
                resolve_template = self._internal_compute_property_inheritance(get_item_instance(template_type, raw_template), deep + 1, inheritance_cache=inheritance_cache, templates_to_ignore=templates_to_ignore)
            
            for item_property, resolve_template_value in resolve_template.iteritems():
                if not resolve_template_value or item_property in NO_HERITABLE_ATTR or item_property.startswith('@') or get_origin(resolve_template, item_property) == DEFAULT_FROM_INFO:
                    continue
                item_value = resolve_item.get(item_property, None)
                
                if item_property == SERVICE_OVERRIDE:
                    if not item_value and resolve_template_value:
                        # In service override, we copy BaseItem in links, so don't warning about this please
                        resolve_item[item_property] = fast_deepcopy(resolve_template_value, ignore_warnings=True)
                        resolve_item[u'@metadata'][METADATA.FROM][item_property] = METADATA.get_metadata(resolve_template, METADATA.FROM)[item_property]
                    elif item_value and resolve_template_value and not resolve_template_value[u'has_error']:
                        for tpl_link in resolve_template_value[u'links']:
                            if not [i for i in item_value[u'links'] if (i[u'key'] == tpl_link[u'key'] and i[u'check_link'] == tpl_link[u'check_link'] and i.get(u'dfe_key', u'') == tpl_link.get(u'dfe_key', u''))]:
                                item_value[u'links'].append(fast_deepcopy(tpl_link))
                        
                        resolve_item[item_property] = item_value
                        resolve_item[u'@metadata'][METADATA.FROM][item_property] = METADATA.get_metadata(resolve_template, METADATA.FROM)[item_property]
                elif item_property in properties_with_plus:
                    has_plus = has_plus_inherited.get(item_property, False)
                    template_links = resolve_template_value[u'links']
                    if has_plus:
                        current_links = item_value[u'links']
                        # I have a plus and i so I need to extend my links with my template_links
                        for template_link in template_links:
                            if template_link not in current_links:
                                current_links.append(template_link)
                                # now find the FromInfo from the template and add it in our from
                                item_id_or_name = template_link.get('_id', template_link.get('name'))
                                from_infos = [f for f in METADATA.get_metadata(resolve_template, METADATA.FROM)[item_property] if f.value == item_id_or_name]
                                for from_info in from_infos:
                                    safe_add_to_dict_but_uniq(resolve_item[u'@metadata'][METADATA.FROM], item_property, from_info)
                                has_plus_inherited[item_property] = resolve_template_value.get('has_plus', False)
                    elif not item_value and template_links:
                        if item_value is None:
                            # setup default dict link if the son has no links
                            resolve_item[item_property] = {'has_plus': False, 'links': []}
                        resolve_item[item_property]['links'].extend(resolve_template_value['links'])
                        has_plus_inherited[item_property] = resolve_template_value.get('has_plus', False)
                        resolve_item[u'@metadata'][METADATA.FROM][item_property] = METADATA.get_metadata(resolve_template, METADATA.FROM)[item_property]
                else:
                    if not item_value and resolve_template_value:
                        resolve_item[item_property] = resolve_template_value
                        resolve_item[u'@metadata'][METADATA.FROM][item_property] = METADATA.get_metadata(resolve_template, METADATA.FROM)[item_property]
        inheritance_cache.add(item_key)
        return resolve_item
    
    
    def _set_default_value(self, item, item_type):
        props_links = DEF_ITEMS[item_type]['props_links']
        if not DEF_ITEMS[item_type].get('default_values', None):
            default_values = {}
            DEF_ITEMS[item_type]['default_values'] = default_values
            cls = DEF_ITEMS[item_type]['class']
            
            for prop_name, prop_data in cls.properties.iteritems():
                if prop_name in props_links and prop_data.has_default and prop_data.default:
                    # the default value should be a link, and we need to build it
                    if prop_name not in ('check_command', 'event_handler'):
                        default_link = _build_prop_link_for_value(prop_data.default, item_type, self._find_item_by_name)
                        default_values[make_unicode(prop_name)] = default_link
                elif prop_name in props_links:
                    continue
                elif prop_data.has_default:
                    default_values[make_unicode(prop_name)] = make_unicode(prop_data.default)
        else:
            default_values = DEF_ITEMS[item_type]['default_values']
        
        for prop_name, prop_data in default_values.iteritems():
            if prop_name not in item:
                try:
                    from_info = [FromInfo(prop_data, DEFAULT_FROM_INFO, item_type)]
                except Exception:
                    from_info = [FromInfo('', DEFAULT_FROM_INFO, item_type)]
                    logger.warning('Invalid default value item_type [%s] prop_name [%s] prop_data [%s] for item [%s]' % (item_type, prop_name, prop_data, item))
                item.set_value(prop_name, prop_data, from_info, default_value=True)
                # type_to_link = [key for key, _type_list in DEF_ITEMS[item_type]['item_links'].iteritems() if prop_name in _type_list]prop_name in props_links
                for prop_type in props_links.get(prop_name, []):
                    self._linkify_value(item, prop_name, prop_type, LINKIFY_MANAGE_STATES, False)
                # Default realm must be retrieved from configuration
                if self.app and prop_name == 'realm':
                    default_realm = self.app.conf.realms.get_default()
                    realm_name = default_realm.get_name()
                    realm_from_info = [FromInfo(realm_name, DEFAULT_FROM_INFO, item_type)]
                    item.set_value(prop_name, realm_name, realm_from_info, default_value=True)
    
    
    def _relink_non_existing_links(self, item, already_save, item_name=None, item_type=None, inheritance_cache=None):
        start_time = time.time()
        item_name = item_name or item.get_name()
        item_type = item_type or item.get_type()
        item_link_with_name = self.items_linked_with_name.find_one({'name': item_name, 'type': item_type})
        added_links = set()
        items_to_save = []
        
        # logger.debug('_relink_non_existing_links for item [%s] with [%s]' % (item, ','.join(
        #     ('%s-%s-%s' % (reverse_link.item_id, reverse_link.item_type, reverse_link.item_state) for reverse_link in (item_link_with_name['reverse_links'] if item_link_with_name else [])))))
        
        if not item_link_with_name:
            return
        
        items_not_existing_anymore = set()
        for item_def_need_my_link in item_link_with_name['reverse_links']:
            to_update_item = self._find_item_by_id(item_def_need_my_link.item_id, item_def_need_my_link.item_type, item_def_need_my_link.item_state)
            if to_update_item:
                if to_update_item.update_link(item_def_need_my_link.key, item):
                    items_to_save.append(to_update_item)
                    added_links.add(item_def_need_my_link)
            else:
                # the item does not exists anymore
                items_not_existing_anymore.add(item_def_need_my_link)
        
        for to_del in items_not_existing_anymore:
            item_link_with_name['reverse_links'].remove(to_del)
        
        item_link_with_name['reverse_links'] = item_link_with_name['reverse_links'] - added_links
        # if no more named links exists remove it from
        if not item_link_with_name['reverse_links']:
            self.items_linked_with_name.remove({'name': item_name, 'type': item_type})
        # re-save modified items
        # logger.debug('_relink_non_existing_links for item [%s] with [%s] items to save' % (item, len(items_to_save)))
        for item_to_save in items_to_save:
            self._save_item_with_updated_links(item_to_save, already_save, inheritance_cache)
        logger.log_perf(start_time, self, '_relink_non_existing_links [%s] [%s]' % (item_name, item_type), 0.01)
    
    
    def _save_item_with_updated_links(self, item_to_save, already_save, inheritance_cache):
        # type: (BaseItem, Set[Tuple[unicode, unicode, unicode]], Set[unicode]) -> None
        item_type = item_to_save.get_type()
        self.objects[item_to_save.get_state()][item_type].regenerate_index(item_to_save)
        if ITEM_TYPE.is_template(item_type):
            self._regenerate_son_items(item_to_save, inheritance_cache)
        self._linkify_host_check(item_to_save, on_update=True, already_save=already_save, inheritance_cache=inheritance_cache)
        raw_item_to_save = item_to_save.get_raw_item(flatten_links=False)
        raw_item_for_mongo = fast_deepcopy(raw_item_to_save) if self.app and self.app.conf.protect_fields__activate_encryption else raw_item_to_save
        self.data_provider.save_item(raw_item_for_mongo, item_to_save.get_type(), item_to_save.get_state())
    
    
    def _regenerate_item(self, raw_item, item_type, item_state, already_save, inheritance_cache=None):
        # type: (Dict, unicode, unicode, Set[Tuple[unicode, unicode, unicode]], Set[unicode]) -> BaseItem
        start_time = time.time()
        item_id = raw_item['_id']
        item_name = get_name_from_type(item_type, raw_item)
        item_in_rg = self._find_item_by_id(item_id, item_type, item_state)
        already_exist = item_in_rg is not None
        same_but_in_previous_state = None
        if not already_exist and ITEM_TYPE.has_work_area(item_type):
            if item_state == ITEM_STATE.WORKING_AREA:
                same_but_in_previous_state = self._find_item_by_id(item_id, item_type, ITEM_STATE.STAGGING)
            else:
                same_but_in_previous_state = self._find_item_by_id(item_id, item_type, ITEM_STATE.WORKING_AREA)
        if inheritance_cache is None:
            # keep track of all already inherited objects, avoiding recomputing them again and again during the ascendant recursion
            inheritance_cache = set()
        old_item_in_rg_use_only_exist = ''
        
        item_source_hash = {}
        source_info = {}
        if item_state == ITEM_STATE.MERGE_SOURCES:
            item_source_hash = METADATA.get_metadata(raw_item, METADATA.SOURCE_HASH)
            source_info = METADATA.get_metadata(raw_item, METADATA.SOURCE_INFO)
        
        old_item_in_rg = {}
        to_keep_metadata = {}
        old_reverse_links = {}
        if already_exist:
            if hasattr(item_in_rg, 'get_use_only_exist'):
                old_item_in_rg_use_only_exist = item_in_rg.get_use_only_exist()
            old_item_in_rg = get_item_instance(item_type, item_in_rg.get_raw_item(keep_metadata=True, flatten_links=False))
            old_reverse_links = METADATA.get_metadata(old_item_in_rg, METADATA.REVERSE_LINKS, set())
            for to_keep_metadata_name, default_value in TO_KEEP_METADATA:
                to_keep_metadata[to_keep_metadata_name] = METADATA.get_metadata(old_item_in_rg, to_keep_metadata_name, default_value)
            item_in_rg.clear()
        else:
            item_in_rg = get_item_instance(item_type)
            if same_but_in_previous_state:
                old_reverse_links = METADATA.get_metadata(same_but_in_previous_state, METADATA.REVERSE_LINKS, set())
                to_keep_metadata = {}
                for to_keep_metadata_name, default_value in TO_KEEP_METADATA:
                    to_keep_metadata[to_keep_metadata_name] = METADATA.get_metadata(same_but_in_previous_state, to_keep_metadata_name, default_value)
                
                reverse_links = same_but_in_previous_state.get_reverse_links().copy()
                for reverse_link in reverse_links:
                    item_to_relink = self._find_item_by_id(reverse_link.item_id, item_type=reverse_link.item_type, item_state=reverse_link.item_state)
                    if item_to_relink:
                        for link in item_to_relink.get_links(reverse_link.key, linked_items_type=item_type):
                            if link[u'exists'] and link[u'_id'] == item_id:
                                link[u'@link'][item_state] = item_in_rg
                to_keep_metadata[METADATA.CHECKS] = []
                for check in METADATA.get_metadata(same_but_in_previous_state, METADATA.CHECKS, []):
                    v = check.copy()
                    if v[u'from'] == same_but_in_previous_state:
                        v[u'from'] = item_in_rg
                        reverse_check_link = METADATA.get_metadata(v[u'check'], METADATA.REVERSE_LINKS_CHECKS)
                        for reverse_host in reverse_check_link[:]:
                            if reverse_host == same_but_in_previous_state:
                                reverse_check_link.append(item_in_rg)
                    to_keep_metadata[METADATA.CHECKS].append(v)
        
        # STEP - Generate the item metadata
        new_metadata = self._initialize_metadata(item_name, item_state, item_type)
        item_definition = DEF_ITEMS[item_type]
        
        if already_exist or same_but_in_previous_state:
            new_metadata[METADATA.REVERSE_LINKS] = fast_deepcopy(old_reverse_links)
            for to_keep_metadata_name, value in to_keep_metadata.iteritems():
                if value:
                    new_metadata[to_keep_metadata_name] = value
            if item_state == ITEM_STATE.MERGE_SOURCES:
                new_metadata[METADATA.SOURCE_HASH] = item_source_hash
                new_metadata[METADATA.SOURCE_INFO] = source_info
        
        item_in_rg.set_value('_id', item_id, FromInfo(item_id, item_id, item_type), default_value=True)
        item_in_rg.set_value(u'@metadata', new_metadata)
        
        # STEP - Building raw prop item
        # Drop @metadata in raw_item : be sure to not erase the new_metadata :)
        raw_item.pop(u'@metadata', None)
        for prop, value in raw_item.iteritems():
            item_in_rg.set_value(prop, value)
        modified_data_keys = self._get_changed_data_keys(old_item_in_rg, item_in_rg)
        # STEP - compute_change
        self._update_item_with_changes(item_in_rg, item_type)
        
        # STEP - Update or create index
        if already_exist:
            self.objects[item_state][item_type].regenerate_index(item_in_rg)
        else:
            self.objects[item_state][item_type].append(item_in_rg)
        
        # STEP - Create a tree of objects links.
        self._linkify_item(item_in_rg, item_definition)
        
        # STEP - find all items that can link on my name before I exists
        if item_state in LINKIFY_MANAGE_STATES:
            self._relink_non_existing_links(item_in_rg, already_save, item_name=item_name, item_type=item_type, inheritance_cache=inheritance_cache)
        
        # STEP - If item was a template we update item son
        if ITEM_TYPE.is_template(item_type):
            for item_state in self.inheritance_handle_states:
                self._regenerate_son_items(item_in_rg, inheritance_cache)
        
        # STEP - Linkify host and check.
        self._linkify_host_check(item_in_rg, on_update=True, already_save=already_save, inheritance_cache=inheritance_cache, old_item_in_rg_use_only_exist=old_item_in_rg_use_only_exist, modified_data_keys=modified_data_keys)
        
        # STEP - Foreach inherited objects, take all attributes  and set them to my object.
        self._set_attr_with_inheritance(item_in_rg, inheritance_cache=inheritance_cache)
        
        # STEP - Get default values from heritage
        self._set_default_value(item_in_rg, item_type)
        
        logger.log_perf(start_time, self, 'regenerate [%s] [%s] [%s]' % (item_state, item_type, item_name))
        return item_in_rg
    
    
    @staticmethod
    def _get_changed_data_keys(old_item, new_item):
        # type: (BaseItem, BaseItem) -> Set[unicode]
        # Here we build two sets of "key value" pairs (tuples) of Data both items.
        old_data_keys_values = set(old_item.get_data_as_list()) if old_item else set()
        new_data_keys_values = set(new_item.get_data_as_list()) if new_item else set()
        
        # Here we use the 'symmetric difference' of two sets to get all items not in common in the sets.
        keys_values_that_changed = old_data_keys_values.symmetric_difference(new_data_keys_values)
        # This means that we got here only Data that changed from old to new item. It includes removed and added ones.
        # The main goal is to get only the keys of changed Data so we get the keys (index 0 of tuples) with a list
        # comprehension and we cast it into a set to only get one occurrence of each.
        uniques_keys_that_changes = set([key_value[0] for key_value in keys_values_that_changed])
        return uniques_keys_that_changes
    
    
    def _update_item_with_changes(self, item, item_type):
        item_id = item['_id']
        transaction = get_transaction_object()
        if transaction:
            raw_change = self.data_provider.find_in_transaction(item_id, item_type=item_type, item_state=ITEM_STATE.CHANGES)
        else:
            raw_change = self.data_provider.find_item_by_id(item_id, item_type, ITEM_STATE.CHANGES)
        if raw_change and 'changes' in raw_change:
            DataProviderMetadata._unserialize_change(item_type, raw_change)
            METADATA.update_metadata(item, METADATA.CHANGES, make_unicode(raw_change['changes']))
            if item.have_manual_changes():
                METADATA.update_metadata(item, METADATA.SOURCE_STATE, SOURCE_STATE.CHANGE)
    
    
    @staticmethod
    def _initialize_metadata(item_name, item_state, item_type):
        item_definition = DEF_ITEMS[item_type]
        new_metadata = {
            METADATA.FROM         : dict(),
            METADATA.RAW_ITEM     : dict(),
            METADATA.STATE        : make_unicode(item_state),
            METADATA.ITEM_TYPE    : make_unicode(item_type),
            METADATA.NAME         : make_unicode(item_name),
            METADATA.TABLE        : make_unicode(item_definition['table']),
            METADATA.SOURCE_STATE : SOURCE_STATE.NEW if item_state == ITEM_STATE.NEW else SOURCE_STATE.IMPORT,
            METADATA.REVERSE_LINKS: set(),
        }
        return new_metadata
    
    
    def _regenerate_son_items(self, item_in_rg, inheritance_cache):
        # type: (Union[BaseItem, UseMixin], Set) -> None
        
        for son_item in item_in_rg.get_sons():
            son_raw_item = son_item.get_raw_item(keep_metadata=True, flatten_links=False)
            son_type = son_item.get_type()
            son_raw_item = get_item_instance(son_type, son_raw_item)
            
            son_raw_item[u'@metadata'][u'from'] = dict()
            self._build_initial_from_info(son_raw_item)
            
            self._linkify_item(son_raw_item, DEF_ITEMS[son_type])
            self._set_attr_with_inheritance(son_raw_item, inheritance_cache=inheritance_cache)
            
            if son_raw_item != son_item:
                son_item.clear()
                son_item.update(son_raw_item)
                if ITEM_TYPE.is_template(son_type):
                    self._regenerate_son_items(son_item, inheritance_cache)
            self._set_default_value(son_item, son_type)
    
    
    @staticmethod
    def _unserialize_change(item_type, raw_change):
        for property_name, change_prop in raw_change['changes'].iteritems():
            _separator = get_property_separator(item_type, property_name)
            if len(change_prop) > 2:
                _from_dict = change_prop[2]
                if isinstance(_from_dict, SourceInfoProperty):
                    continue
            else:
                raw_change['changes'][property_name].append(None)
                # some change can be in very old format
                _from_dict = {
                    'property_key'  : property_name,
                    'property_type' : SourceInfoProperty.UNIQUE_TYPE,
                    'property_value': [],
                    'property_force': False
                }
            raw_change['changes'][property_name][2] = SourceInfoProperty.from_dict(_from_dict, _separator)
    
    
    def find_in_transaction(self, item_id, item_type, item_state):
        return self.data_provider.find_in_transaction(item_id, item_type, item_state)
    
    
    def start_transaction(self, transaction_uuid):
        self.rw_lock.open_transaction()
        self.data_provider.start_transaction(transaction_uuid)
    
    
    def commit_transaction(self, transaction_uuid):
        return_val = self.data_provider.commit_transaction(transaction_uuid)
        self.rw_lock.close_transaction()
        return return_val
    
    
    def save_history(self, history_entry):
        # type: (HistoryEntry) -> NoReturn
        return self.data_provider.save_history(history_entry)
    
    
    def find_history_for_item(self, item_id, item_type):
        # type: (unicode, unicode) -> List
        return self.data_provider.find_history_for_item(item_id, item_type)
