#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2018:
# This file is part of Shinken Enterprise, all rights reserved.

import copy
import threading
import time
import uuid
from threading import RLock

import item_lists
import massive_change
import shinkensolutions.shinkenjson as json
import try_check
from item_lists import elements_generic
from massive_change import massive_change_call
from shinken.log import logger
from shinken.misc.fast_copy import fast_deepcopy
from shinken.misc.type_hint import Optional, List, TYPE_CHECKING
from shinken.objects.host import VIEW_CONTACTS_DEFAULT_VALUE
from shinken.objects.resultmodulation import ModulationRule
from shinken.property import AclShareProp
from shinken.util import make_unicode, blank_context
from try_check import run_check
from ...business.item_controller import exceptions
from ...business.item_controller.exceptions import BusinessException, UnauthorizedSaveException, SaveException, ItemNotFoundException
from ...business.item_controller.item_running_state import ItemContext, ACTIONS
from ...business.source.sourceinfoproperty import SourceInfoProperty
from ...business.sync_ui_common import syncuicommon
from ...dao.checks_inheritance import lookup_items_templates, inherited_checks_from_hosts_templates, TEMPLATE_STATUS
from ...dao.crypto import FrontendCipher, _match_protected_property, PROTECTED_DEFAULT_TAG, PROTECTED_TAG_FROM_CHANGE
from ...dao.datamanagerV2 import DataManagerV2
from ...dao.def_items import ITEM_STATE, ITEM_TYPE, DEF_ITEMS, METADATA, LINKIFY_MANAGE_STATES, SERVICE_OVERRIDE, NOT_TO_LOOK, UNFLATTENNED_PROPERTIES_IN_FRONT
from ...dao.helpers import get_name_from_type, get_default_value, get_inherited_without_default, update_frontend_links_into_links, escape_XSS
from ...dao.item_saving_formatter import update_frontend_service_override_links_into_links
from ...dao.items import BaseItem, get_item_instance
from ...dao.transactions.transactions import DBTransaction
from ...front_end import helper
from ...front_end.helper_frontend_response import FrontEndSaveResponse
from ...front_end.object.messages import Messages, Message, MESSAGE, ValidatorMessages

if TYPE_CHECKING:
    from ...synchronizerdaemon import Synchronizer

MULTI_MASS_ACTION_UPDATE = {}

# Identify the dummy property for submit /reject comments
object_edition_lock = RLock()

# These vars are set by SynchronizerDaemon at runtime through the set_app() function
app = None  # type: Optional[Synchronizer]
datamanagerV2 = None  # type: Optional[DataManagerV2]


def set_app(_app):
    global app
    global datamanagerV2
    
    app = _app
    datamanagerV2 = _app.datamanagerV2
    
    item_lists.app = _app
    item_lists.load_column_configuration()
    
    try_check.app = _app
    try_check.datamanagerV2 = datamanagerV2
    
    massive_change.app = _app
    massive_change.datamanagerV2 = datamanagerV2
    
    helper.app = _app
    exceptions.app = _app


def elements_checksbytemplates():
    app.response.content_type = 'application/json'
    user = app.get_user_auth()
    
    use_links = [item_link for item_link in json.loads(app.request.POST.get('links', '{}').decode('utf-8', 'ignore'))['links'] if item_link]
    my_name = app.request.POST.get('my_name', '').decode('utf-8', 'ignore')
    item_type = app.request.POST.get('item_type', '')
    my_uuid = app.request.POST.get('uuid', '')
    
    use_names = update_frontend_links_into_links('use', use_links, item_type, datamanagerV2)
    
    inherited_checks = inherited_checks_from_hosts_templates(use_names, my_name, my_uuid, item_type, datamanagerV2, with_superfluous_checks=True)
    
    for host_name, checks in inherited_checks.iteritems():
        inherited_checks[host_name] = [app.frontend_cipher.cipher(e, item_type=item_type, user=user) for e in checks]
    
    return json.dumps(inherited_checks, ensure_ascii=False).encode('utf8')


def _get_specific_item(item_id, item_type, current_page=None):
    user = app.get_user_auth()
    is_admin = user.is_admin()
    item_is_new = (app.request.GET.get('new', '0') == '1')
    
    if current_page == 'working_area':
        item_state = ITEM_STATE.NEW if item_is_new else ITEM_STATE.WORKING_AREA
        backup_state = ITEM_STATE.STAGGING
        _working_area = True
        flatten_links_states = (ITEM_STATE.WORKING_AREA, ITEM_STATE.STAGGING)
    else:
        current_page = 'stagging'
        item_state = ITEM_STATE.NEW if item_is_new else ITEM_STATE.STAGGING
        backup_state = ITEM_STATE.WORKING_AREA
        _working_area = False
        flatten_links_states = (ITEM_STATE.STAGGING, ITEM_STATE.WORKING_AREA)
    
    item = datamanagerV2.find_item_by_id(item_id, item_type, item_state)
    # If a user ask from staging a item create in working area it isn't in ITEM_STATE.STAGGING but in ITEM_STATE.WORKING_AREA base
    if not item:
        item = datamanagerV2.find_item_by_id(item_id, item_type, backup_state)
    
    if not item:
        logger.info("[%s-%s] not found" % (item_type, item_id))
        return app.abort(404, app._('element.no_item_with_this_id') % (app._('type.' + item_type[:-1]), item_id))
    
    result = syncuicommon.check_acl(item, user, current_page)
    can_edit = result['can_edit']
    can_view = result['can_view']
    block_acl_cause = result['cause']
    
    if not can_view:
        return app.abort(404, app._('element.no_item_with_this_id') % (app._('type.' + item_type[:-1]), item_id))
    
    changes = {}
    if is_admin and not (current_page == 'stagging' and item_type == ITEM_TYPE.HOSTS):
        changes = fast_deepcopy(item.get_changes(), additional_dispatcher={SourceInfoProperty: SourceInfoProperty.fast_copy_source_info_property})
        for _property_name in changes:
            changes[_property_name][0] = escape_XSS(changes[_property_name][0])
            changes[_property_name][1] = escape_XSS(changes[_property_name][1])
        app.frontend_cipher.cipher({'changes': changes}, item_type=item_type, item_state=ITEM_STATE.CHANGES, user=user)
    
    unflattened_item = item.get_raw_item(flatten_links=False)
    service_overrides_links = []
    for override in unflattened_item.get(SERVICE_OVERRIDE, {}).get('links', []):
        service_overrides = copy.copy(override)
        service_overrides['key'] = override['key']
        service_overrides['value'] = datamanagerV2.flatten_overridden_property(DEF_ITEMS[item_type]['check_type'], override)
        service_overrides['value_with_link'] = override['value']
        service_overrides_links.append(service_overrides)
    
    services_in_service_overrides = []
    if item_type in (ITEM_TYPE.HOSTS, ITEM_TYPE.HOSTTPLS, ITEM_TYPE.CLUSTERS, ITEM_TYPE.CLUSTERTPLS):
        check_types = set([DEF_ITEMS[item_type]['check_type'], DEF_ITEMS[DEF_ITEMS[item_type]['template']]['check_type']])
        for check_type in check_types:
            services_in_service_overrides.extend(item.get_link_items(SERVICE_OVERRIDE, only_exist=True, item_type=check_type))
    
    item = item.get_raw_item(flatten_links=flatten_links_states, keep_metadata=[METADATA.ITEM_TYPE])
    
    item['editable'] = '1' if can_edit else '0'
    
    for item_property, item_value in item.iteritems():
        if isinstance(item_value, bool):
            item[item_property] = '1' if item_value else '0'
        # Trail leading _ in duplicate foreach field
        if item_property == 'duplicate_foreach':
            item[item_property] = item_value[1:]
        elif item_property in UNFLATTENNED_PROPERTIES_IN_FRONT:
            item[item_property] = copy.copy(unflattened_item[item_property])
            if item_property == SERVICE_OVERRIDE:
                item[item_property]['links'] = service_overrides_links
        elif isinstance(item_value, list) and item_property != '_SYNC_KEYS' and item_property != SERVICE_OVERRIDE:
            item[item_property] = ','.join(item_value)
    
    METADATA.update_metadata(item, METADATA.NEW, item_is_new)
    METADATA.update_metadata(item, METADATA.HAS_DATA, DEF_ITEMS[item_type]['has_data'])
    
    if item_type in (ITEM_TYPE.HOSTS, ITEM_TYPE.HOSTTPLS, ITEM_TYPE.CLUSTERS, ITEM_TYPE.CLUSTERTPLS):
        shinken_elements_lists = _build_shinken_elements_lists([
            ITEM_TYPE.TIMEPERIODS,
            ITEM_TYPE.CONTACTS,
            ITEM_TYPE.CONTACTGROUPS,
            ITEM_TYPE.BUSINESSIMPACTMODULATIONS,
            ITEM_TYPE.MACROMODULATIONS,
            ITEM_TYPE.RESULTMODULATIONS,
            ITEM_TYPE.COMMANDS,
            ITEM_TYPE.ESCALATIONS,
        ], services_in_service_overrides)
    else:
        shinken_elements_lists = {}
    
    return {
        'app'                   : app,
        'user'                  : user,
        'item'                  : app.frontend_cipher.cipher(item, item_type=item_type, user=user),
        'shinken_elements_lists': shinken_elements_lists,
        'helper'                : app.helper,
        'is_tpl'                : ITEM_TYPE.is_template(item_type),
        'protected_fields'      : DEF_ITEMS[item_type].get('protected_fields', []),
        'item_type'             : item_type,
        'name'                  : get_name_from_type(item_type, item).replace(' ', u'\u00a0'),  # Here the name is only use for to show and need to be unbreakable
        'changes'               : changes,
        'pending'               : item_is_new,
        'working_area'          : _working_area,
        'can_view'              : can_view,
        'can_edit'              : can_edit,
        'block_acl_cause'       : block_acl_cause
    }


def _build_shinken_elements_lists(types_to_list, services_in_service_overrides=None):
    services_in_service_overrides = services_in_service_overrides if services_in_service_overrides else []
    shinken_elements_lists = {}
    
    for linked_item_type in types_to_list:
        name_field = DEF_ITEMS[linked_item_type]['key_name']
        shinken_elements_lists[linked_item_type] = []
        for linked_item in datamanagerV2.find_merge_state_items(linked_item_type, (ITEM_STATE.NEW, ITEM_STATE.STAGGING, ITEM_STATE.WORKING_AREA)):
            linked_item_for_list = _format_item_for_shinken_elements_lists(linked_item, name_field, linked_item_type)
            shinken_elements_lists[linked_item_type].append(linked_item_for_list)
    
    for service in services_in_service_overrides:
        linked_item_type = METADATA.get_metadata(service, METADATA.ITEM_TYPE)
        name_field = DEF_ITEMS[linked_item_type]['key_name']
        linked_item_for_list = _format_item_for_shinken_elements_lists(service, name_field, linked_item_type)
        
        if not shinken_elements_lists.get(linked_item_type, None):
            shinken_elements_lists[linked_item_type] = []
        shinken_elements_lists[linked_item_type].append(linked_item_for_list)
    
    return shinken_elements_lists


def _format_item_for_shinken_elements_lists(linked_item, name_field, linked_item_type):
    linked_item_for_list = {
        name_field: linked_item[name_field],
        '_id'     : linked_item['_id'],
        'enabled' : linked_item.get('enabled', '1')
    }
    METADATA.update_metadata(linked_item_for_list, METADATA.ITEM_TYPE, linked_item_type)
    
    linked_item_state = METADATA.get_metadata(linked_item, METADATA.STATE)
    METADATA.update_metadata(linked_item_for_list, METADATA.STATE, linked_item_state)
    
    if linked_item_type in (ITEM_TYPE.RESULTMODULATIONS, ITEM_TYPE.BUSINESSIMPACTMODULATIONS, ITEM_TYPE.MACROMODULATIONS):
        linked_item_for_list['modulation_period'] = linked_item.flatten_prop('modulation_period', LINKIFY_MANAGE_STATES)
    
    if linked_item_type == ITEM_TYPE.CONTACTS:
        if linked_item.is_admin():
            linked_item_for_list['is_admin'] = '1'
        if linked_item.is_expert():
            linked_item_for_list['is_expert'] = '1'
    return linked_item_for_list


def list_elements(item_type):
    return elements_generic(item_type)


def new_host():
    return new_object(ITEM_TYPE.HOSTS)


def new_host_tpl():
    return new_object(ITEM_TYPE.HOSTTPLS)


def new_hostgroup():
    return new_object(ITEM_TYPE.HOSTGROUPS)


def new_cluster():
    return new_object(ITEM_TYPE.CLUSTERS)


def new_cluster_tpl():
    return new_object(ITEM_TYPE.CLUSTERTPLS)


def new_services_host():
    return new_object(ITEM_TYPE.SERVICESHOSTS)


def new_services_host_tpl():
    return new_object(ITEM_TYPE.SERVICESHOSTTPLS)


def new_services_cluster():
    return new_object(ITEM_TYPE.SERVICESCLUSTERS)


def new_services_cluster_tpl():
    return new_object(ITEM_TYPE.SERVICESCLUSTERTPLS)


def new_service_tpl():
    return new_object(ITEM_TYPE.SERVICETPLS)


def new_contact():
    return new_object(ITEM_TYPE.CONTACTS)


def new_contact_tpl():
    return new_object(ITEM_TYPE.CONTACTTPLS)


def new_contactgroup():
    return new_object(ITEM_TYPE.CONTACTGROUPS)


def new_escalation():
    return new_object(ITEM_TYPE.ESCALATIONS)


def new_notificationway():
    return new_object(ITEM_TYPE.NOTIFICATIONWAYS)


def new_businessimpactmodulation():
    return new_object(ITEM_TYPE.BUSINESSIMPACTMODULATIONS)


def new_macromodulation():
    return new_object(ITEM_TYPE.MACROMODULATIONS)


def new_resultmodulation():
    item = new_object(ITEM_TYPE.RESULTMODULATIONS)
    item['modulation_rules'] = [ModulationRule()]
    return item


def new_command():
    return new_object(ITEM_TYPE.COMMANDS)


def new_timeperiod():
    return new_object(ITEM_TYPE.TIMEPERIODS)


def new_object(item_type):
    user = app.get_user_auth()
    is_tpl = ITEM_TYPE.is_template(item_type)
    item = {'_id': uuid.uuid1().hex}
    METADATA.update_metadata(item, METADATA.ITEM_TYPE, item_type)
    METADATA.update_metadata(item, METADATA.IN_CREATION, True)
    METADATA.update_metadata(item, METADATA.HAS_DATA, DEF_ITEMS[item_type]['has_data'])
    
    if item_type == ITEM_TYPE.HOSTS:
        _default_view_contacts = get_default_value(ITEM_TYPE.HOSTS, 'view_contacts')[0] or VIEW_CONTACTS_DEFAULT_VALUE.NOBODY
        
        if _default_view_contacts != VIEW_CONTACTS_DEFAULT_VALUE.EVERYONE:
            item['view_contacts'] = user['contact_name']
        item['notification_contacts'] = user['contact_name']
        item['edition_contacts'] = user['contact_name']
    
    if item_type in (ITEM_TYPE.HOSTS, ITEM_TYPE.HOSTTPLS, ITEM_TYPE.CLUSTERS, ITEM_TYPE.CLUSTERTPLS):
        shinken_elements_lists = _build_shinken_elements_lists([
            ITEM_TYPE.TIMEPERIODS,
            ITEM_TYPE.CONTACTS,
            ITEM_TYPE.CONTACTGROUPS,
            ITEM_TYPE.BUSINESSIMPACTMODULATIONS,
            ITEM_TYPE.MACROMODULATIONS,
            ITEM_TYPE.RESULTMODULATIONS,
            ITEM_TYPE.COMMANDS,
            ITEM_TYPE.ESCALATIONS
        ])
    else:
        shinken_elements_lists = {}
    
    return {
        'app'                   : app,
        'user'                  : user,
        'item'                  : app.frontend_cipher.cipher(item, item_type, user=user),
        'shinken_elements_lists': shinken_elements_lists,
        'protected_fields'      : DEF_ITEMS[item_type].get('protected_fields', []),
        'name'                  : '',
        'helper'                : app.helper,
        'is_tpl'                : is_tpl,
        'is_new_object'         : True,
        'item_type'             : item_type,
        'changes'               : {},
        'pending'               : False,
        'can_edit'              : True,
        'can_view'              : True,
        'block_acl_cause'       : '',
    }


def edit_host(item_id):
    return _get_specific_item(item_id, ITEM_TYPE.HOSTS)


def edit_hosttpl(item_id):
    return _get_specific_item(item_id, ITEM_TYPE.HOSTTPLS)


def edit_hostgroup(item_id):
    return _get_specific_item(item_id, ITEM_TYPE.HOSTGROUPS)


def edit_cluster(item_id):
    return _get_specific_item(item_id, ITEM_TYPE.CLUSTERS)


def edit_clustertpl(item_id):
    return _get_specific_item(item_id, ITEM_TYPE.CLUSTERTPLS)


def edit_services_clusters(item_id):
    return _get_specific_item(item_id, item_type=ITEM_TYPE.SERVICESCLUSTERS)


def edit_services_cluster_tpls(item_id):
    return _get_specific_item(item_id, item_type=ITEM_TYPE.SERVICESCLUSTERTPLS)


def edit_services_hosts(item_id):
    return _get_specific_item(item_id, item_type=ITEM_TYPE.SERVICESHOSTS)


def edit_services_hosts_override(item_id):
    return _get_specific_item(item_id, item_type=ITEM_TYPE.SERVICESHOSTS)


def edit_services_host_tpls(item_id):
    return _get_specific_item(item_id, item_type=ITEM_TYPE.SERVICESHOSTTPLS)


def edit_service_tpls(item_id):
    return _get_specific_item(item_id, ITEM_TYPE.SERVICETPLS)


def edit_contact(item_id):
    return _get_specific_item(item_id, ITEM_TYPE.CONTACTS)


def edit_contacttpl(item_id):
    return _get_specific_item(item_id, ITEM_TYPE.CONTACTTPLS)


def edit_contactgroup(item_id):
    return _get_specific_item(item_id, ITEM_TYPE.CONTACTGROUPS)


def edit_command(item_id):
    return _get_specific_item(item_id, ITEM_TYPE.COMMANDS)


def edit_timeperiod(item_id):
    return _get_specific_item(item_id, ITEM_TYPE.TIMEPERIODS)


def edit_businessimpactmodulation(item_id):
    return _get_specific_item(item_id, ITEM_TYPE.BUSINESSIMPACTMODULATIONS)


def edit_macromodulation(item_id):
    return _get_specific_item(item_id, ITEM_TYPE.MACROMODULATIONS)


def edit_resultmodulation(item_id):
    return _get_specific_item(item_id, ITEM_TYPE.RESULTMODULATIONS)


def edit_notificationway(item_id):
    return _get_specific_item(item_id, ITEM_TYPE.NOTIFICATIONWAYS)


def edit_escalation(item_id):
    return _get_specific_item(item_id, ITEM_TYPE.ESCALATIONS)


def get_service_override(host_type, host_id, service_type, service_id, apply_on_type):
    user = app.get_user_auth()
    host_cluster_templates = [
        datamanagerV2.find_item_by_name(tpl_name, ITEM_TYPE.get_template_type(apply_on_type), ITEM_STATE.STAGGING) or
        datamanagerV2.find_item_by_name(tpl_name, ITEM_TYPE.get_template_type(apply_on_type), ITEM_STATE.NEW) for tpl_name in
        app.request.GET.get('tpl', '').split(',') if tpl_name]
    check_name = make_unicode(app.request.GET.get('name', ''))
    dfe_name = make_unicode(app.request.GET.get('dfe_name', ''))
    force_clean = make_unicode(app.request.GET.get('force_clean', ''))
    force_disable = make_unicode(app.request.GET.get('force_disable', ''))
    dom_uuid = make_unicode(app.request.GET.get('dom-uuid', ''))
    is_host_editable = app.request.GET.get('editable', '1')
    check_context = ItemContext(app, service_id, service_type)
    host_context = ItemContext(app, host_id, host_type)
    _item_state = make_unicode(app.request.GET.get('item-state', host_context.from_state))
    checks = datamanagerV2.find_merge_state_items(item_type=service_type, item_states=LINKIFY_MANAGE_STATES, where={'_id': service_id})  # type: List[BaseItem]
    if not checks:
        return app.abort(404, 'There is no [%s] with this id [%s] in [%s]' % (app._('type.' + service_type[:-1]), service_id, check_context.item_state))
    
    host = datamanagerV2.find_item_by_id(host_id, item_type=host_type, item_state=_item_state)  # type: BaseItem
    if not host and not host_context.in_creation:
        return app.abort(404, 'There is no [%s] with this id [%s] in [%s]' % (app._('type.' + host_type[:-1]), host_id, _item_state))
    
    check = checks[0]
    data_on_check = set()
    inherited_without_defaults = get_inherited_without_default(datamanagerV2, check, data_on_item=data_on_check)
    from_info = {}
    data_not_on_check = set()
    for tpl in host_cluster_templates:
        if not tpl:
            continue
        resolve_template_value = tpl.get(SERVICE_OVERRIDE, {})
        if not resolve_template_value:
            continue
        for override in resolve_template_value['links']:
            link_to_service = override['check_link']
            if (link_to_service.get('_id', '') == service_id and dfe_name == override.get('dfe_key', '')) or link_to_service.get('name', '') == check_name:
                if override['key'] == 'check_command_args' and inherited_without_defaults.get('check_command', None):
                    cmd = inherited_without_defaults['check_command'].split('!')[0]
                    inherited_without_defaults['check_command'] = '%s!%s' % (cmd, override['value'])
                    from_info['check_command'] = {'item_name': tpl.get_name(), 'item_id': tpl['_id'], 'item_type': tpl.get_type()}
                else:
                    flattened_value = datamanagerV2.flatten_overridden_property(check.get_type(), override)
                    inherited_without_defaults[override['key']] = flattened_value
                    from_info[override['key']] = {'item_name': tpl.get_name(), 'item_id': tpl['_id'], 'item_type': tpl.get_type()}
                    if override['key'] not in data_on_check and override['key'].startswith('_') and override['key'] not in NOT_TO_LOOK:
                        data_not_on_check.add(override['key'])
    
    METADATA.update_metadata(inherited_without_defaults, METADATA.WINDOW_UUID, dom_uuid)
    METADATA.update_metadata(inherited_without_defaults, METADATA.NAME, check_name)
    METADATA.update_metadata(inherited_without_defaults, METADATA.DFE_NAME, dfe_name)
    METADATA.update_metadata(inherited_without_defaults, METADATA.FROM, from_info)
    
    if force_disable:
        can_edit = False
    elif is_host_editable == '0':
        can_edit = False
    elif host_context.in_creation:
        can_edit = True
    else:
        result = syncuicommon.check_acl(host, host_context.user, _item_state)
        can_edit = result['can_edit']
    
    service_overrides = {
        '_id': inherited_without_defaults['_id']
    }
    if not force_clean:
        if host:
            host = host.get_raw_item(keep_metadata=True, flatten_links=False)
        if host and host.get(SERVICE_OVERRIDE):
            for override in host.get(SERVICE_OVERRIDE, {}).get('links', []):
                if not (override['check_link'].get('name', None) == check_name or override['check_link'].get('_id', None) == service_id):
                    continue
                _dfe_key_override = override.get('dfe_key', '')
                if dfe_name and _dfe_key_override and dfe_name != _dfe_key_override:
                    continue
                
                flattened_value = datamanagerV2.flatten_overridden_property(check.get_type(), override)
                service_overrides[override['key']] = flattened_value
    
    service_overrides['editable'] = '1' if can_edit else '0'
    data_not_on_check.update([data for data in service_overrides if data.startswith('_') and data not in NOT_TO_LOOK and data not in data_on_check])
    
    return {
        'app'                       : app,
        'user'                      : host_context.user,
        'shinken_element_overridden': app.frontend_cipher.cipher(service_overrides, item_type=ITEM_TYPE.HOSTS, user=user),
        'shinken_element_original'  : app.frontend_cipher.cipher(inherited_without_defaults, item_type=ITEM_TYPE.HOSTS, user=user),
        'helper'                    : app.helper,
        'dfe_name'                  : dfe_name,
        'is_tpl'                    : ITEM_TYPE.is_template(service_type),
        'protected_fields'          : DEF_ITEMS[service_type].get('protected_fields', []),
        'item_type'                 : service_type,
        'data_not_on_check'         : data_not_on_check,
        'data_on_check'             : data_on_check,
    }


def elements_item_type_tpls_by_names(tpl_item_type, item_type):
    app.response.content_type = 'application/json'
    
    links = [link for link in json.loads(app.request.POST.get('links', '{}').decode('utf-8', 'ignore'))['links']]
    
    my_name = app.request.POST.get('my_name', '').decode('utf-8', 'ignore')
    linear_result = []
    
    names = update_frontend_links_into_links('use', links, item_type, datamanagerV2)
    
    additional_where = {}
    if app.request.POST.get('is_cluster', '0') == '1':
        additional_where = {'is_cluster': '1'}
    
    lookup_items_templates(item_type, names, linear_result, datamanagerV2, current_branch=[(my_name, item_type, TEMPLATE_STATUS.USEFUL)], add_new=True, additional_where=additional_where)
    
    # Encrypt each item in linear_result
    return json.dumps([app.frontend_cipher.cipher(e, item_type=tpl_item_type, user=app.get_user_auth()) for e in linear_result])


def elements_hosttpls_by_names():
    item_type = app.request.POST.get('item_type', '').decode('utf-8', 'ignore')
    return elements_item_type_tpls_by_names(ITEM_TYPE.HOSTTPLS, item_type)


def elements_clustertpls_by_names():
    item_type = app.request.POST.get('item_type', '').decode('utf-8', 'ignore')
    return elements_item_type_tpls_by_names(ITEM_TYPE.CLUSTERTPLS, item_type)


def elements_servicepls_by_names():
    item_type = app.request.POST.get('item_type', '').decode('utf-8', 'ignore')
    return elements_item_type_tpls_by_names(ITEM_TYPE.SERVICETPLS, item_type)


def elements_contacttpls_by_names():
    item_type = app.request.POST.get('item_type', '').decode('utf-8', 'ignore')
    return elements_item_type_tpls_by_names(ITEM_TYPE.CONTACTTPLS, item_type)


def disable_object(item_type, item_id):
    context = ItemContext(app, item_id, item_type=item_type, action=ACTIONS.DISABLE_OBJECT)
    return _set_enabled(item_type, context, '0')


def enable_object(item_type, item_id):
    context = ItemContext(app, item_id, item_type=item_type, action=ACTIONS.ENABLE_OBJECT)
    return _set_enabled(item_type, context, '1')


def _set_enabled(_item_type, context, enabled):
    validator_message = ValidatorMessages()
    try:
        app.state_controller.set_enable(context, enabled)
    except BusinessException as e:
        validator_message.add_message(Message(MESSAGE.STATUS_CRITICAL, e.text))
        response = FrontEndSaveResponse(validator_message, return_code=e.code)
        return response.get_response()
    response = FrontEndSaveResponse(validator_message)
    return response.get_response()


# Cloning object means you clone ALL properties of this objects,
# so ACL too (users permissions)
def clone_object(item_type, item_id, transform=None):
    context = ItemContext(app, item_id, item_type, action=ACTIONS.CLONE_OBJECT)
    try:
        app.state_controller.clone_object(context, transform)
    except ItemNotFoundException as e:
        validator_message = ValidatorMessages()
        validator_message.add_message(Message(MESSAGE.STATUS_CRITICAL, e.text))
        response = FrontEndSaveResponse(validator_message, return_code=e.code)
        return response.get_response()


# Create from: like duplication, but without ACL/permissions
def create_from_object(item_type, item_id):
    context = ItemContext(app, item_id, item_type=item_type, action=ACTIONS.CREATE_FROM_OBJECT)
    
    
    def transform_acl(item, _context):
        # Overwrite users parameters and set current user as view & edition
        # clean groups
        if _context.item_type in ITEM_TYPE.ALL_HOST_CLASS:
            props_to_relink = ['view_contacts', 'edition_contacts']
            props_to_clean = ['notification_contacts', 'view_contact_groups', 'notification_contact_groups', 'edition_contact_groups']
            
            new_link = {'has_plus': False, 'links': [{'_id': _context.user['_id'], 'item_type': ITEM_TYPE.CONTACTS, 'exists': True}]}
            for prop in props_to_relink:
                item[prop] = new_link
            
            for prop in props_to_clean:
                item.pop(prop, None)
    
    
    try:
        app.state_controller.clone_object(context, transform=transform_acl)
    except ItemNotFoundException as e:
        validator_message = ValidatorMessages()
        validator_message.add_message(Message(MESSAGE.STATUS_CRITICAL, e.text))
        response = FrontEndSaveResponse(validator_message, return_code=e.code)
        return response.get_response()


def save_object(item_type, item_id=''):
    app.response.content_type = 'application/json'
    
    if not item_id:
        error_message = ValidatorMessages()
        error_message.add_message(Message(MESSAGE.STATUS_CRITICAL, app._('element.missing_id')))
        response = FrontEndSaveResponse(error_message)
        return response.get_response()
    
    context = ItemContext(app, item_id, item_type, action=ACTIONS.SAVE_OBJECT)
    
    try:
        item_name, sent_object, validation_messages = _format_form_to_item(app.request.forms, context)
    except (BusinessException) as e:
        response = FrontEndSaveResponse(ValidatorMessages(), return_code=e.code)
        return response.get_response(text=e.text)
    
    try:
        result = app.state_controller.update(context, sent_object)
    except (BusinessException) as e:
        validator_message = ValidatorMessages()
        validator_message.add_message(Message(MESSAGE.STATUS_CRITICAL, e.text))
        response = FrontEndSaveResponse(validator_message, return_code=e.code)
        return response.get_response()
    
    if result['previously_deleted']:
        validation_messages.add_message(Message(MESSAGE.STATUS_WARNING, app._('element.previously_deleted_item'), weight=MESSAGE.WEIGHT_HIGH))
    if result['previously_imported']:
        validation_messages.add_message(Message(MESSAGE.STATUS_WARNING, app._('element.previously_imported_item'), weight=MESSAGE.WEIGHT_HIGH))
    
    response = FrontEndSaveResponse(validation_messages)
    
    return response.get_response()


def _format_form_to_item(request_forms, context):
    # type: (dict, ItemContext) -> (str, dict, Messages)
    warning_messages = []
    deleted_properties = []
    item_id = context.item_id
    item_type = context.item_type
    prev_elt = context.prev_elt
    
    # Use a FrontendCipher from the protected_fields__substrings_list sent by Front to decipher data
    protect_fields__substrings_list = ','.join(json.loads(request_forms['protect_fields__substrings_matching_fields']))
    protect_fields__activate_interface_encryption = json.loads(request_forms['protect_fields__activate_interface_encryption'])
    protect_fields__are_viewable_by_admin_si = json.loads(request_forms['protect_fields__are_viewable_by_admin_si'])
    forms_item = json.loads(request_forms['item'])
    
    # If the item is not in creation but, we haven't the previous item, somebody delete it before we save it. In this case, we can't find the protected data if the user doesn't modify it because front send TAGs
    if not context.in_creation and not context.prev_elt:
        for item_property, value in forms_item.items():
            if _match_protected_property(item_property, protect_fields__substrings_list, item_type) and value in (PROTECTED_TAG_FROM_CHANGE, PROTECTED_DEFAULT_TAG):
                deleted_properties.append(item_property)
                del forms_item[item_property]
    
    if deleted_properties:
        _deleted_html = '<ul><li><span class="shinken-data-user">%s</span></li></ul>' % '</span></li><li><span class="shinken-data-user">'.join(deleted_properties)
        warning_messages.append(app._('element.data_loose') % _deleted_html)
    
    format_item_frontend_cipher = FrontendCipher(protect_fields__activate_interface_encryption, protect_fields__substrings_list, protect_fields__are_viewable_by_admin_si)
    forms_item = format_item_frontend_cipher.uncipher(forms_item, item_type=item_type, old_item=prev_elt, user=context.user)
    
    item_name = get_name_from_type(item_type, forms_item)
    logger.info('[save_object] Save object [%s]-[%s]' % (item_type, item_name))
    
    # The dfe value must be uppercase and prefixed by _
    if 'duplicate_foreach' in forms_item:
        forms_item['duplicate_foreach'] = '_%s' % forms_item['duplicate_foreach'].upper().lstrip('_')
    
    # frontend links are not the same as backend links because
    # in link of item to save id can not exist if item was delete or an item can be created before the save
    
    # As there is no static get_links() method, cast forms_item into a BaseItem
    # DO NOT RE-USE THIS base_item
    base_item = get_item_instance(item_type, default=forms_item)
    METADATA.update_metadata(base_item, METADATA.ITEM_TYPE, item_type)
    for linking_property in DEF_ITEMS[item_type]['props_links']:
        if linking_property != SERVICE_OVERRIDE:
            update_frontend_links_into_links(linking_property, base_item.get_links(linking_property), item_type, datamanagerV2, linkify=False)
    
    if SERVICE_OVERRIDE in forms_item:
        update_frontend_service_override_links_into_links(forms_item, item_type, context.to_state, datamanagerV2)
    
    # pass the validator on object and raise error if not validate
    _validation = syncuicommon.validator.validate(item_type, forms_item)
    _validation_messages = ValidatorMessages(_validation)
    if _validation_messages.has_critical():
        
        raise SaveException(400, _validation_messages.get_html())
    
    # check ACL
    if item_type in [ITEM_TYPE.CONTACTS, ITEM_TYPE.CONTACTTPLS]:
        acl_share_prop = AclShareProp()
        for acl_share_prop_name in ['acl_share_private', 'acl_share_group', 'acl_share_everybody']:
            if acl_share_prop_name in forms_item:
                acl_share = forms_item[acl_share_prop_name]
                acl_share = acl_share_prop.unpythonize(acl_share)
                forms_item[acl_share_prop_name] = acl_share
    
    _update_from_previous(item_type, prev_elt, forms_item)
    
    prev_elt_name = prev_elt.get_name() if prev_elt else None
    prev_elt_address = prev_elt.get('address') if prev_elt else None
    forms_item['_SYNC_KEYS'] = _recompute_sync_keys(forms_item, item_id, item_name, item_type, prev_elt_name, prev_elt_address)
    
    # make sure syncui is in the source list
    if 'syncui' not in forms_item.get('sources', "syncui"):
        forms_item['sources'] += ',syncui'
    elif forms_item.get('sources', '') == '':
        forms_item['sources'] = 'syncui'
    
    for mandatory_field, value in DEF_ITEMS[item_type].get('mandatory_fields', {}).iteritems():
        forms_item[mandatory_field] = value
    _validation_messages.add_messages([Message(MESSAGE.STATUS_WARNING, w) for w in warning_messages])
    return item_name, forms_item, _validation_messages


def _update_from_previous(item_type, prev_elt, sent_object):
    if not prev_elt:
        return
    if 'work_area_info' in prev_elt:
        sent_object['work_area_info'] = prev_elt['work_area_info']
    previous_raw_elt = prev_elt.get_raw_item()
    cls = DEF_ITEMS[item_type]['class']
    prop_dict = getattr(cls, 'passthrough', {})
    for prop in prop_dict.iterkeys():
        if (prop in previous_raw_elt) and (prop not in sent_object):
            sent_object[prop] = previous_raw_elt[prop]
    
    sync_keys = prev_elt.get('_SYNC_KEYS', [])
    sent_object['_SYNC_KEYS'] = map(lambda key: key.lower(), sync_keys)
    if '_SE_UUID' in prev_elt:
        sent_object['_SE_UUID'] = prev_elt['_SE_UUID']


def _recompute_sync_keys(sent_object, item_id, item_name, item_type, prev_elt_name, prev_elt_address):
    # We will update sync_keys for our save item.
    # sync_keys are a list of string in lower case.
    # Minimal sync-key are :
    # * For dedicated checks : a se_uuid format
    # * For template : name with '-tpl' suffix
    # * For other : name
    
    is_tpl = ITEM_TYPE.is_template(item_type)
    
    sync_keys = sent_object.get('_SYNC_KEYS', [])
    if isinstance(sync_keys, basestring):
        _sync_keys = [sync_keys]
    
    sync_keys = set((i.lower() for i in sync_keys))
    
    if not sync_keys:
        if sent_object.get('service_description', ''):
            possible_sync_key = 'core-%s-%s' % (item_type, item_id)
        elif is_tpl:
            possible_sync_key = item_name + '-tpl'
        else:
            possible_sync_key = item_name
        sync_keys.add(possible_sync_key.lower())
    
    # We remove old name in sync_keys and add new name if user change the item name
    # We must keep old name in sync keys if it was previous address because some source use address as sync keys like discovery
    elif not sent_object.get('service_description', '') and prev_elt_name and prev_elt_name != item_name:
        prev_elt_key = prev_elt_name + '-tpl' if is_tpl else prev_elt_name
        prev_elt_key = prev_elt_key.lower()
        new_key = item_name + '-tpl' if is_tpl else item_name
        new_key = new_key.lower()
        
        if prev_elt_address != prev_elt_key and prev_elt_key != new_key and prev_elt_key in sync_keys:
            sync_keys.remove(prev_elt_key)
        sync_keys.add(new_key)
    
    return list(sync_keys)


def delete_ui_entry(item_type, item_id=''):
    logger.debug("Delete UI entry %s:%s" % (item_type, item_id))
    app.response.content_type = 'application/json'
    if not item_id:
        return app.abort(400, 'Missing the element _id')
    
    context = ItemContext(app, item_id, item_type, action=ACTIONS.DELETE_UI_ENTRY)
    validator_message = ValidatorMessages()
    delete = 'OK'
    
    # if item in work area, try to delete it from work area
    if context.from_state == ITEM_STATE.WORKING_AREA and not context.bypass_work_area:
        return delete_item_working_area(context.item_type, context.item_id)
    try:
        app.state_controller.delete(context, context.from_state)
    except UnauthorizedSaveException as e:
        delete = 'SUICIDE' if e.code == 406 else 'PROTECTED'
        validator_message.add_message(Message(MESSAGE.STATUS_CRITICAL, e.text))
        response = FrontEndSaveResponse(validator_message, return_code=e.code, delete=delete)
        return response.get_response()
    except BusinessException as e:
        delete = 'UNKNOWN' if e.code == 'element-unknown' else 'PROTECTED'
        validator_message.add_message(Message(MESSAGE.STATUS_CRITICAL, e.text))
        response = FrontEndSaveResponse(validator_message, return_code=e.code, delete=delete)
        return response.get_response()
    
    response = FrontEndSaveResponse(validator_message, delete=delete, default_text=app._('element.object_deleted'))
    return response.get_response()


def import_from_source(item_type):
    with object_edition_lock:
        _ids = []
        for k in app.request.forms:
            if app.request.forms.get(k):
                _ids.append(k)
        
        logger.debug("[Import] massive import type [%s] - [%s]" % (item_type, _ids))
        
        for _id in _ids:
            context = ItemContext(app, _id, item_type, action=ACTIONS.IMPORT_FROM_SOURCE)
            app.state_controller.import_one_object(context)


def _do_async_mass_action(action):
    user = app.get_user_auth()
    raw_data = app.request.body.read()
    data = json.loads(raw_data)
    object_count = data.get('object_count', 0)
    logger.debug("[Import] massive %s of many types and items, %s items concerned" % (action, object_count))
    mass_action_uuid = uuid.uuid1().hex
    
    MULTI_MASS_ACTION_UPDATE[mass_action_uuid] = {'total': object_count, 'current': 0}
    
    contexts = {}
    elements = data.get('elements', {})
    for item_type, item_ids in elements.iteritems():
        for item_id in item_ids:
            context = ItemContext(app, item_id, item_type, action=action)
            if item_type not in contexts:
                contexts[item_type] = {}
            contexts[item_type][item_id] = context
    launch_many_mass_action_thread(action, mass_action_uuid, elements, contexts, user)
    return {'uuid': mass_action_uuid}


def import_many_from_source():
    return _do_async_mass_action(ACTIONS.IMPORT_FROM_SOURCE)


def validate_many_changes():
    return _do_async_mass_action(ACTIONS.VALIDATE_CHANGES)


def launch_many_mass_action_thread(action, action_uuid, elements, contexts, user):
    thread_name = 'many-mass-action-thread-%s-%s' % (action, action_uuid)
    t = threading.Thread(None, target=_launch_many_mass_action_thread, name=thread_name, args=(action, action_uuid, elements, contexts, user))
    t.daemon = True
    t.start()


def _launch_many_mass_action_thread(action, action_uuid, elements, contexts, user):
    try:
        _real_launch_many_mass_action_thread(action, action_uuid, elements, contexts, user)
    except Exception as e:
        logger.print_stack()


def _real_launch_many_mass_action_thread(action, action_uuid, elements, contexts, user):
    mass_actions_functions = {
        ACTIONS.IMPORT_FROM_SOURCE: (app.state_controller.import_one_object, blank_context),
        ACTIONS.VALIDATE_CHANGES  : (app.state_controller.validate_changes, app.MASS_IMPORT_LOCK),
    }
    mass_action_function, _lock = mass_actions_functions[action]
    with _lock:
        with DBTransaction(user=user):
            inheritance_cache = set()
            MULTI_MASS_ACTION_UPDATE[action_uuid]['counter_error'] = 0
            MULTI_MASS_ACTION_UPDATE[action_uuid]['counter_warning'] = 0
            for item_type, item_ids in elements.iteritems():
                for item_id in item_ids:
                    context = contexts[item_type][item_id]
                    return_values = mass_action_function(context, inheritance_cache=inheritance_cache)
                    validation = return_values[0]
                    item_name = return_values[1]
                    if validation['has_messages']:
                        validation_message = ''.join(['<li>%s</li>' % _validation_msg for _validation_msg in validation['messages']])
                        element_type = app._('type.%s' % context.item_type[:-1])
                        param_message = {}
                        if validation['has_critical']:
                            param_message = {
                                "type_message"      : "critical-error-message",
                                "type_tag"          : 'error',
                                "type_tag_text"     : app._('element.critical-error'),
                                "element_type"      : element_type,
                                "item_name"         : item_name,
                                "validation_message": validation_message,
                            }
                            MULTI_MASS_ACTION_UPDATE[action_uuid]['counter_error'] = MULTI_MASS_ACTION_UPDATE[action_uuid]['counter_error'] + 1
                        elif validation['has_error']:
                            param_message = {
                                "type_message"      : "error-message",
                                "type_tag"          : 'error',
                                "type_tag_text"     : app._('element.error'),
                                "element_type"      : element_type,
                                "item_name"         : item_name,
                                "validation_message": validation_message,
                            }
                            MULTI_MASS_ACTION_UPDATE[action_uuid]['counter_error'] = MULTI_MASS_ACTION_UPDATE[action_uuid]['counter_error'] + 1
                        elif validation['has_warning']:
                            param_message = {
                                "type_message"      : "warning-message",
                                "type_tag"          : 'warning',
                                "type_tag_text"     : app._('element.warning_saving_log'),
                                "element_type"      : element_type,
                                "item_name"         : item_name,
                                "validation_message": validation_message,
                            }
                            MULTI_MASS_ACTION_UPDATE[action_uuid]['counter_warning'] = MULTI_MASS_ACTION_UPDATE[action_uuid]['counter_warning'] + 1
                        
                        message = '''
                        <div class="shinken-saving-log-message %(type_message)s">
                            <div class="shinken-saving-log-message-header">
                                <span class="tag %(type_tag)s">%(type_tag_text)s</span>
                                <span class="tag element-type">%(element_type)s</span>
                                <span class="element-name">%(item_name)s</span> :
                            </div><ul>%(validation_message)s</ul>
                        </div>''' % param_message
                        
                        if 'validation_messages' not in MULTI_MASS_ACTION_UPDATE[action_uuid]:
                            MULTI_MASS_ACTION_UPDATE[action_uuid]['validation_messages'] = []
                        MULTI_MASS_ACTION_UPDATE[action_uuid]['validation_messages'].append(message)
                    
                    MULTI_MASS_ACTION_UPDATE[action_uuid]['current'] = MULTI_MASS_ACTION_UPDATE[action_uuid]['current'] + 1
            # logger.debug("Mass action %s done progress %s" % (action_uuid, MULTI_MASS_ACTION_UPDATE[action_uuid]['current']))
        logger.debug("Mass action %s done" % action_uuid)
        # if no validation error are presents we can delete the update now. If validation the frontend need to get it when it will receive the last update
        _time = time.time()
        time.sleep(1)
        if 'validation_messages' in MULTI_MASS_ACTION_UPDATE:
            while action_uuid in MULTI_MASS_ACTION_UPDATE and time.time() - _time < 10:
                time.sleep(1)
        MULTI_MASS_ACTION_UPDATE.pop(action_uuid, None)


def get_many_mass_action_progress():
    ###################################################################################
    #
    #  NO LOCK FOR THIS ROUTE === DO NOT CHANGE DATA
    #
    ###################################################################################
    
    mass_action_uuid = app.request.GET.get('uuid', None)
    if not mass_action_uuid:
        return app.abort(404, 'The mass action "uuid" was not given')
    mass_action_progress = MULTI_MASS_ACTION_UPDATE.get(mass_action_uuid, None)
    if mass_action_progress is None:
        # the mass action is probably done, return a mock info to simulate the end of the mass action
        return {'total': 1, 'current': 1}
    if mass_action_progress['current'] >= mass_action_progress['total']:
        MULTI_MASS_ACTION_UPDATE.pop(mass_action_uuid, None)
    return mass_action_progress


def save_in_work_area(item_type, item_id):
    app.response.content_type = 'application/json'
    if not item_id:
        error_message = ValidatorMessages()
        error_message.add_message(Message(MESSAGE.STATUS_CRITICAL, app._('element.missing_id')))
        response = FrontEndSaveResponse(error_message)
        return response.get_response()
    
    context = ItemContext(app, item_id, item_type=item_type, action=ACTIONS.SAVE_IN_WORK_AREA)
    
    try:
        item_name, sent_object, validation_messages = _format_form_to_item(app.request.forms, context)
        app.state_controller.save_in_work_area(context, item_name, sent_object)
    except BusinessException as e:
        response = FrontEndSaveResponse(ValidatorMessages(), return_code=e.code)
        return response.get_response(text=e.text)
    
    response = FrontEndSaveResponse(validation_messages)
    
    return response.get_response()


def submit_to_staging(item_type, item_id):
    context = ItemContext(app, item_id, item_type, action=ACTIONS.SUBMIT_TO_STAGING)
    try:
        app.state_controller.submit_to_staging(context)
    except BusinessException as e:
        validator_message = ValidatorMessages()
        validator_message.add_message(Message(MESSAGE.STATUS_CRITICAL, e.text))
        response = FrontEndSaveResponse(validator_message, return_code=e.code)
        return response.get_response()


def accept_submit(item_type, item_id):
    app.response.content_type = 'application/json'
    logger.debug("[work_area] accept submission for item [%s-%s]" % (item_type, item_id))
    
    context = ItemContext(app, item_id, item_type, action=ACTIONS.ACCEPT_SUBMIT)
    try:
        app.state_controller.accept_submit(context)
    except BusinessException as e:
        validator_message = ValidatorMessages()
        validator_message.add_message(Message(MESSAGE.STATUS_CRITICAL, e.text))
        response = FrontEndSaveResponse(validator_message, return_code=e.code)
        return response.get_response()
    
    response = FrontEndSaveResponse(ValidatorMessages(), default_text=app._('element.apply_work_area_OK'))
    return response.get_response()


def reject_submit(item_type, item_id):
    app.response.content_type = 'application/json'
    
    forms = app.request.forms
    reject_comment_text = forms.get('comment', '').decode('utf8', 'ignore').strip()
    context = ItemContext(app, item_id, item_type, action=ACTIONS.REJECT_SUBMIT)
    try:
        app.state_controller.reject_submit(context, reject_comment_text)
    except BusinessException as e:
        validator_message = ValidatorMessages()
        validator_message.add_message(Message(MESSAGE.STATUS_CRITICAL, e.text))
        response = FrontEndSaveResponse(validator_message, return_code=e.code)
        return response.get_response()
    
    logger.debug('[work_area] reject_submit item [%s-%s] it will be resend in work area' % (item_type, item_id))
    response = FrontEndSaveResponse(ValidatorMessages(), default_text=app._('element.reject_submit_OK'), was_reject=True)
    return response.get_response()


# unlock_work_area AKA Cancel modifications
def unlock_work_area(item_type, item_id):
    app.response.content_type = 'application/json'
    context = ItemContext(app, item_id, item_type, action=ACTIONS.UNLOCK_WORK_AREA)
    has_item_in_staging = app.state_controller.unlock_work_area(context)
    
    response = FrontEndSaveResponse(ValidatorMessages(), has_item_in_staging=has_item_in_staging, delete='OK', default_text=app._('element.unlock_item_OK'))
    return response.get_response()


def working_area(item_type):
    user = app.get_user_auth()
    logger.debug('[work_area] call working_area for user [%s]' % (user['contact_name']))
    return elements_generic(item_type, items_state=ITEM_STATE.WORKING_AREA)


def my_working_area(item_type):
    user = app.get_user_auth()
    logger.debug('[work_area] call my_working_area for user [%s]' % (user['contact_name']))
    
    return elements_generic(item_type, items_state=ITEM_STATE.WORKING_AREA, only_my_item=True)


def get_item_working_area(item_type, item_id):
    return_value = _get_specific_item(item_id, item_type, current_page='working_area')
    return return_value


def delete_item_working_area(item_type, item_id):
    app.response.content_type = 'application/json'
    logger.debug("[work_area] delete entry [%s-%s]" % (item_type, item_id))
    if not item_id:
        return app.abort(400, 'Missing the element _id')
    context = ItemContext(app, item_id, item_type, action=ACTIONS.DELETE_ITEM_WORKING_AREA)
    validator_message = ValidatorMessages()
    delete = 'OK'
    
    try:
        item_in_staging = app.state_controller.delete_item_working_area(context)
    except UnauthorizedSaveException as e:
        delete = 'SUICIDE' if e.code == 406 else 'PROTECTED'
        validator_message.add_message(Message(MESSAGE.STATUS_CRITICAL, e.text))
        response = FrontEndSaveResponse(validator_message, return_code=e.code, delete=delete)
        return response.get_response()
    except BusinessException as e:
        delete = 'UNKNOWN' if e.code == 'element-unknown' else 'PROTECTED'
        validator_message.add_message(Message(MESSAGE.STATUS_CRITICAL, e.text))
        response = FrontEndSaveResponse(validator_message, return_code=e.code, delete=delete)
        return response.get_response()
    
    validator_message.add_message(Message(MESSAGE.STATUS_INFO, app._('element.object_deleted')))
    response = FrontEndSaveResponse(validator_message, delete=delete, has_item_in_staging=(item_in_staging is not None))
    return response.get_response()


def add_item_working_area(item_type):
    return_value = new_object(item_type)
    return_value['working_area'] = True
    return return_value


pages = {
    # HOSTS
    new_host                     : {'routes': ['/elements/add/host'], 'view': 'elements_host', 'static': True, 'wrappers': ['auth']},
    edit_host                    : {'routes': ['/elements/hosts/:item_id'], 'view': 'elements_host', 'static': True, 'wrappers': ['auth']},
    # HOSTSTPL
    new_host_tpl                 : {'routes': ['/elements/add/hosttpl'], 'view': 'elements_host', 'static': True},
    edit_hosttpl                 : {'routes': ['/elements/hosttpls/:item_id'], 'view': 'elements_host', 'static': True},
    # HOST GROUPS
    new_hostgroup                : {'routes': ['/elements/add/hostgroup'], 'view': 'elements_hostgroup', 'static': True},
    edit_hostgroup               : {'routes': ['/elements/hostgroups/:item_id'], 'view': 'elements_hostgroup', 'static': True},
    # CLUSTERS
    new_cluster                  : {'routes': ['/elements/add/cluster'], 'view': 'elements_cluster', 'static': True},
    edit_cluster                 : {'routes': ['/elements/clusters/:item_id'], 'view': 'elements_cluster', 'static': True},
    # CLUSTERTPL
    new_cluster_tpl              : {'routes': ['/elements/add/clustertpl'], 'view': 'elements_cluster', 'static': True},
    edit_clustertpl              : {'routes': ['/elements/clustertpls/:item_id'], 'view': 'elements_cluster', 'static': True},
    # SERVICES
    new_services_host            : {'routes': ['/elements/add/serviceshost'], 'view': 'elements_service', 'static': True},
    edit_services_hosts          : {'routes': ['/elements/serviceshosts/:item_id'], 'view': 'elements_service', 'static': True},
    edit_services_hosts_override : {'routes': ['/elements/override2/serviceshosts/:item_id'], 'view': 'elements_service_override', 'static': True},
    # SERVICES HOSTTPL
    new_services_host_tpl        : {'routes': ['/elements/add/serviceshosttpl'], 'view': 'elements_service', 'static': True},
    edit_services_host_tpls      : {'routes': ['/elements/serviceshosttpls/:item_id'], 'view': 'elements_service', 'static': True},
    # SERVICES CLUSTER
    new_services_cluster         : {'routes': ['/elements/add/servicescluster'], 'view': 'elements_service', 'static': True},
    edit_services_clusters       : {'routes': ['/elements/servicesclusters/:item_id'], 'view': 'elements_service', 'static': True},
    # SERVICES CLUSTERTPL
    new_services_cluster_tpl     : {'routes': ['/elements/add/servicesclustertpl'], 'view': 'elements_service', 'static': True},
    edit_services_cluster_tpls   : {'routes': ['/elements/servicesclustertpls/:item_id'], 'view': 'elements_service', 'static': True},
    # SERVICETPL
    new_service_tpl              : {'routes': ['/elements/add/servicetpl'], 'view': 'elements_service', 'static': True},
    edit_service_tpls            : {'routes': ['/elements/servicetpls/:item_id'], 'view': 'elements_service', 'static': True},
    # CONTACTS
    new_contact                  : {'routes': ['/elements/add/contact'], 'view': 'elements_contact', 'static': True},
    edit_contact                 : {'routes': ['/elements/contacts/:item_id'], 'view': 'elements_contact', 'static': True, 'wrappers': ['auth']},
    # CONTACTSTPL
    new_contact_tpl              : {'routes': ['/elements/add/contacttpl'], 'view': 'elements_contact', 'static': True},
    edit_contacttpl              : {'routes': ['/elements/contacttpls/:item_id'], 'view': 'elements_contact', 'static': True},
    # CONTACT GROUPS
    new_contactgroup             : {'routes': ['/elements/add/contactgroup'], 'view': 'elements_contactgroup', 'static': True},
    edit_contactgroup            : {'routes': ['/elements/contactgroups/:item_id'], 'view': 'elements_contactgroup', 'static': True},
    # TIME PERIODS
    new_timeperiod               : {'routes': ['/elements/add/timeperiod'], 'view': 'elements_timeperiod', 'static': True},
    edit_timeperiod              : {'routes': ['/elements/timeperiods/:item_id'], 'view': 'elements_timeperiod', 'static': True},
    # COMMANDS
    new_command                  : {'routes': ['/elements/add/command'], 'view': 'elements_command', 'static': True},
    edit_command                 : {'routes': ['/elements/commands/:item_id'], 'view': 'elements_command', 'static': True},
    # BUSINESS IMPACT MODULATION
    new_businessimpactmodulation : {'routes': ['/elements/add/businessimpactmodulation'], 'view': 'elements_businessimpactmodulation', 'static': True},
    edit_businessimpactmodulation: {'routes': ['/elements/businessimpactmodulations/:item_id'], 'view': 'elements_businessimpactmodulation', 'static': True},
    # MACRO MODULATION
    new_macromodulation          : {'routes': ['/elements/add/macromodulation'], 'view': 'elements_macromodulation', 'static': True},
    edit_macromodulation         : {'routes': ['/elements/macromodulations/:item_id'], 'view': 'elements_macromodulation', 'static': True},
    # RESULT MODULATION
    new_resultmodulation         : {'routes': ['/elements/add/resultmodulation'], 'view': 'elements_resultmodulation', 'static': True},
    edit_resultmodulation        : {'routes': ['/elements/resultmodulations/:item_id'], 'view': 'elements_resultmodulation', 'static': True},
    # NOTIFICATION WAY
    new_notificationway          : {'routes': ['/elements/add/notificationway'], 'view': 'elements_notificationway', 'static': True},
    edit_notificationway         : {'routes': ['/elements/notificationways/:item_id'], 'view': 'elements_notificationway', 'static': True},
    # ESCALATIONS
    new_escalation               : {'routes': ['/elements/add/escalation'], 'view': 'elements_escalation', 'static': True},
    edit_escalation              : {'routes': ['/elements/escalations/:item_id'], 'view': 'elements_escalation', 'static': True},
    
    # TEMPLATING INFORMATION
    # Get checks for templates
    elements_checksbytemplates   : {'routes': ['/elements/checksbytemplates/'], 'method': 'POST', 'static': True, 'wrappers': ['auth']},
    # Get template information
    elements_servicepls_by_names : {'routes': ['/elements/byname/servicetpls/'], 'method': 'POST', 'static': True, 'wrappers': ['auth']},
    elements_contacttpls_by_names: {'routes': ['/elements/byname/contacttpls/'], 'method': 'POST', 'static': True, 'wrappers': ['auth']},
    elements_hosttpls_by_names   : {'routes': ['/elements/byname/hosttpls/'], 'method': 'POST', 'static': True, 'wrappers': ['auth']},
    elements_clustertpls_by_names: {'routes': ['/elements/byname/clustertpls/'], 'method': 'POST', 'static': True, 'wrappers': ['auth']},
    # To run things like checks
    run_check                    : {'routes': ['/element/run'], 'method': 'POST', 'wrappers': ['auth']},
    
    # GENERIC MASS ACTIONS
    disable_object               : {'routes': ['/element/q/:item_type/disable/:item_id'], 'wrappers': ['auth', 'transaction', 'db_transaction']},
    enable_object                : {'routes': ['/element/q/:item_type/enable/:item_id'], 'wrappers': ['auth', 'transaction', 'db_transaction']},
    clone_object                 : {'routes': ['/element/q/:item_type/clone/:item_id'], 'method': 'POST', 'wrappers': ['auth', 'transaction', 'db_transaction']},
    create_from_object           : {'routes': ['/element/q/:item_type/create-from/:item_id'], 'method': 'POST', 'wrappers': ['auth', 'transaction', 'db_transaction']},
    import_from_source           : {'routes': ['/element/q/:item_type/validate/'], 'method': 'POST', 'wrappers': ['auth', 'transaction', 'db_transaction']},
    # GENERIC ACTIONS
    delete_ui_entry              : {'routes': ['/element/q/:item_type/delete/:item_id'], 'method': 'POST', 'wrappers': ['auth', 'transaction', 'db_transaction']},
    
    # STAGING STUFF
    # List Elements
    list_elements                : {'routes': ['/elements/<item_type>'], 'view': '_list_elements', 'static': True, 'wrappers': ['auth']},
    # specific actions
    save_object                  : {'routes': ['/element/q/:item_type/save/:item_id'], 'method': 'POST', 'wrappers': ['auth', 'transaction', 'db_transaction']},
    accept_submit                : {'routes': ['/element/q/:item_type/apply_work_area/:item_id'], 'method': 'POST', 'wrappers': ['auth_admin', 'transaction', 'db_transaction']},
    reject_submit                : {'routes': ['/element/q/:item_type/reject_submit/:item_id'], 'method': 'POST', 'wrappers': ['auth_admin', 'transaction', 'db_transaction']},
    
    # WORKING AREA STUFF
    # List Elements
    working_area                 : {'routes': ['/elements/working_area/<item_type>'], 'view': 'working_area', 'static': True, 'wrappers': ['auth']},
    my_working_area              : {'routes': ['/elements/my_working_area/<item_type>'], 'view': 'my_working_area', 'static': True, 'wrappers': ['auth']},
    # get a specific element in working area
    get_item_working_area        : {'routes': ['/elements/working_area/<item_type>/<item_id>'], 'view': 'elements_host', 'static': True, 'wrappers': ['auth', 'transaction']},
    # mass action work area
    delete_item_working_area     : {'routes': ['/elements/delete/working_area/<item_type>/<item_id>'], 'method': 'POST', 'wrappers': ['auth', 'transaction', 'db_transaction']},
    add_item_working_area        : {'routes': ['/elements/add/working_area/<item_type>'], 'view': 'elements_host', 'static': True, 'wrappers': ['auth', 'transaction', 'db_transaction']},
    # specific actions
    submit_to_staging            : {'routes': ['/element/q/:item_type/submit_to_staging/:item_id'], 'method': 'POST', 'wrappers': ['auth', 'transaction', 'db_transaction']},
    save_in_work_area            : {'routes': ['/element/q/:item_type/save_in_work_area/:item_id'], 'method': 'POST', 'wrappers': ['auth', 'transaction', 'db_transaction']},
    unlock_work_area             : {'routes': ['/element/q/:item_type/unlock_work_area/:item_id'], 'method': 'POST', 'wrappers': ['auth', 'transaction']},
    
    get_service_override         : {'routes': ['/elements/service_overrides/:host_type/:host_id/:service_type/:service_id/:apply_on_type'], 'view': 'elements_service_override', 'static': True, 'wrappers': ['auth']},
    import_many_from_source      : {'routes': ['/elements/q/validate/'], 'method': 'POST', 'wrappers': ['auth', 'transaction', 'json']},
    validate_many_changes        : {'routes': ['/elements/q/validate-changes/'], 'method': 'POST', 'wrappers': ['auth', 'transaction', 'json']},
    
    get_many_mass_action_progress: {'routes': ['/elements/mass_action/progress/'], 'method': 'GET', 'wrappers': ['json']},
    
    # Massive changes
    massive_change_call          : {'routes': ['/elements/massive_change/<item_type>'], 'method': 'POST', 'view': 'massive_change/massive_change', 'wrappers': ['auth']},
}
