import json
import os
import shutil
import time
import traceback
from threading import RLock

from .load import AvgInRange
from .log import logger, get_chapter_string
from .util import format_t_into_dhms_format

VMWARE_CHAPTER = get_chapter_string('VMWARE STATS')
VMWARE_READY_STAT_LIMIT_CRITICAL = 10.0  # over 10% stolen, it's bad
VMWARE_READY_STAT_LIMIT_WARNING = 5.0  # over 10% stolen, it's bad
VMWARE_STATS_FILE_IS_TOO_OLD = 60  # if more than 1min, the stats data are too old
VMWARE_STATS_EXPORT_FILE = '/dev/shm/vmware_stats_export.dat'
IS_WINDOWS = os.name == 'nt'

# We can fake the value, usefull for testing only!
SHINKEN_FAKE_VMWARE_READY_VALUE = os.environ.get('SHINKEN_FAKE_VMWARE_READY_VALUE', '')


class HYPERVISOR(object):
    HYPERV = 'hyper-v'
    VMWARE = 'vmware'
    KVM = 'kvm'


# WARNING: do not edit the values without thinking about multi-versions daemons!
class VMWARE_STATS_KEY(object):
    CONFIGURATION_RECEIVED = 'vmware_configuration_received'
    STATS_ENABLED = 'vmware_stats_enabled'
    IS_VM = 'vmware_vm'
    GUESTLIB_ERROR = 'vmware_guestlib_error'
    GUESTLIB_CALL_ERROR = 'vmware_guestlib_call_error'
    READY_STAT = 'vmware_ready_stat'
    TIME_READ = 'vmware_stats_time'
    LOCAL_TIME = 'vmware_local_time'
    IS_MONITORED = 'vmware_is_monitored'


class VMWareStatsCompute(object):
    def __init__(self):
        self.stats_enable = False
        try:
            from shinkensolutions import vmguestlib
            self.vmware_guestlib = vmguestlib
        except:
            self.vmguestlib = None
        
        self._is_vmware = False
        vmware_info_path = "/sys/class/dmi/id/sys_vendor"
        if os.path.exists(vmware_info_path):
            # Are we a VMWare VM ?
            with open(vmware_info_path) as fd:
                vendor = fd.readline().strip()
            
            self._is_vmware = "vmware" in vendor.lower()
        
        self._reset_stats()
        
        self._is_monitored = not IS_WINDOWS  # windows is NOT monitored currently
        
        self.vmware_lock = RLock()
        
        self._last_update = 0.0
        
        try:
            self.vmware_guestlib = vmguestlib.VMGuestLib() if vmguestlib else None
            self.vmware_guestlib_error = self.vmware_guestlib is None
        except Exception:
            self.vmware_guestlib = None
            self.vmware_guestlib_error = True
        
        self._write_stats_file()
    
    
    def set_enable(self, enable):
        if enable:
            self.stats_enable = True
            logger.info('%s The vmware statistics are enabled in the configuration. You can disable it with the parameter vmware__statistics_compute_enable.' % VMWARE_CHAPTER)
        else:
            logger.info('%s The vmware statistics are disabled from the configuration.' % VMWARE_CHAPTER)
    
    
    def _reset_stats(self):
        self.prev_stolen_ms = 0
        self.prev_used_ms = 0
        self.prev_elapsed_ms = 0
        self.vm_ready_rrd = AvgInRange(60)
        self.vmware_guestlib_call_error = False
        self.vmware_logger_countdown = 0
    
    
    def _is_value_simulated(self):
        return SHINKEN_FAKE_VMWARE_READY_VALUE != ''
    
    
    def _get_simulated_value(self):
        fake_value = float(SHINKEN_FAKE_VMWARE_READY_VALUE)
        logger.info('%s [FAKE] The %%ready value is FAKE %s' % (VMWARE_CHAPTER, fake_value))
        return fake_value
    
    
    def _write_stats_file(self):
        d = self._get_stats()
        # Log current writen value
        if not self._is_monitored:
            logger.info('%s Currently VMWare is not supported' % VMWARE_CHAPTER)
        elif not self._is_vmware and not self._is_value_simulated():
            # Only display once by hour, at the start of the hour
            if int(time.time()) % 3600 == 0:
                logger.info('%s The server is not a VMWare server' % VMWARE_CHAPTER)
        else:
            if not self.stats_enable:
                logger.info('%s Currently the VMWare stats are disabled by the configuration or ther gatherer is not fully started' % VMWARE_CHAPTER)
            else:
                logger.info('%s Current VMWare %%ready 1min average for this server: %s' % (VMWARE_CHAPTER, d[VMWARE_STATS_KEY.READY_STAT]))
        tmp_file = VMWARE_STATS_EXPORT_FILE + '.tmp'
        try:
            with open(tmp_file, 'wb') as f:
                f.write(json.dumps(d))
            shutil.move(tmp_file, VMWARE_STATS_EXPORT_FILE)
        except Exception as exp:
            logger.error('Cannot save VMWare stats file %s: %s' % (VMWARE_STATS_EXPORT_FILE, exp))
    
    
    def update_stats(self):
        if self.vmware_guestlib is None or not self._is_vmware or not self.stats_enable:
            # We don't fetch vmware stats but we need to write the file info for the check
            self._write_stats_file()
            return
        
        # Do not update more than once a second
        now = time.time()
        if now < self._last_update + 1:
            return
        self._last_update = now
        
        self.vmware_logger_countdown += 1
        with self.vmware_lock:
            try:
                self.vmware_guestlib_call_error = False
                self.vmware_guestlib.UpdateInfo()
                
                new_stolen_ms = self.vmware_guestlib.GetCpuStolenMs()
                if not new_stolen_ms:
                    raise Exception("Cannot retrieve CPU Stolen ms")
                new_used_ms = self.vmware_guestlib.GetCpuUsedMs()
                if not new_used_ms:
                    raise Exception("Cannot retrieve CPU Used ms")
                new_elapsed_ms = self.vmware_guestlib.GetElapsedMs()
                if not new_elapsed_ms:
                    raise Exception("Cannot retrieve Elapsed ms")
                
                if self.prev_elapsed_ms == 0:
                    # ok so wait for 1s to compute the diff
                    self.prev_stolen_ms = new_stolen_ms
                    self.prev_used_ms = new_used_ms
                    self.prev_elapsed_ms = new_elapsed_ms
                    
                    time.sleep(1)
                    self.vmware_guestlib.UpdateInfo()
                    new_stolen_ms = self.vmware_guestlib.GetCpuStolenMs()
                    if not new_stolen_ms:
                        raise Exception("Cannot retrieve CPU Stolen ms")
                    new_used_ms = self.vmware_guestlib.GetCpuUsedMs()
                    if not new_used_ms:
                        raise Exception("Cannot retrieve CPU Used ms")
                    new_elapsed_ms = self.vmware_guestlib.GetElapsedMs()
                    if not new_elapsed_ms:
                        raise Exception("Cannot retrieve Elapsed ms")
                
                time_elapsed = new_elapsed_ms - self.prev_elapsed_ms
                used_cpu_pct = 100 * (new_used_ms - self.prev_used_ms) / time_elapsed
                stolen_cpu_pct = 100 * (new_stolen_ms - self.prev_stolen_ms) / time_elapsed
                host_processor_speed = self.vmware_guestlib.GetHostProcessorSpeed()
                if not host_processor_speed:
                    raise Exception("Cannot retrieve host processor speed")
                effective_cpu_mhz = host_processor_speed * (new_used_ms - self.prev_used_ms) / time_elapsed
                self.prev_stolen_ms = new_stolen_ms
                self.prev_used_ms = new_used_ms
                self.prev_elapsed_ms = new_elapsed_ms
                
                logger.info('%s This second %%ready: %s' % (VMWARE_CHAPTER, stolen_cpu_pct))
                self.vm_ready_rrd.update_avg(stolen_cpu_pct)
                self.vmware_logger_countdown = 0
            except Exception:
                self.vmware_guestlib_call_error = True
                if self.vmware_logger_countdown < 600 and self.vmware_logger_countdown % 60 == 0:
                    logger.error(" %s VMGuestlib request failed with error : [%s]" % (VMWARE_CHAPTER, traceback.format_exc()))
                elif self.vmware_logger_countdown % 3600 == 0:
                    logger.error("%s VMGuestlib request failed with error : [%s]. If updating vmware-tools is not solving, you can disable vmware stats with the parameter vmware__statistics_compute_enable in this daemon .cfg file." % (
                        VMWARE_CHAPTER, traceback.format_exc()))
        
        self._write_stats_file()
    
    
    def _get_stats(self):
        
        ready_stat = self.vm_ready_rrd.get_avg(default_value=0.0)
        is_vmware = self._is_vmware
        vmware_guestlib_error = self.vmware_guestlib_error
        vmware_guestlib_call_error = self.vmware_guestlib_call_error
        if self._is_value_simulated():
            ready_stat = self._get_simulated_value()
            is_vmware = True
            vmware_guestlib_error = False  # fake the fact that the lib did return an error
            vmware_guestlib_call_error = False
        return {
            VMWARE_STATS_KEY.IS_MONITORED       : self._is_monitored,
            VMWARE_STATS_KEY.TIME_READ          : int(time.time()),
            VMWARE_STATS_KEY.IS_VM              : is_vmware,
            VMWARE_STATS_KEY.GUESTLIB_ERROR     : vmware_guestlib_error,
            VMWARE_STATS_KEY.GUESTLIB_CALL_ERROR: vmware_guestlib_call_error,
            VMWARE_STATS_KEY.READY_STAT         : ready_stat,
            VMWARE_STATS_KEY.STATS_ENABLED      : self.stats_enable,
            HYPERVISOR.VMWARE                   : True
        }


class VMWareStatsReader(object):
    def __init__(self):
        self._conf_received = False
        self._enabled = False  # by default at startup we do not loop
        
        self._is_vmware = False
        vmware_info_path = "/sys/class/dmi/id/sys_vendor"
        if os.path.exists(vmware_info_path):
            # Are we a VMWare VM ?
            with open(vmware_info_path) as fd:
                vendor = fd.readline().strip()
            
            self._is_vmware = "vmware" in vendor.lower()
    
    
    def get_stats(self):
        stats = {
            VMWARE_STATS_KEY.CONFIGURATION_RECEIVED: self._conf_received,
            VMWARE_STATS_KEY.STATS_ENABLED         : self._enabled,
            VMWARE_STATS_KEY.IS_VM                 : self._is_vmware,
            VMWARE_STATS_KEY.GUESTLIB_ERROR        : False,
            VMWARE_STATS_KEY.GUESTLIB_CALL_ERROR   : False,
            VMWARE_STATS_KEY.READY_STAT            : 0.0,
            # It's important to know if the file/stats are too old
            # so we are giving the file time, and our localtime (can be different)
            # in the poller/arbiter that is asking us if there is a ntp error
            VMWARE_STATS_KEY.TIME_READ             : -1,
            VMWARE_STATS_KEY.LOCAL_TIME            : int(time.time()),
            VMWARE_STATS_KEY.IS_MONITORED          : not IS_WINDOWS,
        }
        
        if os.path.exists(VMWARE_STATS_EXPORT_FILE):
            try:
                with open(VMWARE_STATS_EXPORT_FILE, 'r') as f:
                    vmware_stats = json.loads(f.read())
                    stats.update(vmware_stats)
            except Exception as exp:
                logger.error('Cannot load the vmware stats file %s: %s' % (VMWARE_STATS_EXPORT_FILE, exp))
        return stats
    
    
    # Call from the healthcheck, look at values returned by get_stats and return if ok or not
    @staticmethod
    def check_stats_values(stats):
        vm_text = ''
        vm_status = 'OK'
        
        if not stats:  # if stats is void, means the daemon is not reacheable, skip this test
            return 'OK', ''
        
        # what ever values, if disabled from the conf, do nothing
        if not stats.get(VMWARE_STATS_KEY.STATS_ENABLED):
            vm_status = 'INFO'
            vm_text = "The vmware stats computation is disabled on the gatherer by it's .ini configuration"
            return vm_status, vm_text
        
        # First we are looking if the stats are not too old
        # beware: if LOCAL_TIME is missing, skip the test, the distant daemon is older than us
        file_read_time = stats.get(VMWARE_STATS_KEY.TIME_READ, -1)
        server_local_time = stats.get(VMWARE_STATS_KEY.LOCAL_TIME, None)
        # Maybe the server is a windows, and so it not managed currently (no gatherer on windows)
        # default= False, so patch won't force to monitor windows hosts that are not update (no patch for windows)
        is_monitored = stats.get(VMWARE_STATS_KEY.IS_MONITORED, False)
        
        # For windows hosts, skip and get an OK
        if not is_monitored:
            return 'OK', ''
        
        if server_local_time is not None:
            stats_file_age = server_local_time - file_read_time
            if stats_file_age > VMWARE_STATS_FILE_IS_TOO_OLD:
                vm_text = "The VMWare stats file '%s' seems to be too old (not update since %s > %ss). You must look at the Gatherer log (/var/log/shinken/gatherer.log). Then, only if need, you can restart the Gatherer with 'service shinken-gatherer restart'." % (
                    VMWARE_STATS_EXPORT_FILE,
                    format_t_into_dhms_format(
                        stats_file_age),
                    VMWARE_STATS_FILE_IS_TOO_OLD)
                vm_status = 'WARNING'
                return vm_status, vm_text
        
        if stats.get(VMWARE_STATS_KEY.GUESTLIB_CALL_ERROR, False):
            vm_text = 'This satellite runs on a VMWare VM but it cannot currently communicate with the VCenter.'
            vm_status = 'WARNING'
        if stats.get(VMWARE_STATS_KEY.IS_VM, False):
            vm_text = 'This satellite runs on a VMWare VM'
            vm_status = 'OK'
            # Only check if the configuration is received and is enabled
            if stats.get(VMWARE_STATS_KEY.STATS_ENABLED, True):
                if stats[VMWARE_STATS_KEY.GUESTLIB_ERROR]:
                    vm_text += ' but the VMWare tools are not installed ; please install the package open-vm-tools and reboot the server.'
                    vm_status = 'WARNING'
                elif stats.get(VMWARE_STATS_KEY.READY_STAT, 0) >= VMWARE_READY_STAT_LIMIT_CRITICAL:
                    vm_status = 'CRITICAL'
                    vm_text += ' but the stolen CPU is too high : %d %% - Please reduce VCPU allocation on the VM to lighten hypervisor load' % stats['vmware_ready_stat']
                elif stats.get(VMWARE_STATS_KEY.READY_STAT, 0) >= VMWARE_READY_STAT_LIMIT_WARNING:
                    vm_status = 'WARNING'
                    vm_text += ' but the stolen CPU is too high : %d %% - Please reduce VCPU allocation on the VM to lighten hypervisor load' % stats['vmware_ready_stat']
        
        return vm_status, vm_text


vmware_stats_reader = VMWareStatsReader()
