#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019
# This file is part of Shinken Enterprise, all rights reserved.


import json

from shinken.util import format_t_into_dhms_format
from shinkensolutions.lib_checks.common import BREAK_LINE, COLOR, EXIT_STATUS, HTMLList, HTMLTable, HTMLTag, ParseOptionError, RaiseOnExitOptionParser, Result, ShinkenUtils
from shinkensolutions.lib_checks.graphite import CheckGraphite, TAG_FOR_STATE, TAG_CRITICAL, TAG_OK, TAG_WARNING, GRAPHITE_API_VERSION, GRAPHITE_STATS_KEY, GRAPHITE_STATS_FILE_IS_TOO_OLD, NB_METRICS_COUNT_FILE

VERSION = u'0.1'
DAEMON_TYPE = u'broker'
HTTP_API_STEP = u'HTTP API Stats'

result = Result()

parser = RaiseOnExitOptionParser(u'%prog [options] [--help]', version=u'%prog ' + VERSION)
parser.add_option(u'-H', u'--hostname', dest=u'hostname', help=u'The hostname of the shinken daemon')
parser.add_option(u'-p', u'--port', dest=u'port', type=u'int', help=u'The port of the shinken daemon')
parser.add_option(u'-w', u'--webui', dest=u'webui', help=u'Name of the webui module to check. No default value, this need to be filled')
parser.add_option(u'-t', u'--timeout', dest=u'timeout', type=u'int', default=3, help=u'Timeout ( in minutes ) to connect to the shinken daemon. Default : 3')

parser.add_option(u'-P', u'--ssh-port', dest=u'ssh_port', type=u'int', default=22, help=u'SSH port to connect to. Default : 22')
parser.add_option(u'-i', u'--ssh-key', dest=u'ssh_key_file', default=u'~/.ssh/id_rsa', help=u'SSH key file to use. By default it will take ~/.ssh/id_rsa.')
parser.add_option(u'-u', u'--ssh-user', dest=u'user', default=u'shinken', help=u'Remote user to use. By default shinken.')
parser.add_option(u'-r', u'--passphrase', dest=u'passphrase', default=u'', help=u'SSH key passphrase. By default will use an empty passphrase.')

parser.add_option(u'--shinkenversion', dest=u'shinken_supervisor_version', default=u'', help=u'This shinken version number to compare with the monitored shinken. Mandatory if shinken mode.')


def _parse_args():
    opts = None
    try:
        opts, args = parser.parse_args()
        if args:
            parser.error(u'Does not accept any argument.')
        
        if opts and not opts.hostname:
            parser.error(u'Missing parameter hostname (-H/--hostname)')
        if opts and not opts.port:
            parser.error(u'Missing parameter port (-p/--port)')
    
    except ParseOptionError as e:
        if e.msg:
            result.hard_exit(EXIT_STATUS.CRITICAL, u'Fail to parse command argument : %s %s' % (BREAK_LINE, BREAK_LINE.join(e.msg.split('\n'))))
        exit(0)
    
    return opts


class CheckGraphiteForReader(CheckGraphite):
    
    def __init__(self, graphite_hostname, ssh_user, ssh_passphrase, ssh_key_file, ssh_port):
        super(CheckGraphiteForReader, self).__init__(
            graphite_hostname=graphite_hostname,
            ssh_port=ssh_port,
            ssh_key_file=ssh_key_file,
            passphrase=ssh_passphrase,
            user=ssh_user,
            graphite_location=None,
            graphite_user=None,
            storage_usage_warning=None,
            storage_usage_critical=None,
            graphite_port=None,
            graphite_conf_file=None,
            graphite_cache_name=None,
            graphite_relay_name=None
        )
    
    
    def result_add_check(self, status, output, step_index=1, step_name=u'STEP'):
        self.summary.append(u'%s : %s' % (step_name, output))
        self.result.add_check(status)


def check_graphite_server_read(server, opts):
    check_graphite = CheckGraphiteForReader(opts.hostname, opts.user, opts.passphrase, opts.ssh_key_file, opts.ssh_port)
    check_graphite.result = Result()
    check_graphite.summary = []
    
    _step_name = u'HTTP API status'
    _version = server.get(u'version', None)
    _reachable = server.get(u'reachable', False)
    _nb_metrics = server.get(u'nb_metrics', None)
    _read_time = server.get(GRAPHITE_STATS_KEY.TIME_READ, -1)
    _server_time = server.get(GRAPHITE_STATS_KEY.LOCAL_TIME, None)
    
    if _reachable:
        check_graphite.result_add_check(EXIT_STATUS.OK, u'%s - Can connect to graphite API.' % TAG_OK, step_name=_step_name)
        if not _version or _version != GRAPHITE_API_VERSION:
            check_graphite.result_add_check(EXIT_STATUS.WARNING, u'%s - Graphite backend is not up to date. Please update graphite backend with shinken version %s.' % (TAG_WARNING, opts.shinken_supervisor_version), step_name=HTTP_API_STEP)
        elif _nb_metrics is not None:
            metric_file_is_too_old = ''
            if _server_time is None or _read_time == -1:
                metric_file_is_too_old = u'The graphite stats file "%s" seems to be too old. You must look at the Gatherer log (/var/log/shinken/gatherer.log). ' \
                                         u'Then, only if need, you can restart the Gatherer with "service shinken-gatherer restart".' % NB_METRICS_COUNT_FILE
            else:
                stats_file_age = _server_time - _read_time
                if stats_file_age > GRAPHITE_STATS_FILE_IS_TOO_OLD:
                    metric_file_is_too_old = u'The graphite stats file "%s" seems to be too old (not update since %s > %ss). You must look at the Gatherer log (/var/log/shinken/gatherer.log). ' \
                                             u'Then, only if need, you can restart the Gatherer with "service shinken-gatherer restart".' % (
                                                 NB_METRICS_COUNT_FILE,
                                                 format_t_into_dhms_format(stats_file_age),
                                                 GRAPHITE_STATS_FILE_IS_TOO_OLD)
            if _server_time is not None:
                stats_file_age = _server_time - _read_time
                if stats_file_age > GRAPHITE_STATS_FILE_IS_TOO_OLD:
                    check_graphite.result_add_check(EXIT_STATUS.WARNING, metric_file_is_too_old, step_name=HTTP_API_STEP)
            
            check_graphite.result_add_check(EXIT_STATUS.OK, u'%s - %s metrics found.' % (TAG_OK, _nb_metrics), step_name=HTTP_API_STEP)
            check_graphite.result.add_perf_data(u'nb_metrics', _nb_metrics)
        else:
            check_graphite.result_add_check(EXIT_STATUS.WARNING, u'%s - Can not get stats about metrics and hosts/clusters. The graphite backend may be slow. Please check the logs on graphite backend.' % TAG_WARNING, step_name=HTTP_API_STEP)
    
    else:
        check_graphite.result_add_check(EXIT_STATUS.CRITICAL, u'%s - Fail to request metrics to graphite server.' % TAG_CRITICAL, step_name=_step_name)
    
    return check_graphite.result, [HTMLList.simple_list(check_graphite.summary)]


def metacheck_graphite_server(raw_stats, read_status, opts, webui_name):
    headers = []
    lines = []
    known_realms = raw_stats.get(u'known_realms', [])
    servers = [i for i in read_status if i[u'realm'] in known_realms or i[u'realm'] == u'*']
    servers_state = EXIT_STATUS.OK
    
    unreachable_backends = 0
    graphite_servers = (server for server in servers if server.get(u'module_name', u'') == webui_name)
    for server in graphite_servers:
        check_graphite_server_read_result, summary = check_graphite_server_read(server, opts)
        server_state = check_graphite_server_read_result.status
        if server_state == EXIT_STATUS.CRITICAL:
            unreachable_backends += 1
        
        headers.append(u'%s is %s' % (server[u'host'], TAG_FOR_STATE[server_state]))
        lines.append(summary)
        servers_state = max(server_state, servers_state)
        for perf_name, perf_value in check_graphite_server_read_result.perf_data.iteritems():
            result.add_perf_data(perf_name, perf_value)
    
    metacheck_graphit_server_long_output = HTMLTable.table([], lines, left_headers=headers, title=u'Graphite backend(s)', all_col_same_width=False)
    output = u'All graphite backends are available.'
    if servers_state == EXIT_STATUS.CRITICAL:
        if unreachable_backends == len(servers):
            output = u'All graphite backends are critical.'
        else:
            output = u'Some graphite backends are critical.'
    elif servers_state == EXIT_STATUS.WARNING:
        output = u'Some graphite backends are warning.'
    
    result.add_check(status=servers_state, output=output, long_output=metacheck_graphit_server_long_output)


def metacheck_module(raw_stats, read_status, webui_name):
    modules_state = EXIT_STATUS.OK
    known_realms = raw_stats.get(u'known_realms', [])
    headers = []
    lines = []
    modules_list = {}
    
    for rs in read_status:
        m_name = rs[u'module_name']
        realm = rs[u'realm']
        if m_name not in modules_list:
            modules_list[m_name] = {}
        if realm not in modules_list[m_name]:
            modules_list[m_name][realm] = []
        modules_list[m_name][realm].append(rs)
    
    for module_name, realms_infos in modules_list.iteritems():
        
        if webui_name and webui_name != module_name:
            continue
        
        module_state = EXIT_STATUS.OK
        module_summary = []
        manage_all = False
        
        if len(realms_infos) == 1 and realms_infos.get(u'*', None):
            manage_all = True
            useless_realm = []
            _all_hosts = realms_infos[u'*'][0].get(u'nb_hosts_clusters', None)
            if _all_hosts is None:
                _all_realms = [u'%s : Cannot get hosts/clusters infos. Check graphite backends part.' % TAG_WARNING]
            else:
                _all_realms = [u'Contains %s hosts/clusters with metrics' % _all_hosts]
            module_summary.append(HTMLList.header_list(HTMLTag.color_text(u'All realms'), _all_realms))
        else:
            useless_realm = [r for r in realms_infos.iterkeys() if r not in known_realms]
        
        all_realms = set(useless_realm).union(set(known_realms))
        
        for realm_name in all_realms:
            realm_summary = []
            realm_state = EXIT_STATUS.OK
            
            # Compute exit for realm who are in backend but broker can't manage
            if realm_name in useless_realm:
                realm_state = EXIT_STATUS.WARNING
                realm_summary.append(u'%s : The module %s does not have any data for this realm.' % (TAG_FOR_STATE[realm_state], HTMLTag.color_text(module_name)))
            realm_name = u'All realms' if realm_name == u'*' else realm_name
            
            # Take all realm info if backend is set on '*'
            info = realms_infos['*'] if manage_all else realms_infos.get(realm_name, None)
            
            if info:
                for i in info:
                    if i[u'reachable']:
                        state = EXIT_STATUS.OK if i.get(u'version', None) else EXIT_STATUS.WARNING
                        _nb_hosts = i.get(u'nb_hosts_clusters', None)
                        if _nb_hosts is None:
                            state = EXIT_STATUS.WARNING
                            output = u'Can not get the hosts number. Check the graphite backend.'
                        else:
                            output = u'Contains %s hosts/clusters with metrics' % _nb_hosts
                        
                        _graphite_summary = u'%s : Graphite backend %s ' % (TAG_FOR_STATE[state], HTMLTag.color_text(i[u'host']))
                        if state != EXIT_STATUS.OK:
                            _graphite_summary += u'has some issues. Check graphite backend part.'
                        
                        if manage_all:
                            _realm_summary = _graphite_summary
                        else:
                            _realm_summary = HTMLList.header_list(_graphite_summary, [output])
                    else:
                        state = EXIT_STATUS.CRITICAL
                        _realm_summary = HTMLList.header_list(u'%s : Graphite backend %s. ' % (TAG_FOR_STATE[state], HTMLTag.color_text(i[u'host'])), [u'Server is %s' % HTMLTag.color_text(u'unreachable', color=COLOR.RED)])
                    realm_summary.append(_realm_summary)
                    realm_state = max(realm_state, state)
            else:
                realm_state = EXIT_STATUS.CRITICAL
                realm_summary.append(u'%s : The module %s does not have a graphite backend for this realm.' % (TAG_FOR_STATE[realm_state], HTMLTag.color_text(module_name)))
            module_summary.append(HTMLList.header_list(HTMLTag.color_text(realm_name), realm_summary))
            module_state = max(module_state, realm_state)
        
        headers.append(u'%s is %s' % (module_name, TAG_FOR_STATE[module_state]))
        lines.append([HTMLList.simple_list(module_summary)])
        modules_state = max(modules_state, module_state)
    
    metacheck_graphit_server_long_output = HTMLTable.table([], lines, left_headers=headers, title=u'Module', all_col_same_width=False)
    output = u'''No issues detected for %s''' % webui_name
    if modules_state == EXIT_STATUS.WARNING:
        output = u'%s have some issues.' % webui_name
    elif modules_state == EXIT_STATUS.CRITICAL:
        output = u'%s is critical.' % webui_name
    result.add_check(status=modules_state, output=output, long_output=metacheck_graphit_server_long_output)


def _check_webui_parameter_none(webui_name):
    if webui_name is None:
        arg_name = HTMLTag.color_text(u'"-w"')
        host_template = HTMLTag.color_text(u'Host Template')
        command = HTMLTag.color_text(u'Command')
        output = u'Missing mandatory %s parameter ( webui name ). Check if the %s and the %s are up to date.' % (arg_name, host_template, command)
        result.add_check(EXIT_STATUS.WARNING, output)
        result.exit()


def _validate_webui_opt(raw_stats, webui_name):
    webui_names = [i[u'name'] for i in raw_stats[u'modules_info'] if i[u'type'] == u'webui']
    if webui_name not in webui_names:
        not_found_name = HTMLTag.color_text(webui_name)
        duplicate_foreach = HTMLTag.color_text(u'duplicate foreach')
        var_name = HTMLTag.color_text(u'MODULE_UI_LIST')
        output = u'The webui name "%s" is not known by the Broker. Check the the %s data "%s" of the host.' % (not_found_name, duplicate_foreach, var_name)
        result.add_check(EXIT_STATUS.WARNING, output)
        result.exit()


def main():
    opts = _parse_args()
    
    daemon_adr = opts.hostname
    daemon_port = opts.port
    shinken_supervisor_version = opts.shinken_supervisor_version
    timeout = opts.timeout
    webui_name = opts.webui
    
    _check_webui_parameter_none(webui_name)
    
    if timeout <= 0:
        result.hard_exit(EXIT_STATUS.CRITICAL, u'The --timeout option (%s) must be greater than 0' % timeout)
    
    html, connection_time = ShinkenUtils.request_get_daemon(result, DAEMON_TYPE, u'%s:%s' % (daemon_adr, daemon_port), u'/get_raw_stats', timeout=timeout)
    raw_stats = json.loads(html)
    
    ShinkenUtils.minimal_check(result, raw_stats, DAEMON_TYPE, shinken_supervisor_version)
    _validate_webui_opt(raw_stats, webui_name)

    html, connection_time = ShinkenUtils.request_get_daemon(result, DAEMON_TYPE, u'%s:%s' % (daemon_adr, daemon_port), u'/check_graphite_read_status', timeout=timeout)
    read_status = json.loads(html)
    
    modules_info_webui = [i for i in raw_stats[u'modules_info'] if i[u'type'] == u'webui']
    if not modules_info_webui:
        result.hard_exit(EXIT_STATUS.OK, u'No module "visualisation ui" on this broker')
    
    metacheck_graphite_server(raw_stats, read_status, opts, webui_name)
    result.add_check(long_output=BREAK_LINE)
    metacheck_module(raw_stats, read_status, webui_name)
    
    result.exit()


if __name__ == '__main__':
    main()
