#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2020:
# This file is part of Shinken Enterprise, all rights reserved.


import json
import os
import re

from shinken.misc.type_hint import Any, Dict
from shinkensolutions.api.synchronizer import escape_XSS

try:
    from collections import OrderedDict
except ImportError:
    from ordereddict import OrderedDict

Path = str  # For type hint

MODULE_PATH = '/var/lib/shinken/modules/%s/'
USER_PATH = '/etc/shinken-user/source-data/source-data-%s/configuration/'

LANGUAGES = ('fr', 'en')

# Configuration file
PATH_DEFAULT_MAPPING_RULE = 'mapping/default_mapping_rules%s.json'
PATH_DEFAULT_HOST_TEMPLATE_BINDING_RULE = 'host_template_binding/host_template_binding_rule.json'
PATH_DEFAULT_PROPERTIES_DESCRIPTION = 'collected_fields_from_source/description_of_collected_fields_%s.json'
PATH_DEFAULT_API_PROPERTIES = 'collected_fields_from_source/list_of_collected_fields.json'

# User file
PATH_USER_MAPPING_RULE = 'mapping/user_mapping_rules%s.json'
PATH_USER_HOST_TEMPLATE_BINDING_RULE = 'host_template_binding/host_template_binding_rule.json'
PATH_USER_PROPERTIES_DESCRIPTION = 'collected_fields_from_source/description_of_collected_fields_%s.json'
PATH_USER_API_PROPERTIES = 'collected_fields_from_source/list_of_collected_fields.json'


def _get_configuration_file_path(source_type, file_name):
    # type: (str, str) -> Path
    path = os.path.join(MODULE_PATH % source_type, file_name)
    return path


def _get_user_file_path(source_name, file_name):
    # type: (str, str) -> Path
    path = os.path.join(USER_PATH % source_name, file_name)
    return path


def _read_json_file(path, empty_file_return_empty_dict=False, is_xss_protected=False):
    # type: (Path, bool, bool) -> Any
    with open(path, 'r') as f:
        file_lines = f.readlines()
    
    if not file_lines and empty_file_return_empty_dict:
        return {}
    
    # We need to clean comments on this file, because json is normally without comments
    lines = ['' if line.strip().startswith('#') else line.strip() for line in file_lines]
    file_without_comment = '\n'.join(lines)
    # Remove leading commas, so we can load json file with list as [a,b,c,]
    file_without_leading_commas = re.sub(",[ \t\r\n]+}", "}", file_without_comment)
    file_without_leading_commas = re.sub(",[ \t\r\n]+]", "]", file_without_leading_commas)
    
    # Full comment file are like empty file
    if not file_without_comment.strip() and empty_file_return_empty_dict:
        return {}
    
    # json is read into in ordered dict
    # This is important for rules, so the rules are applied in the order they are written in the file
    json_data = json.loads(file_without_leading_commas, object_hook=OrderedDict)
    if is_xss_protected:
        json_data = escape_loaded_source_file_xss(json_data)
    return json_data


def escape_loaded_source_file_xss(dict_to_escape):
    # type: (OrderedDict) -> OrderedDict
    source_file_xss_escaped = OrderedDict()
    
    for item_type, source_properties in dict_to_escape.iteritems():
        if item_type not in source_file_xss_escaped:
            item_type = escape_XSS(item_type)
            source_file_xss_escaped[item_type] = OrderedDict()
        
        for source_property, source_value in source_properties.iteritems():
            source_file_xss_escaped[item_type][escape_XSS(source_property)] = source_value
    return source_file_xss_escaped


class File(object):
    def __init__(self, path, is_xss_protected=False):
        # type: (Path, bool) -> None
        self.path = path
        self.is_xss_protected = is_xss_protected
    
    
    def exist(self):
        # type: () -> bool
        return os.path.exists(self.path)
    
    
    def load(self):
        # type: () -> Any
        if self.exist():
            return _read_json_file(self.path, empty_file_return_empty_dict=True, is_xss_protected=self.is_xss_protected)
        else:
            return {}


class SourceFiles(object):
    def __init__(self, default_file, user_file):
        # type: (File, File) -> None
        self.default_file = default_file
        self.user_file = user_file


class FileLoader(object):
    def __init__(self, source_type, source_name):
        # type: (str, str) -> None
        self.source_type = source_type
        self.source_name = source_name
        
        self.properties_description = {}  # type: Dict[str,SourceFiles]
        for lang in LANGUAGES:
            self.properties_description[lang] = SourceFiles(
                File(_get_configuration_file_path(self.source_type, PATH_DEFAULT_PROPERTIES_DESCRIPTION % lang), is_xss_protected=True),
                File(_get_user_file_path(self.source_name, PATH_USER_PROPERTIES_DESCRIPTION % lang), is_xss_protected=True)
            )
        
        self.host_template_binding_rule = SourceFiles(
            File(_get_configuration_file_path(self.source_type, PATH_DEFAULT_HOST_TEMPLATE_BINDING_RULE)),
            File(_get_user_file_path(self.source_name, PATH_USER_HOST_TEMPLATE_BINDING_RULE))
        )
        self.api_properties = SourceFiles(
            File(_get_configuration_file_path(self.source_type, PATH_DEFAULT_API_PROPERTIES), is_xss_protected=True),
            File(_get_user_file_path(self.source_name, PATH_USER_API_PROPERTIES), is_xss_protected=True)
        )
    
    
    def get_named_mapping_rule(self, mapper_name):
        suffix = '' if mapper_name == 'default_mapper' else ('_%s' % mapper_name)
        return SourceFiles(
            File(_get_configuration_file_path(self.source_type, PATH_DEFAULT_MAPPING_RULE % suffix)),
            File(_get_user_file_path(self.source_name, PATH_USER_MAPPING_RULE % suffix))
        )
