#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019
# This file is part of Shinken Enterprise, all rights reserved.


import ConfigParser
import json
import os
import random
import shutil
import socket
import time
from subprocess import Popen, PIPE

from shinkensolutions.lib_checks import schecks
from shinkensolutions.lib_checks.common import Result, HTMLTag, COLOR, EXIT_STATUS, HTMLList, ParseOptionError, BREAK_LINE

GRAPHITE_CONF_FILE = '/opt/graphite/conf/carbon.conf'
IOSTATS_FILE = '/dev/shm/__check_graphite_iostats.tmp'
GATHERER_PROCESS_NAME = u'shinken-gatherer'

# Steps names
FILE_PERMISSION_STEP = u'File permission'
PORT_STATUS_STEP = u'Port %s status'

# Action messages
CARBON_CACHE_ACTION = u'receive metrics.'

try:
    import pwd
    import grp
except ImportError:  # windows
    pwd = grp = None


class GRAPHITE_STATS_KEY(object):
    TIME_READ = u'graphite_stats_time'
    LOCAL_TIME = u'graphite_local_time'


GRAPHITE_API_VERSION = u'0.9.11'
GRAPHITE_STATS_FILE_IS_TOO_OLD = 180

TAG_OK = HTMLTag.color_text(u'OK', COLOR.GREEN)
TAG_WARNING = HTMLTag.color_text(u'WARNING', COLOR.ORANGE)
TAG_CRITICAL = HTMLTag.color_text(u'CRITICAL', COLOR.RED)
TAG_UNKNOWN = HTMLTag.color_text(u'UNKNOWN', COLOR.BLACK)

TAG_RUNNING = HTMLTag.color_text(u'Running', COLOR.GREEN)
TAG_NOT_RUNNING = HTMLTag.color_text(u'Not running', COLOR.RED)

CARBON_CACHE_PROCESS_NAME = u'carbon-cache'
CARBON_RELAY_PROCESS_NAME = u'carbon-relay'

TAG_FOR_STATE = {
    EXIT_STATUS.OK      : TAG_OK,
    EXIT_STATUS.WARNING : TAG_WARNING,
    EXIT_STATUS.CRITICAL: TAG_CRITICAL,
    EXIT_STATUS.UNKNOWN : TAG_UNKNOWN,
}

WHISPER_DIR = u'/opt/graphite/storage/whisper'
NB_METRICS_COUNT_FILE = u'/opt/graphite/storage/whisper/.nb_metrics'


class GraphiteMetricsCounter(object):
    UNWANTED_METRIC_DIRECTORIES = (u'carbon', u'.cacheinvalidation', u'.nb_metrics')
    
    
    def __init__(self):
        self.metrics = 0
        self.level_0 = 0
    
    
    def update_count(self):
        self._update_level_0()
        self._do_compute_number_of_metrics_file()
        graphite_stats_time = int(time.time())
        self._write_metric_file(graphite_stats_time)
    
    
    def _update_level_0(self):
        self.level_0 = len([folder for folder in os.listdir(WHISPER_DIR) if folder not in self.UNWANTED_METRIC_DIRECTORIES])
    
    
    # This will be computed by the iostats_collector script
    def _do_compute_number_of_metrics_file(self):
        # type: () -> None
        self.metrics = 0
        
        try:
            # It's fastest to go with a find command instead of doing it ourselves
            cmd = "find %s -type f  -name '*.wsp'  | grep -v '%s/carbon' | wc -l" % (WHISPER_DIR, WHISPER_DIR)
            p = Popen(cmd, stdout=PIPE, stderr=PIPE, shell=True)
            output, stderr = p.communicate()
            self.metrics = int(output.strip())
            return
        except Exception:
            pass
        # oops, something was wrong, switch to a manual mode (maybe we are no mor allowed to launch command, like under apache)
        for root, dirs, files in os.walk(WHISPER_DIR):
            root = root.replace(WHISPER_DIR, '')
            if root.startswith(self.UNWANTED_METRIC_DIRECTORIES):
                continue
            for basename in files:
                if basename.endswith('.wsp'):
                    self.metrics += 1
    
    
    def _get_count_export(self, graphite_stats_time):
        return {'metrics': self.metrics, 'level_0': self.level_0, GRAPHITE_STATS_KEY.TIME_READ: graphite_stats_time}
    
    
    def _write_metric_file(self, graphite_stats_time):
        # type: (int) -> None
        tmp_file = NB_METRICS_COUNT_FILE + '.tmp.%d' % random.randint(1, 100000)
        count = self._get_count_export(graphite_stats_time)
        with open(tmp_file, 'w') as f:
            f.write(json.dumps(count))
        os.chown(tmp_file, pwd.getpwnam("apache").pw_uid, grp.getgrnam("apache").gr_gid)
        shutil.move(tmp_file, NB_METRICS_COUNT_FILE)  # atomic move


def _split_node_graphite(def_node):
    # safe split
    def_node = ('%s:::' % def_node).split(':')
    return def_node[0].strip(), def_node[1].strip(), def_node[2].strip()


class GraphiteConfReader(object):
    def __init__(self, graphite_hostname, graphite_port, ssh_port, ssh_key_file, passphrase, graphite_conf_file, user):
        # type: (unicode, unicode, unicode, unicode, unicode, unicode, unicode) -> None
        self.result = Result()
        self.graphite_hostname = graphite_hostname
        self.graphite_port = graphite_port
        self.ssh_port = ssh_port
        self.user = user
        self.ssh_key_file = ssh_key_file
        self.passphrase = passphrase
        self.graphite_conf_file = graphite_conf_file
        
        self.nodes = []
        self.carbon_conf = None
        self.is_relay = None
        self.client = None
    
    
    def connect(self, quit_on_fail=True, timeout=None):
        if quit_on_fail:
            self.client = schecks.connect(self.graphite_hostname, self.ssh_port, self.ssh_key_file, self.passphrase, self.user, quit_on_fail=quit_on_fail, timeout=timeout)
            return EXIT_STATUS.OK
        
        return_status, client = schecks.connect(self.graphite_hostname, self.ssh_port, self.ssh_key_file, self.passphrase, self.user, quit_on_fail=quit_on_fail, timeout=timeout)
        if return_status == EXIT_STATUS.OK:
            self.client = client
        
        return return_status
    
    
    def close(self):
        self.client.close()
    
    
    def get_mode(self):
        # type: () -> basestring
        return HTMLTag.color_text('relay (sending data to Node)' if self.is_relay else 'cache (where data is stored)', bold=True)
    
    
    def _read_carbon_conf(self, retry=4):
        # type: (int) -> ConfigParser
        stdin, stdout, stderr = self.client.exec_command('LC_ALL=C cat %s' % self.graphite_conf_file)
        self.carbon_conf = ConfigParser.SafeConfigParser({'LINE_RECEIVER_PORT': '--no port--', 'PICKLE_RECEIVER_PORT': '--no port--', 'DESTINATIONS': ''})
        self.carbon_conf.readfp(stdout)
        self.is_relay = self._is_relay_check()
        self.nodes = [_split_node_graphite(i) for i in self.carbon_conf.get('relay', 'DESTINATIONS').split(',')]
        return self.carbon_conf
    
    
    def _is_relay_check(self):
        relay_ports = (
            self.carbon_conf.get('relay', 'LINE_RECEIVER_PORT'),
            self.carbon_conf.get('relay', 'PICKLE_RECEIVER_PORT')
        )
        cache_ports = (
            self.carbon_conf.get('cache', 'LINE_RECEIVER_PORT'),
            self.carbon_conf.get('cache', 'PICKLE_RECEIVER_PORT')
        )
        
        if self.graphite_port not in cache_ports and self.graphite_port not in relay_ports:
            self.result.hard_exit(EXIT_STATUS.CRITICAL, 'Graphite port %s was not found in configuration file %s on %s server.' % (self.graphite_port, self.graphite_conf_file, self.graphite_hostname))
        
        return self.graphite_port in relay_ports


class CheckGraphite(GraphiteConfReader):
    
    def __init__(self, graphite_location, graphite_user, storage_usage_warning, storage_usage_critical, graphite_cache_name, graphite_relay_name, *argv, **kwarg):
        super(CheckGraphite, self).__init__(*argv, **kwarg)
        self.graphite_location = graphite_location
        self.graphite_user = graphite_user
        self.storage_usage_warning = storage_usage_warning
        self.storage_usage_critical = storage_usage_critical
        
        self.graphite_cache_name = graphite_cache_name
        self.graphite_relay_name = graphite_relay_name
        
        self.result = Result()
        self.summary = []
        self.nodes_summary = []
    
    
    def connect(self, quit_on_fail=False, timeout=None):
        state, client_or_message = schecks.connect(self.graphite_hostname, self.ssh_port, self.ssh_key_file, self.passphrase, self.user, quit_on_fail=quit_on_fail, timeout=timeout)
        if state == EXIT_STATUS.OK:
            self.client = client_or_message
        else:
            self.summary.append(client_or_message)
            self.result.add_check(state, client_or_message)
        
        return state
    
    
    def _read_carbon_conf(self, retry=4):
        super(CheckGraphite, self)._read_carbon_conf()
        self.graphite_location = self.graphite_location or self.carbon_conf.get('cache', 'LOCAL_DATA_DIR')
        self.graphite_user = self.graphite_user or self.carbon_conf.get('cache', 'USER')
        self.graphite_process_name = self.graphite_relay_name if self.is_relay else self.graphite_cache_name
        return self.carbon_conf
    
    
    def get_result_check_string(self, status, output, step_name=u'STEP', extra_summary=u''):
        # type: (int, unicode, unicode, unicode) -> unicode
        if status != EXIT_STATUS.OK:
            self.result.add_check(status, u'%s : %s' % (step_name, output))
        return u'%s - %s : %s' % (TAG_FOR_STATE.get(status, TAG_UNKNOWN), step_name, u'%s %s' % (output, extra_summary))
    
    
    def result_add_check(self, status, output, step_name=u'STEP', extra_summary=u''):
        # type: (int, unicode, unicode, unicode) -> None
        self.summary.append(self.get_result_check_string(status, output, step_name, extra_summary))
    
    
    def api_graphite_metric_check(self, step_name, retry=4):
        # type: (unicode, int) -> None
        while True:
            try:
                retry -= 1
                # We are checking graphite query on loopback address
                command = u'curl "http://%s/metrics/get-metrics-count"' % self.graphite_hostname
                stdin, stdout, stderr = self.client.exec_command(command)
                resp = stdout.readline()
                try:
                    count = json.loads(resp)
                    metrics_nb = count['metrics']
                except Exception:  # is not a json, so must be an old graphite
                    # We are checking graphite query on loopback address
                    command = u'curl "http://%s/metrics/index.json"' % self.graphite_hostname
                    stdin, stdout, stderr = self.client.exec_command(command)
                    resp = stdout.readline()
                    metrics = json.loads(resp)
                    metrics = [m for m in metrics if u'carbon.agents' not in m]
                    metrics_nb = len(metrics)
                
                self.result_add_check(EXIT_STATUS.OK, u'%s - %s metrics found.' % (HTMLTag.color_text('Available', COLOR.GREEN), HTMLTag.color_text(metrics_nb)), step_name)
                self.result.add_perf_data(u'nb_metrics', metrics_nb)
                return
            except Exception:
                if retry == 0:
                    self.result_add_check(EXIT_STATUS.CRITICAL, u'%s - Fail to request metric to graphite server.' % TAG_CRITICAL, step_name)
                    return
                else:
                    continue
    
    
    def graphite_port_check(self, step_name, retry=4):
        # type: (unicode, int) -> unicode
        sock = None
        while True:
            try:
                retry -= 1
                sock = socket.create_connection((str(self.graphite_hostname), int(self.graphite_port)), 0.5)
                return self.get_result_check_string(EXIT_STATUS.OK, u'%s - Can receive metrics.' % HTMLTag.color_text(u'OPEN', COLOR.GREEN), step_name)
            except Exception as exp:
                if retry == 0:
                    return self.get_result_check_string(EXIT_STATUS.CRITICAL, u'%s. Cannot receive metrics : [%s]' % (HTMLTag.color_text(u'CLOSE', COLOR.RED), exp), step_name)
                else:
                    continue
            finally:
                if sock is not None:
                    sock.close()
    
    
    def graphite_conf_type_check(self, step_name):
        # type: (unicode) -> unicode
        if self.is_relay:
            conf_type = u'RELAY'
        else:
            conf_type = u'CACHE'
        return self.get_result_check_string(EXIT_STATUS.OK, u'Type %s' % HTMLTag.color_text(conf_type), step_name)
    
    
    def _check_service_status_by_name(self, process_name, retry=4):
        # type: (unicode, int) -> int
        command = u'/usr/sbin/service %s status &> /dev/null; echo $?' % process_name
        exit_status = EXIT_STATUS.CRITICAL
        for _ in range(retry):
            _, stdout, _ = self.client.exec_command(u'LC_ALL=C %s' % command)
            for line in stdout:
                line = line.strip()
                if not line:
                    continue
                if line == u'0':
                    exit_status = EXIT_STATUS.OK
                    break
        return exit_status
    
    
    def graphite_space(self, step_name, retry=4):
        # type: (unicode, int) -> unicode
        while True:
            retry -= 1
            try:
                command_partition_usage = u'LC_ALL=C df -P %s' % self.graphite_location
                stdin, stdout, stderr = self.client.exec_command(command_partition_usage)
                
                used_pct = None
                for line in stdout:
                    line = line.strip()
                    # By pass the first line, we already know about it
                    if not line or line.startswith(u'Filesystem'):
                        continue
                    # Only keep non void elements
                    tmp = [s for s in line.split(u' ') if s]
                    used_pct = int(tmp[4][:-1])
                
                if used_pct is None:
                    exit_status = EXIT_STATUS.CRITICAL
                    output = u'Parse for size of the command [%s] fail' % command_partition_usage
                elif int(used_pct) >= int(self.storage_usage_critical):
                    exit_status = EXIT_STATUS.CRITICAL
                    output = u'%s - Greater than the critical threshold (> %s)' % (HTMLTag.color_text(str(used_pct) + u'%', COLOR.RED), HTMLTag.color_text(u'%s%%' % self.storage_usage_critical))
                elif int(used_pct) >= int(self.storage_usage_warning):
                    exit_status = EXIT_STATUS.WARNING
                    output = u'%s - Greater than the warning threshold (> %s)' % (HTMLTag.color_text(str(used_pct) + u'%', COLOR.ORANGE), HTMLTag.color_text(u'%s%%' % self.storage_usage_warning))
                else:
                    exit_status = EXIT_STATUS.OK
                    output = u'%s - Under the warning threshold ( %s )' % (HTMLTag.color_text(str(used_pct) + u'%', COLOR.GREEN), HTMLTag.color_text(u'< %s%%' % self.storage_usage_warning))

                self.result.add_perf_data(u'%s_graphite_storage_size' % self.graphite_hostname, u'%s%%' % used_pct)
                return self.get_result_check_string(exit_status, output, step_name)
            except Exception as exp:
                if retry == 0:
                    return self.get_result_check_string(EXIT_STATUS.CRITICAL, u'Fail to get server storage usage. [%s]' % exp, step_name)
                else:
                    continue
    
    
    def _graphite_file_permission(self, step_name, retry=4):
        # type: (unicode, int) -> unicode
        while True:
            retry -= 1
            try:
                command_file_permission = u'''LC_ALL=C find '%s' ! -user '%s' | head -11''' % (self.graphite_location, self.graphite_user)
                stdin, stdout, stderr = self.client.exec_command(command_file_permission)
                
                data = [line.strip() for line in stdout if line.strip()]
                have_extra_line = len(data) > 10
                data = data[:10]
                
                if data:
                    exit_status = EXIT_STATUS.CRITICAL
                    output = u'On server %s, the user %s does not have ownership on some files in %s' % (HTMLTag.color_text(self.graphite_hostname), HTMLTag.color_text(self.graphite_user), HTMLTag.color_text(self.graphite_location))
                    if have_extra_line:
                        extra_summary = u' These are the first 10 concerned files/folder: %s' % (HTMLList.simple_list(data))
                    else:
                        extra_summary = u' These are the concerned files/folder: %s' % (HTMLList.simple_list(data))
                else:
                    exit_status = EXIT_STATUS.OK
                    output = u'No problem detected'
                    extra_summary = u''
                
                return self.get_result_check_string(exit_status, output, step_name, extra_summary=extra_summary)
            except Exception as exp:
                if retry == 0:
                    return self.get_result_check_string(EXIT_STATUS.CRITICAL, u'Fail to get server file permission : [%s]' % exp, step_name)
                else:
                    continue
    
    
    def _meta_check_relay(self):
        # type: () -> None
        # STEP 1 - check of Graphite port
        self.graphite_port_check(PORT_STATUS_STEP % HTMLTag.color_text(self.graphite_port))
        # STEP 3 - check of carbon-cache process
        self.add_check_for_service_status(self.graphite_process_name, action_msg=CARBON_CACHE_ACTION)
        # STEP 4 - check API Metrics
        self.api_graphite_metric_check(step_name=u'HTTP API status')
    
    
    def _meta_check_node(self, node, node_index):
        # type: (tuple, int) -> None
        check_graphite_node = self.get_check_node(node)
        
        check_graphite_node.do_all_check()
        node_hostname_html = HTMLTag.color_text(u'%s:%s' % (check_graphite_node.graphite_hostname, check_graphite_node.graphite_port))
        _summary = u'Node %s ( Address: %s ) Type: %s is %s' % (node_index, node_hostname_html, check_graphite_node.get_mode(), TAG_FOR_STATE[check_graphite_node.result.status])
        if check_graphite_node.nodes_summary:
            _summary = u'%s %s' % (_summary, HTMLList.simple_list(check_graphite_node.nodes_summary))
        self.nodes_summary.append(_summary)
        
        if check_graphite_node.result.status != EXIT_STATUS.OK:
            self.result.add_check(status=check_graphite_node.result.status, output=_summary)
    
    
    def get_check_node(self, node):
        # type: (tuple) -> CheckGraphite
        node_graphite_location = self.graphite_location
        node_conf_file = self.graphite_conf_file
        node_graphite_user = self.graphite_user
        node_storage_usage_warning = self.storage_usage_warning
        node_storage_usage_critical = self.storage_usage_critical
        node_cache_name = self.graphite_cache_name
        node_relay_name = self.graphite_relay_name
        check_graphite_node = CheckGraphite(
            graphite_hostname=node[0],
            graphite_port=node[1],
            ssh_port=self.ssh_port,
            ssh_key_file=self.ssh_key_file,
            passphrase=self.passphrase,
            user=self.user,
            graphite_location=node_graphite_location,
            graphite_user=node_graphite_user,
            storage_usage_warning=node_storage_usage_warning,
            storage_usage_critical=node_storage_usage_critical,
            graphite_cache_name=node_cache_name,
            graphite_relay_name=node_relay_name,
            graphite_conf_file=node_conf_file,
        )
        return check_graphite_node
    
    
    def do_all_check(self):
        # type: () -> None
        connect_state = self.connect()
        if connect_state != EXIT_STATUS.OK:
            return
        self._read_carbon_conf()
        
        if self.is_relay:
            self._meta_check_relay()
            for i, node in enumerate(self.nodes):
                self._meta_check_node(node, i + 1)
        else:
            # STEP 1 - check of Graphite port
            self.graphite_port_check(PORT_STATUS_STEP % HTMLTag.color_text(self.graphite_port))
            # STEP 2 - check of carbon-cache process
            self.add_check_for_service_status(self.graphite_process_name, CARBON_CACHE_ACTION)
            # STEP 3 - check API Metrics
            self.api_graphite_metric_check(u'HTTP API status')
            # STEP 4 - check Graphite Storage
            self.summary.append(self.graphite_space(u'Storage size usage'))
            # STEP 5 - check Graphite file permission
            self._graphite_file_permission(FILE_PERMISSION_STEP)
        
        self.close()
    
    
    def add_check_for_service_status(self, process_name, action_msg=u'', error_status=EXIT_STATUS.CRITICAL):
        # type: (unicode, unicode, int) -> tuple
        success_consequence = ' ( To %s )' % action_msg if action_msg else u''
        error_consequence = ' ( Cannot %s )' % action_msg if action_msg else u''
        step_name = u'Process %s status' % HTMLTag.color_text(process_name)
        try:
            graphite_process_status = self._check_service_status_by_name(process_name)
            if graphite_process_status == EXIT_STATUS.OK:
                return graphite_process_status, self.get_result_check_string(graphite_process_status, TAG_RUNNING, step_name, success_consequence)
            else:
                error_color = COLOR.RED if error_status == EXIT_STATUS.CRITICAL else COLOR.ORANGE
                return error_status, self.get_result_check_string(error_status, HTMLTag.color_text('Not running', error_color), step_name, error_consequence)
        except Exception as exp:
            return error_status, self.get_result_check_string(error_status, u'%s - Unexpected error during check. [%s]' % exp.message, error_consequence)
    
    
    def check_and_set_output_result(self):
        # type: () -> None
        self.do_all_check()
        
        title = u'Graphite service as %s' % self.get_mode()
        if self.result.status == EXIT_STATUS.CRITICAL:
            title = u'%s is critical :' % title
        elif self.result.status in (EXIT_STATUS.WARNING, EXIT_STATUS.UNKNOWN):
            title = u'%s is not working fine :' % title
        elif self.result.status == EXIT_STATUS.OK:
            title = u'%s is working fine.' % title
        
        summary_text = HTMLList.header_list(u'Graphite (Address: %s) Type: %s' % (HTMLTag.color_text(self.graphite_hostname), self.get_mode()), self.summary)
        self.result.add_check(long_output=summary_text)
        if self.is_relay:
            nodes_summary = HTMLList.header_list(u'Node information', self.nodes_summary)
            self.result.add_check(long_output=nodes_summary)
        self.result.add_title(title)
        self.result.exit(sorted_by_level=True)


class CheckGraphiteForWriter(CheckGraphite):
    
    def __init__(self, graphite_disks, storage_io_warning, storage_io_critical, *argv, **kwarg):
        super(CheckGraphiteForWriter, self).__init__(*argv, **kwarg)
        self.graphite_disks = graphite_disks
        self.storage_io_warning = storage_io_warning
        self.storage_io_critical = storage_io_critical
    
    
    def get_check_node(self, node):
        node_graphite_location = self.graphite_location
        node_conf_file = self.graphite_conf_file
        node_graphite_user = self.graphite_user
        node_storage_usage_warning = self.storage_usage_warning
        node_storage_usage_critical = self.storage_usage_critical
        node_graphite_disks = self.graphite_disks
        node_storage_io_warning = self.storage_io_warning
        node_storage_io_critical = self.storage_io_critical
        node_cache_name = self.graphite_cache_name
        node_relay_name = self.graphite_relay_name
        check_graphite_node = CheckGraphiteNodeWriter(
            graphite_disks=node_graphite_disks,
            storage_io_warning=node_storage_io_warning,
            storage_io_critical=node_storage_io_critical,
            graphite_hostname=node[0],
            graphite_port=node[1],
            ssh_port=self.ssh_port,
            ssh_key_file=self.ssh_key_file,
            passphrase=self.passphrase,
            user=self.user,
            graphite_location=node_graphite_location,
            graphite_user=node_graphite_user,
            storage_usage_warning=node_storage_usage_warning,
            storage_usage_critical=node_storage_usage_critical,
            graphite_cache_name=node_cache_name,
            graphite_relay_name=node_relay_name,
            graphite_conf_file=node_conf_file,
        )
        return check_graphite_node
    
    
    def do_all_check(self):
        # type: () -> None
        connect_state = self.connect(quit_on_fail=False)
        if connect_state != EXIT_STATUS.OK:
            self.result_add_check(EXIT_STATUS.UNKNOWN, u'Cannot connect to server %s through SSH with user %s to get information.' % (HTMLTag.color_text(self.graphite_hostname), HTMLTag.color_text(self.user)), step_name=u'Backend stats')
            return
        self._read_carbon_conf()
        
        if self.is_relay:
            self._meta_check_relay()
            for i, node in enumerate(self.nodes):
                self._meta_check_node(node, i + 1)
        else:
            self._add_file_permission_checks()
            
            # Carbon cache check
            port_check_title = PORT_STATUS_STEP % HTMLTag.color_text(self.graphite_port)
            carbon_checks = self._get_check_list_with_sub_step(self.graphite_process_name, u'', self.graphite_conf_type_check, port_check_title)
            self.summary.append(HTMLList.header_list(u'Metric reception [%s]' % HTMLTag.color_text(self.graphite_process_name), carbon_checks))
            
            # Shinken gatherer check
            carbon_check_list_title = u'Disk performances [%s]' % HTMLTag.color_text(GATHERER_PROCESS_NAME)
            disc_stats = list()
            disc_stats.append(self.graphite_space(u'Storage size usage'))
            gatherer_checks = self._get_check_list_with_sub_step(GATHERER_PROCESS_NAME, u'get storage I/O statistics', self.check_disk_read_write, carbon_check_list_title, EXIT_STATUS.WARNING)
            disc_stats.extend(gatherer_checks)
            self.summary.append(HTMLList.header_list(u'Storage I/O statistics', disc_stats))
        self.close()
    
    
    def _add_file_permission_checks(self):
        # type: () -> None
        file_permission_check = list()
        file_permission_check.append(self._graphite_file_permission(FILE_PERMISSION_STEP))
        self.summary.append(HTMLList.header_list(FILE_PERMISSION_STEP, file_permission_check))
    
    
    def _get_check_list_with_sub_step(self, process_name, action_msg, sub_step_callback, sub_step_title, error_level=EXIT_STATUS.CRITICAL):
        # type: (unicode, unicode, function, unicode, int) -> list
        checks = list()
        check_status, check_string = self.add_check_for_service_status(process_name, action_msg, error_level)
        checks.append(check_string)
        if check_status == EXIT_STATUS.OK:
            checks.append(sub_step_callback(sub_step_title))
        return checks
    
    
    def check_disk_read_write(self, step_name, retry=4):
        # type: (unicode, int) -> unicode
        data = None
        while True:
            retry -= 1
            try:
                command_io = u'cat %s' % IOSTATS_FILE
                _, stdout, _ = self.client.exec_command(command_io)
                raw = [line for line in stdout]
                data = json.loads(''.join(raw))
                break
            except Exception as _:
                if retry == 0:
                    return self.get_result_check_string(
                        EXIT_STATUS.WARNING,
                        u'%s - On server %s, cannot read the file %s.' % (TAG_WARNING, HTMLTag.color_text(self.graphite_hostname), HTMLTag.color_text(IOSTATS_FILE)),
                        step_name,
                        extra_summary=u'The graphite server may not be up to date, please try to restart shinken-gatherer process (%s).' % HTMLTag.color_text(u'service shinken-gatherer restart')
                    )
                else:
                    continue
        
        if not data:
            return self.get_result_check_string(EXIT_STATUS.UNKNOWN, u'Cannot find any disks in the iostats file.', step_name)
        
        exit_status = EXIT_STATUS.OK
        lines = []
        disk = [d.strip() for d in self.graphite_disks.split(',') if d.strip()]
        for disk_name, data_for_disk in data.iteritems():
            if disk and disk_name not in disk:
                continue
            _exist_status = EXIT_STATUS.OK
            values = [float(v) for v in data_for_disk]
            
            load_range = 0
            load_sum = 0.0
            for value in values:
                load_range += 1
                load_sum += value
            
            avg = load_sum / load_range
            
            # Add perf before continue because avg will be transform into str
            self.result.add_perf_data('%s_graphite_disk_%s_IO' % (self.graphite_hostname, disk_name), '%s%%' % avg)
            
            if avg > int(self.storage_io_critical):
                exit_status = EXIT_STATUS.CRITICAL
                _exist_status = EXIT_STATUS.CRITICAL
                avg = HTMLTag.color_text('%.2f' % avg, COLOR.RED)
            elif avg > int(self.storage_io_warning):
                exit_status = EXIT_STATUS.WARNING if exit_status < EXIT_STATUS.WARNING else exit_status
                _exist_status = EXIT_STATUS.WARNING
                avg = HTMLTag.color_text('%.2f' % avg, COLOR.ORANGE)
            else:
                exit_status = EXIT_STATUS.OK if exit_status < EXIT_STATUS.OK else exit_status
                _exist_status = EXIT_STATUS.OK
                avg = HTMLTag.color_text('%.2f' % avg, COLOR.GREEN)
            
            lines.append(('Disk %s %s : Average : %s%% - Minimum : %s%% - Maximum : %s%%' % (HTMLTag.color_text(disk_name), TAG_FOR_STATE[_exist_status], avg, min(values), max(values))))
        
        if not lines:
            return self.get_result_check_string(EXIT_STATUS.UNKNOWN, u'Cannot find any disks in the iostats file. Maybe the disk filter is not correct. Actual value : %s' % disk, step_name)
        
        output = HTMLList.simple_list(lines)
        return self.get_result_check_string(exit_status, output, step_name)


class CheckGraphiteNodeWriter(CheckGraphiteForWriter):
    def connect(self, quit_on_fail=True, timeout=None):
        return super(CheckGraphiteNodeWriter, self).connect(quit_on_fail, timeout)


def validate_graphite_check_option_parser(parser):
    opts = None
    try:
        opts, args = parser.parse_args()
        if args:
            parser.error('Does not accept any argument.')
        
        if not opts.hostname:
            parser.error('Missing parameter hostname (-H/--hostname)')
        
        if (opts.warning and not opts.critical) or (opts.critical and not opts.warning):
            parser.error('Warning parameter (-w) and Critical parameter (-c) must be used together')
    except ParseOptionError as e:
        result = Result()
        result.hard_exit(EXIT_STATUS.CRITICAL, 'Fail to parse command argument : %s %s' % (BREAK_LINE, BREAK_LINE.join(e.msg.split('\n'))))
    return opts
