#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2018:
#
# This file is part of Shinken Enterprise, all rights reserved.
import re

from shinken.misc.type_hint import Dict

SELECT_PROP_FOR_OVERRIDE = (
    u'contacts',
    u'contact_groups',
    u'escalations',
    u'business_impact_modulations',
    u'macromodulations',
    u'resultmodulations',
    u'notification_period',
    u'maintenance_period'
)

OVERRIDE_PATTERN = re.compile(ur'^([^,]+),\s*([^\s]+)\s+(.*)$', re.UNICODE | re.DOTALL)


class SyntaxServiceOverrideError(SyntaxError):
    def __init__(self, syntax_errors=None, duplicate_errors=None):
        self.syntax_errors = syntax_errors
        self.duplicate_errors = [u'%s:%s' % (prop, value) for prop, value in duplicate_errors] if duplicate_errors else None
        self.message = u''
        if self.syntax_errors:
            self.message = u'syntax errors in service override: %s' % (u','.join(self.syntax_errors))
        if self.duplicate_errors:
            self.message = u'duplicate definition in service override: %s' % (u','.join(self.duplicate_errors))
    
    
    def __str__(self):
        return self.message
    
    
    def __repr__(self):
        return self.message


def parse_service_override_property_list_errors(service_overrides_property):
    # type: (unicode) -> Dict
    # Parses a service_overrides property and returns a dict of dicts {'service_description': { 'prop1': 'value1', 'prop2': 'value2'}}
    # This property is provided in the form $check,$property $value-=#=-$check2,$property2 $value-=#=- ...
    
    from shinken.objects import Host
    
    if not service_overrides_property:
        return {u'parsed_overrides': {}, u'duplicates': [], u'syntax': []}
    
    separator = Host.properties[u'service_overrides'].separator
    service_overrides = [so.strip() for so in service_overrides_property.split(separator)]
    
    duplicate_errors = []
    syntax_errors = []
    parsed_overrides = {}
    for ovr in service_overrides:
        # Checks service override syntax
        match = OVERRIDE_PATTERN.match(ovr)
        if match is None:
            syntax_errors.append(ovr)
            continue
        service_description, prop, value = match.groups()
        if not (service_description and prop and value):
            syntax_errors.append(ovr)
            continue
        service_description = service_description.strip()
        prop = prop.strip()
        value = value.strip()
        if prop in SELECT_PROP_FOR_OVERRIDE:
            value = value.strip(',')
        if service_description not in parsed_overrides:
            parsed_overrides[service_description] = {}
        
        if prop in parsed_overrides[service_description]:
            del parsed_overrides[service_description][prop]
            duplicate_errors.append((prop, service_description))
        else:
            parsed_overrides[service_description][prop] = value
    
    return {u'parsed_overrides': parsed_overrides, u'duplicates': duplicate_errors, u'syntax': syntax_errors}


def parse_service_override_property(service_overrides_property):
    # type: (unicode) -> Dict
    overrides = parse_service_override_property_list_errors(service_overrides_property)
    if overrides[u'syntax']:
        raise SyntaxServiceOverrideError(syntax_errors=overrides[u'syntax'])
    
    if overrides[u'duplicates']:
        raise SyntaxServiceOverrideError(duplicate_errors=overrides[u'duplicates'])
    
    return overrides[u'parsed_overrides']


def unparse_service_override(parsed):
    # type: (Dict) -> unicode
    
    from shinken.objects import Host
    unparsed = []
    for check, so_value in parsed.iteritems():
        for prop_name, value in so_value.iteritems():
            unparsed.append(u'%s, %s %s' % (check, prop_name, value))
    separator = Host.properties[u'service_overrides'].separator
    return separator.join(unparsed)
