#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#     Gabes Jean, naparuba@gmail.com
#     Gerhard Lausser, Gerhard.Lausser@consol.de
#     Gregory Starck, g.starck@gmail.com
#     Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

import math
import re

from shinken.misc.type_hint import TYPE_CHECKING
from shinken.objects.state_id import STATE_ID

if TYPE_CHECKING:
    from shinken.misc.type_hint import List, Optional, Set, Tuple, Any
    from shinken.objects.proxyitem import ProxyItems
    from shinken.objects import SchedulingItem


class DependencyNodeUninitializedException(Exception):
    pass


class DependencyNode(object):
    """
    Here is a node class for dependency_node(s) and a factory to create them
    """
    __state_displays = {2: u'CRITICAL', 3: u'UNKNOWN', 0: u'OK', 1: u'WARNING', -1: u'UNINITIALIZED'}
    
    
    def __init__(self):
        # type: () -> None
        self.operand = None
        self.sons = []
        # Of: values are a triple OK,WARN,CRIT
        self.of_values = (u'0', u'0', u'0')
        self.state_rules = []
        self.output = u''
        self.long_output = u''
        self.is_of_mul = False
        self.configuration_errors = []
        self.not_value = False
    
    
    def __str__(self):
        # type: () -> unicode
        return self.print_item()
    
    
    def __unicode__(self):
        # type: () -> unicode
        return self.print_item()
    
    
    def print_item(self, level=STATE_ID.OK):
        # type: (int) -> unicode
        son_str = []
        for s in self.sons:
            if isinstance(s, DependencyNode):
                son_str.append(s.print_item(level + 1))
            else:  # proxy item
                son_str.append(u' ==element:%s==' % s.name)
        
        op = self.operand
        if op == u'xof':
            op = u'%s of (mul %s)' % (self.of_values, self.is_of_mul)
        return u'''\n[depnode] %s Op:'%s' IsNot:'%s' Sons:'[%s]\'''' % (u'    ' * level, op, self.not_value, u','.join(son_str))
    
    
    @staticmethod
    def get_reverse_state(state):
        # type: (int) -> int
        # Warning is still warning
        if state == STATE_ID.WARN:
            return state
        if state == STATE_ID.OK:
            return STATE_ID.CRIT
        if state == STATE_ID.CRIT:
            return STATE_ID.OK
        # should not go here...
        return state
    
    
    # We will get the state of this node, by looking at the state of our sons, and apply our operand
    def get_state(self, bound_item):
        # type: (DependencyNode) -> int
        # If we are a host or a service, we just got the host/service hard state
        # We dump our sons output as long output
        
        if self.operand == u'proxy_item':
            state = self.get_simple_node_state()
        else:
            # Maybe we have no sons at all? if so, we are unknown
            if len(self.sons) == 0:
                self.output = u'This cluster did not match any element.'
                self.long_output = u''
                return STATE_ID.UNKNOWN
            
            if not self.sons_are_all_initialized(bound_item):
                raise DependencyNodeUninitializedException()
            
            self.long_output = u'<ul>'
            state = self.get_complex_node_state(bound_item)
            self.long_output += u'</ul>'
        
        return state
    
    
    # Returns (is_inherited_flapping, is_partial_flapping) booleans
    def get_flapping_state(self):
        # type: () -> Tuple[bool, bool]
        elements_count = len(self.list_all_elements())
        flapping_count = len([son for son in self.list_all_elements() if son.in_flapping])
        
        if flapping_count == 0:
            return False, False
        elif flapping_count == elements_count:
            return True, False
        else:
            return False, True
    
    
    def get_downtime_state(self):
        # type: () -> Tuple[bool, bool]
        elements_count = len(self.list_all_elements())
        downtime_count = len([son for son in self.list_all_elements() if son.in_downtime or son.in_inherited_downtime or son.in_partial_downtime])
        
        is_inherited_dt = True
        is_not_inherited_dt = False
        is_partial_dt = True
        is_not_partial_dt = False
        
        if downtime_count == 0:
            return is_not_inherited_dt, is_not_partial_dt
        elif downtime_count == elements_count:
            return is_inherited_dt, is_not_partial_dt
        else:
            return is_not_inherited_dt, is_partial_dt
    
    
    # Returns a simple node direct state (such as an host or a service). No
    # calculation is needed
    def get_simple_node_state(self):
        # type: () -> int
        proxy = self.sons[0]
        state = proxy.state
        _type = proxy.type
        
        # The host haven't the same mapping for state id :
        # * 1 is DOWN => must be shown as critical (2)
        # * 2 is UNREACHABLE => must be shown as unknown
        if _type == 'host':
            if state == STATE_ID.WARN:
                state = STATE_ID.CRIT
            elif state == STATE_ID.CRIT:
                state = STATE_ID.UNKNOWN
        context = []
        if proxy.in_downtime or proxy.in_inherited_downtime or proxy.in_inherited_downtime:
            context.append(u'DOWNTIME')
        if proxy.in_acknowledge or proxy.in_inherited_acknowledged:
            context.append(u'ACKNOWLEDGEMENT')
        if proxy.in_flapping:
            context.append(u'FLAPPING')
        status = self.__state_displays.get(state, u'UNKNOWN')
        if context:
            status = u'%s [%s]' % (status, u','.join(context))
        self.output = u'%s %s' % (status, proxy.get_full_name())
        if self.not_value:
            if state == STATE_ID.CRIT:
                return STATE_ID.OK
            if state == STATE_ID.OK:
                return STATE_ID.CRIT
        
        return state
    
    
    def sons_are_all_initialized(self, bound_item):
        # type: (DependencyNode) -> bool
        # look for first uninitialized son, if None found, everyone is initialized
        return next((s for s in self.sons if s.get_state(bound_item) == -1), None) is None
    
    
    # Calculates a complex node state based on its sons state, and its operator
    # Also properly sets the output
    def get_complex_node_state(self, bound_item):
        # type: (DependencyNode) -> int
        states = []
        for s in self.sons:
            state = s.get_state(bound_item)
            if state != 0:
                self.long_output += u'<li>%s</li>' % s.output
            states.append(state)
        
        nb_ok = len([s for s in states if s == STATE_ID.OK])
        nb_warn = len([s for s in states if s == STATE_ID.WARN])
        nb_crit = len([s for s in states if s == STATE_ID.CRIT])
        nb_unknown = len([s for s in states if s == STATE_ID.UNKNOWN])
        if nb_ok > 0:
            self.long_output += u'<li>And [%d] OK.</li>' % nb_ok
        if self.operand == u'|':
            my_state = min(states)
            self.output = u'%s, best of %%s states' % self.__state_displays[my_state]
        
        elif self.operand == u'&':
            if STATE_ID.CRIT in states:
                my_state = STATE_ID.CRIT
            else:
                my_state = max(states)
            self.output = u'%s, worst of %%s states' % self.__state_displays[my_state]
        
        elif self.operand == u'rule':
            my_state = self.get_complex_rule_node_state(nb_ok, nb_warn, nb_crit, nb_unknown)
        # Xof rule
        else:
            my_state = self.get_complex_xof_node_state(nb_ok, nb_warn, nb_crit, nb_unknown)
        
        state_nbs = [nb_ok, nb_warn, nb_crit, nb_unknown]
        state_output = u'[%s]' % u', '.join([u'%d %s' % (state_nbs[i], self.__state_displays[i]) for i in range(4) if state_nbs[i]])
        
        self.output %= state_output
        if self.not_value:
            self.output += u' (reversed)'
            my_state = self.get_reverse_state(my_state)
        source_dt = []
        source_ack = []
        source_flap = []
        all_sons = self.list_all_elements()
        for son in all_sons:
            if son.in_downtime or son.in_inherited_downtime:
                source_dt.append(u'<li>%s</li>' % son.get_full_name())
            if son.in_flapping:
                source_flap.append(u'<li>%s</li>' % son.get_full_name())
            if son.in_acknowledge or son.in_inherited_acknowledged:
                source_ack.append(u'<li>%s</li>' % son.get_full_name())
        
        if bound_item.is_in_downtime() and bound_item.downtimes:
            context = u'DOWNTIME'
            self.output += u'<br>%s context' % context
        elif source_dt:
            context = u'DOWNTIME' if len(source_dt) == len(all_sons) else u'PARTIAL-DOWNTIME'
            self.output += u'<br>%s context from : <ul>%s</ul>' % (context, u''.join(source_dt))
        
        if bound_item.acknowledgement:
            context = u'ACKNOWLEDGED'
            self.output += u'<br>%s context' % context
            if bound_item.acknowledgement.automatic:
                self.output += u' from : <ul>%s</ul>' % (u''.join(source_ack))
        elif source_ack:
            context = u'ACKNOWLEDGE' if len(source_ack) == len(all_sons) else u'PARTIAL-ACKNOWLEDGE'
            self.output += u'<br>%s context from : <ul>%s</ul>' % (context, u''.join(source_ack))
        
        if bound_item.is_flapping:
            context = u'FLAPPING'
            self.output += u'<br>%s context' % context
            if bound_item.inherited_flapping:
                self.output += u' from : <ul>%s</ul>' % u''.join(source_flap)
        elif source_flap and bound_item.flap_detection_enabled:
            context = u'FLAPPING' if len(all_sons) == len(source_flap) else u'PARTIAL-FLAPPING'
            self.output += u'<br>%s context from : <ul>%s</ul>' % (context, u''.join(source_flap))
        return my_state
    
    
    def get_complex_rule_node_state(self, nb_ok, nb_warn, nb_crit, nb_unknown):
        # type: (int, int, int, int) -> int
        self.output = u''
        for rule in self.state_rules:
            if rule.does_apply(nb_ok, nb_warn, nb_crit, nb_unknown):
                if rule.is_default:
                    self.output += u'%s (default rule), calculated from states %%s' % self.__state_displays[rule.out_state_id]
                    return rule.out_state_id
                else:
                    self.output += u'%s (%s %s rule), calculated from states %%s' % (self.__state_displays[rule.out_state_id], rule.of_value.replace(u'%', u'%%'), self.__state_displays[rule.son_state_id])
                    return rule.out_state_id
        # ok, so no matching rule...
        self.output += u'no matching rule and no default found for states %s'
        return 3
    
    
    # Calculates a complex node state with an Xof operand
    def get_complex_xof_node_state(self, nb_ok, nb_warn, nb_crit, nb_unknown):
        # type: (int, int, int, int) -> int
        # We search for OK, WARN or CRIT applications
        # And we will choice between them
        nb_search_ok = self.of_values[0]
        nb_search_warn = self.of_values[1]
        nb_search_crit = self.of_values[2]
        
        # We look for each application
        nb_sons = nb_ok + nb_warn + nb_crit + nb_unknown
        
        # Ok and Crit apply with their own values
        # Warn can apply with warn or crit values
        # so a W C can raise a Warning, but not enough for  a critical
        
        def get_state_for(nb_tot, nb_real, nb_search):
            # type: (int, int, unicode) -> float
            if nb_search.endswith(u'%'):
                nb_search = int(nb_search[:-1])
                if nb_search < 0:
                    # nb_search is negative, so +
                    nb_search = max(100 + nb_search, 0)
                
                apply_for = float(nb_real) / nb_tot * 100 >= nb_search
            else:
                nb_search = int(nb_search)
                if nb_search < 0:
                    # nb_search is negative, so +
                    nb_search = max(nb_tot + nb_search, 0)
                apply_for = nb_real >= nb_search
            return apply_for
        
        
        ok_apply = get_state_for(nb_sons, nb_ok, nb_search_ok)
        warn_apply = get_state_for(nb_sons, nb_warn + nb_crit, nb_search_warn)
        crit_apply = get_state_for(nb_sons, nb_crit, nb_search_crit)
        
        # return the worst state that apply
        if crit_apply:
            my_state = STATE_ID.CRIT
        
        elif warn_apply:
            my_state = STATE_ID.WARN
        
        elif ok_apply:
            my_state = STATE_ID.OK
        
        else:
            # No match: return the worst state
            if nb_crit > 0:
                my_state = STATE_ID.CRIT
            elif nb_unknown > 0:
                my_state = STATE_ID.UNKNOWN
            elif nb_warn > 0:
                my_state = STATE_ID.WARN
            else:
                my_state = STATE_ID.UNKNOWN
        if nb_sons != 0:
            if nb_search_ok.endswith(u'%'):
                # output is formatted 2 times, so to display a u'%' we need u'%%%%' -> u'%%' -> u'%'
                self.output = u'[%s%%%% / %s%%%%] -> %s, calculated from states %%s' % (math.floor(100 * float(nb_ok) / nb_sons), int(nb_search_ok[:-1]), self.__state_displays[my_state])
            else:
                if int(nb_search_warn) > 0 and int(nb_search_crit) > 0:
                    self.output = u'%s, calculated from states %%s' % (self.__state_displays[my_state])
                else:
                    self.output = u'[%s / %s] -> %s, calculated from states %%s' % (int(nb_search_ok), nb_ok, self.__state_displays[my_state])
        return my_state
    
    
    # return a list of all host/service in our node and below
    def list_all_elements(self):
        # type: () -> List[DependencyNode]
        r = []
        # We are a host/service
        if self.operand in [u'proxy_item']:
            return [self.sons[0]]
        
        for s in self.sons:
            r.extend(s.list_all_elements())
        
        # and uniq the result
        return list(set(r))
    
    
    @staticmethod
    def calc_real_of_value(of_value, nb_sons):
        # type: (unicode, int) -> Optional[unicode]
        if not of_value:
            return
        if u'%' in of_value:
            val = int(of_value[:-1])
            if val < 0:
                val += 100
            return str(val) + u'%'
        val = int(of_value)
        if val < 0:
            val += nb_sons
        if val > nb_sons:
            raise Exception(u'Error, xof rule expecting higher value (%s) than the number of sons (%s)' % (of_value, nb_sons))
        return unicode(val)
    
    
    # If we are a of: rule, we can get some 0 in of_values,
    # if so, change them with NB sons instead
    def switch_zeros_of_values(self):
        # type: () -> None
        nb_sons = len(self.sons)
        # Need a list for assignment
        self.of_values = list(self.of_values)
        for i in xrange(3):
            if self.of_values[i] == u'0':
                self.of_values[i] = unicode(nb_sons)
            self.of_values[i] = self.calc_real_of_value(self.of_values[i], nb_sons)
        for rule in self.state_rules:
            rule.of_values = self.calc_real_of_value(rule.of_value, nb_sons)
        self.of_values = tuple(self.of_values)
    
    
    def add_elements_as_son(self, elements, not_value=False):
        # type: (Set[DependencyNode], bool) -> None
        
        # If the node value has a NOT attribute, we apply Boole algebra
        # because the NOT attribute is set on all the sons
        if not_value:
            node = DependencyNode()
            if node.operand == u'&' or self.operand is None:
                node.operand = u'|'
            elif self.operand == u'|':
                node.operand = u'&'
            self.sons.append(node)
        else:
            node = self
        
        for element in elements:
            part_node = DependencyNode()
            part_node.not_value = not_value
            
            part_node.operand = element.__class__.my_type
            part_node.sons.append(element)
            node.sons.append(part_node)


class StateRule(object):
    def __init__(self, of_value, son_state, out_state):
        # type: (unicode, str, str) -> None
        states = {
            u'ok'      : STATE_ID.OK,
            u'warning' : STATE_ID.WARN,
            u'critical': STATE_ID.CRIT,
            u'unknown' : STATE_ID.UNKNOWN,
        }
        self.is_default = son_state == u'default'
        self.of_value = of_value
        self.son_state_id = states.get(son_state, 3)
        self.out_state_id = states.get(out_state, 3)
    
    
    def does_apply(self, nb_ok, nb_warn, nb_crit, nb_unknown):
        # type: (int, int, int, int) -> bool
        if self.is_default:
            return True
        states_nb = (nb_ok, nb_warn, nb_crit, nb_unknown)
        nb_sons = sum(states_nb)
        
        if u'%' in self.of_value:
            nb_search = int(self.of_value[:-1]) * nb_sons / 100.0
        else:
            nb_search = int(self.of_value)
        if nb_search < 0:
            nb_search = max(nb_sons + nb_search, 0)
        nb_search = math.ceil(nb_search)
        return states_nb[self.son_state_id] >= nb_search


# Tokens for all parsable rules.
# Essentially a data container, which associates a regexp match with a type, and can provides accessors to parsed values.
class BpRuleToken(object):
    def __init__(self, tok_type, match_object=None, start=None, end=None, val=None):
        # type: (unicode, Any, Optional[int], Optional[int], Optional[unicode]) -> None
        # The second argument is a regex match ( re.search return ) but I could not find how to properly import 'SRE_Match' type
        self.type = tok_type
        if match_object:
            self.start = match_object.start()
            self.end = match_object.end()
            self.val = match_object.group(0).strip().replace(u'"', u'')
        else:  # take param from a manually forged token
            self.start = start
            self.env = end
            self.val = val.strip()
    
    
    def __unicode__(self):
        # type: () -> unicode
        return u'([%s],[%s])' % (self.type, self.val)
    
    
    def __str__(self):
        # type: () -> unicode
        return unicode(self)
    
    
    def __repr__(self):
        # type: () -> unicode
        return unicode(self)
    
    
    @staticmethod
    def sorted_insert(tok, token_list):
        # type: (BpRuleToken, List[BpRuleToken]) -> None
        i = 0
        for prev_tok in token_list:
            prev_start = prev_tok.start
            if prev_start < tok.start:
                i += 1
            else:
                break
        token_list.insert(i, tok)
    
    
    def get_node_flags(self):
        # type: () -> unicode
        if self.type != u'node':
            raise Exception(u'internal bp rule evaluation error: get_node_flags for non-flags')
        if re.match(re.compile(r'^[gtlr]+:.+'), self.val):
            return self.val.split(u':')[0]
        return u''
    
    
    def get_xof_vals(self):
        # type: () -> Tuple[bool, Tuple[unicode, unicode, unicode]]
        if self.type != u'xof':
            raise Exception(u'Internal BP rule evaluation error: get_xof_vals for non-xof')
        nums = re.findall(r'-?\d+%?', self.val)
        if len(nums) == 1:
            return False, (nums[0], u'0', u'0')
        elif len(nums) == 3:
            return True, (nums[0], nums[1], nums[2])
    
    
    def get_node_expr(self):
        # type: () -> unicode
        if self.type != u'node':
            raise Exception(u'Internal BP rule evaluation error: get_node_expr for non-node')
        if re.match(re.compile(r'^[gtlr]+:.+'), self.val):
            # Remove flags
            expr = u':'.join(self.val.split(u':')[1:])
            if u'r' in self.get_node_flags() and expr[0] == expr[-1] == u'/':
                # Remove regexp delimiters
                expr = expr[1:-1]
            return expr.strip()
        return self.val.strip()
    
    
    # When we have '(' or '[' we want to extract the part which is enclosed and treat it separately.
    def get_sub_part(self, token_list):
        # type: (List[BpRuleToken]) -> List[BpRuleToken]
        if self.val == u'[':
            matching_close = u']'
        elif self.val == u'(':
            matching_close = u')'
        else:
            raise Exception(u'Internal BP rule evaluation error: get_sub_part for non-separator')
        stacked_part = []
        parenthesis_level = 1
        while token_list:
            next_tok = token_list.pop(0)
            # We found another parenthesis ( or bracket ) we take it in our sub part
            if next_tok.val == self.val:
                parenthesis_level += 1
            # We found a closing parenthesis ( or bracket )
            elif next_tok.val == matching_close:
                parenthesis_level -= 1
                # This closing parenthesis ( or bracket ) is the one closing the sub part
                if parenthesis_level == 0:
                    break
            stacked_part.append(next_tok)
        return stacked_part
    
    
    def get_state_rule(self):
        # type: () -> StateRule
        if self.type != u'state_rule':
            raise Exception("Internal BP rule evaluation error: get_state_rule for non-rule")
        num = re.search(r'-?\d+%?', self.val)
        if num:
            num = num.group(0)
        _states = re.findall(r'[a-zA-Z]+', self.val)
        return StateRule(num, _states[0].lower(), _states[1].lower())
    
    
    def get_rules_part(self, token_list):
        # type: (List[BpRuleToken]) -> List[BpRuleToken]
        if self.type != u'state_rule':
            raise Exception(u'Internal BP rule evaluation error: get_rules_part for non-rule')
        stacked_part = [self]
        while token_list:
            next_tok = token_list.pop(0)
            if next_tok.type != u'state_rule_of':
                stacked_part.append(next_tok)
            else:
                break
        return stacked_part


class DependencyNodeFactory(object):
    """ TODO: Add some comment about this class for the doc"""
    HOST_FLAGS = 'grt'
    SERVICE_FLAGS = 'rt'
    COMPLEX_NODE_PATTERN = '()+&|[]'
    # Not all of forbidden chars are used, because a non-escaped regex might use things like ? + *
    FORBIDDEN_CHARS_CLASS = '~!&|<>,()\[\]'
    STATE_RE = r'\s*(ok|warning|critical|unknown)\s*'
    
    
    def __init__(self, bound_item):
        self.bound_item = bound_item
    
    
    def tokenize(self, pattern):
        # type: (unicode) -> List[BpRuleToken]
        # Order matters. First tokens take priority over lower ones in case a pattern matches several.
        tokens = [
            (re.compile(r'(default|-?\d+%%?%(states)s)\s*->\s*%(states)s' % {'states': self.STATE_RE}, re.IGNORECASE), u'state_rule'),
            (re.compile(r'-?\d+%?(, *-?\d*%?, *-?\d*%?)? *of:'), u'xof'),
            (re.compile(r'of\s*:'), u'state_rule_of'),
            (re.compile(r'[%(flags)s]+:/.*?[^\\]/' % {'flags': self.HOST_FLAGS}), u'node'),
            (re.compile(r'([%(flags)s]+:\s*)?"[^"]+"' % {'flags': self.HOST_FLAGS}), u'node'),
            (re.compile(r'(<or>|<and>|<and not>)'), u'part_op'),
            (re.compile(r'([%(flags)s]+:\s*)?[^\s%(forbidden)s][^%(forbidden)s]*' % {'flags': self.HOST_FLAGS, 'forbidden': self.FORBIDDEN_CHARS_CLASS}), u'node'),
            (re.compile(r','), u'comma'),
            (re.compile(r'!'), u'not'),
            (re.compile(r'[&|]'), u'op'),
            (re.compile(r'[()\[\]]'), u'sep'),
            # When we are done tokenizing, only spaces and filling chars must be left, else we need to raise an error
            (re.compile(r'[^~\s]+'), 'unknown_token')
        ]
        found = []
        for token_re, token_type in tokens:
            while True:
                match = re.search(token_re, pattern)
                if match:
                    bp_rule_token = BpRuleToken(token_type, match)
                    BpRuleToken.sorted_insert(bp_rule_token, found)
                    # The '~' is a neutral character that can't match any pattern.
                    # The idea is to fill what we previously matched to avoid pattern overlap and keep track of token order.
                    pattern = pattern[:match.start()] + (u'~' * (match.end() - match.start())) + pattern[match.end():]
                else:
                    break
        return found
    
    
    # the () will be eval in a recursive way, only one level of ()
    
    def eval_cor_pattern(self, pattern, proxyitems, ref=None):
        # type: (unicode, ProxyItems, Optional[SchedulingItem]) -> DependencyNode
        if ref:
            ref = proxyitems.find_by_uuid(ref.uuid)
        pattern = pattern.strip()
        tokens = self.tokenize(pattern)
        try:
            res = self.eval_complex_cor_pattern(tokens, proxyitems, ref)
            
            return res
        except Exception as e:
            node = DependencyNode()
            node.configuration_errors.append(e)
            return node
    
    
    @staticmethod
    def eval_rules_part(tokens):
        # type: (List[BpRuleToken]) -> List[StateRule]
        rules = []
        expect_state_rule = True
        
        while tokens:
            tok = tokens.pop(0)
            tok_type = tok.type
            tok_val = tok.val  # Extracts match
            if expect_state_rule:
                if tok_type != 'state_rule' and tok_val.strip() == '':  # Skip empty tokens
                    continue
                elif tok_type == 'state_rule':
                    rules.append(tok.get_state_rule())
                    expect_state_rule = False
                else:
                    raise Exception('Error, expected state aggregation rule after |')
            else:
                # We have a '|' separating several state rules, expect a state rule
                if tok_val == '|' and not expect_state_rule:
                    expect_state_rule = True
                else:
                    raise Exception('Error, unexpected token %s in a rule' % tok_val)
        return rules
    
    
    # Evaluate a complex correlation expression, such as an &, |, nested expressions in par, and so on.
    def eval_complex_cor_pattern(self, tokens, proxyitems, ref=None):
        # type: (List[BpRuleToken], ProxyItems, Optional[List[unicode]]) -> DependencyNode
        node = DependencyNode()
        
        son_is_not = False  # We keep is the next son will be not or not
        # some token need to know previous tok to known what to do
        previous_tok_type = 'START_OF_PATTERN'
        
        while tokens:
            tok = tokens.pop(0)
            tok_type = tok.type
            tok_val = tok.val  # Extracts match
            
            # Manage the NOT for an expression.
            if tok_type == u'not':
                son_is_not = True
            elif tok_type == u'sep':
                if tok_val == u'(':
                    stacked_part = tok.get_sub_part(tokens)
                    son_node = self.eval_complex_cor_pattern(stacked_part, proxyitems, ref)
                    if son_is_not:
                        son_node.not_value = True
                        son_is_not = False
                    node.sons.append(son_node)
                elif tok_val == u'[':
                    stacked_part = tok.get_sub_part(tokens)
                    # We retrieved all elements from the stack part, now we have to convert them in dep nodes to add them to the expression
                    node.add_elements_as_son(self.eval_stacked_part_item(stacked_part, proxyitems, ref))
                    node.not_value = son_is_not
                    son_is_not = False
                else:
                    raise Exception(u'Error, unexpected separator %s' % tok_val)
            elif tok_type == u'op':
                if node.operand in (u'&', u'|') and tok_val != node.operand:
                    raise Exception(u'Error, mixing & and | in same section is not allowed')
                if not node.operand:
                    node.operand = tok_val
            elif tok_type == u'xof':
                if node.operand:
                    raise Exception(u'Error, unexpected xof operator %s' % tok_val)
                node.operand = u'xof'
                node.is_of_mul, node.of_values = tok.get_xof_vals()
            elif tok_type == u'state_rule':
                if node.operand:
                    raise Exception(u'Error, unexpected state aggregation rule operator %s' % tok_val)
                node.operand = u'rule'
                rules_part = tok.get_rules_part(tokens)
                node.state_rules = self.eval_rules_part(rules_part)
            elif tok_type == u'node':
                # Skip void nodes, should be caught by the comma rule
                if tok_val.strip() == u'':
                    # note: continue to not erase previous_tok_type
                    continue
                # A simple pattern is either a node or node comma node
                stacked_part = [tok]
                if tokens:
                    next_tok = tokens[0]
                    if next_tok.type == u'comma':
                        tokens.pop(0)
                        if not tokens or tokens[0].type != u'node':
                            raise Exception(u'error: expected check matching expression after \'%s,\'' % tok_val)
                        service_tok = tokens.pop(0)
                        stacked_part.append(service_tok)
                
                items = self.eval_simple_cor_pattern(stacked_part, proxyitems, ref)
                node.add_elements_as_son(items, not_value=son_is_not)
                son_is_not = False
            # comma is OK if in fact it's just after an op/xof because it means this part  =====>,SRV_DESC   when you set a bp_rule
            # on a service
            elif tok_type == u'comma':
                if previous_tok_type in [u'op', u'xof', u'not'] or previous_tok_type == u'START_OF_PATTERN':
                    simulated_host_tok = BpRuleToken(u'node', start=-1, end=-1, val=u'')
                    stacked_part = [simulated_host_tok]
                    # void node should be services
                    if tokens:
                        next_tok = tokens[0]
                        if next_tok.type == u'node':
                            service_tok = tokens.pop(0)
                            stacked_part.append(service_tok)
                            my_sons = self.eval_simple_cor_pattern(stacked_part, proxyitems, ref)
                            node.add_elements_as_son(my_sons, not_value=son_is_not)
                son_is_not = False
            else:
                raise Exception(u'Error: unexpected token %s (%s) %s %s' % (tok_val, tok_type, previous_tok_type, tokens))
            
            # Save the token for next turn, as some need to know if they are ok or not based on what was before them
            previous_tok_type = tok_type
        
        if not node.operand:
            node.operand = u'&'
        # We got our nodes, so we can update 0 values of of_values
        # with the number of sons
        node.switch_zeros_of_values()
        
        return node
    
    
    # A stacked part is simply a set of nodes. It is defined as an expression of simple nodes, separated by [ ] ( ) && || or &
    def eval_stacked_part_item(self, tokens, proxyitems, ref=None):
        # type: (List[BpRuleToken], ProxyItems, Optional[List[unicode]]) -> Set(unicode)
        elements = set()
        op = None
        
        while tokens:
            tok = tokens.pop(0)
            tok_type = tok.type
            tok_val = tok.val  # Extracts match
            
            if tok_type == u'sep':
                if tok_val != u'[':
                    raise Exception(u'Error: unexpected separator %s in part expression' % tok_val)
                stacked_part = tok.get_sub_part(tokens)
                sub_elements = self.eval_stacked_part_item(stacked_part, proxyitems, ref)
                if not op and not elements:
                    elements = sub_elements
                elif op:
                    elements = op(sub_elements)
                    op = None
            
            elif tok_type == u'part_op':
                if not op:
                    if tok_val == u'<and>':
                        op = elements.intersection
                    elif tok_val == u'<or>':
                        op = elements.union
                    elif tok_val == u'<and not>':
                        op = elements.difference
                else:
                    raise Exception(u'Error: unexpected operator %s' % tok_val)
            
            elif tok_type == u'node':
                if not tok_val:
                    continue
                # A simple pattern is either a node or node comma node
                stacked_part = [tok]
                if tokens:
                    next_tok = tokens[0]
                    if next_tok.type == u'comma':
                        tokens.pop(0)
                        service_tok = tokens.pop(0)
                        if service_tok.type != u'node':
                            raise Exception(u'Error: expected check matching expression after \'%s,\'' % tok_val)
                        stacked_part.append(service_tok)
                sub_elements = self.eval_simple_cor_pattern(stacked_part, proxyitems, ref)
                if not op and not elements:
                    elements = sub_elements
                elif op:
                    elements = op(sub_elements)
                    op = None
            
            else:
                raise Exception(u'Error: unexpected token %s in part expression.' % tok_val)
        
        return elements
    
    
    # Two cases:
    # node -> host matching rule.
    # node node -> host,service rule.
    # Each rule being either a name, *, flags:name (or flags:regexp)
    # Returns a list of matching elements.
    @staticmethod
    def eval_simple_cor_pattern(tokens, proxyitems, ref=None):
        # type: (List[BpRuleToken], ProxyItems, List[unicode]) -> Set
        # Looks for hosts/services using appropriate filters
        try:
            if len(tokens) > 1:
                # We got a service expression
                host_expr, service_expr = tokens[0].get_node_expr(), tokens[1].get_node_expr()
                host_flags, service_flags = tokens[0].get_node_flags(), tokens[1].get_node_flags()
                items = proxyitems.find_services_by_exprs(host_expr, host_flags, service_expr, service_flags)
                if not items and not (host_flags and host_expr) and not (service_flags and service_expr):
                    if host_flags:
                        host_expr = u'%s:%s' % (host_flags, host_expr)
                    if service_flags:
                        service_expr = u'%s:%s' % (service_flags, service_expr)
                    
                    raise ValueError(u'The check described by [%s, %s] is disabled or does not exist' % (host_expr, service_expr))
            else:
                # We got an host expression
                host_expr = tokens[0].get_node_expr()
                host_flags = tokens[0].get_node_flags()
                items = proxyitems.find_hosts_by_expr(host_expr, host_flags)
        
        except re.error, e:
            raise Exception(u'Business rule uses invalid regex: %s' % e)
        # Should not contains itself
        if ref in items:
            items.remove(ref)
        return set(items)
