#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022:
# This file is part of Shinken Enterprise, all rights reserved.


from shinken.misc.type_hint import TYPE_CHECKING

if TYPE_CHECKING:
    from shinken.log import PartLogger
    from shinken.misc.type_hint import List, Any, Tuple


class AbstractDataHubDriverConfig(object):
    def __init__(self, name, namespace=u''):
        # type: (unicode, unicode) -> None
        self.name = name
        self.namespace = namespace
        
        # set in DataHubFactory.build_driver
        self.data_type = u''
        self.data_hub_id = u''


class AbstractDataHubDriver(object):
    
    def __init__(self, logger, driver_config):
        # type: (PartLogger, AbstractDataHubDriverConfig) -> None
        self._name = driver_config.name
        self._namespace = driver_config.namespace
        self._data_type = driver_config.data_type
        self._data_hub_id = driver_config.data_hub_id
        
        self._logger = logger.get_sub_part(self._name)
        self._logger_init = self._logger.get_sub_part(u'INIT')
        self._logger_reader = self._logger.get_sub_part(u'READ')
        self._logger_writer = self._logger.get_sub_part(u'WRITE')
        
        self.rights_access_correctly_granted = True
        self.rights_access_per_data_id = {}
    
    
    def init(self):
        # type: () -> None
        raise NotImplementedError()
    
    
    def get_number_of_stored_data(self):
        # type: () -> int
        raise NotImplementedError()
    
    
    def get_all_data_id(self):
        # type: () -> List[unicode]
        raise NotImplementedError()
    
    
    def is_data_correct(self, data_id):
        # type: (unicode) -> bool
        raise NotImplementedError()
    
    
    def get_lock(self, data_id):
        # type: (unicode) -> Any
        raise NotImplementedError()
    
    
    def write(self, data_id, data):
        # type: (unicode, Any) -> None
        raise NotImplementedError()
    
    
    def read(self, data_id, log_error=True):
        # type: (unicode, bool) -> Any
        raise NotImplementedError()
    
    
    def read_and_get_last_modification_date(self, data_id, log_error=True):
        # type: (unicode, bool) -> Tuple[Any, int]
        return self.read(data_id, log_error), self.get_last_modification_date(data_id)
    
    
    def remove(self, data_id):
        # type: (unicode) -> None
        raise NotImplementedError()
    
    
    def hard_remove(self, data_id):
        # type: (unicode) -> None
        self.remove(data_id)
    
    
    def get_last_modification_date(self, data_id):
        # type: (unicode) -> int
        raise NotImplementedError()
    
    
    def restore(self, data_id, data):
        # type: (unicode, Any) -> None
        self.write(data_id, data)
    
    
    def sanitize(self, data_id, data):
        # type: (unicode, Any) -> None
        self.write(data_id, data)
    
    
    def destroy(self):
        # type: () -> None
        raise NotImplementedError()
    
    
    def get_total_size(self):
        # type: () -> int
        raise NotImplementedError()
    
    
    def check_right_access_issues_then_fix_them(self):
        # type: () -> None
        pass
    
    
    def force_shinken_rights(self, data_id):
        # type: (unicode) -> None
        pass
