#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022:
# This file is part of Shinken Enterprise, all rights reserved.

import errno
import os
import sys
import threading

from ..portalocker import RLock, LOCK_EX

# Can not import shinkensolutions.system_tools because this module import logger

if sys.platform.startswith('win'):
    
    def set_ownership(path, user='shinken', group='shinken'):  # noqa
        pass

else:
    from pwd import getpwnam
    from grp import getgrnam
    
    
    def set_ownership(path, user='shinken', group='shinken'):
        # Only root can do a chown
        if os.getuid() != 0:
            return
        
        try:
            uid = getpwnam(user).pw_uid
        except KeyError:
            raise OSError(errno.EINVAL, 'This user doesn\'t exists', user)
        try:
            gid = getgrnam(group).gr_gid
        except KeyError:
            raise OSError(errno.EINVAL, 'This group doesn\'t exists', group)
        os.chown(path, uid, gid)


def create_tree(path, user='shinken', group='shinken', mode=0755):
    # Warning! the mode will be modified by os.umask if set
    drive, path = os.path.splitdrive(path)
    full_path = [p for p in path.split(os.sep) if p]
    parents = (u'%s%s' % (drive, os.sep)) if os.path.isabs(path) else ''
    for sub_dir in full_path:
        current_path = os.path.join(parents, sub_dir)
        
        if not os.path.exists(current_path):
            os.mkdir(current_path, mode)
            set_ownership(current_path, user, group)
        # Now, next level
        parents = current_path


class ShinkenInterProcessRLock(object):
    def __init__(self, filename):
        
        self._lock_file = filename
        self._locks = {}
        dirname = os.path.dirname(filename)
        create_tree(dirname)
        if not os.path.exists(filename):
            open(filename, u'w+').close()  # File must exist
            set_ownership(filename)
    
    
    def _get_lock(self):
        # type: () -> RLock
        pid = os.getpid()
        if pid not in self._locks:
            # after a fork, cleanup rogue locks
            for thread_locks in self._locks.itervalues():
                for other_lock in thread_locks.itervalues():
                    if other_lock and other_lock.fh and not other_lock.fh.closed:
                        other_lock.fh.close()
            self._locks = {pid: {}}
        thread_id = threading.current_thread().ident
        if thread_id not in self._locks[pid]:
            self._locks[pid][thread_id] = RLock(self._lock_file, timeout=None, mode='a+b', flags=LOCK_EX)
        
        return self._locks[pid][thread_id]
    
    
    def acquire(self):
        self._get_lock().acquire()
    
    
    def release(self):
        self._get_lock().release()
    
    
    def __enter__(self):
        self._get_lock().acquire()
    
    
    def __exit__(self, exc_type, exc_val, exc_tb):
        self._get_lock().release()
