#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022
# This file is part of Shinken Enterprise, all rights reserved.

import hashlib
import time
import uuid
from collections import namedtuple

import bson
import pymongo
from pymongo.errors import AutoReconnect, ConnectionFailure, PyMongoError

from shinken.log import logger
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.thread_helper import Thread
from ..data_hub.data_hub_exception.data_hub_exception import DataHubItemNotFound

if TYPE_CHECKING:
    from shinken.misc.type_hint import Dict, Any, Union, List, Optional
    from pymongo import MongoClient
    from shinkensolutions.data_hub.data_hub import DataHub
    from .sshtunnelmongomgr import ConnectionResult

DatabaseStatisticsItem = namedtuple(u'DatabaseStatisticsItem', [u'uri_hash', u'data'])


class DatabaseStatistics(Thread):
    def __init__(self, uri, connection_result, data_hub, loop_speed, requestor, get_statistics_interval=60):
        # type:(unicode, ConnectionResult, DataHub, int, unicode, int) -> None
        super(DatabaseStatistics, self).__init__(loop_speed=loop_speed)
        self._uuid = uuid.uuid4().hex  # type: unicode
        self.logger = self.logger.get_sub_part(u'MONGO').get_sub_part(u'THREAD STATISTICS').get_sub_part(self.get_uuid())
        self._uri = uri  # type: unicode
        self._uri_hash = hashlib.md5(self._uri.strip()).hexdigest()  # type: unicode
        self._connection_result = connection_result  # type: ConnectionResult
        self._data_hub = data_hub  # type: DataHub
        self._get_statistics_interval = get_statistics_interval
        self._last_database_statistics_time = 0
        self._is_connected = False
        self._connection_state_message = u'Database information never retrieved yet'
        self.logger.info(u'Creating Thread for %s' % requestor)
    
    
    def get_thread_name(self):
        # type:() -> unicode
        return u'db-stat'
    
    
    def get_uuid(self):
        # type:() -> unicode
        return self._uuid
    
    
    def get_uri(self):
        # type: () -> unicode
        return self._uri
    
    
    def get_uri_hash(self):
        # type: () -> unicode
        return self._uri_hash
    
    
    def get_connection_result(self):
        # type: () -> ConnectionResult
        return self._connection_result
    
    
    def _ping(self):
        mongo_connection = self._connection_result.get_connection()
        mongo_connection.admin.command(u'ping')  # use admin database to get mongo connection state
    
    
    def get_mongo_connection_state(self):
        return self._is_connected, self._connection_state_message
    
    
    def _get_statistics(self):
        # STEP 1 : Read old file and get last results
        databases_statistics, last_modification = self._get_database_statistics_and_last_modification_time()
        
        # If a thread is already managing this uri, but not updating it since 90 seconds, continue and replace this thread
        if databases_statistics and databases_statistics.data[u'file_manager_uuid'] != self.get_uuid() and (time.time() - last_modification) < 90:
            return
        
        # STEP 2 : Update statistics
        databases_statistics = self._update_databases_statistics(databases_statistics)
        
        # STEP 3 : Read old file and get last results
        self._write_databases_statistics_file(databases_statistics)
    
    
    def loop_turn(self):
        # type:() -> None
        try:
            if time.time() - self._last_database_statistics_time >= self._get_statistics_interval:
                self._get_statistics()
                self._last_database_statistics_time = time.time()
            self._ping()
            self._is_connected = True
            self._connection_state_message = u'Connect to MongoDB with address : [%s]' % self._uri
        except AutoReconnect as e:
            self._is_connected = False
            self._connection_state_message = u'Cannot connect to MongoDB with error : [%s]' % e.message
            self.logger.error(u'Cannot connect to MongoDB with error : [%s]. Request will be attempted again on the next loop.' % e.message)
        except ConnectionFailure as exp:
            self._is_connected = False
            self._connection_state_message = u'Cannot connect to mongodb server: %s   (%s)' % (self._uri, exp)
        except PyMongoError as exp:  # TODO : Remove all this auto reconnection handler. We must use the mongoclient. We developed an entire AutoReconnect system, why not use it?
            self._is_connected = False
            self._connection_state_message = u'Mongo error: %s    (%s)' % (self._uri, exp)
    
    
    def _get_database_statistics_and_last_modification_time(self):
        # type: () -> Optional[DatabaseStatisticsItem, float]
        try:
            database_statistics, last_modification = self._data_hub.get_data_and_last_modification(self.get_uri_hash())
            return DatabaseStatisticsItem(uri_hash=self.get_uri_hash(), data=database_statistics), last_modification
        except DataHubItemNotFound:
            return None, time.time()
        except Exception as e:
            raise e.message
    
    
    def _write_databases_statistics_file(self, database_statistics):
        # type:(DatabaseStatisticsItem) -> None
        self._data_hub.save_data(database_statistics.uri_hash, database_statistics.data)
    
    
    @staticmethod
    def _get_mongo_databases_name(mongo_connexion):
        # type:(MongoClient) -> List[unicode]
        return mongo_connexion.database_names()
    
    
    def _update_databases_statistics(self, previous_database_statistics):
        # type: (Optional[DatabaseStatisticsItem]) -> DatabaseStatisticsItem
        
        mongo_connexion = self._connection_result.get_connection()
        
        stats = {}
        
        databases_name = self._get_mongo_databases_name(mongo_connexion)
        if previous_database_statistics:
            pymongo_has_c = previous_database_statistics.data.get(u'pymongo_has_c', {})
            bson_has_c = previous_database_statistics.data.get(u'bson_has_c', {})
            stats = previous_database_statistics.data.get(u'stats', {})
        else:
            pymongo_has_c, bson_has_c = self._check_pymongo_bson_c_extension_installed()
        
        # We want to get dbstats every minute, to not overload the database
        statistics_update_time = time.time()
        self.logger.debug(u'Updating databases statistics [dbstats] for : [%s]' % u', '.join(databases_name))
        for database in databases_name:
            stats[database] = self._adapt_cluster_mongo_data_for_json_format(mongo_connexion[database].command(u'dbstats'))
        self.logger.info(u'Databases statistics updated in %s' % self.logger.format_chrono(statistics_update_time))
        
        if not stats:
            for database in databases_name:
                stats[database] = self._adapt_cluster_mongo_data_for_json_format(mongo_connexion[database].command(u'dbstats'))
        
        return DatabaseStatisticsItem(uri_hash=self.get_uri_hash(), data={
            u'file_manager_uuid': self.get_uuid(),
            u'databases_name'   : databases_name,
            u'address'          : self.get_uri(),
            u'pymongo_has_c'    : pymongo_has_c,
            u'bson_has_c'       : bson_has_c,
            u'stats'            : stats
        })
    
    
    @staticmethod
    def _check_pymongo_bson_c_extension_installed():
        # type: () -> (Dict[unicode, Union[bool, unicode]], Dict[unicode, Union[bool, unicode]])
        logger.debug(u'check_pymongo_c_extension_installed::start')
        pymongo_has_c = {u'installed': True, u'status': u'OK', u'message': u'Your pymongo has C extension installed'}
        bson_has_c = {u'installed': True, u'status': u'OK', u'message': u'Your bson lib has C extension installed'}
        if not pymongo.has_c():
            pymongo_has_c = {u'installed': False, u'status': u'ERROR', u'message': u'Your pymongo lib has not the C extension installed'}
        if not bson.has_c():
            bson_has_c = {u'installed': False, u'status': u'ERROR', u'message': u'Your bson lib has not the C extension installed'}
        
        return pymongo_has_c, bson_has_c
    
    
    @staticmethod
    def _adapt_cluster_mongo_data_for_json_format(stats):
        # type: (Dict) -> Dict[unicode, Any]
        election_id = stats.get(u'$gleStats', {}).get(u'electionId', None)
        last_op_time = stats.get(u'$gleStats', {}).get(u'lastOpTime', None)
        if election_id:
            stats[u'$gleStats'][u'electionId'] = str(election_id)
        if last_op_time:
            stats[u'$gleStats'][u'lastOpTime'] = last_op_time.time
        
        for cluster in stats.get(u'raw', {}):
            election_id = stats[u'raw'][cluster].get(u'$gleStats', {}).get(u'electionId', None)
            last_op_time = stats[u'raw'][cluster].get(u'$gleStats', {}).get(u'lastOpTime', None)
            if election_id:
                stats[u'raw'][cluster][u'$gleStats'][u'electionId'] = str(election_id)
            if last_op_time:
                stats[u'raw'][cluster][u'$gleStats'][u'lastOpTime'] = last_op_time.time
        return stats
