import socket
import time
import httplib
from urllib import urlencode
from django.core.cache import cache
from django.conf import settings
from graphite.render.hashing import compactHash
from graphite.logger import log

try:
  import cPickle as pickle
except ImportError:
  import pickle

import json
import urllib2

class RemoteStore(object):
  lastFailure = 0.0
  retryDelay = settings.REMOTE_STORE_RETRY_DELAY
  available = property(lambda self: time.time() - self.lastFailure > self.retryDelay)

  def __init__(self, host):
    self.host = host
    self._addr = host.split(':',1)[0]


  def find(self, query, search_by_name=True, metric_listing=False):
    request = FindRequest(self, query, search_by_name=search_by_name, metric_listing=metric_listing)
    request.send()
    return request


  def fail(self):
    self.lastFailure = time.time()


  def relay_migration_table(self, migration_table):
      json_data = json.dumps(migration_table)
      graphite_migration_url = "http://%s/migrate/?do_relay=0" % (self._addr)  # we are the last recurvise one
      log.info('[ NEW CONFIGURATION / MIGRATION ] [ RELAY ] : asking for new configuration/migration to relay node %s' % self._addr)
      try:
          result = urllib2.urlopen(graphite_migration_url, data=json_data)
      except (urllib2.HTTPError, urllib2.URLError) as error:
          err = '[ NEW CONFIGURATION / MIGRATION ] [ RELAY ] Error during the new configuration / uuid migration of the relay %s: %s\n' % (self._addr, error)
          log.info(err)
          return {
              'error_count': 1,
              'ok_count'   : 0,
              'nb_migrated': 0,
              'last_error' : err,
              'server_ok'  : [],
          }
      try:
          retval = json.load(result)
      except ValueError:  # not a json? graphite it not up to date!
          err = '[ NEW CONFIGURATION / MIGRATION ] [ RELAY ] Error during the new configuration / uuid migration of the relay %s: the server is not up to date\n' % (self._addr)
          log.info(err)
          return {
              'error_count': 1,
              'ok_count'   : 0,
              'nb_migrated': 0,
              'last_error' : err,
              'server_ok'  : [],
          }
      if result.code != 200:
          err = '[ NEW CONFIGURATION / MIGRATION ] [ RELAY ] Error during the new configuration / uuid migration of the relay %s\n' % self._addr
          log.info(err)
          return {
              'error_count': 1,
              'ok_count'   : 0,
              'nb_migrated': 0,
              'last_error' : err,
              'server_ok'  : [],
          }
    
      return retval



class FindRequest:
  suppressErrors = True

  def __init__(self, store, query, search_by_name=True, metric_listing=False):
    self.store = store
    self.query = query
    self.connection = None
    self.cacheKey = compactHash('find:%s:%s' % (self.store.host, query))
    self.cachedResults = None
    self.search_by_name = search_by_name
    self.metric_listing = metric_listing


  def send(self):
    self.cachedResults = cache.get(self.cacheKey)

    if self.cachedResults:
      return

    self.connection = HTTPConnectionWithTimeout(self.store.host)
    self.connection.timeout = settings.REMOTE_STORE_FIND_TIMEOUT

    query_params = [
      ('local', '1'),
      ('format', 'pickle'),
      ('query', self.query),
      ('search_by_name', '1' if self.search_by_name else '0'),
    ]
    query_string = urlencode(query_params)

    try:
      self.connection.request('GET', '/metrics/find/?' + query_string)
    except:
      self.store.fail()
      if not self.suppressErrors:
        raise


  def get_results(self):
    if self.cachedResults:
      return self.cachedResults

    if not self.connection:
      self.send()

    try:
      response = self.connection.getresponse()
      result_data = response.read()
      assert response.status == 200, "received error response %s - %s %s" % (response.status, response.reason, result_data)
      results = pickle.loads(result_data)

    except:
      self.store.fail()
      if not self.suppressErrors:
        raise
      else:
        results = []

    resultNodes = [ RemoteNode(self.store, node['metric_path'], node['isLeaf']) for node in results ]
    cache.set(self.cacheKey, resultNodes, settings.REMOTE_FIND_CACHE_DURATION)
    self.cachedResults = resultNodes
    return resultNodes



class RemoteNode:
  context = {}

  def __init__(self, store, metric_path, isLeaf):
    self.store = store
    self.fs_path = None
    self.metric_path = metric_path
    self.real_metric = metric_path
    self.name = metric_path.split('.')[-1]
    self.__isLeaf = isLeaf


  def fetch(self, startTime, endTime, search_by_name=True):
    if not self.__isLeaf:
      return []

    query_params = [
      ('target', self.metric_path),
      ('format', 'pickle'),
      ('from', str( int(startTime) )),
      ('until', str( int(endTime) )),
      ('search_by_name', '1' if search_by_name else '0'),
    ]
    query_string = urlencode(query_params)

    connection = HTTPConnectionWithTimeout(self.store.host)
    connection.timeout = settings.REMOTE_STORE_FETCH_TIMEOUT
    connection.request('GET', '/render/?' + query_string)
    response = connection.getresponse()
    rawData = response.read()
    assert response.status == 200, "Failed to retrieve remote data: %d %s %s" % (response.status, response.reason, rawData)

    seriesList = pickle.loads(rawData)
    assert len(seriesList) == 1, "Invalid result: seriesList=%s" % str(seriesList)
    series = seriesList[0]

    timeInfo = (series['start'], series['end'], series['step'])
    return (timeInfo, series['values'])


  def isLeaf(self):
    return self.__isLeaf



# This is a hack to put a timeout in the connect() of an HTTP request.
# Python 2.6 supports this already, but many Graphite installations
# are not on 2.6 yet.

class HTTPConnectionWithTimeout(httplib.HTTPConnection):
  timeout = 30

  def connect(self):
    msg = "getaddrinfo returns an empty list"
    for res in socket.getaddrinfo(self.host, self.port, 0, socket.SOCK_STREAM):
      af, socktype, proto, canonname, sa = res
      try:
        self.sock = socket.socket(af, socktype, proto)
        try:
          self.sock.settimeout( float(self.timeout) ) # default self.timeout is an object() in 2.6
        except:
          pass
        self.sock.connect(sa)
        self.sock.settimeout(None)
      except socket.error, msg:
        if self.sock:
          self.sock.close()
          self.sock = None
          continue
      break
    if not self.sock:
      raise socket.error, msg
