#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2018:
# This file is part of Shinken Enterprise, all rights reserved.


import datetime
import imp
import json
import os
import sys
import threading
import time
import traceback

import shinken.webui.bottlewebui as bottle
from shinken.daemon import Daemon
from shinken.load import AvgForFixSizeCall
from shinken.log import logger
from shinken.message import Message
from shinken.misc.monitoring_item_manager.monitoring_item_manager import datamgr
from shinken.misc.monitoring_item_manager.regenerator import Regenerator
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.modules.base_module.broker_base_module import BrokerBaseModule
from shinken.modules.base_module.sub_module_handler import SubModuleHandler
from shinken.runtime_stats.cpu_stats import cpu_stats_helper
from shinken.webui.bottlewebui import view, route, request, response, template, HTTPError
from shinken.webui.cherrypybackend import CherryPyServerHTTP
from shinkensolutions.lib_modules.configuration_reader_mixin import ConfigurationReaderMixin, SeparatorFormat, ConfigurationFormat, TypeConfiguration

if TYPE_CHECKING:
    from shinken.misc.type_hint import Callable, Dict, Optional
    from shinken.objects.module import Module as ShinkenModuleDefinition
    from shinken.modules.base_sub_module.base_sub_module_livedata import BaseSubModuleLivedata
    from threading import Thread

ONE_HOUR = 3600

bottle.debug(True)

# Import bottle lib to make bottle happy
bottle_dir = os.path.abspath(os.path.dirname(bottle.__file__))
sys.path.insert(0, bottle_dir)

livedata_module_folder = os.path.abspath(os.path.dirname(__file__))
bottle.TEMPLATE_PATH.append(livedata_module_folder)


# called by the plugin manager to get an instance
def get_instance(module_configuration):
    logger.debug(u'Get a BrokerBaseModuleLiveData instance for plugin %s' % module_configuration.get_name())
    
    instance = BrokerModuleLivedata(module_configuration)
    return instance


class EnableCors(object):
    name = 'enable_cors'
    api = 2
    
    
    @staticmethod
    def apply(fn, context):
        def _enable_cors(*args, **kwargs):
            # set CORS headers
            response.headers['Access-Control-Allow-Origin'] = 'http://127.0.0.1:9000'
            response.headers['Access-Control-Allow-Methods'] = 'GET, POST, PUT, OPTIONS, DELETE, PATCH'
            response.headers['Access-Control-Allow-Headers'] = 'Origin, Accept, Content-Type, X-Requested-With, X-CSRF-Token, X-Shinken-Token'
            response.headers['Access-Control-Allow-Credentials'] = 'true'
            if bottle.request.method != 'OPTIONS':
                # actual request; reply with the actual response
                return fn(*args, **kwargs)
        
        
        return _enable_cors


class BrokerModuleLivedata(BrokerBaseModule, Daemon, SubModuleHandler):
    def __init__(self, module_configuration):
        # type: (ShinkenModuleDefinition) -> None
        BrokerBaseModule.__init__(self, module_configuration)
        SubModuleHandler.__init__(self, module_configuration)
        self.http_start_time = int(time.time())
        
        self.plugins = []
        
        self.server_opts = {}
        umask = getattr(module_configuration, 'umask', None)
        if umask is not None:
            self.server_opts['umask'] = int(umask)
        bind_address = getattr(module_configuration, 'bindAddress', None)
        if bind_address:
            self.server_opts['bindAddress'] = str(bind_address)
        self.server_opts['numthreads'] = 64
        
        self.lang = None
        self.host = None
        self.port = None
        self.use_ssl = None
        self.ssl_key = None
        self.ssl_cert = None
        self.token = None
        
        logger_init = self.logger.get_sub_part(u'INITIALISATION')
        configuration_format = [
            SeparatorFormat(u'Module Identity'),
            ConfigurationFormat(u'module_name', u'', TypeConfiguration.STRING, u''),
            
            SeparatorFormat(u'General'),
            ConfigurationFormat(u'broker__module_livedata__lang', u'en', TypeConfiguration.STRING, u'lang'),
            
            SeparatorFormat(u'Listening parameters'),
            ConfigurationFormat(u'broker__module_livedata__listening_address', u'0.0.0.0', TypeConfiguration.STRING, u'host'),
            ConfigurationFormat(u'broker__module_livedata__listening_port', 50100, TypeConfiguration.INT, u'port'),
            
            SeparatorFormat(u'HTTPS parameters'),
            ConfigurationFormat(u'broker__module_livedata__use_ssl', False, TypeConfiguration.BOOL, u'use_ssl'),
            ConfigurationFormat(u'broker__module_livedata__ssl_key', u'/etc/shinken/certs/server.key', TypeConfiguration.STRING, u'ssl_key'),
            ConfigurationFormat(u'broker__module_livedata__ssl_cert', u'/etc/shinken/certs/server.cert', TypeConfiguration.STRING, u'ssl_cert'),
            
            SeparatorFormat(u'API'),
            ConfigurationFormat(u'broker__module_livedata__token', u'change_me', TypeConfiguration.HIDDEN_STRING, u'token'),
            
            SeparatorFormat(u'Modules'),
            ConfigurationFormat(u'modules', [], TypeConfiguration.MODULES_NAMES, u''),
        ]
        ConfigurationReaderMixin.__init__(self, configuration_format, module_configuration, logger_init)
        self.read_configuration()
        self.log_configuration(log_properties=True, show_values_as_in_conf_file=True)
        
        self.average_time_response = AvgForFixSizeCall(time_limit=ONE_HOUR)
        self.nb_errors_by_hour = []
        self.errors = []
        self.errors_per_ip = {}
        self.worst_response_time = {'time': 0, 'response_time': 0.0}
        self.rg = Regenerator(module_configuration)
        self.monitoring_item_manager = None
        self.request = None
        self.abort = None
        self.response = None
        self.template_call = None
        self.srv = None
        
        self.bottle = bottle
        self.data_thread = None  # type: Optional[Thread]
        self.hook_error()
    
    
    def loop_turn(self):
        pass
    
    
    def get_module_info(self):
        module_info = {}
        tmp = BrokerBaseModule.get_submodule_states(self)
        if tmp:
            module_info.update(tmp)
        module_info['https'] = self.use_ssl
        if self.token == "change_me":
            module_info['token'] = 1
        elif self.token == "":
            module_info['token'] = 2
        else:
            module_info['token'] = 0
        
        module_info['port'] = self.port
        time_now = time.time()
        module_info['average_response_time'], request_per_hour = self.average_time_response.get_avg(avg_on_time=False, with_range_size=True)
        if time_now - self.worst_response_time['time'] <= ONE_HOUR and self.worst_response_time['time'] != 0:
            module_info['worst_response_time'] = self.worst_response_time
        module_info['request_per_hour'] = 0
        
        tmp = self.average_time_response.max_call_in_period()
        
        if tmp['nb'] == 0:
            module_info['best_in_one_second'] = tmp
        else:
            module_info['best_in_one_second'] = {'time': datetime.datetime.fromtimestamp(tmp['time']).strftime('%H:%M:%S'), 'nb': tmp['nb']}
        
        module_info['error_per_hour'] = 0
        if len(self.nb_errors_by_hour) >= 1:
            while self.nb_errors_by_hour and time_now - self.nb_errors_by_hour[0] >= ONE_HOUR:
                self.nb_errors_by_hour.pop(0)
            if len(self.nb_errors_by_hour) >= 1:
                module_info['error_per_hour'] = len(self.nb_errors_by_hour)
        
        for error in self.errors:
            if time_now - error['time'] >= ONE_HOUR:
                del error
            else:
                break
        
        module_info['errors'] = self.errors
        module_info['request_per_hour'] = int(request_per_hour) + len(self.errors)
        return module_info
    
    
    def do_loop_turn(self):
        raise NotImplementedError()
    
    
    @staticmethod
    def error_handler(error):
        return_value = error.output
        if error.traceback:
            first_line = True
            for line_stack in error.traceback.splitlines():
                if first_line:
                    logger.error("ERROR stack : %s" % line_stack)
                    first_line = False
                else:
                    logger.error("%s" % line_stack)
        if response and response.content_type == 'application/json':
            return_value = json.dumps(error.output)
        return return_value
    
    
    def hook_error(self):
        
        @bottle.error(500)
        def custom500(error):
            return self.error_handler(error)
        
        
        @bottle.error(400)
        def custom400(error):
            return self.error_handler(error)
        
        
        @bottle.error(404)
        def custom404(error):
            return self.error_handler(error)
        
        
        @bottle.error(401)
        def custom401(error):
            return self.error_handler(error)
        
        
        @bottle.error(403)
        def custom403(error):
            return self.error_handler(error)
        
        
        @bottle.error(409)
        def custom409(error):
            return self.error_handler(error)
    
    
    def init(self):
        self.rg.load_external_queue(self.from_module_to_main_daemon_queue)
    
    
    def want_brok(self, b):
        return self.rg.want_brok(b)
    
    
    def main(self):
        logger.set_name(self.name)
        
        self.datamgr = datamgr
        datamgr.load(self.rg)
        self.monitoring_item_manager = self.datamgr
        
        self.request = request
        self.abort = self._abort
        self.response = response
        self.template_call = template
        
        # the regenerator is ready in the valid process, we can start it's freshness thread
        self.rg.launch_freshness_thread(lang=self.lang)
        try:
            self.do_main()
        except Exception, exp:
            msg = Message(id=0, type='ICrash', data={'name': self.get_name(), 'exception': exp, 'trace': traceback.format_exc()})
            self.from_module_to_main_daemon_queue.put(msg)
            # wait 2 sec so we know that the broker got our message, and die
            time.sleep(2)
            raise
    
    
    @staticmethod
    def _abort(code=500, text='Unknown Error: Application stopped.', warning=False, to_logger=True):
        if to_logger is True:
            logger.error(u'A HTTP error code %s occurred : %s' % (code, text))
        http_error = HTTPError(code, text)
        http_error.warning = warning
        raise http_error
    
    
    def wrap_json(self, f):
        def __wrap(*args, **kwargs):
            self.response.content_type = 'application/json'
            return f(*args, **kwargs)
        
        
        return __wrap
    
    
    # Real main function
    def do_main(self):
        
        # I register my exit function
        self.set_exit_handler()
        
        # We will protect the operations on the non read+write with a lock and 2 int
        self.lock_init(broks_eater=[self.rg.manage_brok], consumer_name='API Requests')
        
        self.data_thread = None
        
        core_plugin_dir = os.path.join(os.path.abspath(os.path.dirname(__file__)), 'plugins')
        
        # Step - Then look at the plugins in toe core and load all we can there
        self.load_plugins(core_plugin_dir)
        
        self.init_sub_modules()
        
        for routes in self.get_sub_module_routes():
            self._load_routes(routes)
        
        # Step - Launch the data thread
        self.data_thread = threading.Thread(None, self.manage_brok_thread, 'datathread')
        self.data_thread.start()
        
        self.rg.memory_management_init()
        # TODO: look for alive and killing
        logger.debug("starting web server")
        self.srv = bottle.run(host=self.host, port=self.port, server=CherryPyServerHTTP, use_ssl=self.use_ssl, ssl_key=self.ssl_key, ssl_cert=self.ssl_cert, **self.server_opts)
        self.srv.start()
    
    
    def update_sub_module(self, sub_module):
        # type: (BaseSubModuleLivedata) -> None
        sub_module.set_token(self.token)
        sub_module.set_monitoring_item_manager(self.monitoring_item_manager)
        sub_module.set_lock(self.consumer_lock)
        sub_module.set_bottle_properties(self.request, self.response, self.abort)
    
    
    def load_plugin(self, fdir, plugin_dir):
        logger.debug(u'Loading %s from %s' % (fdir, plugin_dir))
        try:
            # Put the full qualified path of the module we want to load
            # for example we will give  webui/plugins/eltdetail/
            
            mod_path = os.path.join(plugin_dir, fdir)
            # Then we load the eltdetail.py inside this directory
            m = imp.load_module(u'%s' % fdir, *imp.find_module(fdir, [mod_path]))
            m_dir = os.path.abspath(os.path.dirname(m.__file__))
            sys.path.append(m_dir)
            
            logger.debug(u'module %s is loaded from %s' % (m_dir, str(m)))
            self._load_routes(m.pages)
            
            # And finally register me so the pages can get data and other useful stuff
            m.app = self
        
        except Exception, exp:
            logger.warning(u'Loading plugins: %s' % exp)
            logger.print_stack()
    
    
    def _load_routes(self, route_definitions):
        # type: (Dict) -> None
        for (controller, route_definition) in route_definitions.items():
            routes = route_definition.get(u'routes', None)
            _view = route_definition.get(u'view', None)
            wrappers = route_definition.get(u'wrappers', [u'auth'])
            
            controller_name = controller.__name__
            controller.display_name = controller_name
            for wrapper in wrappers:
                wrap = {u'json': self.wrap_json}.get(wrapper, None)  # type: Callable
                if wrap:
                    controller_name = controller.display_name
                    controller = wrap(controller)
                    controller.display_name = u'%s-w(%s)' % (controller_name, wrapper)
            
            # IMPORTANT: apply VIEW BEFORE route!
            if _view:
                controller_name = controller.display_name
                controller = view(_view)(controller)
                controller.display_name = u'%s-v(%s)' % (controller_name, _view)
            
            # Maybe there is no route to link, so pass
            if routes:
                for _route in routes:
                    method = route_definition.get(u'method', u'GET')
                    logger.debug(u'linking function [%s] and route [%s] for method [%s]' % (getattr(controller, u'display_name', str(controller)), _route, method))
                    
                    # Ok, we will just use the lock for all plugin page, but not for static objects so we set the lock at the function level.
                    controller_name = controller.display_name
                    lock_version = self.lockable_function(controller)
                    lock_version.display_name = controller_name
                    controller = route(_route, callback=lock_version, method=[method, 'OPTIONS'])
    
    
    # Here we will load all plugins (pages) under the webui/plugins directory.
    # Each one can have a page, views and htdocs dir that we must route correctly
    def load_plugins(self, plugin_dir):
        # Load plugin directories
        if not os.path.exists(plugin_dir):
            return
        
        sys.path.append(plugin_dir)
        
        plugin_folders_name = [plugin_folder_name for plugin_folder_name in os.listdir(plugin_dir) if os.path.isdir(os.path.join(plugin_dir, plugin_folder_name))]
        
        # We try to import them, but we keep only the one of our type
        for plugin_folder_name in plugin_folders_name:
            self.load_plugin(plugin_folder_name, plugin_dir)
    
    
    # We want a lock manager version of the plugin functions
    def lockable_function(self, f):
        def lock_version(**args):
            snap = cpu_stats_helper.get_thread_cpu_snapshot()
            start = time.time()
            _function = lock_version.f
            
            # set CORS headers
            response.headers['Access-Control-Allow-Origin'] = 'http://127.0.0.1:9000'
            response.headers['Access-Control-Allow-Methods'] = 'GET, POST, PUT, OPTIONS, DELETE, PATCH'
            response.headers['Access-Control-Allow-Headers'] = 'Origin, Accept, Content-Type, X-Requested-With, X-CSRF-Token, X-Shinken-Token'
            response.headers['Access-Control-Allow-Credentials'] = 'true'
            
            try:
                with self.consumer_lock:
                    t = time.time()
                    if bottle.request.method != 'OPTIONS':
                        # And disable cache on all backend calls
                        self._set_no_cache_headers()
                        return _function(**args)
                    return
            finally:
                logger.log_perf(t, 'perf', 'function %s executed' % getattr(_function, 'display_name', str(_function)))
                if abs(time.time() - start) > 0.05:  # more than 50ms, we log debug the perfs
                    logger.debug(u'[ PERFS ] [ %s ] %s' % (str(f), snap.get_diff()))
        
        lock_version.f = f
        return lock_version
    
    
    # For this request, ask for no cache at all. for example static that have always the
    # same address across versions
    @staticmethod
    def _set_no_cache_headers():
        # The Cache-Control is per the HTTP 1.1 spec for clients and proxies
        # (and implicitly required by some clients next to Expires).
        # The Pragma is per the HTTP 1.0 spec for prehistoric clients.
        # The Expires is per the HTTP 1.0 and 1.1 spec for clients and proxies.
        # In HTTP 1.1, the Cache-Control takes precedence over Expires, so it's after all for HTTP 1.0 proxies only.
        response.headers['Cache-Control'] = 'no-cache,no-store,must-revalidate'
        response.headers['Pragma'] = 'no-cache'
        response.headers['Expires'] = '0'
    
    
    @staticmethod
    def is_check_uuid(uuid):
        return '-' in uuid
