#!/usr/bin/python

# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.


import copy
import random
import re
import sys
import time

from shinken.log import logger
from shinken.macroresolver import MacroResolver
from shinken.modulesmanager import ModulesManager
from shinken.objects.config import Config
from shinken.util import make_unicode

# Always initialize random...

random.seed(time.time())
try:
    import uuid
except ImportError:
    uuid = None


def get_uuid(self):
    if uuid:
        return uuid.uuid1().hex
    # Ok for old python like 2.4, we will lie here :)
    return int(random.random() * sys.maxint)


# Look if the name is a IPV4 address or not
def is_ipv4_addr(name):
    p = r"^([01]?\d\d?|2[0-4]\d|25[0-5])\.([01]?\d\d?|2[0-4]\d|25[0-5])\.([01]?\d\d?|2[0-4]\d|25[0-5])\.([01]?\d\d?|2[0-4]\d|25[0-5])$"
    return (re.match(p, name) is not None)


def by_order(r1, r2):
    if r1.discoveryrule_order == r2.discoveryrule_order:
        return 0
    if r1.discoveryrule_order > r2.discoveryrule_order:
        return 1
    if r1.discoveryrule_order < r2.discoveryrule_order:
        return -1


class DiscoveredHost(object):
    my_type = 'host'  # we fake our type for the macro resolving
    
    macros = {'HOSTNAME': 'name'}
    
    
    def __init__(self, name, runners, rulemanager, trad, merge=False, first_level_only=False):
        self.name = name
        self.data = {}
        self.runners = runners
        self.merge = merge
        self.trad = trad
        self.rulemanager = rulemanager
        
        self.matched_rules = []
        self.launched_runners = []
        
        self.in_progress_runners = []
        self.properties = {}
        self.customs = {}
        self.first_level_only = first_level_only
    
    
    # In final phase, we keep only _ properties and
    # rule based one
    def update_properties(self, final_phase=False):
        d = {}
        if final_phase:
            for (k, v) in self.data.iteritems():
                if k.startswith('_'):
                    d[k] = v
        else:
            d = copy.copy(self.data)
        
        d['host_name'] = self.name
        # Set address directive if an ip exists
        if self.data.has_key('ip'):
            d['address'] = self.data['ip']
        
        # Set nmap data
        nmap = {}
        for data_name in ['fqdn', 'mac', 'macvendor', 'openports', 'os', 'ostype', 'osvendor', 'osversion']:
            nmap[data_name] = self.data.get(data_name, '')
            if data_name == 'openports':
                if nmap[data_name]:
                    nmap[data_name] = map(int, nmap[data_name].split(','))
                    nmap[data_name].sort()
                    nmap[data_name] = map(str, nmap[data_name])
                    nmap[data_name] = ','.join(nmap[data_name])
        
        if nmap['mac']:
            d['_MAC_ADDRESS'] = nmap['mac']
        d['@metadata'] = {'pre_import_table_data': nmap, 'pre_import_table_name': self.trad('discovery.nmap_table'), 'more_informations_on_element': {}}
        
        for rule in self.matched_rules:
            rule.apply_on_host(d)
        
        # Now we need to add all rules on "more informations" part of the host
        rules_applied_on_by = d.get('rules_applied_on_by', {})
        for key in rules_applied_on_by:
            if len(rules_applied_on_by[key]) == 1:
                d['@metadata']['more_informations_on_element'][key] = "%s %s" % (self.trad('discovery.modified_by_unique_rule'), rules_applied_on_by[key][0])
            else:
                d['@metadata']['more_informations_on_element'][key] = "%s %s" % (self.trad('discovery.modified_by_rules'), ', '.join(rules_applied_on_by[key]))
        if rules_applied_on_by:
            del d['rules_applied_on_by']
        # Change join prop list in string with a ',' separator
        prefix_name = d.get('prefix_name', '')
        if prefix_name:
            d['host_name'] = "%s-%s" % (prefix_name, d['host_name'])
            del d['prefix_name']
        for (k, v) in d.iteritems():
            if type(d[k]).__name__ == 'list':
                d[k] = ','.join(d[k])
        
        self.properties = d
        # print 'Update our properties', self.name, d
        
        # For macro-resolving, we should have our macros too
        self.customs = {}
        for (k, v) in self.properties.iteritems():
            self.customs['_' + k.upper()] = v
    
    
    # Manager ask us our properties for the configuration, so
    # we keep only rules properties and _ ones
    def get_final_properties(self):
        self.update_properties(final_phase=True)
        return self.properties
    
    
    def get_to_run(self):
        self.in_progress_runners = []
        
        if self.first_level_only:
            return
        
        for r in self.runners:
            # If we already launched it, we don't want it :)
            if r in self.launched_runners:
                print 'Sorry', r.get_name(), 'was already launched'
                continue
            # First level discovery are for large scan, so not for here
            if r.is_first_level():
                print 'Sorry', r.get_name(), 'is first level'
                continue
            # And of course it must match our data
            # print 'Is ', r.get_name(), 'matching??', r.is_matching_disco_datas(self.properties)
            if r.is_matching_disco_datas(self.properties):
                self.in_progress_runners.append(r)
    
    
    def need_to_run(self):
        return len(self.in_progress_runners) != 0
    
    
    # Now we try to match all our hosts with the rules
    def match_rules(self):
        self.matched_rules = self.rulemanager.match_rules(self)
        self.update_properties()
    
    
    def read_disco_buf(self, buf):
        # print 'Read buf in', self.name
        for l in buf.split('\n'):
            # print ""
            # If it's not a disco line, bypass it
            if not re.search('::', l):
                continue
            # print "line", l
            elts = l.split('::', 1)
            if len(elts) <= 1:
                # print "Bad discovery data"
                continue
            name = elts[0].strip()
            
            # We can choose to keep only the basename
            # of the nameid, so strip the fqdn
            # But not if it's a plain ipv4 addr
            # TODO : gt this! if self.conf.strip_idname_fqdn:
            if not is_ipv4_addr(name):
                name = name.split('.', 1)[0]
            
            data = '::'.join(elts[1:])
            
            # Maybe it's not me?
            if name != self.name:
                if not self.merge:
                    print 'Bad data for me? I bail out data!'
                    data = ''
                else:
                    print 'Bad data for me? Let\'s switch !'
                    self.name = name
            
            # Now get key,values
            if not '=' in data:
                continue
            
            elts = data.split('=', 1)
            if len(elts) <= 1:
                continue
            
            key = elts[0].strip()
            value = elts[1].strip()
            # print "INNER -->", name, key, value
            self.data[key] = value
    
    
    def launch_runners(self):
        for r in self.in_progress_runners:
            print "I", self.name, " is launching", r.get_name(), "with a %d seconds timeout" % 3600
            r.launch(timeout=3600, ctx=[self])
            self.launched_runners.append(r)
    
    
    def wait_for_runners_ends(self):
        all_ok = False
        while not all_ok:
            print 'Loop wait runner for', self.name
            all_ok = True
            for r in self.in_progress_runners:
                if not r.is_finished():
                    # print "Check finished of", r.get_name()
                    r.check_finished()
                b = r.is_finished()
                if not b:
                    # print r.get_name(), "is not finished"
                    all_ok = False
            time.sleep(0.1)
    
    
    def get_runners_outputs(self):
        for r in self.in_progress_runners:
            if r.is_finished():
                # print'Get output', self.name, r.discoveryrun_name, r.current_launch
                if r.current_launch.exit_status != 0:
                    print "Error on run"
        raw_disco_data = '\n'.join(r.get_output() for r in self.in_progress_runners if r.is_finished())
        if len(raw_disco_data) != 0:
            print "Got Raw disco data", raw_disco_data
        else:
            print "Got no data!"
            for r in self.in_progress_runners:
                print "DBG", r.current_launch
        # Now get the data for me :)
        self.read_disco_buf(raw_disco_data)


class DiscoveryManager:
    def __init__(self, path, macros, overwrite, runners, trad, rulemanager, output_dir=None,
                 only_new_hosts=False, backend=None, modules_path='', merge=False, conf=None,
                 first_level_only=False):
        # i am arbiter-like
        self.log = logger
        self.overwrite = overwrite
        self.runners = runners
        self.output_dir = output_dir
        self.only_new_hosts = only_new_hosts
        self.merge = merge
        self.config_files = [path]
        # For specific backend, to override the classic file/db behavior
        self.backend = backend
        self.modules_path = modules_path
        self.first_level_only = first_level_only
        self.trad = trad
        self.rulemanager = rulemanager
        
        if not conf:
            self.conf = Config()
            
            buf = self.conf.read_config(self.config_files)
            
            # Add macros on the end of the buf so they will
            # overwrite the resource.cfg ones
            for (m, v) in macros:
                buf += '\n$%s$=%s\n' % (m, v)
            
            raw_objects = self.conf.read_config_buf(buf)
            self.conf.create_objects_for_type(raw_objects, 'arbiter')
            self.conf.create_objects_for_type(raw_objects, 'module')
            self.conf.early_arbiter_linking()
            self.conf.create_objects(raw_objects)
            self.conf.linkify_templates()
            self.conf.apply_inheritance()
            self.conf.explode()
            self.conf.create_reversed_list()
            self.conf.remove_twins()
            self.conf.apply_implicit_inheritance()
            self.conf.fill_default()
            self.conf.remove_templates()
            self.conf.pythonize()
            self.conf.linkify()
            self.conf.apply_dependencies()
            self.conf.is_correct()
        else:
            self.conf = conf
        
        self.discoveryruns = self.conf.discoveryruns
        
        m = MacroResolver()
        m.init(self.conf)
        
        # Hash = name, and in it (key, value)
        self.disco_data = {}
        # Hash = name, and in it rules that apply
        self.disco_matches = {}
        
        self.init_backend()
    
    
    def add(self, obj):
        pass
    
    
    # We try to init the backend if we got one
    def init_backend(self):
        if not self.backend or not isinstance(self.backend, basestring):
            return
        
        print "Doing backend init"
        for mod in self.conf.modules:
            if getattr(mod, 'module_name', '') == self.backend:
                print "We found our backend", mod.get_name()
                self.backend = mod
        if not self.backend:
            print "ERROR : cannot find the module %s" % self.backend
            sys.exit(2)
        self.modules_manager = ModulesManager('discovery', self.modules_path, [])
        self.modules_manager.set_modules([mod])
        self.modules_manager.load_and_init()
        self.backend = self.modules_manager.get_all_instances()[0]
        print "We got our backend!", self.backend
    
    
    def loop_discovery(self):
        still_loop = True
        i = 0
        while still_loop:
            i += 1
            # print '\n'
            # print 'LOOP'*10, i
            still_loop = False
            for (name, dh) in self.disco_data.iteritems():
                dh.update_properties()
                to_run = dh.get_to_run()
                # print 'Still to run for', name, to_run
                if dh.need_to_run():
                    still_loop = True
                    dh.launch_runners()
                    dh.wait_for_runners_ends()
                    dh.get_runners_outputs()
                    dh.match_rules()
    
    
    def read_disco_buf(self):
        buf = self.raw_disco_data
        for l in buf.split('\n'):
            # print ""
            # If it's not a disco line, bypass it
            if not re.search('::', l):
                continue
            # print "line", l
            elts = l.split('::', 1)
            if len(elts) <= 1:
                # print "Bad discovery data"
                continue
            name = elts[0].strip()
            
            # We can choose to keep only the basename
            # of the nameid, so strip the fqdn
            # But not if it's a plain ipv4 addr
            if self.conf.strip_idname_fqdn:
                if not is_ipv4_addr(name):
                    orig = name
                    name = name.split('.', 1)[0]
                    # Maybe it's only a simple int, so we get back to full name
                    try:
                        _ = int(name)
                        name = orig
                    except:
                        pass
            
            data = '::'.join(elts[1:])
            
            # Register the name
            if not name in self.disco_data:
                self.disco_data[name] = DiscoveredHost(name, self.discoveryruns, self.rulemanager, trad=self.trad,
                                                       merge=self.merge, first_level_only=self.first_level_only)
            
            # Now get key,values
            if not '=' in data:
                continue
            
            elts = data.split('=', 1)
            if len(elts) <= 1:
                continue
            
            dh = self.disco_data[name]
            key = elts[0].strip()
            value = elts[1].strip()
            # print "-->", name, key, value
            dh.data[key] = value
    
    
    # Now we try to match all our hosts with the rules
    def match_rules(self):
        for (name, dh) in self.disco_data.iteritems():
            dh.matched_rules = self.rulemanager.match_rules(dh)
            dh.update_properties()
    
    
    def is_allowing_runners(self, name):
        name = name.strip()
        
        # If we got no value, it's * by default
        if '*' in self.runners:
            return True
        
        # print self.runners
        # If we match the name, ok
        for r in self.runners:
            r_name = r.strip()
            # print "Look", r_name, name
            if r_name == name:
                return True
        
        # Not good, so not run this!
        return False
    
    
    def allowed_runners(self):
        return [r for r in self.discoveryruns if self.is_allowing_runners(r.get_name())]
    
    
    def launch_runners(self):
        allowed_runners = self.allowed_runners()
        
        if len(allowed_runners) == 0:
            logger.info("[Discovery Import] ERROR : there is no matching runners selected!")
            return
        
        for r in allowed_runners:
            logger.info("[Discovery Import] Launching %s with a %d seconds timeout" % (r.get_name(), self.conf.runners_timeout))
            r.launch(timeout=self.conf.runners_timeout)
    
    
    def wait_for_runners_ends(self):
        all_ok = False
        while not all_ok:
            all_ok = self.is_all_ok()
            time.sleep(0.1)
    
    
    def is_all_ok(self):
        all_ok = True
        for r in self.allowed_runners():
            if not r.is_finished():
                # print "Check finished of", r.get_name()
                r.check_finished()
            b = r.is_finished()
            if not b:
                # print r.get_name(), "is not finished"
                all_ok = False
        return all_ok
    
    
    def get_runners_outputs(self):
        for r in self.allowed_runners():
            if r.is_finished():
                logger.debug("[Discovery Import] command %s %s" % (r.discoveryrun_name, make_unicode(r.current_launch)))
                if r.current_launch.exit_status != 0:
                    logger.debug("[Discovery Import] Error on run")
        self.raw_disco_data = '\n'.join(r.get_output() for r in self.allowed_runners() if r.is_finished())
        if len(self.raw_disco_data) != 0:
            logger.debug("[Discovery Import] Got raw nmap data %s" % make_unicode(self.raw_disco_data))
        else:
            logger.debug("[Discovery Import] Got no data!")
            for r in self.allowed_runners():
                logger.debug("[Discovery Import] %s" % (make_unicode(r.current_launch)))
    
    
    # Write all configuration we've got
    def write_config(self):
        # Store host to del in a separate array to remove them after look over items
        items_to_del = []
        still_duplicate_items = True
        managed_element = True
        while still_duplicate_items:
            # If we didn't work in the last loop, bail out
            if not managed_element:
                still_duplicate_items = False
            # print "LOOP"
            managed_element = False
            for name in self.disco_data:
                if name in items_to_del:
                    continue
                managed_element = True
                print('Search same host to merge.')
                dha = self.disco_data[name]
                # Searching same host and update host macros
                for oname in self.disco_data:
                    dhb = self.disco_data[oname]
                    # When same host but different properties are detected
                    if dha.name == dhb.name and dha.properties != dhb.properties:
                        for (k, v) in dhb.properties.iteritems():
                            # Merge host macros if their properties are different
                            if k.startswith('_') and dha.properties.has_key(k) and dha.properties[k] != dhb.properties[
                                k]:
                                dha.data[k] = dha.properties[k] + ',' + v
                                # print('Merged host macro:', k, dha.properties[k])
                                items_to_del.append(oname)
                        
                        # print('Merged '+ oname + ' in ' + name)
                        dha.update_properties()
                    else:
                        still_duplicate_items = False
        
        # Removing merged element
        for item in items_to_del:
            # print('Deleting '+item)
            del self.disco_data[item]
        
        # New loop to reflect changes in self.disco_data since it isn't possible
        # to modify a dict object when reading it.
        for name in self.disco_data:
            # print "Writing", name, "configuration"
            self.write_host_config(name)
    
    
    # We search for all rules of type host, and we merge them
    def write_host_config(self, host):
        dh = self.disco_data[host]
        
        d = dh.get_final_properties()
        final_host = dh.name
        
        if self.backend:
            self.backend.write_host_config_to_db(final_host, d)
