import copy
import json
import re

from shinken.log import logger
from shinken.misc.type_hint import TYPE_CHECKING

try:
    from ordereddict import OrderedDict
except ImportError:
    from collections import OrderedDict

if TYPE_CHECKING:
    from shinkensolutions.ssh_mongodb.mongo_client import MongoClient

ILLEGAL_CHARS = re.compile("""[`~!$%^&*"|'<>?,()=/+]""")

DEFAULT_STATE = 'default'
OVERLOAD_BY_USER_STATE = 'overload_by_user'
INVALID_STATE = 'invalid'
WARNING_STATE = 'warning'
DISABLED_STATE = 'disabled'
DEFINED_MANY_TIMES_STATE = 'defined_many_times'

BAD_CONDITION_ERROR = 'bad_condition'
NOT_APPLIED_ERROR = 'not_applied'
MISSING_NAME_ERROR = 'missing_name'
ILLEGAL_CHARS_ERROR = 'illegal_chars'
BAD_KEYS_ERROR = 'bad_keys'

WARNING_RULES_STATES = [DEFINED_MANY_TIMES_STATE]
DISABLED_RULES_STATES = [DISABLED_STATE, INVALID_STATE]


class SimpleCondition(object):
    def __init__(self, condition):
        self.condition = condition
        self.key = ''
        self.expression = ''
        self.error = ''
        self.regex = "^([^\n\t\r=]+)\s*=\s*(\^?[^\n\t\r=]+\$?)$"
        self.openports_regex = "^(openports)\s*=\s*([0-9](\|?[0-9]+)*)$"
        self.reg_expression = ''
    
    
    @staticmethod
    def __expression_to_regex(expression):
        reg_expression = expression
        startswith = ""
        endswith = ""
        if reg_expression.startswith('^'):
            reg_expression = reg_expression[1:]
            startswith = "^"
        if reg_expression.endswith('$'):
            reg_expression = reg_expression[:-1]
            endswith = "$"
        reg_expression = re.escape(reg_expression)
        reg_expression = "%s(%s)%s" % (startswith, reg_expression, endswith)
        return reg_expression
    
    
    def validate_condition(self):
        tmp_condition = self.condition
        tmp_splited_condition = tmp_condition.split('=')
        tmp_key = tmp_splited_condition[0]
        if tmp_key == "openports":
            match = re.search(self.openports_regex, self.condition)
        else:
            match = re.search(self.regex, self.condition)
        if match:
            self.key = match.group(1)
            self.expression = match.group(2)
            
            if self.expression == '^' or self.expression == '$' or self.expression == '^$':
                self.error = 'error.default'
                return False
            self.reg_expression = self.__expression_to_regex(self.expression)
            return True
        else:
            self.error = 'error.default'
            split = self.condition.split('=')
            if len(split) == 2:
                if not split[0] and not split[1]:
                    self.error = 'error.missing_everything'
                elif not split[0]:
                    self.error = 'error.missing_key'
                elif not split[1]:
                    self.error = 'error.missing_expression'
            elif len(split) > 2:
                self.error = 'error.too_many_equals'
        
        return False


class DiscoveryCondition(object):
    def __init__(self, condition):
        self.expression = condition['expression']
        self.number = condition['number']
        self.conditions = []
        self.valid = True
        
        multi_condition = condition['expression'].split('AND')
        for cond in multi_condition:
            self.conditions.append(SimpleCondition(cond.strip()))
            if not self.conditions[-1].validate_condition():
                self.valid = False
    
    
    @staticmethod
    def __match_data_openports(data, condition):
        ports = condition.expression.split('|')
        result = data.get('openports', '')
        if not result:
            return False
        result_ports = result.split(',')
        for port in ports:
            if port in result_ports:
                return True
        return False
    
    
    def match_data_with_conditions(self, data):
        for condition in self.conditions:
            if condition.key == "openports":
                if not self.__match_data_openports(data, condition):
                    return False
                continue
            result = re.search(condition.reg_expression, data.get(condition.key, ''), re.IGNORECASE)
            if result and result.re.groups != 0:
                continue
            return False
        return True


class DiscoveryRule(object):
    def __init__(self, name, conditions, use, bad_keys, state, errors, prefix_name, from_user=False):
        self.name = name
        self.state = state
        self.errors = errors
        self.raw_conditions = conditions
        self.conditions = self.__parse_conditions(conditions)
        self.from_user = from_user
        self.use = use
        self.prefix_name = prefix_name
        self.bad_keys = bad_keys
        
        if len(self.bad_keys) > 0:
            self.state = INVALID_STATE
            self.errors.append(BAD_KEYS_ERROR)
    
    
    def __parse_conditions(self, raw_conditions):
        conditions = []
        
        if BAD_KEYS_ERROR in self.errors:
            return conditions
        
        for condition in raw_conditions:
            if condition['expression']:
                discovery_condition = DiscoveryCondition(condition)
                if not discovery_condition.valid and BAD_CONDITION_ERROR not in self.errors:
                    self.state = INVALID_STATE
                    self.errors.append(BAD_CONDITION_ERROR)
                conditions.append(discovery_condition)
        
        return conditions
    
    
    def to_dict(self, index):
        dictionary = {'_id': "Rule%s" % index, 'name': self.name, 'conditions': [], 'use': self.use, 'bad_keys': self.bad_keys, 'prefix_name': self.prefix_name, 'state': self.state, 'errors': self.errors, 'order': index}
        
        for discovery_condition in range(len(self.conditions)):
            condition_to_add = {'expression': self.conditions[discovery_condition].expression, 'number': self.conditions[discovery_condition].number}
            for simple_condition in self.conditions[discovery_condition].conditions:
                if not simple_condition.error:
                    condition_to_add[simple_condition.key] = simple_condition.expression
                else:
                    condition_to_add['error'] = simple_condition.error
            dictionary['conditions'].append(condition_to_add)
        return dictionary
    
    
    def is_match(self, dh):
        if self.state in DISABLED_RULES_STATES:
            return False
        
        for condition in self.conditions:
            if condition.match_data_with_conditions(dh.data):
                return True
        return False
    
    
    def apply_on_host(self, host):
        if self.use:
            host['use'] = host.get('use', [])
            for use_to_add in self.use:
                if use_to_add not in host['use']:
                    host['use'].append(use_to_add)
            host['rules_applied_on_by'] = host.get('rules_applied_on_by', {})
            host['rules_applied_on_by']['use'] = host['rules_applied_on_by'].get('use', [])
            host['rules_applied_on_by']['use'].append(self.name)
        if self.prefix_name:
            prefix_name = host.get('prefix_name', '')
            if not prefix_name:
                host['prefix_name'] = self.prefix_name
                host['rules_applied_on_by'] = host.get('rules_applied_on_by', {})
                host['rules_applied_on_by']['host_name'] = [self.name]


class DiscoveryRulesManager:
    STATUS_USER_FILE_OK = 0
    STATUS_USER_FILE_ERROR = 1
    STATUS_USER_FILE_NOT_FOUND = 2
    STATUS_USER_FILE_BAD_JSON = 3
    STATUS_USER_FILE_BAD_KEYS = 4
    SHINKEN_RULES_PATH = '/etc/shinken/_default/sources/discovery/discovery_rules.json'
    
    
    def __init__(self, db, collection_name, user_rules_path):
        # type: (MongoClient, unicode, unicode) -> None
        self.user_rules_path = user_rules_path
        
        self.db = db
        self.collection_name = collection_name
        self._get_collection()
        self.reloading = False
        self.shinken_rules = []
        self.rules = []
        
        self.status = {'default': 0, 'user': 0, 'user_error_trad': ''}
        self.__add_default_rules(self.SHINKEN_RULES_PATH)
        if self.status['default'] == 0:
            self.update_user_rules()
        self.__save_to_database()
    
    
    def _get_collection(self):
        self.collection = self.db.get_collection(self.collection_name)
    
    
    def match_rules(self, dh):
        matched_rules = []
        for rule in self.rules:
            if rule.is_match(dh):
                matched_rules.append(rule)
        return matched_rules
    
    
    def set_mongo_db(self, mongo):
        self.db = mongo
        self._get_collection()
    
    
    def update_user_rules(self):
        if self.reloading:
            return
        self.reloading = True
        self.rules = copy.copy(self.shinken_rules)
        rules = self.__add_rules_from_file(self.user_rules_path)
        if rules['status'] != 'OK':
            self.status['user'] = self.STATUS_USER_FILE_ERROR
            self.status['user_error_trad'] = "error.user_file_error"
            if rules['status'] == 'FILE_NOT_FOUND':
                logger.warning("[Discovery Import] The file %s can't be reached." % self.user_rules_path)
                self.status['user'] = self.STATUS_USER_FILE_NOT_FOUND
                self.status['user_error_trad'] = "error.user_file_not_found"
            elif rules['status'] == 'BAD_JSON':
                logger.warning("[Discovery Import] The file %s is not a json file." % self.user_rules_path)
                self.status['user'] = self.STATUS_USER_FILE_BAD_JSON
                self.status['user_error_trad'] = "error.user_file_bad_json"
            elif rules['status'] == 'EMPTY':
                logger.warning("[Discovery Import] The file %s is empty." % self.user_rules_path)
                self.status['user'] = self.STATUS_USER_FILE_OK
                self.status['user_error_trad'] = "error.user_file_empty"
        else:
            self.status['user'] = self.STATUS_USER_FILE_OK
            user_rule_index = 0
            for rule in rules['rules']:
                to_skip = False
                state = "from_user"
                actual_rules_index = 0
                for actual_rule in self.rules:
                    if rule.get('name', '') and actual_rule.name == rule.get('name', ''):
                        if actual_rule.from_user:
                            if actual_rule.state not in DISABLED_RULES_STATES:
                                actual_rule.state = DEFINED_MANY_TIMES_STATE
                            to_skip = True
                        else:
                            self.rules.pop(actual_rules_index)
                            state = OVERLOAD_BY_USER_STATE
                        break
                    actual_rules_index += 1
                if not to_skip:
                    state, errors = self.__valid_rule(rule, state)
                    self.rules.insert(user_rule_index, DiscoveryRule(
                        name=rule.get('name', ''),
                        conditions=rule.get('conditions', None),
                        use=rule.get('use', []),
                        bad_keys=rule.get('bad_keys', []),
                        state=state,
                        errors=errors,
                        prefix_name=rule.get('prefix_name', ''),
                        from_user=True
                    ))
                    user_rule_index += 1
        self.__save_to_database()
    
    
    def __valid_rule(self, rule, state):
        errors = []
        
        if not rule.get('name', ''):
            errors.append(MISSING_NAME_ERROR)
        if MISSING_NAME_ERROR not in errors and not rule.get('conditions', None) and not rule.get('use', None) and not rule.get('prefix_name', None):
            state = DISABLED_STATE
        if state != DISABLED_STATE and (not rule.get('conditions', None) or (not rule.get('use', None) and not rule.get('prefix_name', None))):
            errors.append(NOT_APPLIED_ERROR)
        for model in rule.get('use', []):
            if self.contains_illegal_chars(model):
                errors.append(ILLEGAL_CHARS_ERROR)
                break
        if ILLEGAL_CHARS_ERROR not in errors and self.contains_illegal_chars(rule.get('prefix_name', '')):
            errors.append(ILLEGAL_CHARS_ERROR)
        
        if len(errors) > 0:
            state = INVALID_STATE
        
        return state, errors
    
    
    def __add_default_rules(self, shinken_rules_path):
        rules = self.__add_rules_from_file(shinken_rules_path, True)
        if rules['status'] is not 'OK':
            self.status['default'] = 1
            if rules['status'] is 'FILE_NOT_FOUND':
                logger.error("[Discovery Import] The file %s can't be reached." % shinken_rules_path)
            elif rules['status'] in ['BAD_JSON', 'EMPTY', 'BAD_KEYS']:
                logger.error("[Discovery Import] The file %s has been modified." % shinken_rules_path)
        else:
            index = 0
            for rule in rules['rules']:
                self.shinken_rules.append(DiscoveryRule(
                    name=rule['name'],
                    conditions=rule['conditions'],
                    use=rule['use'],
                    bad_keys=[],
                    state=DEFAULT_STATE,
                    errors=[],
                    prefix_name=rule.get('prefix_name', '')
                ))
                if self.shinken_rules[index].state == INVALID_STATE:
                    self.status['default'] = 1
                    logger.error("[Discovery Import] The file %s has been modified." % shinken_rules_path)
                    return
                index += 1
            logger.debug('[Discovery Import] Default discovery rules correctly loaded.')
    
    
    def __add_rules_from_file(self, file_path, is_default=False):
        accepted_keys = ['name', 'use', 'condition1', 'condition2', 'condition3', 'condition4', 'condition5',
                         'condition6', 'condition7', 'condition8', 'condition9', 'prefix_name']
        rules = []
        try:
            json_data = self.__load_json_file(file_path)
            raw_rules = json_data.get('rules', None)
            if raw_rules is None:
                return {'rules': [], 'status': 'EMPTY'}
            for raw_rule in raw_rules:
                rule = {}
                conditions = {}
                rule['bad_keys'] = []
                for key in raw_rule:
                    if key not in accepted_keys:
                        rule['bad_keys'].append(key)
                    if key.startswith('condition'):
                        conditions[key] = {'expression': raw_rule[key], 'number': key[-1:]}
                    elif key == 'use':
                        rule[key] = raw_rule[key].split(',')
                    else:
                        rule[key] = raw_rule[key]
                rule['conditions'] = [conditions[k] for k in sorted(conditions)]
                if is_default and (
                        not rule.get('name', '') or not rule.get('use', '') or not rule.get('conditions', None)):
                    return {'rules': [], 'status': 'BAD_KEYS'}
                rules.append(rule)
        except IOError:
            if not self.user_rules_path:
                return {'rules': [], 'status': 'OK'}
            return {'rules': [], 'status': 'FILE_NOT_FOUND'}
        except ValueError:
            return {'rules': [], 'status': 'BAD_JSON'}
        return {'rules': rules, 'status': 'OK'}
    
    
    def __save_to_database(self):
        if self.collection:
            self.collection.drop()
            self._get_collection()
            if self.status['default'] == 0:
                all_rules = []
                index = 1
                for rule in self.rules:
                    all_rules.append(rule.to_dict(index))
                    index += 1
                self.collection.insert_many(all_rules)
        self.reloading = False
    
    
    # Load a specific json file by removing the # at the start of the lines
    @staticmethod
    def __load_json_file(json_file):
        logger.debug('[Discovery Import] Loading configuration file [%s]' % json_file)
        try:
            with open(json_file, 'r') as f:
                file_lines = f.readlines()
            # We need to clean comments on this file, because json is normally without comments
            lines = ['' if line.strip().startswith('#') else line.strip() for line in file_lines]
            file_without_comment = '\n'.join(lines)
            # json is read into in ordered dict
            # This is important for rules, so the rules are applied in the order they are written in the file
            json_data = json.loads(file_without_comment, object_hook=OrderedDict)
            return json_data
        except Exception, exp:
            raise exp
    
    
    @staticmethod
    def contains_illegal_chars(field):
        return re.search(ILLEGAL_CHARS, field)
