#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019
# This file is part of Shinken Enterprise, all rights reserved.

import time

from shinken.misc.type_hint import TYPE_CHECKING
from shinken.thread_helper import Thread

if TYPE_CHECKING:
    from shinken.misc.type_hint import NoReturn, Any, Number, Str, Optional, Callable
    from event_container.ec_database_connection import ECDatabaseConnection
    from shinken.log import PartLogger


class ECComponentStats(Thread):
    def __init__(self, loop_speed, logger=None, database_connection=None):
        # type: (Number, PartLogger, ECDatabaseConnection) -> NoReturn
        super(ECComponentStats, self).__init__(loop_speed=loop_speed, logger=logger)
        self.database_connection = database_connection
        self.cache = {}
    
    
    # The purpose of this function is to take data from cache despite using mongo every time
    def get_from_cache(self, data_name, second_to_keep_cache=1, function_to_call_if_not_in_cache=None):
        # type: (Str, Optional[Number], Callable) -> Any
        data_cache = self.cache.get(data_name, {})
        if data_cache.get('time', 0) != 0 and (second_to_keep_cache == -1 or time.time() - data_cache.get('time', 0) <= second_to_keep_cache):
            return data_cache['data']
        else:
            data = function_to_call_if_not_in_cache()
            self.save_in_cache(data_name, data)
            return data
    
    
    # Here we save data in cache
    def save_in_cache(self, data_name, data):
        # type: (Str, Any) -> None
        self.cache[data_name] = {'time': time.time(), 'data': data}
    
    
    def get_day_keep_data(self):
        return self.get_from_cache('day_keep_data', second_to_keep_cache=-1, function_to_call_if_not_in_cache=self.database_connection.get_day_keep_data)
    
    
    def get_total_event_in_db(self):
        return self.get_from_cache('total_event_in_db', function_to_call_if_not_in_cache=self.database_connection.count_event_in_db)
    
    
    def get_total_size_of_event_in_db(self):
        return self.get_from_cache('total_size_of_event_in_db', function_to_call_if_not_in_cache=self.database_connection.get_size_of_database)
    
    
    def get_oldest_event_in_db(self):
        return self.get_from_cache('oldest_event_in_db', function_to_call_if_not_in_cache=self.database_connection.find_oldest_event_in_db)
