#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022
# This file is part of Shinken Enterprise, all rights reserved.

import re
from copy import copy
from datetime import timedelta

from ec_common import ACK, DOWNTIME, STATE
from shinken.misc.type_hint import List, Dict, Optional, Any, Text
from shinkensolutions.date_helper import get_datetime_with_local_time_zone
from shinkensolutions.lib_modules.exception import EventContainerFilterError


class FILTER_VALUE_TYPE(object):
    STRING = u'STRING'
    INTEGER = u'INTEGER'
    BOOLEAN = u'BOOLEAN'


class EventContainerFilter(object):
    
    def __init__(self, logger):
        self.logger = logger
    
    
    @staticmethod
    def _parse_time_filter(filter_value):
        # NOTE: they are REAL date in database, not epoch
        val = filter_value[0]
        if val.startswith(u'latest') or val.startswith(u'old') or val.startswith(u'in_more_than') or val.startswith(u'in_less_than'):
            operator, delay = val.split('|', 1)
            delay = int(delay)
            operator = u'$gt' if operator == u'latest' or operator == u'in_more_than' else u'$lt'
            now = get_datetime_with_local_time_zone()
            point_in_time = now - timedelta(seconds=delay)
            return {operator: point_in_time}
        
        raise Exception(u'Malformed time filter value: %s' % val)
    
    
    @staticmethod
    def _parse_simple_values(filter_values, filter_value_type=FILTER_VALUE_TYPE.STRING):
        # type: (List[unicode],unicode) -> Any
        
        if filter_value_type == FILTER_VALUE_TYPE.INTEGER:
            filter_value = list(set([int(v) for v in filter_values]))
        elif filter_value_type == FILTER_VALUE_TYPE.BOOLEAN:
            filter_value = list(set([int(v == u'true') for v in filter_values]))
        else:
            filter_value = list(set(filter_values))
        
        if len(filter_value) > 1:
            return {u'$in': filter_value}
        else:
            return filter_value[0]
    
    
    @staticmethod
    def _parse_string_values(filter_value):
        filter_value = filter_value[0]
        pat = re.compile(u'.*%s.*' % re.escape(filter_value), re.I)
        return {u'$regex': pat}
    
    
    @staticmethod
    def _parse_one_filter(visu_filter):
        # type: (Text) -> (Optional[Dict[Text,Any]], Optional[unicode])
        mongo_filter = {}
        have_context_filter = False
        context_filter = {u'$or': []}
        hint = None
        
        for search_type in visu_filter.split('~'):
            split = search_type.split(':', 1)
            filter_tag = split[0]  # .lower()
            filter_value = split[1].split('^^')
            if not split[1]:
                continue
            
            if filter_tag == u'status':
                mongo_filter[u'state_id'] = EventContainerFilter._parse_simple_values(filter_value, filter_value_type=FILTER_VALUE_TYPE.INTEGER)
                filter_value = list(set([int(v) for v in filter_value]))
                
                if STATE.UNKNOWN in filter_value:
                    filter_value.append(STATE.MISSING_DATA)
                    filter_value.append(STATE.SHINKEN_INACTIVE)
                
                if len(filter_value) > 1:
                    mongo_filter[u'state_id'] = {u'$in': filter_value}
                else:
                    mongo_filter[u'state_id'] = filter_value[0]
            
            elif filter_tag == u'is_status_confirmed':
                mongo_filter[u'state_type'] = EventContainerFilter._parse_simple_values(filter_value, filter_value_type=FILTER_VALUE_TYPE.BOOLEAN)
            
            elif filter_tag == u'event_since':
                mongo_filter[u'event_since'] = EventContainerFilter._parse_time_filter(filter_value)
                hint = u'event_since'
            
            elif filter_tag == u'event_hard_since':
                mongo_filter[u'event_hard_since'] = EventContainerFilter._parse_time_filter(filter_value)
            
            elif filter_tag == u'realm':
                mongo_filter[u'realm'] = EventContainerFilter._parse_simple_values(filter_value)
            
            elif filter_tag == u'output':
                mongo_filter[u'output'] = EventContainerFilter._parse_string_values(filter_value)
            
            elif filter_tag == u'long_output':
                mongo_filter[u'long_output'] = EventContainerFilter._parse_string_values(filter_value)
            
            elif filter_tag == u'host_name':
                mongo_filter[u'host_name'] = EventContainerFilter._parse_string_values(filter_value)
            
            elif filter_tag == u'service_description':
                mongo_filter[u'check_name'] = EventContainerFilter._parse_string_values(filter_value)
            
            elif filter_tag == u'type':
                mongo_filter[u'item_type'] = EventContainerFilter._parse_simple_values(filter_value)
            
            elif filter_tag == u'context':
                filter_value = [v.lower() for v in filter_value]
                multi_context = bool(len(filter_value) > 1)
                if u'acknowledged' in filter_value:
                    ack_filter = {
                        u'$or': [
                            {u'partial_ack': True},
                            {u'acknowledged': {u'$in': [ACK.ACTIVE, ACK.INHERITED]}},
                        ],
                    }
                    if multi_context:
                        context_filter[u'$or'].append(ack_filter)
                    else:
                        context_filter = ack_filter
                    have_context_filter = True
                if u'flapping' in filter_value:
                    flapping_filter = {
                        u'$or': [
                            {u'partial_flapping': True},
                            {u'flapping': True},
                        ],
                    }
                    if multi_context:
                        context_filter[u'$or'].append(flapping_filter)
                    else:
                        context_filter = flapping_filter
                    have_context_filter = True
                if u'downtime' in filter_value:
                    dt_filter = {
                        u'$or': [
                            {u'partial_dt': True},
                            {u'downtime': {u'$in': [DOWNTIME.ACTIVE, DOWNTIME.INHERITED]}},
                        ],
                    }
                    if multi_context:
                        context_filter[u'$or'].append(dt_filter)
                    else:
                        context_filter = dt_filter
                    have_context_filter = True
                
                if u'nothing' in filter_value:
                    have_context_filter = False
                    mongo_filter[u'partial_ack'] = False
                    mongo_filter[u'acknowledged'] = ACK.NONE
                    mongo_filter[u'partial_flapping'] = False
                    mongo_filter[u'flapping'] = False
                    mongo_filter[u'partial_dt'] = False
                    mongo_filter[u'downtime'] = DOWNTIME.NONE
            # item_uuids is generated if host_name or service_description is used from webui
            # and will remove theses two other filters
            elif filter_tag == u'item_uuids':
                mongo_filter[u'item_uuid'] = EventContainerFilter._parse_simple_values(filter_value)
                if not hint:
                    hint = u'item_uuid'
            else:
                raise Exception('Unknown filter key: %s' % filter_tag)
        
        if have_context_filter:
            if mongo_filter:
                mongo_filter = {u'$and': [mongo_filter, context_filter]}
            else:
                mongo_filter = context_filter
        
        return mongo_filter, hint
    
    
    def _parse_visu_filter(self, visu_filters):
        # type: (List[Text]) -> (Optional[Dict[Text,Any]], Optional[unicode])
        if not visu_filters:
            return None, None
        try:
            filters = []
            hint = None
            for visu_filter in visu_filters:
                one_filter, one_hint = EventContainerFilter._parse_one_filter(visu_filter)
                if one_filter:
                    filters.append(one_filter)
                    if one_hint:
                        if one_hint == u'event_since' or not hint:
                            hint = one_hint
            if not filters:
                return None, hint
            elif len(filters) > 1:
                return {u'$or': filters}, hint
            else:
                return filters[0], hint
        except Exception as e:
            message = getattr(e, 'message', str(e))
            message = 'Invalid filter:%s. %s' % (visu_filters, message)
            self.logger.warning(message)
            raise EventContainerFilterError(message=message)
    
    
    def _parse_date_filter(self, date_filter):
        # type: (Text) -> Optional[Dict[Text,Any]]
        if not date_filter:
            return None
        try:
            filter_start = None
            filter_end = None
            for raw_filter in date_filter.split('~'):
                _name, _value = raw_filter.split(':')
                if not _value or _value == u'none':
                    continue
                if _name == u'start':
                    filter_start = {u'event_since': {u'$gte': get_datetime_with_local_time_zone(int(_value))}}
                elif _name == u'end':
                    filter_end = {u'event_since': {u'$lt': get_datetime_with_local_time_zone(int(_value))}}
            
            if filter_start and filter_end:
                _filter = {u'$and': [filter_start, filter_end]}
            elif filter_start:
                _filter = filter_start
            elif filter_end:
                _filter = filter_end
            else:
                _filter = None
            
            return _filter
        
        except Exception as e:
            message = getattr(e, 'message', str(e))
            message = 'Invalid date_filter:%s. %s' % (date_filter, message)
            self.logger.warning(message)
            raise EventContainerFilterError(message=message)
    
    
    def from_visu_filters_to_mongo_filters(self, visu_filters, date_filter):
        # type: (List[Text], Text) -> (Optional[Dict[Text,Any]], Optional[unicode])
        visu_filter, hint = self._parse_visu_filter(visu_filters)
        date_filter = self._parse_date_filter(date_filter)
        if date_filter:
            hint = u'event_since'
        return EventContainerFilter.append_to_filter_event_since(visu_filter, date_filter), hint
    
    
    @staticmethod
    def append_to_filter(filters, to_append):
        # type: (Dict[Text,Any], Dict[Text,Any]) -> Optional[Dict[Text,Any]]
        if to_append:
            if filters:
                if u'$and' in to_append:
                    if u'$and' in filters:
                        filters = copy(filters)
                        filters[u'$and'].extend(to_append[u'$and'])
                    else:
                        filters = {u'$and': [filters] + to_append[u'$and']}
                else:
                    if u'$and' in filters:
                        filters = copy(filters)
                        filters[u'$and'].append(to_append)
                    elif u'$or' in filters:
                        filters = {u'$and': [filters, to_append]}
                    else:
                        filters = copy(filters)
                        for k, v in to_append.iteritems():
                            filters[k] = v
            else:
                filters = to_append
        return filters
    
    
    @staticmethod
    def append_to_filter_event_since(filters, to_append):
        # type: (Dict[Text,Any], Dict[Text,Any]) -> Optional[Dict[Text,Any]]
        
        if not to_append:
            return filters
        if not filters:
            return to_append
        
        if u'$and' in to_append:
            if u'$and' in filters:
                filters = copy(filters)
                filters[u'$and'].extend(to_append[u'$and'])
            else:
                filters = {u'$and': [filters] + to_append[u'$and']}
        else:
            if u'$and' in filters:
                filters = copy(filters)
                filters[u'$and'].append(to_append)
            elif u'$or' in filters:
                filters = {u'$and': [filters, to_append]}
            else:
                filters = copy(filters)
                if set(filters.iterkeys()).intersection(set(to_append.iterkeys())):
                    filters = {u'$and': [filters, to_append]}
                else:
                    for k, v in to_append.iteritems():
                        filters[k] = v
        

        
        return filters
