#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019
# This file is part of Shinken Enterprise, all rights reserved.

import threading
import time
import traceback

from ec_common import ACK, DOWNTIME, STATE_TYPE
from ec_database_connection import ECDatabaseConnection
from ec_event import Event
from ec_filter import EventContainerFilter
from ec_reader_stats import ECReaderStats
from shinken.misc.filter import check_user_item_access
from shinken.misc.type_hint import List, Dict, Any, Text, Optional
from shinken.modules.base_module.basemodule import BaseModule
from shinken.modules.base_module.broker_base_module import ContextClass
from shinken.objects.contact import Contact
from shinkensolutions.date_helper import timestamp_from_datetime
from shinkensolutions.lib_modules.configuration_reader import read_int_in_configuration
from shinkensolutions.lib_modules.exception import EventContainerFilterError

EVENTS_EXPORT_MAX_RETRY_DEFAULT_VALUE = 2
EVENTS_EXPORT_MAX_RETRY_KEY_NAME = u'broker__module_webui__module_event_manager_reader__events_export__max_retry'


class EventContainerWebUIModule(BaseModule):
    
    def __init__(self, configuration):
        BaseModule.__init__(self, configuration)
        self.database_connection = ECDatabaseConnection(configuration, self.logger, ttl_index=False)
        self.reader_stats = ECReaderStats(self, self.database_connection)
        self.event_container_filter = EventContainerFilter(self.logger)
        Event.reader_stats = self.reader_stats
        self._init_completed = False
        self.events_export_max_retry = read_int_in_configuration(configuration, EVENTS_EXPORT_MAX_RETRY_KEY_NAME, unicode(EVENTS_EXPORT_MAX_RETRY_DEFAULT_VALUE), log_fct=self.logger)
        if self.events_export_max_retry <= 0:
            self.logger.error(u'%s has not positive value[%s], resetting to default value [%s]' % (EVENTS_EXPORT_MAX_RETRY_KEY_NAME, self.events_export_max_retry, EVENTS_EXPORT_MAX_RETRY_DEFAULT_VALUE))
            self.events_export_max_retry = EVENTS_EXPORT_MAX_RETRY_DEFAULT_VALUE
    
    
    def init(self):
        self.database_connection.init()
        self.reader_stats.start_thread()
        self._init_completed = True
    
    
    def get_events(self, user, filters, date_filter, first_ordering_uuid, page_index, page_size, are_rights_already_managed=False, active_context_lock=None, filters_json=None):
        # type: (Contact, List, str, str, int, int, bool, Optional[ContextClass], Optional[unicode]) -> (List[Dict], bool)
        user_uuid = user.get_uuid()
        event_start = time.time()

        try:
            filters, hint = self.event_container_filter.from_visu_filters_to_mongo_filters(filters, date_filter)
        except EventContainerFilterError:
            self.logger.warning(u'Cannot create filter from user input (%s / %s): %s' % (filters, date_filter, traceback.format_exc()))
            filters = None
            hint = None

        # Items from Regenerator must not be used after this function call (as fair lock will be released)
        _events, have_next = self._get_events(user, filters, hint, first_ordering_uuid, page_index, page_size, are_rights_already_managed, active_context_lock)

        self.logger.log_perf(event_start, u'user=%s ] [ get_events' % user_uuid, u'%s event%s returned with filter:[%s]' % (len(_events), u's' if len(_events) > 1 else u'', filters_json), min_time=0, info_time=0, warn_time=30)

        return map(self._from_event_to_visu_event, _events), have_next
    
    
    def find_downtimes(self, downtime_ids):
        return self.database_connection.find_downtimes(downtime_ids)
    
    
    def find_acknowledge(self, acknowledge_id):
        return self.database_connection.find_acknowledge(acknowledge_id)
    
    
    def _from_event_to_visu_event(self, event):
        # type: (Event) -> Dict[Text, Any]
        
        visu_event = event.__dict__.copy()
        visu_event['event_since'] = timestamp_from_datetime(event.event_since)
        visu_event['event_hard_since'] = timestamp_from_datetime(event.event_hard_since) if event.state_type == STATE_TYPE.HARD else None
        visu_event['context'] = self._get_context(visu_event)
        
        if event.active_downtime_uuids:
            dts = self.find_downtimes(event.active_downtime_uuids)
            if dts:
                visu_downtimes = [{
                    u'author'    : dt[u'author'],
                    u'comment'   : dt[u'comment'],
                    u'end_time'  : dt.get(u'end_time', None),
                    u'start_time': dt.get(u'start_time', None),
                } for dt in dts]
                
                visu_event['downtimes'] = visu_downtimes
                visu_event['downtimes_inherited'] = visu_downtimes
        if event.acknowledge_id:
            a = self.find_acknowledge(event.acknowledge_id)
            if a:
                visu_acknowledges = {
                    u'author'    : a[u'author'],
                    u'comment'   : a[u'comment'],
                    u'start_time': a[u'start_time'],
                }
                visu_event['acknowledgement'] = visu_acknowledges
                visu_event['acknowledgement_inherited'] = visu_acknowledges
                visu_event['partial_acknowledge'] = visu_acknowledges
        
        visu_event['status'] = event.state_id
        visu_event['is_status_confirmed'] = event.state_type == STATE_TYPE.HARD
        visu_event['type'] = event.item_type
        visu_event['service_description'] = event.check_name
        
        visu_event.pop('_id', None)
        visu_event.pop('state_id', None)
        visu_event.pop('state_type', None)
        visu_event.pop('item_type', None)
        visu_event.pop('name', None)
        visu_event.pop('flapping', None)
        visu_event.pop('acknowledged', None)
        visu_event.pop('downtime', None)
        visu_event.pop('partial_flapping', None)
        visu_event.pop('partial_ack', None)
        visu_event.pop('partial_dt', None)
        visu_event.pop('check_name', None)
        visu_event.pop('state_validity_end_time', None)
        visu_event.pop('state_validity_end_datetime', None)
        visu_event.pop('missing_data_activation_time', None)
        visu_event.pop('missing_data_activation_datetime', None)
        visu_event.pop('from_brok', None)
        visu_event.pop('brok_version', None)
        return visu_event
    
    
    @staticmethod
    def _get_context(visu_event):
        context = u'NOTHING'
        if visu_event[u'partial_flapping']:
            context = u'PARTIAL-FLAPPING'
        if visu_event[u'flapping']:
            context = u'FLAPPING'
        if visu_event[u'partial_ack']:
            context = u'PARTIAL-ACKNOWLEDGED'
        if visu_event[u'acknowledged'] == ACK.INHERITED:
            context = u'INHERITED-ACKNOWLEDGED'
        if visu_event[u'acknowledged'] == ACK.ACTIVE:
            context = u'ACKNOWLEDGED'
        if visu_event[u'partial_dt']:
            context = u'PARTIAL-DOWNTIME'
        if visu_event[u'downtime'] == DOWNTIME.INHERITED:
            context = u'INHERITED-DOWNTIME'
        if visu_event[u'downtime'] == DOWNTIME.ACTIVE:
            context = u'DOWNTIME'
        return context
    
    
    @staticmethod
    def _filter_events_by_user_visibility(user, events):
        # type: (Contact, List[Event]) -> List[Event]
        filtered_events = [e for e in events if check_user_item_access(user, e.item_uuid)]
        return filtered_events
    
    
    def _get_events(self, user, filters, hint, first_ordering_uuid, page_index, page_size, are_rights_already_managed, active_context_lock=None):
        # type: (Contact, Dict[Text,Any], Optional[unicode], unicode, int, int, bool, Optional[ContextClass]) -> (List[Event], bool)
        # active_context_lock will be released by this function
        
        start = time.time()
        have_next = False
        if first_ordering_uuid:
            filters = self._set_first_ordering_uuid(filters, first_ordering_uuid)
        
        if user.is_admin or are_rights_already_managed:
            self.logger.debug(u'[%s] Start to get events from the user, with rights already filtered, from page=%s and page_size=%s' % (user.get_name(), page_index, page_size))
            if active_context_lock:
                self.logger.debug(u'_get_events thread[%s] releasing fair lock before DB query' % threading.current_thread().ident)
                active_context_lock.release()
            events = self.database_connection.find_events(filters, hint, skip_value=(page_index * page_size), page_size=page_size + 1)
        else:
            self.logger.debug(u'[%s] Start to get events from the user, but without rights already filtered, from page=%s and page_size=%s' % (user.get_name(), page_index, page_size))
            events = self._get_events_for_non_admin(user, filters, hint, first_ordering_uuid, page_index, page_size, active_context_lock)

        # Items from Regenerator are no more available as fair lock has been released !!
        
        if len(events) > page_size:
            last_event = events.pop()
            have_next = last_event.ordering_uuid
        
        return events, have_next
    
    
    def _get_events_for_non_admin(self, user, filters, hint, first_ordering_uuid, page_index, page_size, active_context_lock=None):
        # type: (Contact, List, Optional[unicode], unicode, int, int, Optional[ContextClass]) -> List[Event]
        
        if active_context_lock:
            self.logger.debug(u'_get_events_for_non_admin thread[%s] releasing fair lock' % threading.current_thread().ident)
            active_context_lock.release()
        events = []
        mongo_page_size = int(page_size * 1.5)
        mongo_page_index = 0
        to_skip = page_index * page_size
        end_index = (to_skip + page_size + 1)  # +1 to known if there are next element
        start = time.time()
        while True:
            start_loop = time.time()
            log_txt = u'[%s] [Time elapsed from start of request:%.3fs] Looking for events, from mongo_page_index=%s with a page size=%s' % (user.get_name(), time.time() - start, mongo_page_index, mongo_page_size)
            log_f = self.logger.debug
            if start_loop - start > 60:
                log_f = self.logger.warning
            log_f(log_txt)
            
            all_events_page = self.database_connection.find_events(filters, hint, skip_value=mongo_page_index, page_size=mongo_page_size)
            
            if not all_events_page:
                return events[to_skip:end_index]
            
            if active_context_lock:
                start_acquire = time.time()
                active_context_lock.acquire()
                self.logger.debug(u'_get_events_for_non_admin thread[%s] reacquired fair lock in %.3fs for user right management' % (threading.current_thread().ident, time.time() - start_acquire))
            
            all_events_page = self._filter_events_by_user_visibility(user, all_events_page)
            
            if active_context_lock:
                self.logger.debug(u'_get_events_for_non_admin thread[%s] releasing fair lock after user right management' % threading.current_thread().ident)
                active_context_lock.release()
            
            if first_ordering_uuid is None and all_events_page:
                first_ordering_uuid = all_events_page[0].ordering_uuid
                filters = self._set_first_ordering_uuid(filters, first_ordering_uuid)
            
            if len(events) + len(all_events_page) > end_index:
                events = events + all_events_page
                return events[to_skip:end_index]
            
            events.extend(all_events_page)
            mongo_page_index += mongo_page_size
            log_f(u'[%s] Time for this request was %.3fs' % (user.get_name(), time.time() - start_loop))
    
    
    @staticmethod
    def _set_first_ordering_uuid(original_filters, first_ordering_uuid):
        filters = original_filters
        if first_ordering_uuid:
            filters = EventContainerFilter.append_to_filter(original_filters, {u'ordering_uuid': {u'$lte': first_ordering_uuid}})
        return filters
    
    
    def get_raw_stats(self, param=''):
        try:
            seconds_of_stats = int(param)
        except:
            seconds_of_stats = 60
        
        status = BaseModule.get_state(self)
        if not self._init_completed:
            status['raw_stats'] = {'is_ready': False}
        else:
            status['raw_stats'] = self.reader_stats.get_raw_stats(seconds_of_stats)
        
        return status
    
    
    def send_stat_to_reader_stats(self, stat_name, stat_endpoint, filters, stat_time):
        self.reader_stats.add_request_stat(stat_name, stat_endpoint, filters, stat_time)
    
    
    def do_loop_turn(self):
        pass
    
    
    def loop_turn(self):
        pass
