#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019
# This file is part of Shinken Enterprise, all rights reserved.


from shinken.misc.type_hint import NoReturn, TYPE_CHECKING
from shinken.objects.module import Module as ShinkenModuleDefinition
from shinken.thread_helper import Thread

from sla_component_manager import ComponentManager

if TYPE_CHECKING:
    from sla_error_handler import SlaErrorHandler


class AbstractComponent(object):
    
    def __init__(self, conf, component_manager):
        # type: (ShinkenModuleDefinition, ComponentManager) -> None
        component_manager.add_to_manager(self)
        self.conf = conf
        self.logger = component_manager.logger
    
    
    def init(self):
        # type: () -> NoReturn
        raise NotImplementedError()
    
    
    def tick(self):
        # type: () -> NoReturn
        raise NotImplementedError()
    
    
    def after_fork_cleanup(self):
        self.conf = None
        self.logger = None


class ThreadComponent(AbstractComponent, Thread):
    
    def __init__(self, conf, component_manager, error_handler, only_one_thread_by_class=False, loop_speed=1, stop_thread_on_error=True):
        # type: (ShinkenModuleDefinition, ComponentManager, SlaErrorHandler, bool, float, bool) -> None
        AbstractComponent.__init__(self, conf, component_manager)
        Thread.__init__(self, only_one_thread_by_class=only_one_thread_by_class, loop_speed=loop_speed, stop_thread_on_error=stop_thread_on_error, error_handler=error_handler, logger=component_manager.logger)
    
    
    def init(self):
        # type: () -> NoReturn
        raise NotImplementedError()
    
    
    def get_thread_name(self):
        # type: () -> str
        raise NotImplementedError()
    
    
    def loop_turn(self):
        # type: () -> NoReturn
        raise NotImplementedError()
    
    
    def tick(self):
        # type: () -> NoReturn
        self.start_thread()
    
    
    def after_fork_cleanup(self):
        AbstractComponent.after_fork_cleanup(self)
        Thread.after_fork_cleanup(self)
