#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2021
# This file is part of Shinken Enterprise, all rights reserved.

import logging
import time
from datetime import datetime, timedelta

from shinken.ipc.share_item import ShareItem
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.thread_helper import async_call
from sla.component.sla_abstract_component import AbstractComponent

if TYPE_CHECKING:
    from shinken.misc.type_hint import Any, Callable, Dict, Union
    from sla.component.sla_component_manager import ComponentManager
    from sla.component.sla_database_connection import SLADatabaseConnection
    from shinken.objects.module import Module as ShinkenModuleDefinition


class SlaUniqueElementsConsts(object):
    ID = u'sla_unique_archived_elements_count'
    
    class KEYS(object):
        COUNT = u'total_archived_elements_count'
        TO_ADD = u'nb_elements_to_add'
        TO_REMOVE = u'nb_elements_to_remove'
        LAST_UPDATE = u'last_update'


class SlaUniqueArchivedElements(ShareItem):
    
    def init_from_dict(self, unique_archived_elements_document):
        # type: (Dict) -> None
        setattr(self, u'_id', unique_archived_elements_document[u'_id'])
        setattr(self, SlaUniqueElementsConsts.KEYS.COUNT, unique_archived_elements_document[SlaUniqueElementsConsts.KEYS.COUNT])
        setattr(self, SlaUniqueElementsConsts.KEYS.TO_ADD, unique_archived_elements_document[SlaUniqueElementsConsts.KEYS.TO_ADD])
        setattr(self, SlaUniqueElementsConsts.KEYS.TO_REMOVE, unique_archived_elements_document[SlaUniqueElementsConsts.KEYS.TO_REMOVE])
        setattr(self, SlaUniqueElementsConsts.KEYS.LAST_UPDATE, unique_archived_elements_document[SlaUniqueElementsConsts.KEYS.LAST_UPDATE])
    
    
    @classmethod
    def get_default_document(cls):
        return {
            u'_id'                                  : SlaUniqueElementsConsts.ID,
            SlaUniqueElementsConsts.KEYS.COUNT      : -1,
            SlaUniqueElementsConsts.KEYS.TO_ADD     : 0,
            SlaUniqueElementsConsts.KEYS.TO_REMOVE  : 0,
            SlaUniqueElementsConsts.KEYS.LAST_UPDATE: 0
        }
    
    
    def to_dict(self):
        # type: () -> Dict[unicode, Union[unicode, int]]
        return {
            u'_id'                                  : self._id,
            SlaUniqueElementsConsts.KEYS.COUNT      : self.total_archived_elements_count,
            SlaUniqueElementsConsts.KEYS.TO_ADD     : self.nb_elements_to_add,
            SlaUniqueElementsConsts.KEYS.TO_REMOVE  : self.nb_elements_to_remove,
            SlaUniqueElementsConsts.KEYS.LAST_UPDATE: self.last_update
        }
    
    
    def update_total_archive_elements_count(self, new_element_count):
        # type: (int) -> None
        # Using setattr because attributes are not initialized in __init__
        setattr(self, SlaUniqueElementsConsts.KEYS.LAST_UPDATE, time.time())
        setattr(self, SlaUniqueElementsConsts.KEYS.COUNT, new_element_count)
        self.reset_sub_counters()
    
    
    def reset_sub_counters(self):
        # type: () -> None
        # Using setattr because attributes are not initialized in __init__
        setattr(self, SlaUniqueElementsConsts.KEYS.TO_ADD, 0)
        setattr(self, SlaUniqueElementsConsts.KEYS.TO_REMOVE, 0)
    
    
    def compute_total(self):
        # type: () -> int
        element_count = getattr(self, SlaUniqueElementsConsts.KEYS.COUNT, None)
        if element_count is None:
            return -1
        if element_count in (0, -1):
            return element_count
        return (element_count + self.nb_elements_to_add) - self.nb_elements_to_remove
    
    
    def is_too_old(self):
        # type: () -> bool
        last_update = getattr(self, SlaUniqueElementsConsts.KEYS.LAST_UPDATE, None)
        if last_update is None:
            return True
        yesterday = datetime.now() - timedelta(days=1)
        last_update = datetime.fromtimestamp(last_update)
        return last_update < yesterday
    
    
    def get_last_update_iso(self):
        # type: () -> unicode
        return datetime.fromtimestamp(self.last_update).strftime(u'%Y-%m-%d %H:%M:%S')


class SLACollectionsStats(AbstractComponent):
    MAX_TRIES = 3
    IGNORED_EXCEPTIONS = (u'AutoReconnect', u'ConnectionFailure', u'TimeoutError')
    
    
    def __init__(self, conf, component_manager, sla_database_connection, should_maintain_share_item=False):
        # type: (ShinkenModuleDefinition, ComponentManager,  SLADatabaseConnection, bool) -> None
        super(SLACollectionsStats, self).__init__(conf, component_manager)
        self.sla_database_connection = sla_database_connection
        self.should_maintain_share_item = should_maintain_share_item
        
        self.unique_archived_elements = SlaUniqueArchivedElements(key_name=u'sla_collections_stats_unique_elements_in_archive', reinit=should_maintain_share_item)  # type: SlaUniqueArchivedElements
        
        self.sla_stats_logger = self.logger.get_sub_part(u'SLA COLLECTIONS STATS')
        self.unique_elements_logger = self.sla_stats_logger.get_sub_part(u'UNIQUE ELEMENTS IN ARCHIVE')
    
    
    def init(self):
        # type: () -> None
        if not self.should_maintain_share_item:
            return
        
        self._retry_on_fail(self._loading_elements_count_document, u'Getting the previous count of unique elements in archive from "sla_collections_stats" collection')
        if self.unique_archived_elements.total_archived_elements_count == -1:
            self.unique_elements_logger.info(u'Creating the database document for the count element in archive')
            self.sla_database_connection.col_sla_collections_stats.save(SlaUniqueArchivedElements.get_default_document())
            self.unique_archived_elements.init_from_dict(SlaUniqueArchivedElements.get_default_document())
            self.update_count_elements_in_archive()
        else:
            if self.unique_archived_elements.is_too_old():
                self.unique_elements_logger.info(u'The count of unique elements in archive needs to be refreshed because it is older than one day')
                self.update_count_elements_in_archive()
    
    
    def _loading_elements_count_document(self):
        # type: () -> None
        unique_archived_elements_documents = self.sla_database_connection.col_sla_collections_stats.find_one({u'_id': SlaUniqueElementsConsts.ID})
        if unique_archived_elements_documents is None:
            self.unique_archived_elements.total_archived_elements_count = -1
            self.unique_elements_logger.warning(u'No count of unique elements in archive found in "sla_collections_stats" collection')
        else:
            self.unique_archived_elements.init_from_dict(unique_archived_elements_documents)
            self.unique_elements_logger.info(u'Found [ %s ] unique elements in archive in "sla_collections_stats" collection, dating from [ %s ]' % (self.unique_archived_elements.compute_total(), self.unique_archived_elements.get_last_update_iso()))
    
    
    def _retry_on_fail(self, callback, action_log):
        # type: (Callable, unicode) -> Any
        for try_number in range(1, self.MAX_TRIES + 1):
            try:
                self.unique_elements_logger.info(u'%s: Try %s/%s' % (action_log, try_number, self.MAX_TRIES))
                return callback()
            except Exception as exc:
                if try_number < self.MAX_TRIES:
                    self.unique_elements_logger.warning(u'%s failed because of this error: %s' % (action_log, exc))
                    if type(exc).__name__ not in self.IGNORED_EXCEPTIONS:
                        self.unique_elements_logger.print_stack(level=logging.WARNING)
                else:
                    self.unique_elements_logger.error(u'%s failed because of this error: %s' % (action_log, exc))
                    if type(exc).__name__ not in self.IGNORED_EXCEPTIONS:
                        self.unique_elements_logger.print_stack()
    
    
    # @async_call
    # def update_count_elements_in_archive(self):
    #     # type: () -> None
    #     total_elements_archived_found = self._retry_on_fail(self._compute_total_unique_elements_stored, u'Getting the count of unique elements in archive from archive database')
    #     if total_elements_archived_found not in (0, -1):
    #         self.unique_elements_logger.info(u'Count of unique elements in archive found: [ %d ]' % total_elements_archived_found)
    #
    #     self.unique_archived_elements.update_total_archive_elements_count(total_elements_archived_found)
    #     self.sla_database_connection.col_sla_collections_stats.update(entry_id=SlaUniqueElementsConsts.ID, update=self.unique_archived_elements.to_dict())

    @async_call
    def update_count_elements_in_archive(self):
        # type: () -> None
        total_elements_archived_found = 0
        
        self.unique_archived_elements.update_total_archive_elements_count(total_elements_archived_found)
        self.sla_database_connection.col_sla_collections_stats.update(entry_id=SlaUniqueElementsConsts.ID, update=self.unique_archived_elements.to_dict())
    
    
    def _compute_total_unique_elements_stored(self):
        # type: () -> int
        col_archive_data = self.sla_database_connection.col_archive.aggregate([
            {u'$group': {u'_id': 0, u'uuids': {u'$addToSet': u'$uuid'}}},
            {u'$project': {u'uniqueUuidCount': {u'$size': u'$uuids'}}}
        ])
        try:
            # pymongo version >= 3
            result = col_archive_data.next()
            return result[u'uniqueUuidCount']
        except StopIteration:
            return 0
        except AttributeError:
            if col_archive_data and len(col_archive_data[u'result']) > 0:
                return col_archive_data[u'result'][0][u'uniqueUuidCount']
            return 0
    
    
    def update_nb_elements_to_add(self, nb_elements_to_add):
        # type: (int) -> None
        if self.unique_archived_elements.is_too_old():
            self.update_count_elements_in_archive()
        else:
            self.unique_archived_elements.nb_elements_to_add += nb_elements_to_add
            self.sla_database_connection.col_sla_collections_stats.update(entry_id=SlaUniqueElementsConsts.ID, update=self.unique_archived_elements.to_dict())
    
    
    def update_nb_elements_to_remove(self, nb_elements_to_remove):
        # type: (int) -> None
        if self.unique_archived_elements.is_too_old():
            self.update_count_elements_in_archive()
        else:
            self.unique_archived_elements.nb_elements_to_remove += nb_elements_to_remove
            self.sla_database_connection.col_sla_collections_stats.update(entry_id=SlaUniqueElementsConsts.ID, update=self.unique_archived_elements.to_dict())
    
    
    def get_unique_elements_in_sla_archive_count(self):
        # type: () -> int
        return self.unique_archived_elements.compute_total()
    
    
    def tick(self):
        # inherited method intentionally left empty
        pass
