#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019
# This file is part of Shinken Enterprise, all rights reserved.


from collections import namedtuple
from ctypes import c_bool, c_int64, c_double
from multiprocessing import Value

FutureState = namedtuple(u'FutureState', [u'date', u'to_report', u'state', u'output', u'long_output', u'ack_uuid', u'active_downtime_uuids', u'missing_data_activation_time'])  # This tuple and this values are immutable
CURRENT_ARCHIVE_VERSION = 3
LIST_STATUS = (u'ok', u'warn', u'crit', u'unknown', u'missing', u'inactive')


class LOG_PART(object):
    LOADING_CONFIG = u'LOADING CONFIG'
    INITIALISATION = u'INITIALISATION'
    DAILY_CLEAN = u'DAILY_CLEAN'
    ARCHIVE_COLLECTION = u'ARCHIVE COLLECTION'
    FUTURE_STATE = u'REPORT STATE'
    SLA_UPDATE = u'SLA UPDATE'
    MANAGE_BROKS = u'MANAGE BROKS'
    ARCHIVE_MIGRATION = u'ARCHIVE MIGRATION'
    RAW_SLA_MIGRATION = u'RAW SLA MIGRATION'


class RAW_SLA_KEY(object):
    HNAME = u'0'
    SDESC = u'1'
    TYPE = u'2'
    ACK_UUID = u'4'
    ACK = u'5'
    DT = u'6'
    STATUS = u'7'
    START = u'8'
    END = u'9'
    FLAPPING = u'10'
    CONTEXT_AND_STATUS = u'11'
    UUID = u'12'
    OUTPUT = u'13'
    LONG_OUTPUT = u'14'
    DOWNTIMES_UUID = u'15'
    MISSING_DATA_ACTIVATION_TIME = u'16'


class STATUS(object):
    OK = 0
    WARN = 1
    CRIT = 2
    UNKNOWN = 3
    MISSING_DATA = 4
    SHINKEN_INACTIVE = 5
    
    STATUS_MAP = {
        u'OK'      : OK,
        u'WARNING' : WARN,
        u'CRITICAL': CRIT,
        u'UNKNOWN' : UNKNOWN,
    }
    
    PRINT_STATE = {
        OK              : u'STATUS_OK',
        WARN            : u'STATUS_WARN',
        CRIT            : u'STATUS_CRIT',
        UNKNOWN         : u'STATUS_UNKNOWN',
        MISSING_DATA    : u'STATUS_MISSING_DATA',
        SHINKEN_INACTIVE: u'STATUS_SHINKEN_INACTIVE',
    }


# ---------------- Shared Data ----------------
class SharedData(object):
    def __init__(self):
        self._migration_daily_done = Value(c_bool, False)
        self._migration_archive_done = Value(c_bool, False)
        # This parameter is always 'False'. The only issue to set it 'True', is for recompute the SLA from the mongo collection 'has_been_archive_day_year' and not the 'day_year'
        self._already_archived = Value(c_bool, False)
        
        self._default_sla_thresholds = (99, 97)
        
        # Archive stats
        self._archive_in_progress = Value(c_bool, False)
        self._total_sla_archived = Value(c_int64, 0)
        self._total_sla_current_archive = Value(c_int64, 0)
        self._sla_archived_during_current_archive = Value(c_int64, 0)
        self._archive_progression_date = Value(c_int64, 0)
        self._latest_archive_start_time = Value(c_double, 0)
        self._latest_archive_execution_time = Value(c_double, 0)
        self._previous_archive_start_time = Value(c_double, 0)
        self._previous_archive_execution_time = Value(c_double, 0)
        self._previous_archive_sla_archived = Value(c_int64, 0)
        
        # Migration stats
        self._migration_in_progress = Value(c_bool, False)
        self._total_sla_to_migrate = Value(c_int64, 0)
        self._nb_sla_left_to_migrate = Value(c_int64, 0)
        self._execution_time_last_migration = Value(c_double, 0)
        
        # Daily clean stats
        self._daily_clean_in_progress = Value(c_bool, False)
        self._total_sla_to_remove = Value(c_int64, 0)
        self._nb_sla_left_to_remove = Value(c_int64, 0)
        self._execution_time_last_daily_clean = Value(c_double, 0)
        self._total_unique_elements_valid = Value(c_bool, False)
    
    
    def set_default_values(self, default_properties_values):
        def_for_host = default_properties_values.get(u'host', {})
        self._default_sla_thresholds = (float(def_for_host.get(u'sla_warning_threshold', self._default_sla_thresholds[0])), float(def_for_host.get(u'sla_critical_threshold', self._default_sla_thresholds[1])))
    
    
    def get_default_sla_thresholds(self):
        return self._default_sla_thresholds
    
    
    def get_already_archived(self):
        return self._already_archived.value
    
    
    def set_already_archived(self, value):
        self._already_archived.value = value
    
    
    def get_migration_daily_done(self):
        return self._migration_daily_done.value
    
    
    def set_migration_daily_done(self, value):
        self._migration_daily_done.value = value
    
    
    def get_migration_archive_done(self):
        return self._migration_archive_done.value
    
    
    def set_migration_archive_done(self, value):
        self._migration_archive_done.value = value
    
    
    # -------------------- Archive
    
    def get_archive_in_progress(self):
        return self._archive_in_progress.value
    
    
    def set_archive_in_progress(self, value):
        self._archive_in_progress.value = value
    
    
    def get_total_sla_archived(self):
        return self._total_sla_archived.value
    
    
    def set_total_sla_archived(self, value):
        self._total_sla_archived.value = value
    
    
    def get_total_sla_current_archive(self):
        return self._total_sla_current_archive.value
    
    
    def set_total_sla_current_archive(self, value):
        self._total_sla_current_archive.value = value
    
    
    def get_sla_archived_during_current_archive(self):
        return self._sla_archived_during_current_archive.value
    
    
    def set_sla_archived_during_current_archive(self, value):
        self._sla_archived_during_current_archive.value = value
    
    
    def get_archive_progression_date(self):
        return self._archive_progression_date.value
    
    
    def set_archive_progression_date(self, value):
        self._archive_progression_date.value = value
    
    
    def get_latest_archive_start_time(self):
        return self._latest_archive_start_time.value
    
    
    def set_latest_archive_start_time(self, value):
        self._latest_archive_start_time.value = value
    
    
    def get_latest_archive_execution_time(self):
        return self._latest_archive_execution_time.value
    
    
    def set_latest_archive_execution_time(self, value):
        self._latest_archive_execution_time.value = value
    
    
    def get_previous_archive_start_time(self):
        return self._previous_archive_start_time.value
    
    
    def set_previous_archive_start_time(self, value):
        self._previous_archive_start_time.value = value
    
    
    def get_previous_archive_execution_time(self):
        return self._previous_archive_execution_time.value
    
    
    def set_previous_archive_execution_time(self, value):
        self._previous_archive_execution_time.value = value
    
    
    def get_previous_archive_sla_archived(self):
        return self._previous_archive_sla_archived.value
    
    
    def set_previous_archive_sla_archived(self, value):
        self._previous_archive_sla_archived.value = value
    
    
    # -------------------- Migration
    
    def get_migration_in_progress(self):
        return self._migration_in_progress.value
    
    
    def set_migration_in_progress(self, value):
        self._migration_in_progress.value = value
    
    
    def get_total_sla_to_migrate(self):
        return self._total_sla_to_migrate.value
    
    
    def set_total_sla_to_migrate(self, value):
        self._total_sla_to_migrate.value = value
    
    
    def get_nb_sla_left_to_migrate(self):
        return self._nb_sla_left_to_migrate.value
    
    
    def set_nb_sla_left_to_migrate(self, value):
        self._nb_sla_left_to_migrate.value = value
    
    
    def get_execution_time_last_migration(self):
        return self._execution_time_last_migration.value
    
    
    def set_execution_time_last_migration(self, value):
        self._execution_time_last_migration.value = value
    
    
    # -------------------- Daily clean
    
    def get_daily_clean_in_progress(self):
        return self._daily_clean_in_progress.value
    
    
    def set_daily_clean_in_progress(self, value):
        self._daily_clean_in_progress.value = value
    
    
    def get_total_sla_to_remove(self):
        return self._total_sla_to_remove.value
    
    
    def set_total_sla_to_remove(self, value):
        self._total_sla_to_remove.value = value
    
    
    def get_nb_sla_left_to_remove(self):
        return self._nb_sla_left_to_remove.value
    
    
    def set_nb_sla_left_to_remove(self, value):
        self._nb_sla_left_to_remove.value = value
    
    
    def get_execution_time_last_daily_clean(self):
        return self._execution_time_last_daily_clean.value
    
    
    def set_execution_time_last_daily_clean(self, value):
        self._execution_time_last_daily_clean.value = value
    
    
    def get_total_unique_elements_valid(self):
        return self._total_unique_elements_valid.value
    
    
    def set_total_unique_elements_valid(self, value):
        self._total_unique_elements_valid.value = value


shared_data = SharedData()
