#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019
# This file is part of Shinken Enterprise, all rights reserved.

import logging
import time
import traceback
from collections import namedtuple

from shinken.basesubprocess import QueueHandler
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.modules.base_module.basemodule import ModuleState
from shinken.subprocess_helper.error_handler import ErrorHandler, ERROR_LEVEL

if TYPE_CHECKING:
    from shinken.misc.type_hint import NoReturn, Union
    from shinken.log import PartLogger

Error = namedtuple(u'Error', [u'time', u'level', u'message', u'exception', u'stack'])


class SlaErrorHandler(QueueHandler, ErrorHandler):
    
    def __init__(self, module_name):
        QueueHandler.__init__(self, max_size=100, queue_name=u'sla-error-queue [%s]' % module_name)
        self.internal_state = ModuleState.OK
        self.fatal_error_message = u''
    
    
    def reset(self):
        QueueHandler.reset(self)
        self.internal_state = ModuleState.OK
        self.fatal_error_message = u''
    
    
    def on_add_item(self, item):
        if item.level == ERROR_LEVEL.FATAL:
            self.internal_state = ModuleState.FATAL
            self.fatal_error_message = item.message
        if self.internal_state == ModuleState.FATAL:
            return
        
        if item.level == ERROR_LEVEL.ERROR:
            self.internal_state = ModuleState.CRITICAL
        if self.internal_state == ModuleState.CRITICAL:
            return
        
        if item.level == ERROR_LEVEL.WARNING:
            self.internal_state = ModuleState.WARNING
        if self.internal_state == ModuleState.WARNING:
            return
    
    
    @staticmethod
    def _get_stack():
        return traceback.format_exc().splitlines()
    
    
    def handle_exception(self, message, exception, logger, level=ERROR_LEVEL.ERROR):
        # type: (unicode, Union[unicode,Exception], PartLogger, unicode) -> NoReturn
        
        if level == ERROR_LEVEL.ERROR or level == ERROR_LEVEL.FATAL:
            logger.error(message)
            logger.print_stack()
        
        if level == ERROR_LEVEL.WARNING:
            logger.warning(message)
            logger.print_stack(level=logging.WARNING)
        
        error = Error(time=time.time(), level=level, message=message, exception=str(exception), stack=self._get_stack())
        
        self.put(error)
        # Wait for the get
        time.sleep(2)
    
    
    def get_internal_state(self):
        return self.internal_state
    
    
    def get_errors(self):
        return self.all_items
    
    
    def after_fork_cleanup(self):
        QueueHandler.after_fork_cleanup(self)
