#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019
# This file is part of Shinken Enterprise, all rights reserved.

import time
import uuid

import sla_common
from shinken.ipc.share_item import ShareItem
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.objects.module import Module as ShinkenModuleDefinition
from sla_abstract_component import AbstractComponent, ThreadComponent
from sla_common import LIST_STATUS
from sla_component_manager import ComponentManager
from sla_info import SLAInfo

if TYPE_CHECKING:
    from shinken.misc.type_hint import Number, Optional
    from ..sla_migrator import Migrator
    from .sla_error_handler import SlaErrorHandler


class SLAMigrationConfUpdateNotifier(ShareItem):
    def __init__(self, prefix_name=None):
        # type: (Optional[unicode]) -> None
        if prefix_name:
            prefix_name = u'%s-' % prefix_name
        key_name = u'%s%s' % (prefix_name, type(self).__name__)
        super(SLAMigrationConfUpdateNotifier, self).__init__(key_name=key_name, reinit=False)
    
    
    def get_last_update(self):
        return getattr(self, u'last_update', 0)
    
    
    def set_last_update(self, update_time=None):
        # type: (Optional[Number]) -> None
        if not update_time:
            update_time = int(time.time())
        setattr(self, u'last_update', update_time)


class SLAMigrationConfChecker(ThreadComponent):
    def __init__(self, conf, component_manager, error_handler, migrator, share_item):
        # type: (ShinkenModuleDefinition, ComponentManager, SlaErrorHandler, Migrator, SLAMigrationConfUpdateNotifier) -> None
        super(SLAMigrationConfChecker, self).__init__(conf, component_manager, error_handler, only_one_thread_by_class=False, loop_speed=60, stop_thread_on_error=False)
        self.migrator = migrator
        self.logger = migrator.logger
        self.share_item = share_item
        self.last_update = 0
    
    
    def init(self):
        pass
    
    
    def get_thread_name(self):
        # type: () -> str
        return 'sla-migr-cnf-chk'
    
    
    def loop_turn(self):
        # type: () -> None
        last_update = self.share_item.get_last_update()
        if self.last_update != last_update:
            self.last_update = last_update
            self.migrator.configuration_uuid = uuid.uuid4().hex
            sla_common.shared_data.set_migration_archive_done(False)
            self.logger.info(u'New configuration received. Ask for a recheck of archive version.')


class SLAMigration(AbstractComponent):
    
    def __init__(self, conf, component_manager, sla_info):
        # type: (ShinkenModuleDefinition, ComponentManager,  SLAInfo) -> None
        super(SLAMigration, self).__init__(conf, component_manager)
        self.sla_info = sla_info
    
    
    @staticmethod
    def must_migrate(archive):
        current_archive_version = archive.get('version', -1)
        return sla_common.CURRENT_ARCHIVE_VERSION != current_archive_version
    
    
    def migrate_archive(self, archive_day):
        archive_name = '%s (%s-%s)' % (archive_day.get('uuid', ''), archive_day.get('hname', ''), archive_day.get('sdesc', ''))
        self.logger.debug('[MIGRATION] ' ' -> %s ' 'start migrate archive' % archive_name)
        current_archive_version = archive_day.get('version', -1)
        
        if not self.must_migrate(archive_day):
            return False
        
        if current_archive_version < 1:
            self.logger.debug('[MIGRATION] ' ' -> %s ' 'migration to version 1' % archive_name)
            current_archive_version = 1
            new_archive_day = {
                '_id'    : archive_day['_id'],
                'ranges' : archive_day['ranges'],
                'type'   : archive_day['type'],
                'hname'  : archive_day['hname'],
                'year'   : archive_day['year'],
                'yday'   : archive_day['yday'],
                'version': current_archive_version,
            }
            if archive_day['type'] == 'service':
                new_archive_day['sdesc'] = archive_day['sdesc']
            if archive_day.get('missing', False):
                new_archive_day['missing'] = archive_day['missing']
            _total = 0
            for prefix in LIST_STATUS:
                _sum = archive_day.get('dt_sum_%s' % prefix, 0) + archive_day.get('%s_dt_nack_sum' % prefix, 0) + archive_day.get('%s_dt_ack_sum' % prefix, 0)
                if _sum:
                    new_archive_day['history_dt_%s' % prefix] = _sum
            
            for prefix in LIST_STATUS:
                _sum = archive_day.get('%s_sum' % prefix, 0)
                _total += _sum
                if _sum:
                    new_archive_day['archive_%s' % prefix] = _sum
                    new_archive_day['sla_%s' % prefix] = _sum
                    new_archive_day['history_%s' % prefix] = _sum
            
            new_archive_day['archive_total'] = _total
            new_archive_day['sla_total'] = _total
            new_archive_day['history_total'] = _total
            new_archive_day['archive_format'] = '0000'
            new_archive_day['sla_format'] = '0000'
            new_archive_day['history_format'] = '0000'
            
            archive_day.clear()
            archive_day.update(new_archive_day)
        
        if sla_common.CURRENT_ARCHIVE_VERSION == current_archive_version:
            return True
        
        if current_archive_version < 2:
            self.logger.debug('[MIGRATION] ' ' -> %s ' 'migration to version 2' % archive_name)
            item_uuid = self.sla_info.get_uuid(archive_day['hname'], archive_day.get('sdesc', ''))
            if item_uuid:
                current_archive_version = 2
                archive_day['version'] = current_archive_version
                archive_day['uuid'] = item_uuid
                del archive_day['type']
                del archive_day['hname']
                archive_day.pop('sdesc', None)
            else:
                self.logger.debug('[MIGRATION] ' ' -> %s ' 'migration to version 2 fail. Unable to find uuid for name %s-%s' % (archive_name, archive_day['hname'], archive_day.get('sdesc', '')))
                return False
        
        if sla_common.CURRENT_ARCHIVE_VERSION == current_archive_version:
            return True
        
        if current_archive_version < 3:
            self.logger.debug('[MIGRATION] ' ' -> %s ' 'migration to version 3' % archive_name)
            current_archive_version = 3
            archive_day['version'] = current_archive_version
            archive_day['thresholds'] = self.sla_info.get_sla_thresholds(archive_day['uuid'])
            archive_day['total_ranges'] = len(archive_day['ranges'])
        
        # If element does not exists in SLA_INFO (UUID not found), shouldn't we remove this archive here ?
        return True
    
    
    def init(self):
        pass
    
    
    def tick(self):
        pass
