#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022:
# This file is part of Shinken Enterprise, all rights reserved.


import base64
import httplib
import json
import os
import re
import sys
import time
import uuid

from _ssl import SSLError
from pymongo import MongoClient

from shinken.log import logger
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.modules.base_module.basemodule import BaseModule, SOURCE_STATE

try:
    from shinken.synchronizer.business.source.source import Sources
    from shinken.synchronizer.component.component_manager import component_manager
    from shinken.synchronizer.dao.def_items import METADATA, ITEM_TYPE
except ImportError:
    from synchronizer.business.source.source import Sources
    from synchronizer.component.component_manager import component_manager
    from synchronizer.dao.def_items import METADATA, ITEM_TYPE

if TYPE_CHECKING:
    from shinken.misc.type_hint import Optional
    from synchronizer.synchronizerdaemon import Synchronizer

properties = {
    'daemons': ['synchronizer'],
    'type'   : 'synchronizer-collector-linker',
    'tabs'   : True
}

app = None  # type: Optional[Synchronizer]

NOT_IMPORT_PROPERTIES = ['_SE_UUID_HASH', 'sources', '_id']
ILLEGAL_CHARS = re.compile("""[`~!$%^&*"|'<>?,()=/+]""")
ILLEGAL_CHARS_ADDRESS = re.compile("""[<>&"\'/]""")


class TranslateService(object):
    def __init__(self):
        self.initialized = False
        self.langs_path = ''
        self.langs = {'en': None, 'fr': None}
        self.select_langs = 'en'
    
    
    def init(self, select_lang, trad_files):
        self.select_langs = select_lang
        self.langs_path = trad_files
        self.initialized = True
    
    
    # Translate call
    def _(self, s):
        if not self.initialized:
            raise Exception('TranslateService not initialized')
        # First find lang of the current call, and by default or if it fail take the global lang parameter
        lang = self.select_langs
        
        d = self.langs.get(lang, None)
        # If missing, load the file
        if d is None:
            pth = os.path.join(self.langs_path, lang + '.js')
            if not os.path.exists(pth):
                logger.error('Cannot load the lang file %s: no such file' % pth)
                return ''
            f = open(pth, 'r')
            buf = f.read()
            f.close()
            lines = buf.splitlines()
            new_lines = []
            for line in lines:
                line = line.strip()
                if line.startswith('//'):
                    continue
                if line.startswith('var lang'):
                    line = '{'
                if line.startswith('};'):
                    line = '}'
                new_lines.append(line)
            buf = '\n'.join(new_lines)
            o = json.loads(buf)
            self.langs[lang] = o
        o = self.langs[lang]
        elts = [e.strip() for e in s.split('.') if e.strip()]
        for e in elts:
            o = o.get(e, None)
            if o is None:
                logger.error('Traduction: cannot find %s in the lang %s' % (s, lang))
                return 'TO_TRAD(%s)' % s
        return o


translate_service = TranslateService()


def set_app(_app):
    global app
    app = _app
    translate_service.init(app.lang, os.path.join(app.modules_dir, properties["type"], 'htdocs', 'js', 'traductions'))


# called by the plugin manager to get a module
def get_instance(plugin):
    logger.info("[synchronizer-collector-linker] Get a synchronizer-collector-linker module for plugin %s" % plugin.get_name())
    
    mongodb_uri = plugin.mongodb_uri
    mongodb_database = plugin.mongodb_database
    instance = SynchronizerCollectorLinker(plugin, mongodb_uri, mongodb_database)
    return instance


class SynchronizerCollectorLinker(BaseModule):
    def __init__(self, mod_conf, mongodb_uri, database):
        BaseModule.__init__(self, mod_conf)
        self.listener_port = 7777
        self.synchronizer_address = None
        self.mongodb_uri = mongodb_uri
        self.mongodb_database = database
        self.mongodb_con = self.mongodb_db = None
        self.syncdaemon = None
        self.last_synchronizations = None
        self.remote_hosts = []
        self.tabs = []
        self.source_name = mod_conf.get_name()[7:]
        self.db_conf_name = self.source_name + "_confs"
    
    
    # TODO: en faire une vraie methode dans une classe pour les modules de sources du synchronizer
    def get_my_source(self):
        for source in self.syncdaemon.sources:
            for source_module in source.modules:
                if source_module == self:
                    return source
        return None
    
    
    def init(self):
        logger.info("[synchronizer-collector-linker] Initialization of the synchronizer-collector-linker module")
    
    
    def load(self, daemon):
        self.syncdaemon = daemon
        self.__load_data_backend()
        translate_service.init(self.syncdaemon.lang, os.path.join(self.syncdaemon.modules_dir, properties["type"], 'htdocs', 'js', 'traductions'))
        # This line list all custom tabs which will be displayed in source page
        self.tabs = [{'name': "synchronizers-list", 'displayed_name': translate_service._('common.sync_list_title'), 'content': "synchronizers_list", "warnings": 0, "errors": 0}]
        self.update_tabs()
    
    
    # Give try to load the mongodb and turn until we are killed or the monogo goes up
    def __load_data_backend(self):
        logger.info("[synchronizer-collector-linker] Load data from database")
        start = time.time()
        while not self.interrupted:
            try:
                self.__load_data_backend_try()
                # connect was ok? so bail out :)
                return
            except Exception:
                now = time.time()
                time_out = getattr(self.syncdaemon, 'mongodb_retry_timeout', 60)
                if now > start + time_out:  # more than 1min without mongo? not good!
                    logger.error('[synchronizer-collector-linker] Cannot contact the mongodb server for more than %ss, bailing out'
                                 % time_out)
                    logger.error('[synchronizer-collector-linker] Make sure the MongoDB server is running correctly')
                    sys.exit(2)
                
                # hummer the connection if need
                time.sleep(1)
    
    
    def __load_data_backend_try(self):
        try:
            if self.mongodb_con is not None:
                try:
                    self.mongodb_con.close()
                except Exception, exp:
                    logger.error('[synchronizer-collector-linker] Closing mongodb : %s' % exp)
            self.mongodb_con = MongoClient(self.mongodb_uri, fsync=True)
            # force a fsync connection, so all access are forced even if it's slower on writes
            self.mongodb_db = getattr(self.mongodb_con, self.mongodb_database)
        except Exception, e:
            logger.error("[synchronizer-collector-linker] Mongodb Module: Error %s:" % e)
            raise
        logger.debug('[synchronizer-collector-linker] Connection OK for %s' % self.mongodb_db)
    
    
    def get_number_sync(self):
        nb = self.mongodb_db[self.db_conf_name].find({"enabled": True}).count()
        nb_total = self.mongodb_db[self.db_conf_name].count()
        return nb, nb_total
    
    
    def update_tabs(self):
        nb, nb_total = self.get_number_sync()
        number = '[<span class="shinken-number-active-conf">%d</span>/%d]' % (nb, nb_total)
        self.tabs[0]['displayed_name'] = "%s %s" % (translate_service._('common.sync_list_title'), number)
        self.tabs[0]['errors'] = nb_total - nb
    
    
    def get_listener_headers(self, sync):
        # this couple of login password is define in the listener-shinken default login and password from the synchronizer's listener-shinken module
        login = sync['contact_name']
        password = self.syncdaemon.database_cipher.decipher_value(sync['password'])
        # base64 encode the username and password
        auth = base64.encodestring('%s:%s' % (login, password)).replace('\n', '')
        _listener_headers = {
            'Authorization': "Basic %s" % auth,
            "Content-type" : "application/json",
        }
        return _listener_headers
    
    
    def get_listener_conn(self, address, use_ssl):
        if use_ssl:
            http_conn = httplib.HTTPSConnection(address, self.listener_port, timeout=3)
        else:
            http_conn = httplib.HTTPConnection(address, self.listener_port, timeout=3)
        return http_conn
    
    
    def _get_remote_hosts(self, sync):
        conn = self.get_listener_conn(sync['address'], sync['use_ssl'])
        try:
            conn.request("GET", "/shinken/listener-shinken/v1/get_remote_hosts", headers=self.get_listener_headers(sync))
            response = conn.getresponse()
            if response:
                if response.status != httplib.OK:
                    if response.status == httplib.FORBIDDEN:
                        return {'error': translate_service._('sync.bad_logins')}
            hosts = json.loads(response.read())
            conn.close()
        except SSLError:
            return {'error': translate_service._('sync.https_error')}
        except Exception:
            return {'error': translate_service._('sync.verify_configuration')}
        return hosts
    
    
    def get_objects(self):
        res = {
            'state'   : SOURCE_STATE.OK,
            'output'  : "%s" % translate_service._('common.sync_output'),
            'objects' : {'host': []},
            'errors'  : [],
            'warnings': []
        }
        sync_list = _get_synchronizer_linker_confs(self.db_conf_name, self.mongodb_db)
        if not sync_list:
            # In case no sync are configured, change the status
            res = {
                'state'   : SOURCE_STATE.NOT_CONFIGURED,
                'output'  : translate_service._('sync.output_no_sync'),
                'objects' : {'host': []},
                'errors'  : [],
                'warnings': []
            }
            return res
        # These numbers are useful to determine if we have a critical issue
        number_of_active_sync = 0
        number_of_fail = 0
        sync_linker_db = self.mongodb_db[self.db_conf_name]
        for sync in sync_list:
            conf = sync_linker_db.find_one({'_id': sync['_id']})
            if not sync['enabled']:
                conf['last_import'] = "deactivated"
                conf['last_import_text'] = translate_service._('sync.deactivated')
                sync_linker_db.save(conf)
                continue
            number_of_active_sync += 1
            hosts_lists = self._get_remote_hosts(sync)
            # We will report each problem we encounter
            if 'error' in hosts_lists.keys():
                number_of_fail += 1
                conf['last_import'] = "warning"
                conf['last_import_text'] = hosts_lists['error']
                sync_linker_db.save(conf)
                res['warnings'].append("%s %s" % (translate_service._('sync.access_error') % sync['sync_name'], hosts_lists['error']))
                continue
            # We need to change the uuid and the source
            for host in hosts_lists['data']:
                new_uuid = "%s-%s" % (sync['_suuid'], host['_id'])
                host['_SYNC_KEYS'] = new_uuid.lower()
                host['_SE_UUID'] = 'core-%s-%s' % (ITEM_TYPE.HOSTS, host['_id'])
                if sync['prefix']:
                    host['host_name'] = u"%s-%s" % (sync['prefix'], host['host_name'])
                
                for not_import_property in NOT_IMPORT_PROPERTIES:
                    host.pop(not_import_property, None)
                res['objects']['host'].append(host)
            conf['last_import'] = "ok"
            conf['last_import_text'] = "OK"
            sync_linker_db.save(conf)
        # If we can't communicate with any synchronizer, it's critical
        if number_of_active_sync == 0:
            # In case no sync are configured, change the status
            res = {
                'state'   : SOURCE_STATE.NOT_CONFIGURED,
                'output'  : translate_service._('sync.output_no_sync'),
                'objects' : {'host': []},
                'errors'  : [],
                'warnings': []
            }
            return res
        if number_of_fail == number_of_active_sync:
            res['output'] = translate_service._('sync.output_error_cant_access_sync')
            res['state'] = SOURCE_STATE.CRITICAL
            res['errors'] = res['warnings']
            res['warnings'] = []
        # If we can't communicate to at least 1 synchronizer, it's not critical
        elif number_of_fail != 0:
            res['output'] = translate_service._('sync.output_warning_cant_access_sync').format(number_of_fail)
            res['state'] = SOURCE_STATE.WARNING
        return res
    
    
    def compute_state(self, source):
        source.output = ''
        source.summary_output = ''
        if self.is_all_confs_disabled(source.get_name()):
            source.state = SOURCE_STATE.NOT_CONFIGURED
            source.output = self.get_trad_no_conf_enabled()
        else:
            source.state = SOURCE_STATE.READY_FOR_IMPORT
    
    
    def set_conf_enabled(self, conf_id, enabled):
        enabled = '1' == enabled
        dc = self.mongodb_db[self.db_conf_name].find_one({'_id': conf_id})
        logger.debug('[synchronizer-collector-linker] set_conf_enabled [%s]:[%s]' % (conf_id, enabled))
        if dc:
            dc['enabled'] = enabled
            self.mongodb_db[self.db_conf_name].save(dc)
    
    
    def delete_conf(self, conf_id):
        self.mongodb_db[self.db_conf_name].remove({'_id': conf_id})
    
    
    def save_conf(self, conf_id, new_conf, sname):
        self.mongodb_db[self.db_conf_name].save(new_conf)
        return "ok"
    
    
    def is_all_confs_disabled(self, sname):
        all_conf = self.mongodb_db[self.db_conf_name].find({'source_name': sname, 'enabled': True})
        return all_conf.count() == 0
    
    
    def get_trad_no_conf_enabled(self):
        return translate_service._('sync.output_no_sync')


def _get_synchronizer_linker_confs(db_conf_name, mongodb_db):
    sync_linker_db = mongodb_db.get_collection(db_conf_name)
    sync_linker_confs = list(sync_linker_db.find(sort=[(u'sync_name', 1)]))
    return sync_linker_confs


def _get_own_module(sname):
    return next((mod for mod in app.source_module_instances if mod.name[7:] == sname), None)


def synchronizers_list(sname):
    db_conf_name = sname + "_confs"
    sync_linker_confs = _get_synchronizer_linker_confs(db_conf_name, component_manager.get_mongo_component())
    user = app.get_user_auth()
    return {
        'sync_linker_confs': sync_linker_confs,
        'sname'            : sname,
        'app'              : app,
        'translate_service': translate_service,
        'user'             : user,
    }


def _get_source_data_linker(source_name):
    mongo_component = component_manager.get_mongo_component()
    source = next((s for s in app.sources if s.get_name() == source_name.strip()), None)
    if not source:
        return None
    source.update_from_last_synchronization(mongo_component.col_last_synchronization)
    return source


def new_sync_conf(sname):
    user = app.get_user_auth()
    _app = app
    conf = {
        '_id'      : uuid.uuid1().hex,
        'sync_name': '',
        'enabled'  : True,
        'use_ssl'  : False
    }
    source = _get_source_data_linker(sname)
    
    if source is None:
        return app.abort(404, app._('source.no_source_with_this_name') % sname)
    
    METADATA.update_metadata(conf, METADATA.IN_CREATION, True)
    METADATA.update_metadata(conf, METADATA.ITEM_TYPE, ITEM_TYPE.REMOTE_SYNCHRONIZER)
    
    return {
        'source'           : source,
        'app'              : _app,
        'translate_service': translate_service,
        'user'             : user,
        'item'             : conf,
        'source_name'      : sname,
        'commentary'       : translate_service._('common.connection_warning')
    }


def valide_no_special_char_input_field(field, pattern):
    return not re.search(pattern, field)


def get_remote_sync_uuid(use_ssl, address, contact_name, password):
    to_return = {'error': "", 'suuid': ""}
    if use_ssl:
        conn = httplib.HTTPSConnection(address, "7777", timeout=3)
    else:
        conn = httplib.HTTPConnection(address, "7777", timeout=3)
    try:
        auth = base64.encodestring('%s:%s' % (contact_name, password)).replace('\n', '')
        headers = {
            'Authorization': "Basic %s" % auth,
            "Content-type" : "application/json",
        }
        conn.request("GET", "/shinken/listener-shinken/v1/get_sync_uuid", headers=headers)
        response = conn.getresponse()
        
        if response.status != 200:
            to_return['error'] = translate_service._('common.bad_logins')
            return to_return
        
        data = json.loads(response.read())
        conn.close()
    except SSLError:
        to_return['error'] = translate_service._('common.ssl_error')
        return to_return
    except Exception:
        to_return['error'] = translate_service._('common.error_connection')
        return to_return
    
    to_return['suuid'] = data['data']['suuid']
    return to_return


def post_sync_conf(sname):
    db_conf_name = sname + "_confs"
    col_sync_confs = component_manager.get_mongo_component().get_collection(db_conf_name)
    forms_item = json.loads(app.request.forms['item'])
    _id = forms_item.get('_id', '')
    sync_name = forms_item.get('sync_name', '').strip()
    address = forms_item.get('address', sync_name).strip()
    contact_name = forms_item.get('contact_name', '').strip()
    password = forms_item.get('password', '').strip()
    if app.conf.protect_fields__activate_encryption:
        password = base64.decodestring(password)
    use_ssl = forms_item.get('use_ssl', '0') == '1'
    prefix = forms_item.get('prefix', sync_name).strip()
    notes = forms_item.get('notes', '').strip()
    enabled = forms_item.get('enabled', '1') == '1'
    _warning = ""
    
    if not sync_name:
        app.response.status = 400
        return {'type': 'error', 'message': translate_service._('common.missing_name')}
    else:
        if not valide_no_special_char_input_field(sync_name, ILLEGAL_CHARS):
            app.response.status = 400
            return {'type': 'error', 'message': translate_service._('common.invalid_characters') % (ILLEGAL_CHARS.pattern[1:-1], translate_service._('sync.sync_name'))}
    
    if not address:
        app.response.status = 400
        return {'type': 'error', 'message': translate_service._('common.missing_address')}
    else:
        if not valide_no_special_char_input_field(address, ILLEGAL_CHARS_ADDRESS):
            app.response.status = 400
            return {'type': 'error', 'message': translate_service._('common.invalid_characters') % (ILLEGAL_CHARS_ADDRESS.pattern[1:-1], translate_service._('sync.address'))}
    
    if not contact_name:
        app.response.status = 400
        return {'type': 'error', 'message': translate_service._('common.missing_login')}
    else:
        if not valide_no_special_char_input_field(contact_name, ILLEGAL_CHARS):
            app.response.status = 400
            return {'type': 'error', 'message': translate_service._('common.invalid_characters') % (ILLEGAL_CHARS.pattern[1:-1], translate_service._('sync.contact_name'))}
    
    if not password:
        app.response.status = 400
        return {'type': 'error', 'message': translate_service._('common.missing_password')}
    
    if prefix:
        if not valide_no_special_char_input_field(prefix, ILLEGAL_CHARS):
            app.response.status = 400
            return {'type': 'error', 'message': translate_service._('common.invalid_characters') % (ILLEGAL_CHARS.pattern[1:-1], translate_service._('sync.prefix'))}
    
    conf_name = col_sync_confs.find_one({'source_name': sname, 'sync_name': sync_name})
    conf_prefix = col_sync_confs.find_one({'source_name': sname, 'prefix': prefix})
    conf_address = col_sync_confs.find_one({'source_name': sname, 'address': address})
    if conf_name and conf_name['_id'] != _id:
        app.response.status = 400
        return {'type': 'error', 'message': translate_service._('common.sync_already_exist') % sync_name}
    if conf_prefix and conf_prefix['_id'] != _id:
        app.response.status = 400
        return {'type': 'error', 'message': translate_service._('common.sync_prefix_already_exist') % (prefix, conf_prefix['sync_name'])}
    if conf_address and conf_address['_id'] != _id:
        app.response.status = 400
        return {'type': 'error', 'message': translate_service._('common.sync_address_already_exist') % (address, conf_address['sync_name'])}
    
    conf = col_sync_confs.find_one({'_id': _id})
    if conf is None or conf['address'] != address:
        ret = get_remote_sync_uuid(use_ssl, address, contact_name, password)
        if ret['error']:
            app.response.status = 400
            return {'type': 'error', 'message': ret['error']}
        _suuid = ret['suuid']
    elif enabled:
        ret = get_remote_sync_uuid(use_ssl, address, contact_name, password)
        if ret['error']:
            app.response.status = 400
            _suuid = conf['_suuid']
            _warning = ret['error']
        else:
            _suuid = ret['suuid']
    else:
        _suuid = conf['_suuid']
    
    if _suuid == app._suuid:
        app.response.status = 400
        return {'type': 'error', 'message': translate_service._('common.error_local_address')}
    
    conf_suuid = col_sync_confs.find_one({'source_name': sname, '_suuid': _suuid})
    if conf_suuid and conf_suuid['_id'] != _id:
        app.response.status = 400
        return {'type': 'error', 'message': translate_service._('common.sync_uuid_already_exist') % conf_suuid['sync_name']}
    
    if conf is None:
        conf = {
            '@metadata'       : {'crypted': True} if app.database_cipher.enable else {'uncrypted': True},
            '_id'             : uuid.uuid4().hex,
            'source_name'     : sname,
            'last_import'     : "never-import",
            'last_import_text': translate_service._('sync.never-import')
        }
    else:
        if conf.get('prefix', '') != prefix and conf.get('prefix', '') != '':
            Sources.update_sync_keys(app, sname, conf['prefix'] + '-', prefix + '-')
    
    # Now update it if need
    conf['_suuid'] = _suuid
    conf['sync_name'] = sync_name
    conf['address'] = address
    conf['prefix'] = prefix
    conf['notes'] = notes
    conf['enabled'] = enabled
    conf['password'] = app.database_cipher.cipher_value(password)
    conf['contact_name'] = contact_name
    conf['use_ssl'] = use_ssl
    
    app.save_source_conf_to_backend(sname, conf['_id'], conf)
    
    mod = _get_own_module(sname)
    mod.update_tabs()
    if _warning:
        return {'type': 'save_with_warning', 'message': translate_service._('common.warning_save_connection_fail')}


def _get_source_data(source_name):
    mongo_component = component_manager.get_mongo_component()
    sources = [s for s in app.sources]
    source = next((s for s in sources if s.get_name() == source_name.strip()), None)
    if not source:
        return None
    source.update_from_last_synchronization(mongo_component.col_last_synchronization)
    return source


def get_sync_conf(sname, id):
    if not id:
        return app.abort(404, translate_service._('common.no_synchronizer_with_this_id') % id)
    
    user = app.get_user_auth()
    source = _get_source_data(sname)
    
    if source is None:
        return app.abort(404, app._('source.no_source_with_this_name') % sname)
    
    db_conf_name = sname + "_confs"
    conf = component_manager.get_mongo_component().get_collection(db_conf_name).find_one({'_id': id})
    if conf is None:
        app.abort(404, translate_service._('common.no_synchronizer_with_this_id') % id)
    
    if "password" in conf.keys():
        conf['password'] = app.database_cipher.decipher_value(conf['password'])
        if app.conf.protect_fields__activate_encryption:
            conf['password'] = base64.b64encode(conf['password'])
    
    METADATA.update_metadata(conf, METADATA.IN_CREATION, False)
    METADATA.update_metadata(conf, METADATA.ITEM_TYPE, ITEM_TYPE.REMOTE_SYNCHRONIZER)
    
    return {
        'source'           : source,
        'app'              : app,
        'user'             : user,
        'item'             : conf,
        'source_name'      : sname,
        'translate_service': translate_service,
        'commentary'       : translate_service._('common.connection_warning')
    }


def enable_sync_conf(sname, conf_id):
    enable_bool = app.request.forms['isEnabled'] == 'true'
    logger.debug('[synchronizer-collector-linker] enable_sync_conf [%s-%s]:[%s]' % (sname, conf_id, enable_bool))
    app.set_source_conf_enabled_to_backend(sname, conf_id, enable_bool)
    
    mod = _get_own_module(sname)
    mod.update_tabs()
    return {'isEnabled': enable_bool, 'synchronizer_id': conf_id, 'source_name': sname}


def delete_sync_conf(sname):
    conf_id = app.request.forms.get('_id', '')
    if not conf_id:
        return app.abort(404, translate_service._('common.no_synchronizer_with_this_id') % id)
    
    app.delete_source_conf_to_backend(sname, conf_id)
    
    mod = _get_own_module(sname)
    mod.update_tabs()


pages = {
    synchronizers_list: {'routes': ['/sources/:sname/synchronizers-list/synchronizers_list'], 'view': 'synchronizers_list', 'static': True, 'wrappers': ['json']},
    get_sync_conf     : {'routes': ['/sources/:sname/synchronizer-linker/:id'], 'view': 'source_sync_conf', 'static': True},
    new_sync_conf     : {'routes': ['/sources/:sname/synchronizer-linker/add/'], 'view': 'source_sync_conf', 'static': True},
    post_sync_conf    : {'routes': ['/sources/:sname/synchronizer-linker/save/'], 'method': 'POST', 'view': None, 'static': True, 'wrappers': ['auth', 'transaction']},
    enable_sync_conf  : {'routes': ['/sources/:sname/synchronizer-list/enabled/:conf_id'], 'method': 'POST', 'view': None, 'static': True},
    delete_sync_conf  : {'routes': ['/sources/:sname/synchronizer-linker/delete/'], 'method': 'POST', 'view': None, 'static': True, 'wrappers': ['auth', 'transaction']},
}
