#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2020:
# This file is part of Shinken Enterprise, all rights reserved.

from shinken.misc.type_hint import TYPE_CHECKING

if TYPE_CHECKING:
    from .configuration_component import ConfigurationComponent
    from .logger_component import LoggerComponent
    from .mongo_component import MongoComponent
    from synchronizer.component.abstract_component import AbstractComponent
    from shinken.misc.type_hint import List, Optional


class ComponentManager(object):
    __instance__ = None  # type: ComponentManager
    
    
    def __init__(self):
        self._ready = False
        self._all_components = []  # type: List[AbstractComponent]
        
        self._configuration_component = None  # type: Optional[ConfigurationComponent]
        self._logger_component = None  # type: Optional[LoggerComponent]
        self._mongo_component = None  # type: Optional[MongoComponent]
        
        if ComponentManager.__instance__ is None:
            ComponentManager.__instance__ = self
        else:
            raise Exception('You cannot create another ComponentManager class')
    
    
    @staticmethod
    def get_instance():
        # type: () -> ComponentManager
        if ComponentManager.__instance__ == 'killed':
            raise Exception('Instance is killed !!!')
        if not ComponentManager.__instance__:
            ComponentManager()
        return ComponentManager.__instance__
    
    
    @staticmethod
    def is_ready():
        return ComponentManager.get_instance()._ready
    
    
    @staticmethod
    def kill_instance():
        # type: () -> None
        ComponentManager.__instance__ = 'killed'
    
    
    @staticmethod
    def get_configuration_component():
        # type: () -> ConfigurationComponent
        if not ComponentManager.get_instance()._ready:
            raise Exception('ComponentManager is not ready')
        return ComponentManager.get_instance()._configuration_component
    
    
    @staticmethod
    def get_logger_component():
        # type: () -> LoggerComponent
        if not ComponentManager.get_instance()._ready:
            raise Exception('ComponentManager is not ready')
        return ComponentManager.get_instance()._logger_component
    
    
    @staticmethod
    def get_mongo_component():
        # type: () -> MongoComponent
        if not ComponentManager.get_instance()._ready:
            raise Exception('ComponentManager is not ready')
        return ComponentManager.get_instance()._mongo_component
    
    
    @staticmethod
    def set_component(component_name, component):
        component_manager = ComponentManager.get_instance()
        component_manager._all_components.append(component)
        setattr(component_manager, '_%s' % component_name, component)
    
    
    @staticmethod
    def init():
        component_manager = ComponentManager.get_instance()
        for component in component_manager._all_components:
            component.init()
        
        component_manager._ready = True
