#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2020:
# This file is part of Shinken Enterprise, all rights reserved.

import sys

from pyVmomi import vim, vmodl, VmomiSupport

from shinken.basemodule import SOURCE_STATE
from shinkensolutions.api.synchronizer import ITEM_TYPE
from shinkensolutions.api.synchronizer.source import CollectorModule, DisplayOriginItemFormat, SourceConfiguration, SERVICE_MODE, GenericDictMapperOriginToSource, SourceException, ApiItemProperties, OriginItemDescription
from shinkensolutions.lib_modules.configuration_reader import read_string_in_configuration, read_int_in_configuration

if (sys.version_info[0], sys.version_info[1]) == (2, 6):
    from patch_lib_pyvomi_2_6 import SmartConnectNoSSL
else:
    from patch_lib_pyvomi_2_7 import SmartConnectNoSSL
    
from shinken.log import logger

properties = {
    'daemons': ['synchronizer'],
    'type'   : 'synchronizer_collector_vmware',
    'tabs'   : True,
}

MAX_STEP = 5

dcif_tabular = DisplayOriginItemFormat(DisplayOriginItemFormat.TABULAR)

source_configuration = SourceConfiguration()
source_configuration.set_mapping_origin_to_source(SERVICE_MODE.ON, GenericDictMapperOriginToSource, ['vm', 'esx'])
source_configuration.set_display_origin_item(SERVICE_MODE.ON, dcif_tabular)
source_configuration.set_api_item_properties(SERVICE_MODE.ON)


class MACHINE_TYPE(object):
    VIRTUAL_MACHINE = 'VIRTUAL_MACHINE'
    HOST = 'HOST'


# called by the plugin manager to get a module
def get_instance(plugin):
    return SourceVMWare(plugin, source_configuration)


class VMWareOriginItemDescription(OriginItemDescription):
    def validate(self):
        host_system_properties = PyVmomiFunctions.get_property_list(vim.HostSystem)
        virtual_machine_properties = PyVmomiFunctions.get_property_list(vim.VirtualMachine)
        known_properties = host_system_properties | virtual_machine_properties | set(ShinkenFormatter.EXTRA_PROPERTIES.keys())
        
        properties_description = self.origin_item_properties_description.get(ITEM_TYPE.HOSTS, {}).keys()
        unknown_properties_with_description = [property_description for property_description in properties_description if property_description not in known_properties]
        if unknown_properties_with_description:
            _properties = '<br>'.join(unknown_properties_with_description)
            self.add_warning(self._translator.translate('vmware_error.unknown_key_with_description', self.user_file.path, self.user_other_lang_file.path, _properties))


source_configuration.set_origin_item_description_class(VMWareOriginItemDescription)


class VMWareApiItemProperties(ApiItemProperties):
    def validate(self):
        host_system_properties = PyVmomiFunctions.get_property_list(vim.HostSystem)
        virtual_machine_properties = PyVmomiFunctions.get_property_list(vim.VirtualMachine)
        known_properties = host_system_properties | virtual_machine_properties | set(ShinkenFormatter.EXTRA_PROPERTIES.keys())
        
        to_query_properties = self.properties_allows.get(ITEM_TYPE.HOSTS, [])
        unknown_to_query_properties = [to_query_property for to_query_property in to_query_properties if to_query_property not in known_properties]
        if unknown_to_query_properties:
            _properties = '<br>'.join(unknown_to_query_properties)
            self.add_warning(self._translator.translate('vmware_error.unknown_key_to_query', self.user_file.path, _properties))


source_configuration.set_api_item_properties_class(VMWareApiItemProperties)


class PyVmomiFunctions(object):
    
    @staticmethod
    def get_property_list(vim_type):
        prop_spec = vmodl.query.PropertyCollector.PropertySpec(type=vim_type, all=False)
        
        result = PyVmomiFunctions._get_prop_spec_list(prop_spec)
        result.add('name')
        
        return result
    
    
    @staticmethod
    def _get_prop_spec_list(prop_spec, properties_parent_name=None):
        if properties_parent_name is None:
            properties_parent_name = []
        _return = set()
        for item in prop_spec.type._propList:
            if item.name not in properties_parent_name:
                properties_parent_name_cp = properties_parent_name[:]
                properties_parent_name_cp.append(item.name)
                if isinstance(item.type, VmomiSupport.LazyType) and len(properties_parent_name_cp) <= 5:
                    _return = _return | PyVmomiFunctions._get_prop_spec_list(item, properties_parent_name_cp)
                else:
                    prop_path = '.'.join(properties_parent_name_cp)
                    _return.add(prop_path)
        return _return


class ShinkenFormatter(object):
    EXTRA_PROPERTIES = {
        'shinken.machine_type'                                                        : '',
        'shinken.network'                                                             : 'network',
        'shinken.datastore'                                                           : 'datastore',
        'shinken.guest.disk'                                                          : 'guest.disk',
        'shinken.config.hardware.device.labelSummary'                                 : 'config.hardware.device',
        'shinken.runtime.networkRuntimeInfo.netStackInstanceRuntimeInfo.instanceState': 'runtime.networkRuntimeInfo.netStackInstanceRuntimeInfo',
        'shinken.runtime.powerState'                                                  : 'runtime.powerState',
        'shinken.config.network.vnic.first'                                           : 'config.network.vnic',
        'shinken.config.network.vnic'                                                 : 'config.network.vnic'
    }
    
    
    def format(self, item, property_name, value):
        property_name = property_name.replace('.', '__')
        try:
            result = getattr(self, 'format_%s' % property_name)(value)
        except Exception:
            result = value
        item[property_name] = result
    
    
    @staticmethod
    def format_shinken__machine_type(value):
        return MACHINE_TYPE.VIRTUAL_MACHINE if value == vim.VirtualMachine else MACHINE_TYPE.HOST
    
    
    @staticmethod
    def format_shinken__network(value):
        return ','.join(i.name for i in value)
    
    
    @staticmethod
    def format_shinken__datastore(value):
        datastores_summaries = []
        for datastore in value:
            name = datastore.name
            summary_capacity = datastore.summary.capacity
            summary_type = datastore.summary.type
            summary_free_space = datastore.summary.freeSpace
            summary_url = datastore.summary.url
            summary_uncommitted = datastore.summary.uncommitted
            datastores_summaries.append(
                '(name = %s, summary_capacity = %s, summary_type = %s, summary_freeSpace = %s, summary_url = %s, summary_uncommitted = %s)' % (name, summary_capacity, summary_type, summary_free_space, summary_url, summary_uncommitted))
        return ','.join(i for i in datastores_summaries)
    
    
    @staticmethod
    def format_shinken__guest__disk(value):
        disks_summaries = []
        for disk in value:
            capacity = disk.capacity
            disk_path = disk.diskPath
            free_space = disk.freeSpace
            disks_summaries.append('(capacity = %s, diskPath = %s, freeSpace = %s)' % (capacity, disk_path, free_space))
        return ','.join(i for i in disks_summaries)
    
    
    @staticmethod
    def format_shinken__config__hardware__device__labelSummary(value):
        labels_summaries = []
        for device_object in value:
            label = device_object.deviceInfo.label
            summary = device_object.deviceInfo.summary
            labels_summaries.append('(label = %s, summary = %s)' % (label, summary))
        return ','.join(i for i in labels_summaries)
    
    
    @staticmethod
    def format_shinken__runtime__networkRuntimeInfo__netStackInstanceRuntimeInfo__instanceState(value):
        net_stack_infos = []
        for net_stack_info in value:
            instance = net_stack_info.netStackInstanceKey
            state = net_stack_info.state
            net_stack_infos.append('(instance = %s, state = %s)' % (instance, state))
        return ','.join(i for i in net_stack_infos)
    
    
    @staticmethod
    def format_shinken__runtime__powerState(value):
        return '1' if value == vim.HostSystem.PowerState.poweredOn else '0'
    
    
    @staticmethod
    def format_shinken__config__network__vnic__first(value):
        return value[0].spec.ip.ipAddress
    
    
    @staticmethod
    def format_shinken__config__network__vnic(value):
        ip_address_as_list = []
        for network in value:
            ip_address_as_list.append(network.spec.ip.ipAddress)
        return ip_address_as_list


class SourceVMWare(CollectorModule):
    vmware_server_username = ''
    vmware_server_password = ''
    vmware_server_ip = ''
    vmware_connection_timeout = 60
    cfg_path = ''
    extra_properties_to_build = []
    to_query_properties = []
    
    
    def query_esx_elements(self, smart_connect, root_folder, vim_type):
        container = smart_connect.session.content.viewManager.CreateContainerView(root_folder, [vim_type], True)
        
        # In api_item_properties there are EXTRA_PROPERTIES so we convert this extra_property in vmware property
        properties_list = [ShinkenFormatter.EXTRA_PROPERTIES.get(_property, _property) for _property in self.to_query_properties]
        all_properties_for_vim_type = PyVmomiFunctions.get_property_list(vim_type)
        
        # We filter properties in description file with properties know in api because if we query properties unknown by the api, the item are not fetch.
        properties_to_query = [prop for prop in properties_list if prop in all_properties_for_vim_type]
        
        view = self.collect_data(smart_connect, container, vim_type, properties_to_query)
        
        container.Destroy()
        return view
    
    
    def collect_data(self, smart_connect, container, vim_type, properties_to_query):
        obj_spec = vmodl.query.PropertyCollector.ObjectSpec()
        obj_spec.obj = container
        obj_spec.skip = True
        
        traversal_spec = vmodl.query.PropertyCollector.TraversalSpec()
        traversal_spec.name = 'traverseEntities'
        traversal_spec.path = 'view'
        traversal_spec.skip = False
        traversal_spec.type = container.__class__
        obj_spec.selectSet = [traversal_spec]
        
        property_spec = vmodl.query.PropertyCollector.PropertySpec()
        property_spec.type = vim_type
        if properties_to_query:
            property_spec.pathSet = properties_to_query
        else:
            property_spec.all = True
        
        filter_spec = vmodl.query.PropertyCollector.FilterSpec()
        filter_spec.objectSet = [obj_spec]
        filter_spec.propSet = [property_spec]
        
        api_items = smart_connect.session.content.propertyCollector.RetrieveContents([filter_spec])
        self.source_import_progression.set_step_progress_max(len(api_items))
        items = []
        shinken_formatter = ShinkenFormatter()
        for progress, api_item in enumerate(api_items, 1):
            smart_connect.smartConnectTestConnection(self.logger, self.translator.translate)
            self.source_import_progression.set_step_progress(progress)
            item = {}
            api_item_prop_set = api_item.propSet
            for property_name in self.to_query_properties:
                is_extra_prop = property_name in ShinkenFormatter.EXTRA_PROPERTIES
                if is_extra_prop:
                    if property_name == 'shinken.machine_type':
                        shinken_formatter.format(item, property_name, vim_type)
                        continue
                    api_item_prop_name = ShinkenFormatter.EXTRA_PROPERTIES[property_name]
                else:
                    api_item_prop_name = property_name
                
                api_item_prop = next((_api_item_prop for _api_item_prop in api_item_prop_set if _api_item_prop.name == api_item_prop_name), None)
                if not api_item_prop:
                    continue
                
                shinken_formatter.format(item, property_name, api_item_prop.val)
            
            items.append(item)
        
        return items
    
    
    def load_configuration(self, configuration):
        self.cfg_path = configuration.imported_from.split(':')[0]
        self.vmware_server_ip = read_string_in_configuration(configuration, 'vmware_server_ip', '')
        self.vmware_server_username = read_string_in_configuration(configuration, 'vmware_server_username', '')
        self.vmware_server_password = read_string_in_configuration(configuration, 'vmware_server_password', '')
        self.vmware_connection_timeout = read_int_in_configuration(configuration, 'vmware_connection_timeout', 10)
        self.source_import_progression.set_max_step(MAX_STEP)
    
    
    def import_source_items(self):
        self.logger.info('start importing')
        self.to_query_properties = self.api_item_properties.properties_allows.get(ITEM_TYPE.HOSTS, [])
        self.extra_properties_to_build = [extra_property for extra_property in ShinkenFormatter.EXTRA_PROPERTIES.iterkeys() if extra_property in self.to_query_properties]
        translate = self.translator.translate
        self.source_import_progression.set_current_step(translate('vmware_progress.connection_to_vmware_server'), 1)
        
        if self.vmware_server_ip == '' or self.vmware_server_username == '' or self.vmware_server_password == '':
            empty_champ = []
            if self.vmware_server_ip == '':
                empty_champ.append('vmware_server_ip')
            if self.vmware_server_username == '':
                empty_champ.append('vmware_server_username')
            if self.vmware_server_password == '':
                empty_champ.append('vmware_server_password')
            raise SourceException(translate('vmware_error.get_esx_setting_fail') % (','.join(empty_champ), self.cfg_path), level=SOURCE_STATE.NOT_CONFIGURED)
        
        # Step get all origin items
        try:
            smart_connect = SmartConnectNoSSL(host=self.vmware_server_ip, user=self.vmware_server_username, pwd=self.vmware_server_password, connectionPoolTimeout=self.vmware_connection_timeout, socketTimeout=self.vmware_connection_timeout)
            
            self.logger.info('get list of hosts and vms')
            self.source_import_progression.set_current_step(translate('vmware_progress.query_esx'), 2)
            vmware_esx = self.query_esx_elements(smart_connect, smart_connect.session.content.rootFolder, vim.HostSystem)
            
            self.source_import_progression.set_current_step(translate('vmware_progress.query_vm'), 3)
            vmware_vms = self.query_esx_elements(smart_connect, smart_connect.session.content.rootFolder, vim.VirtualMachine)
        
        except Exception as e:
            raise SourceException(translate('vmware_error.get_esx_element_fail') % (self.vmware_server_ip, e))
        
        # Step map origin -> source
        self.logger.info('cast hosts and vms from pyVmomi format (source lib format) to shinken format : Step 03 map origin item properties in source item')
        self.source_import_progression.set_current_step(translate('vmware_progress.prepare_item_for_shinken'), 4)
        source_items = [self.mapper_origin_to_source.map(ITEM_TYPE.HOSTS, item, None, 'vm') for item in vmware_vms]
        source_items += [self.mapper_origin_to_source.map(ITEM_TYPE.HOSTS, item, None, 'esx') for item in vmware_esx]
        
        self.logger.info('cast hosts and vms from pyVmomi format (source lib format) to shinken format : DONE')
        items_containers = self._add_source_items_to_items_containers(source_items)
        
        self.logger.info('add all vms and hosts in return container')
        self.source_import_progression.set_current_step(translate('vmware_progress.building_of_the_execution_result'), 5)
        return self.prepare_finale_source_result(items_containers)
    
    
    def _add_source_items_to_items_containers(self, source_items):
        items_containers = self.build_items_containers()
        for source_item in source_items:
            items_containers.add_item(ITEM_TYPE.HOSTS, source_item)
        return items_containers
