#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2018:
# This file is part of Shinken Enterprise, all rights reserved.

import time
import json
import urllib

from shinken.basemodule import BaseModule, SOURCE_STATE
from shinken.log import logger
from shinken.synchronizer.dao.def_items import DEF_ITEMS, ITEM_STATE
from shinken.synchronizer.dao.helpers import get_name_from_type

properties = {
    'daemons': ['synchronizer'],
    'type'   : 'syncui-import',
}


# called by the plugin manager to get a module
def get_instance(plugin):
    logger.info("[SyncUI Import] Get a SYNCUI import module for plugin %s" % plugin.get_name())
    
    instance = SyncUI_Import(plugin)
    return instance


class SyncUI_Import(BaseModule):
    def __init__(self, mod_conf):
        BaseModule.__init__(self, mod_conf)
        self.syncdaemon = None
    
    
    # Called by Arbiter to say 'let's prepare yourself guy'
    def init(self):
        logger.info("[SyncUI Import] Initialization of source SyncUI")
    
    
    def load(self, daemon):
        logger.info("[SyncUI Import] Load")
        self.syncdaemon = daemon
    
    
    def get_objects(self):
        logger.info("[SyncUI Import] Asking for objects")
        
        params = urllib.urlencode({
            'private_key': self.syncdaemon.get_private_http_key(),
            'states'     : ','.join([ITEM_STATE.WORKING_AREA, ITEM_STATE.STAGGING]),
        })
        url = "/internal/get_conf?%s" % params
        
        nb_try = 0
        max_try = 5
        objects = {}
        while nb_try < max_try:  # do the try for max_try times
            nb_try += 1
            try:
                conn = self.syncdaemon.get_synchronizer_syncui_connection()
                conn.request("GET", url)
                response = conn.getresponse()
                if response.status != 200:
                    objects = {}
                else:
                    buf = response.read()
                    objects = json.loads(buf)
                conn.close()
                break
            except Exception as exp:
                logger.error('[SyncUI Import] cannot get the Synchronizer objects %s data: "%s" still %d tries' % (url, str(exp), max_try - nb_try))
                time.sleep(nb_try * 2)
        
        ret_objects = {}
        for item_type, item_list in objects.iteritems():
            item_class = DEF_ITEMS[item_type]['table']
            if not ret_objects.get(item_class, None):
                ret_objects[item_class] = []
            for item in item_list:
                _SYNC_KEYS = item.get('_SYNC_KEYS', [])
                if not _SYNC_KEYS:
                    continue
                if '_SE_UUID' not in item:
                    se_uuid = 'core-%s-%s' % (item_type, item['_id'])
                    if se_uuid not in _SYNC_KEYS:
                        _SYNC_KEYS.append(se_uuid)
                    item['_SE_UUID'] = se_uuid
                item['_SYNC_KEYS'] = (','.join(_SYNC_KEYS)).lower()
                item['imported_from'] = 'syncui [ <a href="/elements/%s/%s">%s</a> ]' % (item_type, item['_id'], get_name_from_type(item_type, item))
                ret_objects[item_class].append(item)
        
        res = {
            'state': SOURCE_STATE.OK,
            'output': self.syncdaemon._('import-syncui.output_ok_load_successful'),
            'objects': ret_objects,
            'errors': [],
            'warnings': []
        }
        return res
