#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2009-2022:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

import httplib
import json
import threading
import time
import urllib

from shinken.log import logger, LoggerFactory
from shinken.misc.filter import is_authorized
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.toolbox.url_helper import BaseUrl, ShinkenBaseUrlException
from shinken.util import get_sec_from_morning, get_wday
from shinken.webui.bottlewebui import HTTPError

if TYPE_CHECKING:
    from shinken.misc.type_hint import Callable, Dict, List, Optional, Union
    from shinken.objects.host import Host
    from shinken.objects.service import Service
    from webui.module import WebuiBroker

# ## Will be populated by the UI with it's own value
app = None  # type: Optional[WebuiBroker]
widget_display_name = 'Graphs'
widget_desc = '''Show the perfdata graph'''
default_options = [
    {'name': 'search', 'value': '', 'type': 'hst_AND_srv_AND_met', 'label': 'Metrics', 'required': True},
    {'name': 'show_warning', 'value': False, 'type': 'bool', 'label': 'Show warning thresholds (if available)', 'required': False},
    {'name': 'show_critical', 'value': False, 'type': 'bool', 'label': 'Show critical thresholds (if available)', 'required': False},
    {'name': 'y_min', 'value': '', 'type': 'int', 'label': 'Graph minimum', 'required': False},
    {'name': 'y_max', 'value': '', 'type': 'int', 'label': 'Graph maximum', 'required': False},
]

HTTP_ERROR_GRAPHITE_SERVER_NOT_FOUND = 512
HTTP_ERROR_GRAPHITE = 513

DEFAULT_GRAPHITE_BACKENDS_REALM = u'*'

last_graphite_errors = {}

GRAPHITE_PING_TIMEOUT = 10
GRAPHITE_DATA_TIMEOUT = 20
MIN_TIME_BETWEEN_GRAPHITE_RETRY = 120

metrology_logger = LoggerFactory.get_logger().get_sub_part(u'METROLOGY')


# clean error count oldest than 24h
def _clean_http_graphite_error(_app):
    try:
        hours_since_epoch = int(time.time()) / 3600
        for key in _app.graphs_errors.keys():
            if (int(key) < (hours_since_epoch - 24)) or int(key) > hours_since_epoch:
                del _app.http_errors_count[key]
    except Exception:
        # if an error occurs here, do nothing
        pass


# record errors in app.graphs_errors and count them by hour
def _add_http_graphite_error(_app, error):
    hours_since_epoch = int(time.time()) / 3600
    if hours_since_epoch in _app.graphs_errors:
        _app.graphs_errors[int(hours_since_epoch)] += 1
    else:
        _app.graphs_errors[int(hours_since_epoch)] = 1


# Internal
def _parse_common_graphite_return(data, check_infos):
    # type: (Union[unicode, List], Dict) -> List
    # The data parameter is a Json, no need to parse it
    if isinstance(data, list):
        return data
    
    res = []
    # Now output each line
    try:
        for line in data.splitlines():
            metric_res = {u'trending': False}
            elts = line.split('|')
            meta = elts[0]
            raw_values = elts[1]
            elts = meta.split(',')
            name = elts[0]
            start = int(elts[1])
            end = int(elts[2])
            step = int(elts[3])
            
            metric_res[u'name'] = name
            metric_res[u'host_name'] = check_infos.get(u'host_name', None)
            metric_res[u'check_name'] = check_infos.get(u'check_name', None)
            metric_res[u'mname'] = name.split('.')[-1]
            metric_res[u'metric_name'] = metric_res[u'mname'].replace(u'SLASH', u'/')
            metric_res[u'start'] = start
            metric_res[u'end'] = end
            metric_res[u'has_value'] = False
            metric_res[u'uuid'] = check_infos.get(u'uuid', None)
            metric_res[u'interval'] = step
            
            values = []
            elts = raw_values.split(u',')
            elts.reverse()
            
            pos = end
            last_val = end
            for e in elts:
                if not e.startswith(u'None'):
                    v = float(e)
                    t = [pos * 1000, v]
                    values.append(t)
                    metric_res[u'has_value'] = True
                    last_val = pos
                elif last_val - pos > (check_infos[u'check_interval'] * 2 * 60):
                    values.append([pos * 1000, None])
                pos -= step
            
            values.reverse()
            metric_res[u'values'] = values
            res.append(metric_res)
    except:
        logger.error(u'data from graphite : [%s]' % data.decode(u'utf8'))
        logger.print_stack()
        return app.abort(HTTP_ERROR_GRAPHITE_SERVER_NOT_FOUND, u'broker error see log')
    return res


def _parse_host_graphite_return(data, host_infos):
    _clean_http_graphite_error(app)
    if host_infos.get(u'is_service', True):
        return app.abort(404, u'[ _parse_host_graphite_return ] object %s was not found' % host_infos[u'uuid'])
    logger.debug(u'[ _parse_host_graphite_return ] check time:[%s]' % host_infos[u'check_interval'])
    return _parse_common_graphite_return(data, host_infos)


def _parse_service_graphite_return(data, check_infos):
    _clean_http_graphite_error(app)
    
    if not check_infos.get(u'is_service', False):
        return app.abort(404, u'[ _parse_service_graphite_return ] object %s was not found' % check_infos[u'uuid'])
    logger.debug(u'[_parse_service_graphite_return] check time:[%s]' % check_infos[u'check_interval'])
    return _parse_common_graphite_return(data, check_infos)


def _get_graphite_base_url(current_realm):
    # type: (unicode) -> Optional[BaseUrl]
    _clean_http_graphite_error(app)
    graphite_backends = app.graphite_backends.split(u',')
    backends = []
    for gb in graphite_backends:
        elements = gb.split(u':', 1)
        if len(elements) != 2:
            metrology_logger.error(u'The Graphite backend [ %s ] is not well formatted. It needs at least a realm and a host : <REALM>:<HOSTNAME>' % gb)
            continue
        backends.append((elements[0].strip(), elements[1].strip()))
    
    if not backends:
        metrology_logger.error(u'No valid Graphite backends found')
        return None
    
    url = _find_graphite_backend(backends, current_realm)
    if url is None:
        url = _find_graphite_backend(backends, DEFAULT_GRAPHITE_BACKENDS_REALM)
    
    # In all cases, give back what we can
    return url


def _find_graphite_backend(backends, realm_name):
    # type: (List, unicode) -> Optional[BaseUrl]
    for realm, url in backends:
        if realm == realm_name:
            return BaseUrl.from_url(url)
    return None


def _sort_services(s1, s2):
    return cmp(s1.service_description.lower(), s2.service_description.lower())


def _sort_have_metrics(e1, e2):
    return cmp(e2[u'have_metrics'], e1[u'have_metrics'])


def _get_timezone_offset():
    is_dst = time.daylight and time.localtime().tm_isdst > 0
    return int(time.altzone if is_dst else time.timezone) / 60


# We will try to query the front page of graphite in a small time 5s, and if fail, kill this server
# for a while (60s)
def _get_graphite_connection(base_url):
    # type: (BaseUrl) -> Optional[httplib.HTTPConnection]
    graphite_logger = metrology_logger.get_sub_part(u'graphite:%s' % base_url.get_url())
    now = int(time.time())
    
    min_time_between_graphite_retry = int(getattr(app.myconf, u'metrology__after_error_wait_before_retry', MIN_TIME_BETWEEN_GRAPHITE_RETRY))
    graphite_ping_timeout = int(getattr(app.myconf, u'metrology__ping_timeout', GRAPHITE_PING_TIMEOUT))
    graphite_data_timeout = int(getattr(app.myconf, u'metrology__query_timeout', GRAPHITE_DATA_TIMEOUT))
    
    # If we are too close of a graphite error
    if now < last_graphite_errors.get(base_url.get_url(), 0.0) + min_time_between_graphite_retry:
        return None
    # Try a ping
    ping_conn = httplib.HTTPConnection(host=base_url.get_host(), port=base_url.get_port(), timeout=graphite_ping_timeout)
    try:
        ping_conn.request("GET", "/")  # ping, small static page
        response = ping_conn.getresponse()
        response.read()  # be sure to read, to be sure we did test the whole connection way
        ping_conn.close()
    except Exception:
        last_graphite_errors[base_url.get_url()] = int(time.time())
        graphite_logger.warning(u'Graphite server currently unavailable. Will not try to recontact it until %d seconds' % min_time_between_graphite_retry)
        _add_http_graphite_error(app, HTTP_ERROR_GRAPHITE)
        return None
    
    # We did succeed, but the conn is too low in timeout, need a bigger one to allow
    data_conn = httplib.HTTPConnection(host=base_url.get_host(), port=base_url.get_port(), timeout=graphite_data_timeout)
    return data_conn


def _get_metrics_from_graphite(base_url, graphite_query):
    # type: (BaseUrl, unicode) -> List[Dict]
    conn = _get_graphite_connection(base_url)
    if conn is None:
        return app.abort(HTTP_ERROR_GRAPHITE, u'[ _get_metrics_from_graphite ] Cannot request the graphite server')
    
    logger.debug(u'[ _get_metrics_from_graphite ] target : %s' % graphite_query)
    
    t0 = time.time()
    try:
        # search_by_name: let graphite know we are using uuids directly
        conn.request(u'GET', u'/metrics/find/?query=%s&search_by_name=0' % graphite_query)
        response = conn.getresponse()
        graphite_return = response.read()
        conn.close()
    except Exception as err:
        logger.info(u'GRAPHITE ERROR: %s %s %s' % (err, type(err), time.time() - t0))
        return app.abort(HTTP_ERROR_GRAPHITE_SERVER_NOT_FOUND, err.message)
    
    # graphite return an error
    if response.status == 500:
        # if the data1 contain some string, return some special errors
        if u'[Errno 13] Permission denied' in graphite_return:
            _add_http_graphite_error(app, HTTP_ERROR_GRAPHITE)
            return app.abort(HTTP_ERROR_GRAPHITE, u'[ _get_metrics_from_graphite ] cannot access metrology data, check the /opt/graphite/storage/whisper/ read access for the host %s.' % graphite_query.replace(u'.', u'/'))
        else:
            return app.abort(HTTP_ERROR_GRAPHITE, u'[ _get_metrics_from_graphite ] %s' % graphite_return)
    
    return json.loads(graphite_return)


def _compute_metric_infos(metrics):
    metric_infos = []
    metric_names = map(lambda _metric: _metric[u'text'], metrics)
    for metric in filter(lambda _metric: not (_metric[u'text'].endswith(u'_warn') or _metric[u'text'].endswith(u'_crit')), metrics):
        metric_dict = {}
        name = metric[u'text']
        t = name.split('.')
        if len(t) == 0:
            continue
        mname = t[-1]
        mname = mname.replace(u'SLASH', u'/')
        metric_dict[u'name'] = mname
        metric_dict[u'warn'] = (u'%s_warn' % name) in metric_names
        metric_dict[u'crit'] = (u'%s_crit' % name) in metric_names
        
        metric_infos.append(metric_dict)
    return metric_infos


def _get_cached_item_infos(item, get_html_id=None):
    # type: (Union[Host, Service], Optional[Callable[Union[Host,Service]], unicode]) -> Dict[u'uuid':unicode, u'is_service': bool, u'host_name': unicode, u'check_name': unicode, u'check_interval': int, u'state': unicode, u'realm': unicode, Optional[u'_id': unicode]]
    global app
    if hasattr(item, u'host'):
        is_service = True
        host_name = item.host.get_name()
        check_name = item.get_name()
        realm = item.host.get_realm()
    else:
        is_service = False
        host_name = item.get_name()
        check_name = app._(u'graphics.check_host_alive')
        realm = item.get_realm()
    
    check_infos = {
        u'uuid'          : item.get_instance_uuid(),
        u'is_service'    : is_service,
        u'host_name'     : host_name,
        u'check_name'    : check_name,
        u'check_interval': item.check_interval,
        u'state'         : item.state.lower(),
        u'realm'         : realm
    }
    if get_html_id:
        check_infos.update({u'_id': get_html_id(item)})
    return check_infos


def _get_service_metric_info(check_infos):
    _clean_http_graphite_error(app)
    
    realm_name = check_infos[u'realm']
    try:
        base_url = _get_graphite_base_url(realm_name)
    except ShinkenBaseUrlException as err:
        err_msg = u'The graphite_backend for the realm [ %s ] has errors : %s' % (realm_name, err)
        metrology_logger.error(err_msg)
        return app.abort(HTTP_ERROR_GRAPHITE, err_msg)
    
    if not base_url:
        metrology_logger.error(u'[ _get_host_metric_info ] DETAIL: element=%s not found in graphite server data with current backends=%s' % (check_infos, app.graphite_backends))
        return app.abort(HTTP_ERROR_GRAPHITE_SERVER_NOT_FOUND, u'%s not found in what graphite server datas are.' % check_infos[u'host_name'])
    graphite_query = u'%s.*' % (check_infos[u'uuid'].replace(u'-', u'.'))
    graphite_return = _get_metrics_from_graphite(base_url, graphite_query)
    
    metrics = _parse_service_graphite_return(graphite_return, check_infos)
    
    return _compute_metric_infos(metrics)


def _get_host_metric_info(check_infos):
    _clean_http_graphite_error(app)
    base_url = _get_graphite_base_url(check_infos[u'realm'])
    if not base_url:
        metrology_logger.error(u'[ _get_host_metric_info ] DETAIL: element=%s not found in graphite server data with current backends=%s' % (check_infos, app.graphite_backends))
        return app.abort(HTTP_ERROR_GRAPHITE_SERVER_NOT_FOUND, u'%s not found in what graphite server datas are.' % check_infos[u'host_name'])
    
    graphite_query = u'%s.__HOST__.*' % (check_infos[u'uuid'].replace(u'-', u'.'))
    graphite_return = _get_metrics_from_graphite(base_url, graphite_query)
    
    metrics = _parse_host_graphite_return(graphite_return, check_infos)
    
    return _compute_metric_infos(metrics)


# iframe
def get_graph_json(uuid):
    _clean_http_graphite_error(app)
    user = app.get_user_auth()
    item = app.datamgr.get_host_by_uuid(uuid)
    _item_is_a_service = False
    
    if not item:
        item = app.datamgr.get_service_by_uuid(uuid)
        _item_is_a_service = True
    
    if item is None:
        item = {
            u'uuid'      : uuid,
            u'is_unknown': True
        }
        return app.abort(404, item, True)
    
    if not is_authorized(item, user, app):
        logger.info(u'[ get_graph_json ] User is not authorized to view this element')
        item = {
            u'uuid'      : uuid,
            u'is_unknown': False
        }
        return app.abort(403, item, True)
    check_infos = _get_cached_item_infos(item)
    
    request = app.request
    metric = request.GET.get(u'metric', DEFAULT_GRAPHITE_BACKENDS_REALM)
    _from = request.GET.get(u'from', u'-1d')
    do_trending = (request.GET.get(u'trending', u'0') == u'1')
    
    _from_orig = _from
    # If it's not a -, it's a +, so in the future, so get from few past
    if not _from.startswith(u'-'):
        _from = u'-1h'
    
    # Hook metric / into SLASH
    metric = metric.replace(u'/', u'SLASH')
    _realm_name = item.host.realm if _item_is_a_service else item.realm  # realm string in the host
    
    base_url = _get_graphite_base_url(_realm_name)
    if not base_url:
        # NOTE: only log when we didn't find it, so we don't overload logs
        metrology_logger.error(u'[ _get_metric_info ] DETAIL: element=%s (realm=%s) not found in graphite server data with current backends=%s' % (item.get_full_name(), _realm_name, app.graphite_backends))
        return app.abort(HTTP_ERROR_GRAPHITE_SERVER_NOT_FOUND, u'get_graph_json can not contact graphite', True)
    
    # Regenerator items must not be used when faire lock has been released
    item = None  # noqa: yes not used anymore
    user = None  # noqa: yes not used anymore
    
    app.consumer_lock.release()
    logger.debug(u'[ get_graph_json ] thread[%s] released fair lock before connecting to graphite' % threading.current_thread().ident)
    
    conn = _get_graphite_connection(base_url)
    if conn is None:
        return app.abort(HTTP_ERROR_GRAPHITE, u'[ get_graph_json ] Cannot request the graphite server')
    
    targets = []
    if _item_is_a_service:
        if metric == DEFAULT_GRAPHITE_BACKENDS_REALM:
            targets.append(u'target=%s.%s' % (uuid.replace(u'-', u'.'), metric))
        else:
            for suf in [u'', u'_crit', u'_warn']:
                targets.append(u'target=%s.%s%s' % (uuid.replace(u'-', u'.'), metric, suf))
    else:
        if metric == DEFAULT_GRAPHITE_BACKENDS_REALM:
            targets.append(u'target=%s__HOST__.%s' % (uuid, metric))
        else:
            for suf in [u'', u'_crit', u'_warn']:
                targets.append(u'target=%s.__HOST__.%s%s' % (uuid, metric, suf))
    target = u'&'.join(targets)
    logger.debug(u'[ get_graph_json ] target %s' % target)
    
    try:
        # search_by_name: let graphite know we are using uuids directly
        conn.request(u'GET', u'/render/?%s&rawData=true&from=%s&search_by_name=0' % (target, _from))
        response = conn.getresponse()
        graphite_return = response.read()
        conn.close()
    except:
        return app.abort(HTTP_ERROR_GRAPHITE_SERVER_NOT_FOUND, u'[ get_graph_json ] can not contact graphite', True)
    
    # graphite return an error
    if response.status == 500:
        # if the data1 contain some string, return some special errors
        if u'[Errno 13] Permission denied' in graphite_return:
            logger.error(u'[ get_graph_json ] Error code 513, cannot access metrology data, check the /opt/graphite/storage/whisper/ read access for the host %s.' % uuid)
            _add_http_graphite_error(app, HTTP_ERROR_GRAPHITE)
            return app.abort(HTTP_ERROR_GRAPHITE, u"Graphite don't have access to your whisper storage", True)
    
    if _item_is_a_service:
        datas = _parse_service_graphite_return(graphite_return, check_infos)
    else:
        datas = _parse_host_graphite_return(graphite_return, check_infos)
    
    metrics_to_trend = []
    for data in datas:
        elts = data[u'name'].split(u',', 1)
        name = elts[0]
        elts = name.split(u'.')
        if len(elts) < 3:  # Bad name?
            continue
        host_uuid = elts[0]
        
        metric = elts[-1]
        if metric.endswith(u'_crit') or metric.endswith(u'_warn'):  # not a real metric
            continue
        check_uuid = u'.'.join(elts[1:-1])  # take between
        logger.debug(u'[ get_graph_json ] LOOKING FOR TRENDING FOR %s %s %s' % (host_uuid, check_uuid, metric))
        metrics_to_trend.append((host_uuid, check_uuid, metric, data[u'start'], data[u'end']))
    
    if do_trending:
        logger.debug(u'[ get_graph_json ] do_trending')
        # first look at the module and so the db
        db = None
        trending_collection = None
        for mod in app.modules_manager.get_all_instances():
            if mod.properties[u'type'] != u'trending':
                continue
            # todo : use the MongoClient !
            db = getattr(mod, u'db', None)
            if db:
                trending_collection = getattr(db, u'trending', None)
        if trending_collection is None:
            logger.error(u'[ get_graph_json ] No trending module found')
        else:  # do the job :)
            logger.debug(u'[ get_graph_json ] metrics_to_trend %s ' % metrics_to_trend)
            for (host_uuid, check_uuid, metric, start, end) in metrics_to_trend:
                # maybe we are asking for forecast, if so hack the future :)
                if not _from_orig.startswith('-'):
                    now = int(time.time())
                    start = now - 3600  # minus 1 hour
                    if _from_orig == u'1d':
                        end = now + 86400
                    if _from_orig == u'1w':
                        end = now + 86400 * 7
                    if _from_orig == u'1month':
                        end = now + 86400 * 31
                    if _from_orig == u'1y':
                        end = now + 86400 * 366
                metric_res = {
                    u'name'    : '.'.join([host_uuid, check_uuid, u'%s_trend' % metric]),
                    u'mname'   : u'%s_trend' % metric,
                    u'start'   : start,
                    u'end'     : end,
                    u'trending': True,
                }
                logger.debug(u'[ get_graph_json ] metric_res %s ' % metric_res)
                values = []
                metric_res[u'values'] = values
                chunk_interval = 900
                t = (start / 900) * 900
                # We will got all db entries to do just one query instead of numerous ones
                all_trend = {}
                for e in trending_collection.find({u'hname': host_uuid.replace(u' ', u'_').replace(u'DOT', u'_'), u'sdesc': check_uuid.replace(u' ', u'_'), u'metric': metric, u'cycle': u'week'}):
                    all_trend[e[u'_id']] = e
                while t <= end:
                    # if we got a forecast, look in the future from now
                    if not _from_orig.startswith(u'-'):
                        nb_week_offset = int((t - start) / (86400 * 7))
                    else:
                        nb_week_offset = int((end - t) / (86400 * 7))
                    # but if past, look from the end, so end
                    sec_from_morning = get_sec_from_morning(t)
                    wday = get_wday(t)
                    chunk_nb = sec_from_morning / chunk_interval
                    t += chunk_interval
                    key = u'%s.%s.%s.week.%d.Vtrend.%d' % (host_uuid.replace(u'.', u'DOT').replace(u' ', u'_'), check_uuid.replace(u' ', u'_'), metric, wday, chunk_nb)
                    doc = all_trend.get(key, None)
                    v = None
                    if doc:
                        v = doc[u'VtrendSmooth']
                    # If we got a value, apply the number of week offset
                    if v:
                        if not _from_orig.startswith(u'-'):
                            v += nb_week_offset * doc[u'VevolutionSmooth']
                        else:
                            v -= nb_week_offset * doc[u'VevolutionSmooth']
                    
                    values.append([t * 1000, v])
                datas.append(metric_res)
    
    return json.dumps(datas)


# Widget
def get_host_metric_list(uuid):
    _clean_http_graphite_error(app)
    item = app.datamgr.get_host_by_uuid(uuid)
    if not item:
        logger.debug(u'[ get_host_metric_list ] host not found')
        return app.abort(404, [], True)
    
    realm = item.realm
    host_name = item.get_name()
    checks_infos = []
    for service in item.services:
        checks_infos.append(_get_cached_item_infos(service))
    
    host_infos = None
    if not item.got_business_rule:
        host_infos = _get_cached_item_infos(item)
    
    # Regenerator items are not available when fair lock has been released
    item = None
    
    app.consumer_lock.release()
    logger.debug(u'[ get_host_metric_list ] thread[%s] released fair lock before connecting to graphite' % threading.current_thread().ident)
    
    list_checks = []
    for check_infos in checks_infos:
        try:
            metric_info = _get_service_metric_info(check_infos)
        except HTTPError:
            metric_info = []
        
        data = {
            u'uuid'               : check_infos[u'uuid'],
            u'service_description': check_infos[u'check_name'],
            u'metrics'            : metric_info,
            u'sort_key'           : check_infos[u'check_name']
        }
        list_checks.append(data)
    if host_infos:
        _host_metric_infos = _get_host_metric_info(host_infos)
        data_host = {
            u'uuid'               : host_infos[u'uuid'],
            u'service_description': app._(u'graphics.check_host_alive'),
            u'metrics'            : _host_metric_infos,
            # We want this metric to appear at the top, so we use the first printable character (other than space) at the start of the sork key
            u'sort_key'           : u'!%s' % app._(u'graphics.check_host_alive')
        }
        list_checks.insert(0, data_host)
    
    return json.dumps(list_checks)


def loading_graphs_widget(uuid_dashboard, uuid_widget):
    width = app.request.GET.get('width', '')
    height = app.request.GET.get('height', '')
    metrics = app.request.GET.get('metrics', '')
    graph_min = app.request.GET.get('graph-min', '')
    graph_max = app.request.GET.get('graph-max', '')
    
    param_url = {
        "uuid_dashboard": uuid_dashboard,
        "uuid_widget"   : uuid_widget
    }
    
    param_request = {
        'width'    : width,
        'height'   : height,
        "type"     : "widget",
        "metrics"  : metrics,
        "graph_min": graph_min,
        "graph_max": graph_max
    }
    return {
        'app'                     : app,
        'load'                    : 'api/widget/graphs/page',
        'param_url'               : param_url,
        'offset_time_zone_backend': _get_timezone_offset(),
        'param_request'           : param_request
    }


def get_graphs_widget(uuid_dashboard, uuid_widget):
    width = app.request.GET.get('width', '250')
    height = app.request.GET.get('height', '275')
    
    return {
        'code_http'      : 200,
        'uuid_dashboard' : uuid_dashboard,
        'uuid_widget'    : uuid_widget,
        'widget_width'   : width,
        'widget_height'  : height,
        'colors_graphics': app.colors_graphics,
        'app'            : app,
        'name_elt_info'  : json.dumps({'code_http': 200})
    }


# Panel
def loading_graphs_panel(uuid):
    width = app.request.GET.get('width', '')
    height = app.request.GET.get('height', '')
    metric = app.request.GET.get('metric', '')
    
    if metric:
        metric = urllib.quote_plus(metric)
    
    param_request = {
        'width'     : width,
        'type'      : "detail",
        'height'    : height,
        'metric'    : metric,
        'width_host': app.request.GET.get('width_host', '200')
    }
    
    return {
        'app'                     : app,
        'load'                    : 'api/graphs/panel/detail/page',
        'offset_time_zone_backend': _get_timezone_offset(),
        'param_url'               : uuid,
        'param_request'           : param_request
    }


def get_graphs_panel(uuid):
    _clean_http_graphite_error(app)
    user = app.get_user_auth()
    
    # Look for an host or a service?
    host = None
    sdesc = ""
    if app.is_check_uuid(uuid):
        check = app.datamgr.get_service_by_uuid(uuid)
        if check:
            host = check.host
            sdesc = check.service_description
    else:
        host = app.datamgr.get_host_by_uuid(uuid)
    
    if not host:
        logger.info(u'[ get_graphs_panel ] host not found')
        return {u'app': app, u'code_http': 404, u'uuid': uuid, u'translator': app._}
    
    if not is_authorized(host, user, app):
        logger.info(u'[ get_graphs_panel ] User is not authorized to view this element')
        return {u'app': app, u'code_http': 403, u'uuid': uuid, u'translator': app._}
    
    width = int(app.request.GET.get(u'width', u'500'))
    height = int(app.request.GET.get(u'height', u'571'))
    metric = app.request.GET.get(u'metric', u'')
    
    services = host.services[:]
    services.sort(_sort_services)
    
    data = {
        u'uuid'    : host.uuid,
        u'hname'   : host.get_name(),
        u'services': []
    }
    realm = host.realm
    checks_infos = []
    for service in services:
        checks_infos.append(_get_cached_item_infos(service, app.helper.get_html_id))
    
    host_infos = None
    if not host.got_business_rule:
        host_infos = _get_cached_item_infos(host, app.helper.get_html_id)
    
    # Regenerator items must not be used when fair lock has been released
    services = None
    host = None
    user = None
    
    app.consumer_lock.release()
    logger.debug(u'[ get_graphs_panel ] thread[%s] released fair lock before connecting to graphite' % threading.current_thread().ident)
    
    for check_infos in checks_infos:
        try:
            tmp_metric_info = _get_service_metric_info(check_infos)
        except HTTPError as err:
            logger.info(err.message)
            return {u'app': app, u'code_http': err.status, u'uuid': uuid, u'translator': app._}
        
        data_service = {
            u'sdesc'       : check_infos[u'check_name'],
            u'state'       : check_infos[u'state'],
            u'_id'         : check_infos[u'_id'],
            u'uuid'        : check_infos[u'uuid'],
            u'metrics'     : tmp_metric_info,
            u'have_metrics': len(tmp_metric_info) != 0
        }
        
        data[u'services'].append(data_service)
    data[u'services'].sort(_sort_have_metrics)
    
    if host_infos:
        _host_metric_infos = _get_host_metric_info(host_infos)
        data_host = {
            u'sdesc'       : app._(u'graphics.check_host_alive'),
            u'state'       : host_infos[u'state'],
            u'_id'         : host_infos[u'_id'],
            u'uuid'        : host_infos[u'uuid'],
            u'metrics'     : _host_metric_infos,
            u'have_metrics': len(_host_metric_infos) != 0
        }
        data[u'services'].insert(0, data_host)
    
    return {
        u'translator'     : app._,
        u'http_start_time': app.http_start_time,
        u'show_trending'  : app.show_trending,
        u'width'          : width,
        u'height'         : height,
        u'uuid'           : uuid,
        u'sdesc'          : sdesc,
        u'colors_graphics': app.colors_graphics,
        u'metric_selected': metric,
        u'metric'         : metric,
        u'data'           : data,
        u'code_http'      : 200,
        u'width_host'     : app.request.GET.get('width_host', '200')
    }


def get_graphs_errors():
    return json.dumps(sum(app.graphs_errors.itervalues()))


pages = {
    get_graph_json       : {'routes': ['/api/graphs/data/:uuid'], 'view': None, 'static': False, 'wrappers': ['auth', 'json']},
    
    get_host_metric_list : {'routes': ['/api/widget/graphs/checks-with-metrics/:uuid'], 'view': None, 'static': False, 'wrappers': ['auth', 'json']},
    loading_graphs_widget: {
        'routes'             : ['/api/widget/graphs/loading/:uuid_dashboard/:uuid_widget'],
        'view'               : 'loading_graphs',
        'static'             : True,
        'widget'             : ['dashboard'],
        'widget_display_name': widget_display_name,
        'widget_desc'        : widget_desc,
        'widget_name'        : 'graphs',
        'widget_size'        : {'width': 6, 'height': 4},
        'widget_options'     : default_options,
        'widget_favoritable' : False,
        'resizable'          : False,
        'wrappers'           : []
    },
    get_graphs_widget    : {'routes': ['/api/widget/graphs/page/:uuid_dashboard/:uuid_widget'], 'view': 'widget_graphs', 'static': True},
    
    loading_graphs_panel : {'routes': ['/api/graphs/panel/detail/loading/:uuid'], 'view': 'loading_graphs', 'static': True},
    get_graphs_panel     : {'routes': ['/api/graphs/panel/detail/page/:uuid'], 'view': 'detail_graphs', 'static': True},
    get_graphs_errors    : {'routes': ['/api/graphs/errors'], 'static': True, 'wrappers': []},
    
}
