#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2009-2022:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

import json
import time

from shinken.log import logger
from shinken.misc.filter import is_authorized
from shinken.misc.filter import only_related_to
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.objects.proxyitem import proxyitemsgraph

if TYPE_CHECKING:
    from shinken.misc.type_hint import Optional
    from webui.module import WebuiBroker

app = None  # type: Optional[WebuiBroker]
default_options = [{'name': 'imp_name', 'value': '', 'type': 'host', 'label': 'Select Host/Cluster', 'required': True}]
widget_display_name = '360'
widget_desc = '''Show a 3d view of an object relations
'''


# Sort hosts and services by impact, states and co
def hst_srv_sort(s1, s2):
    if s1.business_impact > s2.business_impact:
        return -1
    if s2.business_impact > s1.business_impact:
        return 1
    # ok, here, same business_impact
    # Compare warn and crit state
    if s1.state_id > s2.state_id:
        return -1
    if s2.state_id > s1.state_id:
        return 1
    # Ok, so by name...
    return s1.get_full_name() > s2.get_full_name()


class Linkator(object):
    def __init__(self):
        pass
    
    
    def get_all_linked_elts(self, elt, levels=3):
        my = set([elt])
        if levels == 0:
            return my
        for i in app.helper.get_all_my_fathers(elt):
            my |= self.get_all_linked_elts(i, levels=levels - 1)
        return my
    
    
    def find_elt_id(self, all_elts, elt):
        try:
            return all_elts.index(elt)
        except ValueError:
            return -1
    
    
    # Need to create a X level higher and lower to the element
    def create_json_dep_graph(self, elt, user, levels=3):
        if elt is None:
            return json.dumps([])
        
        print "START create_json_dep_graph::", elt.get_name()
        t0 = time.time()
        # First we need ALL elements
        all_elts = self.get_all_linked_elts(elt, levels=levels)
        
        print "create_json_dep_graph:: all", len(all_elts)
        all_elts = list(all_elts)
        res = []
        i = 0
        for e in all_elts:
            links = []
            links_name = []
            icon = app.helper.get_tag_icon(app, e)
            authorized = is_authorized(e, user, app)
            # BP rule/cluster got a special picture
            if e.got_business_rule:
                icon = '/static/%d/depgraph/images/cloud.png' % app.http_start_time
            name = e.get_name()
            business_impact = e.business_impact
            _type = 'host'
            if e.__class__.my_type == 'service':
                name = e.host.get_name()
                _type = 'service'
            d = {'name': name, 'id': i, 'icon': icon, 'links': links, 'state': e.state.lower(), 'business_impact': business_impact,
                 'type': _type, 'links_name': links_name, 'authorized': authorized}
            res.append(d)
            
            # also append summry thing
            __d = app.helper.get_summary(e)
            d['status'] = __d['status']
            d['summary'] = __d['summary']
            
            child_dependencies = list(e.child_dependencies)
            # and for cluster, look elsewhere
            if e.got_business_rule:
                fathers = proxyitemsgraph.son_to_fathers[e.get_instance_uuid()]  # warning, not .get() because will be None
                for father_uuid in fathers:
                    father = app.helper.get_elt_from_uuid(father_uuid)
                    if father is not None:
                        child_dependencies.append(father)
            
            for c in child_dependencies:
                idx = []
                if c.__class__.my_type == 'service':
                    idx.append(self.find_elt_id(all_elts, c))
                    
                    if e != c:
                        links_name.append(c.get_full_name())
                else:
                    idx.append(self.find_elt_id(all_elts, c))
                    if e != c:
                        links_name.append(c.get_name())
                for j in idx:
                    # If found and if not us, add it in our links
                    if j != -1 and j != i:
                        links.append(j)
            i += 1
        
        # Merge service into hosts
        new_res = []
        all_hosts = [d for d in res if d['type'] == 'host']
        for d in all_hosts:
            new_res.append(d)
            hname = d['name']
            links = d['links']
            links_name = d['links_name']
            bi = d['business_impact']
            state = d['state']
            this_host_srv = [o for o in res if o['type'] == 'service' and o['name'] == hname]
            # logger.debug('MAP %s => %s' % (hname, this_host_srv))
            # If the host is up, take the worse state available
            if state == 'up':
                for _s in ['unknown', 'warning', 'critical']:
                    _got_any = len([o['state'] for o in this_host_srv if o['state'] == 'unknown']) > 0
                    if _got_any:
                        d['state'] = _s
            for s in this_host_srv:
                # Look at the business impact is we need to
                if s['business_impact'] > bi:
                    d['business_impact'] = s['business_impact']
                # Get the link of the service if possible
                for l in s['links_name']:
                    links_name.append(l)
                    # links_name.extend(o['links_name'])
            links_name = list(set(links_name))
            d['links_name'] = links_name
            # logger.debug('MAP %s Result links_name: %s' % (hname, links_name))
            for ln in links_name:
                # Strip service and directly get hosts
                if '/' in ln:
                    ln = ln.split('/', 1)[0]
                _tmp = [o['id'] for o in res if o['name'] == ln]
                if len(_tmp) > 0:
                    _i = _tmp[0]
                    if _i not in links:
                        links.append(_i)
            links = list(set(links))
        
        # Now factorize all services into their host element
        i = 0
        # Will get previousI-> newI
        map_ids = {}
        for d in new_res:
            prev_id = d['id']
            map_ids[prev_id] = i
            i += 1
        
        # logger.debug("MAPPING %s" % map_ids)
        # logger.debug("NEW RES %s" % new_res)
        # Now our mapping is done, change all d inners
        # so we do not have a spare distribution
        # logger.debug('MAP %s' % new_res)
        for d in new_res:
            new_links = []
            did = map_ids.get(d['id'])
            d['id'] = did
            # logger.debug('MAP %s' % d['name'])
            
            for i in d['links']:
                new_i = map_ids.get(i, None)
                # logger.debug("MAP %s %s" % (i, new_i))
                if new_i is not None:
                    new_links.append(new_i)
            d['links'] = new_links
        
        j = json.dumps(new_res)
        return j


def show_3d(uuid=''):
    # First we look for the user sid
    # so we bail out if it's a false one
    user = app.get_user_auth()
    
    if not user:
        app.bottle.redirect("/")
    
    l = Linkator()
    
    imp_name = ''
    if uuid:
        logger.debug('show_3d uuid %s' % uuid)
        imp_name = app.get_name_from_uuid(uuid)[0]
    
    all_imp_impacts = only_related_to(app.datamgr.get_important_elements(), user, app)
    all_imp_impacts.sort(hst_srv_sort)
    
    if not imp_name and len(all_imp_impacts) > 0:
        imp_name = all_imp_impacts[0].get_full_name()
    
    impact = app.datamgr.get_host(imp_name)
    graph = l.create_json_dep_graph(impact, user, 10)
    
    return {
        'app'       : app,
        'graph'     : graph,
        'impact'    : impact,
        'impacts'   : all_imp_impacts,
        'imp_name'  : imp_name,
        'valid_user': True,
        'user'      : user
    }


def widget_show_3d_(uuid):
    # First we look for the user sid
    # so we bail out if it's a false one
    user = app.get_user_auth()
    
    if not user:
        logger.info('widget_show_3d_ User is not auth')
        return {'app': app, 'code_http': 401, 'uuid': uuid}
    
    l = Linkator()
    
    wid = app.request.GET.get('wid', 'widget_threed_' + str(int(time.time())))
    collapsed = (app.request.GET.get('collapsed', 'False') == 'True')
    
    try:
        imp_name = app.get_name_from_uuid(uuid)[0]
    except:
        return {'app': app, 'code_http': 404, 'uuid': uuid}
    
    options = {'imp_name': {'value': imp_name, 'type': 'hst_srv', 'label': 'Element name'}}
    
    all_imp_impacts = only_related_to(app.datamgr.get_important_elements(), user, app)
    all_imp_impacts.sort(hst_srv_sort)
    
    if not imp_name and len(all_imp_impacts) > 0:
        imp_name = all_imp_impacts[0].get_full_name()
    
    impact = app.datamgr.get_host(imp_name)
    if not impact:
        logger.info('widget_show_3d_ host not found')
        return {'app': app, 'code_http': 404, 'uuid': uuid}
    if not is_authorized(impact, user, app):
        logger.info('widget_show_3d_ User is not authorized to view this element')
        return {'app': app, 'code_http': 403, 'uuid': uuid}
    
    graph = l.create_json_dep_graph(impact, user, 10)
    
    title = '360 view for %s' % imp_name
    
    return {
        'app'          : app,
        'graph'        : graph,
        'impact'       : impact,
        'impacts'      : all_imp_impacts,
        'imp_name'     : imp_name,
        'valid_user'   : True,
        'user'         : user,
        'wid'          : wid,
        'collapsed'    : collapsed,
        'options'      : options,
        'base_url'     : '/widget/360',
        'title'        : title,
        'code_http'    : 200,
        'name_elt_info': json.dumps({'hname': imp_name, 'uuid': uuid, 'code_http': 200})
    }


pages = {
    show_3d        : {'routes': ['/360/:uuid', '/360'], 'view': '360', 'static': True, 'wrappers': []},
    widget_show_3d_: {
        'routes'             : ['/widget/360/:uuid'],
        'view'               : 'widget_360',
        'static'             : True,
        'widget'             : ['dashboard'],
        'widget_display_name': widget_display_name,
        'widget_desc'        : widget_desc,
        'widget_name'        : '360',
        'widget_picture'     : '/static/threed/img/widget_360.png',
        'widget_size'        : {'width': 8, 'height': 6},
        'widget_options'     : default_options,
        'widget_favoritable' : False,
        'resizable'          : False,
        'wrappers'           : []
    },
}
