#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2009-2022:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

import json
import time

from shinken.log import logger
from shinken.misc.filter import is_authorized
from shinken.misc.type_hint import TYPE_CHECKING

if TYPE_CHECKING:
    from shinken.misc.type_hint import Optional
    from webui.module import WebuiBroker

app = None  # type: Optional[WebuiBroker]


def depgraph(uuid):
    # First we look for the user sid
    # so we bail out if it's a false one
    user = app.get_user_auth()
    
    if not user:
        return app.abort(401, 'depgraph_host user is not authenticated', True)
    
    # Ok we are in a detail page but the user ask for a specific search
    search = app.request.GET.get('global_search', None)
    loop = bool(int(app.request.GET.get('loop', '0')))
    loop_time = int(app.request.GET.get('loop_time', '10'))
    levelDistance = int(app.request.GET.get('levelDistance', '125'))
    
    if search:
        new_h = app.datamgr.get_host(search)
        if new_h:
            app.bottle.redirect("/depgraph/" + search)
    
    elt = None
    if app.is_check_uuid(uuid):
        elt = app.datamgr.get_service_by_uuid(uuid)
    else:
        elt = app.datamgr.get_host_by_uuid(uuid)
    return {'app'          : app,
            'elt'          : elt,
            'jgraph'       : create_json_dep_graph(elt, levels=10, user=user),
            'user'         : user,
            'valid_user'   : True,
            'loop'         : loop,
            'loop_time'    : loop_time,
            'levelDistance': levelDistance,
            'deep'         : 10
            }


def get_depgraph_widget(uuid):
    # First we look for the user sid
    # so we bail out if it's a false one
    user = app.get_user_auth()
    if not user:
        logger.info('get_depgraph_widget User is not authenticated')
        return {'app': app, 'code_http': 401, 'uuid': uuid}
    
    levelDistance = int(app.request.GET.get('levelDistance', '145'))
    deep = app.request.GET.get('deep', '5')
    if deep == '' or deep == 'null' or deep == None:
        deep = 5
    else:
        deep = int(deep)
    deep = max(1, deep)
    
    width = int(app.request.GET.get('width', '500'))
    height = int(app.request.GET.get('height', '610'))
    
    back_color = app.request.GET.get('back_color', "#EFECE5")
    
    if not uuid:
        # Ok look for the first host we can find
        hosts = app.datamgr.get_hosts()
        for h in hosts:
            uuid = h.uuid
            break
    
    # Look for an host or a service?
    elt = None
    name = ""
    hname = ""
    sdesc = ""
    if app.is_check_uuid(uuid):
        elt = app.datamgr.get_service_by_uuid(uuid)
        if elt:
            name = "%s/%s" % (elt.host.host_name, elt.service_description)
            hname = elt.host.host_name
            sdesc = elt.service_description
    else:
        elt = app.datamgr.get_host_by_uuid(uuid)
        if elt:
            name = elt.host_name
            hname = elt.host_name
    
    if not elt:
        logger.info('get_depgraph_widget Item not found.')
        return {'app': app, 'code_http': 404, 'uuid': uuid}
    
    if not is_authorized(elt, user, app):
        logger.info('get_depgraph_widget User is not authorized to view this element')
        return {'app': app, 'code_http': 403, 'uuid': uuid}
    
    wid = app.request.GET.get('wid', 'widget_depgraph_' + str(int(time.time())))
    collapsed = (app.request.GET.get('collapsed', 'False') == 'True')
    
    options = {'search': {'value': name, 'type': 'host', 'label': 'Search an element'}, }
    title = 'Relation graph for %s' % name
    
    return {
        'app'          : app,
        'elt'          : elt,
        'user'         : user,
        'wid'          : wid,
        'jgraph'       : create_json_dep_graph(elt, levels=deep, user=user),
        'collapsed'    : collapsed,
        'options'      : options,
        'base_url'     : '/widget/depgraph',
        'title'        : title,
        'levelDistance': levelDistance,
        'deep'         : deep,
        'width'        : width,
        'height'       : height,
        'back_color'   : back_color,
        'code_http'    : 200,
        'hname'        : hname,
        'sdesc'        : sdesc,
        'uuid'         : uuid,
        'name_elt_info': json.dumps({'hname': hname, 'sdesc': sdesc, 'uuid': uuid, 'code_http': 200})
    }


# Need to create a X level higher and lower to the element
def create_json_dep_graph(elt, levels=3, user=None):
    # First we need ALL elements
    all_elts = get_all_linked_elts(elt, levels=levels)
    dicts = get_dep_graph_struct(all_elts, user)
    return json.dumps(dicts)


def refresh_depgraph(uuid):
    user = app.get_user_auth()
    
    levels = int(app.request.GET.get('levels', '10'))
    
    if app.is_check_uuid(uuid):
        elt = app.datamgr.get_service_by_uuid(uuid)
    else:
        elt = app.datamgr.get_host_by_uuid(uuid)
    
    if elt is None:
        return app.abort(404, 'no such element uuid: %s' % uuid)
    
    if not is_authorized(elt, user, app):
        return app.abort(403, 'user is not authorized to view this element : %s' % uuid)
    
    return create_json_dep_graph(elt, levels, user)


# Return all linked elements of this elt, and 2 level
# higher and lower :)
def get_all_linked_elts(elt, levels=3):
    my = set([elt])
    if levels == 0:
        return my
    for i in app.helper.get_all_my_fathers(elt):
        my |= get_all_linked_elts(i, levels=levels - 1)
    return my


def get_dep_graph_struct(all_elts, user):
    res = []
    for elt in all_elts:
        t = elt.__class__.my_type
        
        # We set the values for webui/plugins/depgraph/htdocs/js/eltdeps.js
        # so a node with important data for rendering
        # type = custom, business_impact and img_src.
        __sum = app.helper.get_summary(elt)
        state = elt.state.lower()
        status = __sum['status']
        # lie about cluster in degraded state
        if elt.__class__.my_type == 'host' and elt.got_business_rule:
            if elt.bp_state == 1:
                state = 'warning'
                status = elt.bp_state
            elif elt.bp_state == 3:
                state = 'unknown'
                status = elt.bp_state
        
        authorized = is_authorized(elt, user, app)
        
        d = {
            'id'         : elt.get_dbg_name(),
            'name'       : elt.get_name(),
            
            'data'       : {
                'uuid'                         : elt.uuid,
                '$type'                        : 'custom',
                'business_impact'              : elt.business_impact,
                'img_src'                      : app.helper.get_tag_icon(app, elt, not_founded='hardware'),
                'is_problem'                   : elt.is_problem,
                'got_business_rule'            : elt.got_business_rule,
                'state'                        : state,
                'in_scheduled_downtime'        : elt.in_scheduled_downtime,
                'problem_has_been_acknowledged': elt.problem_has_been_acknowledged,
                'is_flapping'                  : elt.is_flapping,
                'status'                       : status,
                'context'                      : __sum['summary'],
                'authorized'                   : authorized,
                'url'                          : "/hosts/%s/checks/detail/%s-%s" % (elt.host.uuid, elt.host.uuid, elt.uuid) if t == 'service' else "/hosts/%s" % elt.uuid
            },
            'adjacencies': []
        }
        res.append(d)
        
        # Set the right info panel
        business_impact_img = "/images/dollar.png" if elt.business_impact >= 3 else "/img/icons/star.png"
        business_impact_alt = "dollar" if elt.business_impact >= 3 else "start"
        if authorized:
            d['data']['infos-hover'] = r'''%s
            <b class="%s" style="color:black;">
                <img width="20px;" height="20px;" class="depgraph-icon" src="%s"/>
                <small>(since %s)</small>
            </b>
            <div style="float:right;">
                <a href="%s" class="btn hide">
                    <i class="icon-search"></i> Details
                </a>
            </div>''' % \
                                       (('<img src="/static/' + str(app.http_start_time) + business_impact_img + '" alt="' + business_impact_alt + '">') * (elt.business_impact + 1),
                                        state, app.helper.get_icon_state(app.helper.get_small_icon_state(elt)),
                                        app.helper.print_duration(elt.last_state_change, just_duration=True, x_elts=2),
                                        app.helper.get_link_dest(elt))
            d['data']['infos'] = r'''%s
            <b class="%s" style="color:black;">
                <span style='min-width:20px'>
                    <img width="20px;" height="20px;" class="depgraph-icon" src="%s"/>
                </span> <img width="20px;" height="20px;" class="depgraph-icon" src="%s"/>
                <small>(since %s)</small>
                <span style="white-space: pre-wrap;">%s</span>
            </b>
            <div style="float:right;">
                <a href="%s" class="btn hide">
                    <i class="icon-search"></i>
                    Details
                </a>
            </div>''' % \
                                 (('<img src="/static/' + str(app.http_start_time) + business_impact_img + '" alt="' + business_impact_alt + '">') * (elt.business_impact + 1),
                                  elt.state.lower(),
                                  '%s' % app.helper.get_icon_state(__sum['summary'].lower()),
                                  '%s' % app.helper.get_icon_state(state),
                                  app.helper.print_duration(elt.last_state_change, just_duration=True, x_elts=2),
                                  elt.get_full_name(),
                                  app.helper.get_link_dest(elt))
        else:
            d['data']['infos-hover'] = r'''
            <b class="%s" style="color:black;">
                <img width="20px;" height="20px;" class="depgraph-icon" src="%s"/>
                <small>(since %s)</small>
            </b>
            ''' % (
                state,
                app.helper.get_icon_state(app.helper.get_small_icon_state(elt)),
                app.helper.print_duration(elt.last_state_change, just_duration=True, x_elts=2))
            
            d['data']['infos'] = r'''
                <b class="%s" style="color:black;">
                    <span style='min-width:20px'>
                        <img width="20px;" height="20px;" class="depgraph-icon" src="%s"/>
                    </span> <img width="20px;" height="20px;" class="depgraph-icon" src="%s"/>
                    <small>(since %s)</small>
                    <span style="white-space: pre-wrap;">%s</span>
                </b>
                ''' % (
                elt.state.lower(),
                '%s' % app.helper.get_icon_state(__sum['summary'].lower()),
                '%s' % app.helper.get_icon_state(state),
                app.helper.print_duration(elt.last_state_change, just_duration=True, x_elts=2),
                elt.get_full_name())
        
        d['data']['elt_type'] = elt.__class__.my_type
        d['data']['is_problem'] = elt.is_problem
        d['data']['state_id'] = elt.state_id
        
        d['data']['checks'] = {'total': 0}
        if t == 'host':
            svcs = elt.services
            d['data']['checks']['CRITICAL'] = len([_s for _s in svcs if _s.state == 'CRITICAL'])
            d['data']['checks']['WARNING'] = len([_s for _s in svcs if _s.state == 'WARNING'])
            d['data']['checks']['UNKNOWN'] = len([_s for _s in svcs if _s.state == 'UNKNOWN' or _s.state == 'PENDING'])
            d['data']['checks']['OK'] = len([_s for _s in svcs if _s.state == 'OK'])
            d['data']['checks']['total'] = len(svcs)
        
        if elt.state in ['OK', 'UP', 'PENDING']:
            d['data']['circle'] = 'none'
        elif elt.state in ['UNREACHABLE', 'UNKNOWN'] or (elt.got_business_rule and elt.bp_state == 3):
            d['data']['circle'] = 'gray'
        elif elt.state in ['DOWN', 'CRITICAL']:
            d['data']['circle'] = 'red'
        elif elt.state in ['WARNING']:
            d['data']['circle'] = 'orange'
        
        else:
            d['data']['circle'] = 'none'
        # Now put in adj our parents
        for p in app.helper.get_all_my_fathers(elt):
            # if not in the graph skip it :)
            if p not in all_elts:
                continue
            
            # If a link represents a host/host network dependency, represent the line as dashed
            type = "line"
            if t == p.__class__.my_type == 'host' and not elt.got_business_rule and not p.got_business_rule:
                type = "dashed"
            # Ok a basic link with the element and elt so
            pd = {'nodeTo': p.get_dbg_name(),
                  'data'  : {"$type": type, "$direction": [elt.get_dbg_name(), p.get_dbg_name()]}
                  }
            
            p_status = app.helper.get_summary(p)['status']
            if p_status == 1 and (p.__class__.my_type == 'service' or p.got_business_rule):
                pd['data']["$color"] = '#F6A039'
            elif p_status == 0:
                pd['data']["$color"] = 'SeaGreen'
            elif p_status == 3:
                pd['data']["$color"] = 'gray'
            else:
                pd['data']["$color"] = '#EB5E4F'
            d['adjacencies'].append(pd)
    
    for node in res:
        if node['data']['got_business_rule']:
            for adj in node['adjacencies']:
                father = adj['nodeTo']
                for other_node in res:
                    if other_node['id'] == father:
                        other_node['data']['is_in_cluster'] = True
                        break
    
    return res


def get_depgraph_inner(uuid):
    # First we look for the user sid
    # so we bail out if it's a false one
    user = app.get_user_auth()
    
    only_graph = app.request.GET.get('only_graph', '').strip()
    levelDistance = int(app.request.GET.get('levelDistance', '125'))
    
    if not user:
        return app.abort(401, 'get_depgraph_inner user is not authenticated', True)
    
    # Look for an host or a service?
    elt = None
    if app.is_check_uuid(uuid):
        elt = app.datamgr.get_service_by_uuid(uuid)
    else:
        elt = app.datamgr.get_host_by_uuid(uuid)
    
    return {'app'          : app,
            'elt'          : elt,
            'jgraph'       : create_json_dep_graph(elt, levels=10, user=user),
            'user'         : user,
            'only_graph'   : only_graph,
            'levelDistance': levelDistance,
            'deep'         : 10
            }


default_options = [
    {'name': 'search', 'value': '', 'type': 'host', 'label': 'Select an host/cluster as origin', 'required': True},
    {'name': 'deep', 'value': 5, 'type': 'int', 'label': 'SELECT tree depth FROM the origin', 'min': 1, 'max': 10, 'required': True},
]

widget_display_name = 'Dependency Graph'
widget_desc = '''Show a graph of an object relations
'''

pages = {
    refresh_depgraph   : {'routes': ['/refresh_depgraph/:uuid'], 'wrappers': ['auth', 'json']},
    depgraph           : {'routes': ['/depgraph/:uuid'], 'view': 'depgraph', 'static': True},
    get_depgraph_widget: {
        'routes'             : ['/widget/depgraph/:uuid'],
        'view'               : 'widget_depgraph',
        'static'             : True,
        'widget'             : ['dashboard'],
        'widget_display_name': widget_display_name,
        'widget_desc'        : widget_desc,
        'widget_name'        : 'depgraph',
        'widget_picture'     : '/static/depgraph/img/widget_depgraph.png',
        'widget_size'        : {'width': 6, 'height': 8},
        'widget_options'     : default_options,
        'widget_favoritable' : False,
        'resizable'          : False
    },
    get_depgraph_inner : {'routes': ['/inner/depgraph/:uuid'], 'view': 'inner_depgraph', 'static': True},
}
