#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2018:
# This file is part of Shinken Enterprise, all rights reserved.

import copy
from collections import namedtuple

from shinken.misc.type_hint import TYPE_CHECKING
from ..item_controller.work_area_helper import get_item_state, WORK_AREA_INFO_KEY
from ...dao.def_items import ITEM_STATE, HISTORY_ACTION, DEF_ITEMS

if TYPE_CHECKING:
    from shinken.misc.type_hint import NoReturn
    from ...synchronizerdaemon import Synchronizer
    from ...dao.items.contactitem import ContactItem
    from ...dao.def_items import ItemState, ItemType

State = namedtuple(u'State', [u'state', u'wa_bypass'])


class ACTIONS(object):
    DISABLE_OBJECT = u'DISABLE_OBJECT'
    ENABLE_OBJECT = u'ENABLE_OBJECT'
    CLONE_OBJECT = u'CLONE_OBJECT'
    CREATE_FROM_OBJECT = u'CREATE_FROM_OBJECT'
    IMPORT_FROM_SOURCE = u'IMPORT_FROM_SOURCE'
    DELETE_UI_ENTRY = u'DELETE_UI_ENTRY'
    DELETE_ITEM_WORKING_AREA = u'DELETE_ITEM_WORKING_AREA'
    LIST_ELEMENTS = u'LIST_ELEMENTS'
    SAVE_OBJECT = u'SAVE_OBJECT'
    ACCEPT_SUBMIT = u'ACCEPT_SUBMIT'
    REJECT_SUBMIT = u'REJECT_SUBMIT'
    VALIDATE_CHANGES = u'VALIDATE_CHANGES'
    SUBMIT_TO_STAGING = u'SUBMIT_TO_STAGING'
    SAVE_IN_WORK_AREA = u'SAVE_IN_WORK_AREA'
    UNLOCK_WORK_AREA = u'UNLOCK_WORK_AREA'

    BYPASS_WORK_AREA_ACTIONS = (CLONE_OBJECT, CREATE_FROM_OBJECT, ACCEPT_SUBMIT, REJECT_SUBMIT, SUBMIT_TO_STAGING, UNLOCK_WORK_AREA)


DEF_ACTIONS = {
    ACTIONS.DISABLE_OBJECT          : {
        u'transitions': {
            State(ITEM_STATE.WORKING_AREA, False): ITEM_STATE.WORKING_AREA,
            State(ITEM_STATE.STAGGING, False)    : ITEM_STATE.WORKING_AREA,
            State(ITEM_STATE.WORKING_AREA, True) : ITEM_STATE.WORKING_AREA,
            State(ITEM_STATE.STAGGING, True)     : ITEM_STATE.STAGGING,
        }
    },
    ACTIONS.ENABLE_OBJECT           : {
        u'transitions': {
            State(ITEM_STATE.WORKING_AREA, False): ITEM_STATE.WORKING_AREA,
            State(ITEM_STATE.STAGGING, False)    : ITEM_STATE.WORKING_AREA,
            State(ITEM_STATE.WORKING_AREA, True) : ITEM_STATE.WORKING_AREA,
            State(ITEM_STATE.STAGGING, True)     : ITEM_STATE.STAGGING,
        }
    },
    ACTIONS.CLONE_OBJECT            : {
        u'transitions': {
            State(ITEM_STATE.WORKING_AREA, False): ITEM_STATE.WORKING_AREA,
            State(ITEM_STATE.STAGGING, False)    : ITEM_STATE.WORKING_AREA,
            State(ITEM_STATE.WORKING_AREA, True) : ITEM_STATE.STAGGING,
            State(ITEM_STATE.STAGGING, True)     : ITEM_STATE.STAGGING,
        }
    },
    ACTIONS.CREATE_FROM_OBJECT      : {
        u'transitions': {
            State(ITEM_STATE.WORKING_AREA, False): ITEM_STATE.WORKING_AREA,
            State(ITEM_STATE.STAGGING, False)    : ITEM_STATE.WORKING_AREA,
            State(ITEM_STATE.WORKING_AREA, True) : ITEM_STATE.STAGGING,
            State(ITEM_STATE.STAGGING, True)     : ITEM_STATE.STAGGING,
        }
    },
    ACTIONS.IMPORT_FROM_SOURCE      : {
        u'transitions': {
            State(ITEM_STATE.NEW, False): ITEM_STATE.WORKING_AREA,
            State(ITEM_STATE.NEW, True) : ITEM_STATE.STAGGING
        }
    },
    ACTIONS.DELETE_UI_ENTRY         : {
        u'transitions': {
            State(ITEM_STATE.WORKING_AREA, False): ITEM_STATE.WORKING_AREA,
            State(ITEM_STATE.STAGGING, False)    : ITEM_STATE.WORKING_AREA,
            State(ITEM_STATE.WORKING_AREA, True) : ITEM_STATE.STAGGING,
            State(ITEM_STATE.STAGGING, True)     : ITEM_STATE.STAGGING,
        }
    },
    ACTIONS.DELETE_ITEM_WORKING_AREA: {
        u'transitions': {
            State(ITEM_STATE.WORKING_AREA, False): ITEM_STATE.WORKING_AREA,
            State(ITEM_STATE.STAGGING, False)    : ITEM_STATE.WORKING_AREA,
            State(ITEM_STATE.WORKING_AREA, True) : ITEM_STATE.STAGGING,
            State(ITEM_STATE.STAGGING, True)     : ITEM_STATE.STAGGING,
        }
    },
    ACTIONS.LIST_ELEMENTS           : {
        u'transitions': {
            State(ITEM_STATE.WORKING_AREA, False): ITEM_STATE.WORKING_AREA,
            State(ITEM_STATE.STAGGING, False)    : ITEM_STATE.WORKING_AREA,
            State(ITEM_STATE.WORKING_AREA, True) : ITEM_STATE.STAGGING,
            State(ITEM_STATE.STAGGING, True)     : ITEM_STATE.STAGGING,
        }
    },
    ACTIONS.SAVE_OBJECT             : {
        u'transitions': {
            State(ITEM_STATE.WORKING_AREA, False): ITEM_STATE.WORKING_AREA,
            State(ITEM_STATE.STAGGING, False)    : ITEM_STATE.WORKING_AREA,
            State(ITEM_STATE.WORKING_AREA, True) : ITEM_STATE.STAGGING,
            State(ITEM_STATE.STAGGING, True)     : ITEM_STATE.STAGGING,
        }
    },
    ACTIONS.ACCEPT_SUBMIT           : {
        u'transitions': {
            State(ITEM_STATE.WORKING_AREA, False): ITEM_STATE.STAGGING,
            State(ITEM_STATE.STAGGING, False)    : ITEM_STATE.STAGGING,
            State(ITEM_STATE.WORKING_AREA, True) : ITEM_STATE.STAGGING,
            State(ITEM_STATE.STAGGING, True)     : ITEM_STATE.STAGGING,
        }
    },
    ACTIONS.REJECT_SUBMIT           : {
        u'transitions': {
            State(ITEM_STATE.WORKING_AREA, False): ITEM_STATE.WORKING_AREA,
            State(ITEM_STATE.STAGGING, False)    : ITEM_STATE.WORKING_AREA,
            State(ITEM_STATE.WORKING_AREA, True) : ITEM_STATE.WORKING_AREA,
            State(ITEM_STATE.STAGGING, True)     : ITEM_STATE.WORKING_AREA,
        }
    },
    ACTIONS.VALIDATE_CHANGES        : {
        u'transitions': {
            State(ITEM_STATE.WORKING_AREA, False): ITEM_STATE.WORKING_AREA,
            State(ITEM_STATE.STAGGING, False)    : ITEM_STATE.WORKING_AREA,
            State(ITEM_STATE.WORKING_AREA, True) : ITEM_STATE.STAGGING,
            State(ITEM_STATE.STAGGING, True)     : ITEM_STATE.STAGGING,
        }
    },
    ACTIONS.SUBMIT_TO_STAGING       : {
        u'transitions': {
            State(ITEM_STATE.WORKING_AREA, False): ITEM_STATE.WORKING_AREA,
            State(ITEM_STATE.STAGGING, False)    : ITEM_STATE.WORKING_AREA,
            State(ITEM_STATE.WORKING_AREA, True) : ITEM_STATE.WORKING_AREA,
            State(ITEM_STATE.STAGGING, True)     : ITEM_STATE.WORKING_AREA,
        }
    },
    ACTIONS.SAVE_IN_WORK_AREA       : {
        u'transitions': {
            State(ITEM_STATE.WORKING_AREA, False): ITEM_STATE.WORKING_AREA,
            State(ITEM_STATE.STAGGING, False)    : ITEM_STATE.WORKING_AREA,
            State(ITEM_STATE.WORKING_AREA, True) : ITEM_STATE.WORKING_AREA,
            State(ITEM_STATE.STAGGING, True)     : ITEM_STATE.WORKING_AREA,
        }
    },
    ACTIONS.UNLOCK_WORK_AREA        : {
        u'transitions': {
            State(ITEM_STATE.WORKING_AREA, False): ITEM_STATE.WORKING_AREA,
            State(ITEM_STATE.STAGGING, False)    : ITEM_STATE.WORKING_AREA,
            State(ITEM_STATE.WORKING_AREA, True) : ITEM_STATE.WORKING_AREA,
            State(ITEM_STATE.STAGGING, True)     : ITEM_STATE.WORKING_AREA,
        }
    },
}


class EditItemContext(object):
    
    def __init__(self, app, item_id, item_type, action=None, item_state=None, user=None, bypass_work_area=False):
        # type: (Synchronizer, unicode, ItemType, unicode, unicode, ContactItem, bool) -> NoReturn
        
        user = user or app.get_user_auth()
        
        self.item_type = item_type
        self.item_id = item_id
        self.action = action
        self.user = user
        self.is_admin = user.is_admin()
        self.is_expert = user.is_expert()
        self.is_new = (app.request.GET.get(u'pending', u'0') == u'1') or (app.request.GET.get(u'item_state') == u'new')
        self.in_creation = (app.request.GET.get(u'in_creation', u'0') == '1')
        self.bypass_work_area = bypass_work_area or (app.request.GET.get(u'bypass_work_area', u'0') == u'1')
        self.must_submit_to_staging = (app.request.GET.get(u'submit_to_stagging', u'0') == u'1')
        self.prev_elt = None  # it will be computed on _compute_current_state_for_item
        self.from_state = self._compute_current_state_for_item(app)
        self.with_change = (app.request.GET.get(u'with_change', u'0') == u'1')
        self.to_state = self._compute_next_state_for_item()
        self.item_state = item_state if item_state else self.from_state
        self.bypass_check_acl = False  # This will allow avoiding check_acl if we need to push the save without regarding the acl
    
    
    def __str__(self):
        prop_to_print = (
            u'item_type',
            u'item_id',
            u'action',
            u'from_state',
            u'to_state',
            u'user',
            u'is_admin',
            u'is_expert',
            u'is_new',
            u'in_creation',
            u'bypass_work_area',
            u'must_submit_to_staging',
            u'prev_elt',
            u'with_change',
            u'item_state',
            u'bypass_check_acl'
        )
        tpl = u','.join(u'%s:[%%s]' % i for i in prop_to_print)
        values = tuple([getattr(self, i, u'UNSET') for i in prop_to_print])
        return u'EditItemContext[%s]' % (tpl % values)
    
    
    def __repr__(self):
        return self.__str__()
    
    
    # we need to update the work_area_info of the staging object compute a new context with a new action and to state
    # Return a copy of the context with the destination state set to staging
    def update_work_area(self):
        # type: () -> EditItemContext
        staging_context = copy.copy(self)
        staging_context.prev_elt = self.prev_elt
        staging_context.to_state = ITEM_STATE.STAGGING
        staging_context.action = HISTORY_ACTION.UPDATE_WORK_AREA_INFO
        staging_context.with_change = False
        staging_context.bypass_check_acl = True
        return staging_context
    
    
    def _compute_current_state_for_item(self, app):
        # type: (Synchronizer) -> ItemState
        has_work_area = DEF_ITEMS[self.item_type][u'has_work_area']
        if self.is_new:
            states_to_lookup = [ITEM_STATE.NEW, ITEM_STATE.STAGGING]
            if has_work_area:
                states_to_lookup.append(ITEM_STATE.WORKING_AREA)
            items = app.datamanagerV2.find_merge_state_items(where={u'_id': self.item_id}, item_type=self.item_type, item_states=states_to_lookup)
            if items:
                self.prev_elt = items[0]
            
            return app.request.GET.get(u'item_state', ITEM_STATE.NEW)
        
        if not has_work_area:
            self.prev_elt = app.datamanagerV2.find_item_by_id(self.item_id, self.item_type, ITEM_STATE.STAGGING)
            return ITEM_STATE.STAGGING
        
        self.prev_elt = app.datamanagerV2.find_item_by_id(self.item_id, self.item_type, ITEM_STATE.WORKING_AREA)
        if self.prev_elt:
            work_area_info = self.prev_elt.get(WORK_AREA_INFO_KEY, None)
            return get_item_state(work_area_info)
        
        self.prev_elt = app.datamanagerV2.find_item_by_id(self.item_id, self.item_type, ITEM_STATE.STAGGING)
        if self.prev_elt:
            work_area_info = self.prev_elt.get(WORK_AREA_INFO_KEY, None)
            return get_item_state(work_area_info)
        else:
            return ITEM_STATE.WORKING_AREA
    
    
    # Computes the state in which the item will be after the action
    def _compute_next_state_for_item(self):
        # type: () -> ItemState
        if self.is_admin and self.bypass_work_area:
            return ITEM_STATE.STAGGING
        
        if DEF_ITEMS[self.item_type][u'has_work_area']:
            return ITEM_STATE.WORKING_AREA
        
        return ITEM_STATE.STAGGING
