#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2018
# This file is part of Shinken Enterprise, all rights reserved.

import time

from shinken.log import logger
from .exceptions import WorkAreaInfoUnknownStatus
from ...dao.def_items import WORKING_AREA_LAST_ACTION, ITEM_STATE, WORKING_AREA_STATUS
from ...dao.helpers import compute_diff

WORK_AREA_INFO_KEY = 'work_area_info'


def init_work_area(item, user):
    logger.debug(u'[work_area] init_work_area for [%s]' % item[u'_id'])
    work_area_info = {
        u'can_submit'            : True,
        u'status'                : WORKING_AREA_STATUS.WORKING,
        
        u'get_by_user'           : [user[u'_id']],
        u'get_by_user_name'      : [user[u'contact_name']],
        u'get_at'                : time.time(),
        
        u'submit_by_user'        : u'',
        u'submit_by_user_name'   : u'',
        u'submit_at'             : u'',
        u'submit_new_item'       : False,
        u'submit_del_item'       : False,
        
        u'reject_by_user'        : u'',
        u'reject_by_user_name'   : u'',
        u'reject_at'             : u'',
        
        u'validated_by_user'     : u'',
        u'validated_by_user_name': u'',
        u'validated_at'          : u'',
        u'last_action'           : WORKING_AREA_LAST_ACTION.CREATED
    }
    return work_area_info


def reset_work_area(item):
    item.set_value(WORK_AREA_INFO_KEY, {
        'diff_item'             : [],
        'can_submit'            : False,
        'status'                : WORKING_AREA_STATUS.VALIDATED,
        
        'get_by_user'           : [],
        'get_by_user_name'      : [],
        'get_at'                : '',
        
        'submit_by_user'        : '',
        'submit_by_user_name'   : '',
        'submit_at'             : '',
        'submit_del_item'       : False,
        'submit_new_item'       : False,
        
        'reject_by_user'        : '',
        'reject_by_user_name'   : '',
        'reject_at'             : '',
        
        'comments'              : {},
        
        'validated_by_user'     : '',
        'validated_by_user_name': '',
        'validated_at'          : '',
        'last_action'           : ''
    })


def set_in_working_on_work_area_save(item_in_staging, item_type, prev_elt, item_in_work_area, user):
    if prev_elt and WORK_AREA_INFO_KEY in prev_elt and prev_elt[WORK_AREA_INFO_KEY]['status'] != WORKING_AREA_STATUS.VALIDATED:
        can_submit = prev_elt[WORK_AREA_INFO_KEY].get('can_submit', False)
        # User can submit item to staging if it is a new item or it have difference with old item
        can_submit = can_submit or (item_in_staging is None) or (len(compute_diff(item_in_staging, item_in_work_area, item_type)) != 0)
        
        item_in_work_area_info = prev_elt[WORK_AREA_INFO_KEY]
        item_in_work_area_info['status'] = WORKING_AREA_STATUS.WORKING
        item_in_work_area_info['can_submit'] = can_submit
        item_in_work_area[WORK_AREA_INFO_KEY] = item_in_work_area_info
    else:
        work_area_info = init_work_area(item_in_work_area, user)
        item_in_work_area[WORK_AREA_INFO_KEY] = work_area_info
        if item_in_staging:
            work_area_info_staging = item_in_work_area[WORK_AREA_INFO_KEY]
            work_area_info_staging['last_action'] = WORKING_AREA_LAST_ACTION.MODIFIED
            item_in_work_area[WORK_AREA_INFO_KEY] = work_area_info_staging


# SETTINGS COMMON TO A STATUS

def set_commons_submit(comment, user, work_area_info):
    work_area_info['submit_by_user'] = user['_id']
    work_area_info['submit_by_user_name'] = user['contact_name']
    work_area_info['submit_at'] = time.time()
    _comments = work_area_info.get('comments', {})
    _comments['proposed_comment'] = comment
    work_area_info['comments'] = _comments
    work_area_info['can_submit'] = False


def set_commons_rejected(comment, user, work_area_info):
    work_area_info['reject_by_user'] = user['_id']
    work_area_info['reject_by_user_name'] = user['contact_name']
    work_area_info['reject_at'] = time.time()
    _comments = work_area_info.get('comments', {})
    _comments['reject_comment'] = comment
    work_area_info['comments'] = _comments


def set_commons_validated(user, work_area_info):
    work_area_info['validated_by_user'] = user['_id']
    work_area_info['validated_by_user_name'] = user['contact_name']
    work_area_info['validated_at'] = time.time()
    work_area_info['comments'] = {}


# SETTINGS FOR "PROPOSED" STATUS

def set_proposed_created(comment, user, work_area_info):
    set_commons_submit(comment, user, work_area_info)
    work_area_info['submit_new_item'] = True
    work_area_info['submit_del_item'] = False
    work_area_info['status'] = WORKING_AREA_STATUS.PROPOSED


def set_proposed_suppressed(comment, user, work_area_info):
    set_commons_submit(comment, user, work_area_info)
    work_area_info['submit_new_item'] = False
    work_area_info['submit_del_item'] = True
    work_area_info['status'] = WORKING_AREA_STATUS.PROPOSED


def set_proposed_modified(comment, user, work_area_info):
    set_commons_submit(comment, user, work_area_info)
    work_area_info['submit_new_item'] = False
    work_area_info['submit_del_item'] = False
    work_area_info['status'] = WORKING_AREA_STATUS.PROPOSED


# SETTINGS FOR "EDITING" STATUS

def set_editing_suppressed(comment, user, work_area_info):
    work_area_info['can_submit'] = True
    work_area_info['status'] = WORKING_AREA_STATUS.WORKING
    work_area_info['last_action'] = WORKING_AREA_LAST_ACTION.SUPPRESSED
    if user['_id'] not in work_area_info['get_by_user']:
        work_area_info['get_by_user'].append(user['_id'])
    if user['contact_name'] not in work_area_info['get_by_user_name']:
        work_area_info['get_by_user_name'].append(user['contact_name'])


def set_editing_modified(comment, user, work_area_info, work_area_last_action):
    work_area_info['can_submit'] = True
    work_area_info['status'] = WORKING_AREA_STATUS.WORKING
    work_area_info['last_action'] = work_area_last_action
    if user['_id'] not in work_area_info['get_by_user']:
        work_area_info['get_by_user'].append(user['_id'])
    if user['contact_name'] not in work_area_info['get_by_user_name']:
        work_area_info['get_by_user_name'].append(user['contact_name'])


# SETTINGS FOR "VALIDATED" STATUS

def set_validated(user, work_area_info):
    set_commons_validated(user, work_area_info)
    work_area_info['submit_new_item'] = False
    work_area_info['submit_del_item'] = False
    work_area_info['can_submit'] = False
    work_area_info['status'] = WORKING_AREA_STATUS.VALIDATED


def set_validated_modified(user, work_area_info):
    set_commons_validated(user, work_area_info)
    work_area_info['submit_new_item'] = False
    work_area_info['submit_del_item'] = False
    work_area_info['can_submit'] = False
    work_area_info['status'] = WORKING_AREA_STATUS.VALIDATED
    work_area_info['last_action'] = WORKING_AREA_LAST_ACTION.MODIFIED


# SETTINGS FOR "REJECTED" STATUS

def set_rejected(comment, user, work_area_info):
    set_commons_rejected(comment, user, work_area_info)
    work_area_info['can_submit'] = True
    work_area_info['submit_del_item'] = False
    work_area_info['submit_new_item'] = False
    # If the item in working we don't want to set it to validated it keep in working
    current_status = work_area_info['status']
    new_status = WORKING_AREA_STATUS.REJECTED if current_status == WORKING_AREA_STATUS.PROPOSED else current_status
    work_area_info['status'] = new_status
    work_area_info['last_action'] = get_last_action(work_area_info)


# MISC UTILITIES

def clean_diff(work_area_info):
    work_area_info['diff_item'] = []


def clean_work_area_users(work_area_info):
    work_area_info['get_by_user'] = []
    work_area_info['get_by_user_name'] = []


def clean_last_action(work_area_info):
    work_area_info['last_action'] = []


def get_last_action(work_area_info):
    if work_area_info.get('status', WORKING_AREA_STATUS.VALIDATED) == WORKING_AREA_STATUS.VALIDATED:
        return ''
    
    last_action = work_area_info.get('last_action', None)
    
    if last_action:
        return last_action
    
    # For item with work_area_info without last_action create before V02.04.00 RC170
    if work_area_info.get('exist_in', 'working_area') == 'working_area':
        return WORKING_AREA_LAST_ACTION.CREATED
    
    if work_area_info.get('exist_in', 'working_area') == 'stagging' and work_area_info.get('status') != WORKING_AREA_STATUS.WORKING:
        return WORKING_AREA_LAST_ACTION.SUPPRESSED
    
    return WORKING_AREA_LAST_ACTION.MODIFIED


def create_comment(txt, user, on):
    if on not in [WORKING_AREA_STATUS.REJECTED, WORKING_AREA_STATUS.PROPOSED]:
        raise Exception("create_comment: bad on argument, must be in WORKING_AREA_STATUS.REJECTED or WORKING_AREA_STATUS.PROPOSED")
    comment = {'text'     : txt,
               'user_id'  : user['_id'],
               'user_name': user['contact_name'],
               'at'       : int(time.time()),
               'on'       : on,
               }
    logger.debug('[work_area] %s comment by [%s] at [%s] : [%s]' % ({WORKING_AREA_STATUS.REJECTED: 'rejected', WORKING_AREA_STATUS.PROPOSED: 'proposed'}.get(on), comment['user_name'], comment['at'], txt))
    return comment


def get_item_state(work_area_info):
    if not work_area_info:
        return ITEM_STATE.STAGGING
    
    if work_area_info['status'] == WORKING_AREA_STATUS.VALIDATED:
        return ITEM_STATE.STAGGING
    
    last_action = get_last_action(work_area_info)
    
    if last_action == WORKING_AREA_LAST_ACTION.CREATED:
        return ITEM_STATE.WORKING_AREA
    
    if last_action == WORKING_AREA_LAST_ACTION.MODIFIED:
        return ITEM_STATE.WORKING_AREA
    
    if last_action == WORKING_AREA_LAST_ACTION.REJECTED_SUPPRESSED:
        return ITEM_STATE.WORKING_AREA
    
    if last_action == WORKING_AREA_LAST_ACTION.SUPPRESSED:
        return ITEM_STATE.WORKING_AREA
    else:
        text = "Work_area_info: %s ; Unknown value for last_action" % work_area_info
        logger.debug("[get_item_state] : %s ", text)
        raise WorkAreaInfoUnknownStatus(text=text)
