#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022
# This file is part of Shinken Enterprise, all rights reserved.

import threading
import time
import urllib
import uuid
from ctypes import c_bool
from multiprocessing import Value

from shinken.log import logger
from shinken.misc.type_hint import Optional, Dict
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.modules.base_module.basemodule import SOURCE_STATE
from shinken.objects.item import Item, Items
from shinken.property import BoolProp, IntegerProp, StringProp, ListProp, EditableListProp
from shinkensolutions.ssh_mongodb import ASCENDING
from ...dao.datamanagerV2 import get_type_item_from_class, FALLBACK_USER
from ...dao.dataprovider.dataprovider_mongo import DataProviderMongo
from ...dao.def_items import ITEM_STATE, ITEM_TYPE

try:
    from shinken.synchronizer.component.component_manager import component_manager
except ImportError:
    from synchronizer.component.component_manager import component_manager

if TYPE_CHECKING:
    from shinkensolutions.api.synchronizer.source.abstract_module.source_module import SourceModule
    from shinkensolutions.api.synchronizer.source.display_origin_item.display_origin_item_format import DisplayOriginItemFormat
    from shinkensolutions.api.synchronizer.source.display_origin_item.abstract_display_origin_item import AbstractDisplayOriginItem
    from shinkensolutions.api.synchronizer.source.tab.tab_container import TabContainer
    from shinkensolutions.api.synchronizer.source.route.route_container import RouteContainer

if TYPE_CHECKING:
    from .synchronizerdaemon import Synchronizer
    from typing import Optional, List, Dict, Union


class Source(Item):
    id = 1  # zero is always special in database, so we do not take risk here
    my_type = 'source'
    
    # Properties
    if TYPE_CHECKING:
        source_name = u''
        modules = []
        order = 1
        import_interval = 1
        description = u''
        internal = False
        always_enabled = False
        not_stored_properties = []
        last_import_detailed = False
        put_in_staging = False
        put_in_staging_user = u''
    
    properties = Item.properties.copy()
    properties.update({
        'source_name'          : StringProp(),
        'modules'              : EditableListProp(default=''),
        'order'                : IntegerProp(default='1'),
        'import_interval'      : IntegerProp(default='1'),
        'enabled'              : BoolProp(default='1'),
        'description'          : StringProp(default=''),
        'internal'             : BoolProp(default='0'),
        'always_enabled'       : BoolProp(default='0'),
        'not_stored_properties': ListProp(default=''),
        'last_import_detailed' : BoolProp(default='0'),
        'put_in_staging'       : BoolProp(default='0'),
        'put_in_staging_user'  : StringProp(default=''),
    })
    
    running_properties = Item.running_properties.copy()
    running_properties.update({
        'last_import'            : IntegerProp(default=0),
        'objects'                : StringProp(default={}),
        'output'                 : StringProp(default=''),
        'source_import_output'   : StringProp(default=''),
        'summary_output'         : StringProp(default=''),
        'errors'                 : StringProp(default=[]),
        'warnings'               : StringProp(default=[]),
        'merged_warnings'        : StringProp(default=[]),
        'merged_errors'          : StringProp(default=[]),
        'state'                  : StringProp(default=SOURCE_STATE.PENDING),
        'source_import_state'    : StringProp(default=SOURCE_STATE.PENDING),
        'nb_elements'            : IntegerProp(default=0),
        'nb_elements_ok'         : IntegerProp(default=0),
        'nb_elements_warning'    : IntegerProp(default=0),
        'nb_elements_error'      : IntegerProp(default=0),
        'force_import_flag'      : BoolProp(default=False),
        'data_lock'              : StringProp(default=None),
        'current_import_nb'      : IntegerProp(default=0),
        'last_imported_import_nb': IntegerProp(default=0),
        'clean_import'           : BoolProp(default=False),
        'fully_merged'           : BoolProp(default=True),
        'next_import'            : IntegerProp(default=0),
        'already_import'         : BoolProp(default=False),
        'type'                   : StringProp(default='source'),
        'unmerged_new_nb'        : IntegerProp(default=0),
        'merge_ongoing'          : BoolProp(default=False),
        'merge_asked'            : BoolProp(default=False),
        'was_clean'              : BoolProp(default=False),
        'import_state'           : StringProp(default=SOURCE_STATE.PENDING),
        'app'                    : StringProp(default='')  # set in load_config_file of the synchronizer daemon
    })
    macros = {}
    
    
    def __init__(self, params=None, skip_useless_in_configuration=False):
        # Here we set the _enable BEFORE to call super because it will be used in the enabled setter (see @enabled.setter)
        
        if params is None:
            params = {}
        
        # Running properties
        if TYPE_CHECKING:
            self.last_import = 0
            self.objects = {}
            self.output = u''
            self.source_import_output = u''
            self.summary_output = u''
            self.errors = []
            self.warnings = []
            self.merged_warnings = []
            self.merged_errors = []
            self.state = u''
            self.source_import_state = u''
            self.nb_elements = 0
            self.nb_elements_ok = 0
            self.nb_elements_warning = 0
            self.nb_elements_error = 0
            self.force_import_flag = False
            self.data_lock = None  # type: Optional[threading.RLock]
            self.current_import_nb = 0
            self.last_imported_import_nb = 0
            self.clean_import = False
            self.fully_merged = False
            self.next_import = 0
            self.already_import = False
            self.type = u''
            self.unmerged_new_nb = 0
            self.merge_ongoing = False
            self.merge_asked = False
            self.was_clean = False
            self.import_state = 0
            self.app = None  # type: Optional[Synchronizer]
            self.saved = {}
            
            # Properties defined elsewhere...
            self.configuration_errors = []  # type: List[unicode]
        
        self._enable = Value(c_bool, True)
        self.modules = []
        self.last_import = 0
        self.state = SOURCE_STATE.PENDING
        self.was_clean = False
        self.output = ''
        self.summary_output = ''
        self.errors = []
        self.warnings = []
        self.merged_warnings = []
        self.merged_errors = []
        self.current_import_nb = 0
        self.next_import = 0
        self.nb_elements = 0
        self.nb_elements_ok = 0
        self.nb_elements_warning = 0
        self.nb_elements_error = 0
        self.already_import = False
        self.source_import_output = ''
        self.source_import_state = ''
        self.import_state = ''
        self.force_import_flag = False
        self.objects = []
        self.configuration_errors = []
        self.clean_import = False
        self.import_interval = 0
        self.put_in_staging = False
        self.app = None
        self.put_in_staging_user = ''
        self.always_enabled = False
        self.source_name = ''
        self.last_imported_import_nb = 0
        self.fully_merged = False
        self.data_lock = None
        self.order = 0
        self.not_stored_properties = []
        self.internal = False
        self.saved = {}
        
        super(Source, self).__init__(params or {})
        
        self.skipped_conf_keys = ['skipped_conf_keys', 'view_conf', 'data_backend', 'saved', 'id', 'customs', '_enable', '_is_correct__is_error_collected']
        self.view_conf = []
        
        for key in self.__dict__.keys():
            if key in self.properties.keys() or key in self.running_properties.keys() or key in self.skipped_conf_keys:
                continue
            self.view_conf.append(key)
    
    
    def get_module(self):
        # type: () -> SourceModule
        return self.modules[0]
    
    
    def get_tabs(self):
        return getattr(self.get_module(), 'tabs', [])
    
    
    def get_tab_container(self):
        # type: () -> TabContainer
        return getattr(self.get_module(), 'tab_container', None)
    
    
    def get_route_container(self):
        # type: () -> RouteContainer
        return getattr(self.get_module(), 'route_container', None)
    
    
    def build_origin_item_info(self, origin_item_info):
        # type: (Dict) -> Optional[AbstractDisplayOriginItem]
        get_display_origin_item_format = getattr(self.get_module(), 'get_display_origin_item_format', None)
        if not get_display_origin_item_format or not get_display_origin_item_format() or not origin_item_info:
            return None
        display_origin_item_format = get_display_origin_item_format()  # type: DisplayOriginItemFormat
        display_origin_item_class = display_origin_item_format.display_origin_item_class
        return display_origin_item_class.from_json(self.get_module().logger, self.get_module().translator, self.get_module(), origin_item_info)
    
    
    @property
    def enabled(self):
        return self._enable.value
    
    
    @enabled.setter
    def enabled(self, value):
        self._enable.value = Source.properties['enabled'].pythonize(value)
    
    
    def get_name(self):
        return getattr(self, 'source_name', '')
    
    
    # The synchronizer have retention data for us, thanks
    def restore_retention_data(self, data):
        self.last_import = data.get('synchronization_time', 0)
        self.state = data.get('state', SOURCE_STATE.PENDING)
        self.was_clean = data.get('was_clean', False)
        self.output = data.get('output', '')
        self.summary_output = data.get('summary_output', '')
        self.errors = data.get('errors', [])
        self.warnings = data.get('warnings', [])
        self.merged_warnings = data.get('merged_warnings', [])
        self.merged_errors = data.get('merged_errors', [])
        self.current_import_nb = data.get('current_import_nb', 0)
        self.next_import = data.get('next_import', 0)
        self.nb_elements = data.get('nb_elements', 0)
        self.nb_elements_ok = data.get('nb_elements_ok', 0)
        self.nb_elements_warning = data.get('nb_elements_warning', 0)
        self.nb_elements_error = data.get('nb_elements_error', 0)
        
        if self.last_import != 0:
            self.already_import = True
        if self.state == SOURCE_STATE.RUNNING:  # was saved during a run, get back to pending
            self.state = SOURCE_STATE.PENDING
        if self.next_import == -1:  # same for next import
            self.next_import = 0
        if self.nb_elements_error or self.nb_elements_warning:
            self.state = SOURCE_STATE.WARNING
        else:
            self.summary_output = ''
        
        self.source_import_output = data.get('source_import_output', self.output)
        self.source_import_state = data.get('source_import_state', self.state)
    
    
    # Someone asks me for running NOW
    def set_force_import(self):
        # If not enabled, I don't care about force or not
        if not self.enabled:
            return self.state
        # if already set, exit
        if self.force_import_flag:
            return self.state
        self.force_import_flag = True
    
    
    # Someone asks me to clean my result from staging
    def set_clean_import(self):
        # If not enabled, I don't care about force or not
        if not self.enabled:
            return self.state
        
        self.objects = {}
        self.nb_elements = 0
        self.nb_elements_ok = 0
        self.nb_elements_warning = 0
        self.nb_elements_error = 0
        
        self.clean_import = True
        self.force_import_flag = True
    
    
    def is_import_in_progress(self):
        return self.state == SOURCE_STATE.RUNNING or not self.fully_merged
    
    
    def need_import(self):
        if not self.enabled:
            return False
        
        # Maybe our current import is not merged, so do not need to run it again until we load the previous run
        if not self.fully_merged:
            return False
        
        # Maybe it's already running, if so not need:)
        if self.state == SOURCE_STATE.RUNNING:
            return False
        
        # If set, consume our import flag
        if self.force_import_flag:
            return True
        
        # Maybe the normal import scheduled is disabled :)
        if self.import_interval == 0:
            return False
        
        # We are not ready to run
        if self.next_import == -1:
            return False
        
        # Ok so now look at the time :)
        now = int(time.time())
        r = now > self.next_import
        return r
    
    
    def _empty_source(self, source_state, source_output):
        # type: (unicode, unicode) -> None
        self.import_state = source_state
        self.source_import_state = self.import_state
        
        self.output = source_output
        self.summary_output = source_output
        self.source_import_output = source_output
        
        self.objects = {}
        
        self.errors = []
        self.warnings = []
        self.merged_warnings = []
        self.merged_errors = []
        
        self.nb_elements = 0
        self.nb_elements_ok = 0
        self.nb_elements_warning = 0
        self.nb_elements_error = 0
        self.next_import = 60 * self.import_interval if self.import_interval else 0
    
    
    def set_merged(self, import_nb):
        self.last_imported_import_nb = import_nb
        # logger.info('[DEBUG SOURCE] [%s] merge tell source import is done. It merges import [%s] and last import to merge is [%s] so the source is fully_merged:[%s]' % (self.get_name(), import_nb, self.current_import_nb, self.last_imported_import_nb >= self.current_import_nb))
        if self.last_imported_import_nb >= self.current_import_nb:
            self.fully_merged = True
    
    
    def hook_post_merged(self):
        if self.put_in_staging:
            logger.info('The source %s must put in staging this data' % self.get_name())
            info = self.app.call_internal_put_source_in_staging(self.put_in_staging_user or FALLBACK_USER['contact_name'], self.source_name)
            
            text_import_ok = '<br>%s' % self.app.t('source.import_in_staging_OK')
            text_import_ko = '<br>%s' % self.app.t('source.import_in_staging_fail') % info['msg']
            self.output = self.output.replace(text_import_ok, '')
            self.output = self.output.replace(text_import_ko, '')
            
            if info['code'] == 200:
                self.output += text_import_ok
            else:
                self.state = SOURCE_STATE.WARNING
                self.output += text_import_ko
    
    
    def get_data_lock(self):
        # If not exiting, create it
        if self.data_lock is None:
            self.data_lock = threading.RLock()
        return self.data_lock
    
    
    # Is the source enabled or not, and when changed, switch the other properties
    # Return if we did change or not (because caller must to things if we do)
    def set_enabled(self, b):
        # an always enabled source can't be disabled
        if self.always_enabled:
            return False
        
        if isinstance(b, basestring):
            b = b == '1'
        
        # No change, ... no change
        
        if b == self.enabled:
            return False
        
        # Ok we will change ^^
        self.enabled = b
        
        if self.enabled:
            # also force a source current_import_nb increase to show that this source will be merged if needed
            self.current_import_nb += 1
            # We need to be taken into account for the merge
            self.fully_merged = False
        else:
            self.fully_merged = True
        
        return True
    
    
    # Set a specific SUB conf enabled or not, like a specific discovery
    def compute_state(self):
        mongo_component = component_manager.get_mongo_component()
        if hasattr(self.get_module(), 'compute_state'):
            self.get_module().compute_state(self)
        else:
            self.update_from_last_synchronization(mongo_component.col_last_synchronization)
    
    
    def do_after_fork(self):
        if hasattr(self.get_module(), 'do_after_fork'):
            self.get_module().do_after_fork()
    
    
    def set_conf_enabled(self, conf_id, enabled):
        if hasattr(self.get_module(), 'set_conf_enabled'):
            self.get_module().set_conf_enabled(conf_id, enabled)
        self.compute_state()
    
    
    def delete_conf(self, conf_id):
        if hasattr(self.get_module(), 'delete_conf'):
            self.get_module().delete_conf(conf_id)
        self.compute_state()
    
    
    def save_conf(self, conf_id, new_conf):
        _to_return = ""
        if hasattr(self.get_module(), 'save_conf'):
            _to_return = self.get_module().save_conf(conf_id, new_conf, self.get_name())
        
        if _to_return == 'ok':
            self.compute_state()
        
        return _to_return
    
    
    def compute_next_import(self):
        if self.import_interval != 0 and self.last_import:
            self.next_import = self.last_import + 60 * self.import_interval
        else:
            self.next_import = -1
    
    
    def done_import(self):
        # type: () -> Dict[unicode, Union[unicode, int, List[unicode]]]
        if self.last_import <= 0 and self.state != SOURCE_STATE.NOT_CONFIGURED:
            self.state = SOURCE_STATE.NEVER_IMPORT
            self._empty_source(self.state, self.source_import_output)
            synchronization_time = 0
        else:
            self.compute_next_import()
            synchronization_time = int(time.time())
        
        entry = {
            '_id'                 : '%s-%d' % (self.get_name(), self.current_import_nb),
            'source_name'         : self.get_name(),
            'state'               : self.state,
            'synchronization_time': synchronization_time,
            'errors'              : self.errors,
            'merged_errors'       : self.merged_errors,
            'warnings'            : self.warnings,
            'merged_warnings'     : self.merged_warnings,
            'output'              : self.output,
            'source_import_output': self.source_import_output,
            'source_import_state' : self.source_import_state,
            'summary_output'      : self.summary_output,
            'type'                : self.my_type,  # source for source, but will be listener for listeners
            'was_clean'           : self.was_clean,
            'nb_elements'         : self.nb_elements,
            'nb_elements_ok'      : self.nb_elements_ok,
            'nb_elements_warning' : self.nb_elements_warning,
            'nb_elements_error'   : self.nb_elements_error,
            'current_import_nb'   : self.current_import_nb,
        }
        
        return entry
    
    
    def update_from_last_synchronization(self, last_synchronization):
        
        e = last_synchronization.find_one({'source_name': self.get_name()})
        if e:
            self.last_import = e['synchronization_time']
            self.state = e['state']
            self.output = e['output']
            self.source_import_state = e.get('source_import_state', self.state)
            self.source_import_output = e.get('source_import_output', self.output)
            self.errors = e['errors']
            self.warnings = e['warnings']
            self.merged_warnings = e.get('merged_warnings', [])
            self.merged_errors = e.get('merged_errors', [])
            self.saved = e['saved']
        else:
            self.last_import = 0
            self.state = SOURCE_STATE.PENDING
            self.source_import_state = SOURCE_STATE.PENDING
            self.output = ''
            self.source_import_output = ''
            self.errors = []
            self.warnings = []
            self.merged_warnings = []
            self.merged_errors = []
            self.saved = {}
    
    
    def _import_failed(self, state, message):
        self._empty_source(state, message)
        self.errors = [message]
        
        # We have a new run, so currently we are not fully merged without doubt
        # the merger thread will set us when it did finish an import_nb, so we will know
        # if it did merge the last one or not
        self.fully_merged = False
        # Increase the current_import_nb to remember about when we did run last
        self.current_import_nb += 1
        self.already_import = True
        self.last_import = int(time.time())
        self.next_import = -1
    
    
    # Really launch the import from module
    def do_import(self, need_types):
        now = int(time.time())
        if self.clean_import:
            self._empty_source(SOURCE_STATE.OK, self.app.t(u'source.import_clean'))
            self.fully_merged = False
            self.was_clean = True
            self.clean_import = False
            return
        
        self.objects = {}
        self.was_clean = False
        self.force_import_flag = False
        
        # If I have configuration errors, show it
        if self.configuration_errors:
            logger.error(u'The source %s have configuration errors: %s, skipping it' % (self.get_name(), str(self.configuration_errors)))
            self._import_failed(SOURCE_STATE.CRITICAL, u','.join(self.configuration_errors))
            return
        
        logger.debug(u'[import_thread][%s] Will have to launch an import' % self.get_name())
        f = getattr(self.get_module(), u'get_objects', None)
        if f is None:
            self._import_failed(SOURCE_STATE.CRITICAL, u'The module \"%s\" have not the \"get_objects\" function. Are you sure this module are a source import module ?' % self.get_module().get_name())
            return
        
        self.fully_merged = False
        self.state = SOURCE_STATE.RUNNING
        self.output = u''
        logger.debug(u'[import_thread][%s] Launching the get_objects' % self.get_name())
        try:
            res = self.get_module().get_objects()
        except Exception as exp:
            error_message = u'The source [%s] fail with error : %s' % (self.get_name(), str(exp))
            logger.error(error_message)
            logger.print_stack()
            self._import_failed(SOURCE_STATE.CRITICAL, error_message)
            return
        
        objs = res.get(u'objects', {})
        self.import_state = res.get(u'state', SOURCE_STATE.UNKNOWN)
        self.output = res.get(u'output', u'')
        
        if self.import_state in (SOURCE_STATE.NOT_CONFIGURED_BUT_CAN_LAUNCH_IMPORT, SOURCE_STATE.NOT_CONFIGURED):
            self._import_failed(self.import_state, self.output)
            return
        
        self.output = res.get(u'output', u'')
        self.source_import_output = self.output
        self.source_import_state = self.import_state
        self.errors = res.get(u'errors', [])
        self.warnings = res.get(u'warnings', [])
        self.merged_warnings = []
        self.merged_errors = []
        for _type in need_types:
            items = objs.get(_type, [])
            if not items:
                continue
            for item in items:
                # TAG this host with the source data name
                item[u'source'] = self.get_name()
                item[u'import_date'] = now
                item[u'_id'] = uuid.uuid4().hex
                if item.get(u'_SYNC_KEYS', None):
                    if isinstance(item[u'_SYNC_KEYS'], (set, list, tuple)):
                        item[u'_SYNC_KEYS'] = [sync_key.lower() for sync_key in item[u'_SYNC_KEYS']]
                    else:
                        item[u'_SYNC_KEYS'] = item[u'_SYNC_KEYS'].lower()
                if not item.get(u'imported_from', None):
                    item[u'imported_from'] = self.get_name()
                for _prop in self.not_stored_properties:
                    item.pop(_prop, None)
            self.objects[_type] = items
            # logger.debug('[%s] found [%d][%s]' % (self.get_name(), len(items), _type))
        
        # Increase the current_import_nb to remember about when we did run last
        self.current_import_nb += 1
        # logger.info('[DEBUG SOURCE][%s] import done so the next import to merge is:[%s]' % (self.get_name(), self.current_import_nb))
        self.summary_output = ''
        # We have a new run, so currently we are not fully merged without doubt
        # the merger thread will set us when it did finish an import_nb, so we will know
        # if it did merge the last one or not
        self.fully_merged = False
        self.already_import = True
        self.last_import = int(time.time())
    
    
    @staticmethod
    def get_ip_ranges(source_name, source):
        _return = {}
        mongo_component = component_manager.get_mongo_component()
        if source.module_type in ('discovery-import', 'server-analyzer'):
            m = source.modules[0]
            if source.module_type == 'discovery-import':
                ip_ranges = m.get_confs()
            else:
                ip_ranges = mongo_component.col_discovery_confs.find({}, sort=[('discovery_name', ASCENDING)])
            _return['ip_range'] = []
            for ip_range in ip_ranges:
                if source_name == ip_range['source_name']:
                    _range = {
                        'discovery_enabled': ip_range['enabled'],
                        'source_name'      : ip_range['source_name'],
                        'state'            : ip_range['state'],
                        'notes'            : ip_range['notes'],
                        'discovery_name'   : ip_range['discovery_name'],
                        '_id'              : ip_range['_id'],
                        'iprange'          : ip_range['iprange']
                    }
                    _return['ip_range'].append(_range)
        return _return
    
    
    def get_api_infos(self):
        source_module = self.get_module()
        
        # get relative time
        _last_import = self.last_import - time.time() if self.last_import else 0
        if self.force_import_flag:
            _next_import = 0
        elif self.next_import == -1 or not self.already_import:
            _next_import = -1
        else:
            _next_import = self.next_import - time.time()
        
        _type = 'collector' if self.my_type == 'source' else self.my_type
        _module_type = source_module.properties.get('type', None) if source_module.properties else None
        
        # logger.info('[DEBUG SOURCE] source:[%s] state:[%s] fully_merged (all import are merge):[%s]' % (self.get_name(), self.state, self.fully_merged))
        if self.state == SOURCE_STATE.RUNNING and hasattr(source_module, 'source_import_progression'):
            source_import_progression = source_module.get_source_import_progression()
            self.output = source_import_progression.get_current_step()
        
        result = {
            'source_name'        : self.get_name(),
            'already_import'     : self.already_import,
            'fully_merged'       : self.fully_merged,
            'order'              : self.order,
            'state'              : self.state,
            'enabled'            : self.enabled,
            'import_interval'    : self.import_interval,
            'nb_elements'        : self.nb_elements,
            'nb_elements_ok'     : self.nb_elements_ok,
            'nb_elements_warning': self.nb_elements_warning,
            'nb_elements_error'  : self.nb_elements_error,
            'last_import'        : _last_import,
            'next_import'        : _next_import,
            'output'             : self.output,
            'summary_output'     : self.summary_output,
            'type'               : _type,
            'module_type'        : _module_type,
            'internal'           : self.internal,
            'always_enabled'     : self.always_enabled
        }
        
        return result
    
    
    def get_configuration_fields(self):
        raise NotImplementedError()
    
    
    def is_correct(self):
        state = super(Source, self).is_correct()
        _logger_error_msg = []
        _error_msg_dollar = 'You cannot use a dollar in your source name. '
        _error_msg_bad_name = 'Name of the sources must have only ASCII characters.'
        if isinstance(self.source_name, str):
            _source_name_unicode = self.source_name.decode('utf-8', 'ignore')
            _source_name_ascii = _source_name_unicode.encode('ascii', 'ignore')
            if '$' in _source_name_ascii:
                _logger_error_msg.append(_error_msg_dollar)
                state = False
            if _source_name_ascii != self.source_name:
                state = False
                _logger_error_msg.append(_error_msg_bad_name)
        if isinstance(self.source_name, unicode):
            _source_name_ascii = self.source_name.encode('ascii', 'ignore')
            if '$' in _source_name_ascii:
                state = False
                _logger_error_msg.append(_error_msg_dollar)
            _source_name_unicode = _source_name_ascii.decode('utf-8', 'ignore')
            if self.source_name != _source_name_unicode:
                state = False
                _logger_error_msg.append(_error_msg_bad_name)
        if _logger_error_msg:
            self.configuration_errors.append(_logger_error_msg)
            logger.error("[item::%s] %s" % (self.get_name(), ''.join(_logger_error_msg)))
        return state


class Sources(Items):
    name_property = "source_name"
    inner_class = Source
    
    
    def linkify(self, modules):
        # TODO source can own only 1 module so we must remove modules to source and make it module
        self.linkify_s_by_plug(modules)
    
    
    def linkify_modules_instances(self, mod_manager):
        for s in self:
            new_modules = []
            for plug in s.modules:
                plug_name = plug.get_name()
                # don't tread void names
                if plug_name == '':
                    continue
                mod = None
                for m in mod_manager.get_all_alive_instances():
                    if m.get_name() == plug_name:
                        mod = m
                
                if mod is not None:
                    new_modules.append(mod)
                else:
                    err = "Error: the module %s is unknown for %s" % (plug_name, s.get_name())
                    for mod_conf, _ in mod_manager.modules_assoc:
                        if mod_conf.get_name() == plug_name:
                            err = "Error: the source %s has no defined module_type" % s.get_name()
                            break
                    s.configuration_errors.append(err)
            s.modules = new_modules
    
    
    @staticmethod
    def update_sync_keys(app, source_name, old_pattern, new_pattern):
        conn = app.get_synchronizer_syncui_connection()
        # NOTE: internal calls are protected
        params = urllib.urlencode({
            'private_key': app.get_private_http_key(),
            'source_name': source_name,
            'old_pattern': old_pattern.encode('utf8'),
            'new_pattern': new_pattern.encode('utf8'),
        })
        conn.request("GET", "/internal/update_sync_keys?%s" % params)
        response = conn.getresponse()
        response.read()
        conn.close()
    
    
    @staticmethod
    def find_existing_item_match(app, item, item_class):
        _sync_keys = item['_SYNC_KEYS'].split(',')
        _where = {'_SYNC_KEYS': {'$in': _sync_keys}}
        mongo_component = component_manager.get_mongo_component()
        provider_mongo = DataProviderMongo(mongo_component, app.database_cipher)
        item_type = get_type_item_from_class(item_class + 's', item)
        ret = provider_mongo.find_items(item_type, ITEM_STATE.STAGGING, where=_where)
        if not ret and ITEM_TYPE.has_work_area(item_type):
            ret = provider_mongo.find_items(item_type, ITEM_STATE.WORKING_AREA, where=_where)
        return ret
