#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022
# This file is part of Shinken Enterprise, all rights reserved.

import copy
import itertools
import json
import os
import threading
import time
import urllib
import uuid
from collections import OrderedDict
from datetime import datetime
from hashlib import md5

from pymongo.errors import BulkWriteError

from shinken.log import logger, LoggerFactory
from shinken.misc.type_hint import Optional
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.modules.base_module.basemodule import SOURCE_STATE
from shinken.objects.resultmodulation import ModulationParser
from shinken.property import ListProp, BoolProp
from shinkensolutions.beacon import Beacon
from shinkensolutions.service_override_parser import parse_service_override_property, unparse_service_override, SyntaxServiceOverrideError
from shinkensolutions.ssh_mongodb.mongo_error import ShinkenMongoException
from shinkensolutions.time_period_parser import TimePeriodParser
from shinkensolutions.toolbox.box_tools_string import ToolsBoxString
from ...business.source.source import Source
from ...business.source.sourceinfo import SourceInfo
from ...business.source.sourceinfoproperty import SourceInfoProperty
from ...component.component_manager import component_manager
from ...dao.database_migration import migrate_contact_contactgroups, migrate_hosts_contacts_properties, migrate_to_clusters
from ...dao.datamanagerV2 import get_type_item_from_class, get_class_from_type, get_name_from_type, FALLBACK_USER, DataManagerV2
from ...dao.dataprovider.dataprovider_memory import DataProviderMemory
from ...dao.def_items import NAGIOS_TABLES, NAGIOS_TABLE_KEYS, DEF_ITEMS, ITEM_STATE, ITEM_TYPE, METADATA, LINKIFY_MANAGE_STATES, SERVICE_OVERRIDE, ADVANCED_PERIODS
from ...dao.helpers import split_and_strip_list, safe_add_to_dict, add_unique_value_and_handle_plus_and_null
from ...dao.item_comparator import compute_item_diff_existing_vs_from_source
from ...dao.item_saving_formatter import build_link
from ...dao.parsers.bp_rule_parser import BpRuleParser
from ...dao.parsers.complex_exp_parser import parse_complex_exp, get_unparse_complex_exp
from ...dao.validators.import_validator import ImportValidator
from ...dao.validators.merge_validator import MergeValidator, Validator, DO_DEBUG_VALIDATION_STEP_TIMES
from ...front_end.service.collapse_content import CollapsedContent

if TYPE_CHECKING:
    from ..source import Source
    from ..synchronizerdaemon import Synchronizer
    from shinken.misc.type_hint import NoReturn, Dict, List, Any, Optional

EXCLUDE_HASH_PROP = set(['_id', 'import_date', 'last_modification', '@metadata', 'work_area_info', 'to_merge'])
IGNORED_PROPERTIES_FOR_MERGE = set(['@metadata', 'last_modification', 'work_area_info', '__ALREADY_PACKED__', '__MY_BROTHERS__', '__XX__', '__XX_ALREADY_LOOP__', '__SYNC_IDX__', 'source', 'source_order', 'presence_protection'])
SOURCE_TYPE_WITH_CONFIGURATION = ('analyzer', 'listener')

INTERVAL_TO_LOG = 1
ALL_SOURCES = u'ALL_SOURCES'


def make_item_hash(item):
    m = md5()
    for item_prop, value in item.iteritems():
        if item_prop in EXCLUDE_HASH_PROP:
            continue
        if isinstance(value, unicode):
            m.update(''.join(map(bin, bytearray(value, 'utf-8'))))
        else:
            m.update(str(value))
    
    METADATA.update_metadata(item, METADATA.HASH, m.hexdigest())
    return item


class CacheNodeTree(object):
    def __init__(self, item=None, original_hash=None, children=None):
        self.item = item
        self.children = children or list()
        self.hash = original_hash or METADATA.get_metadata(item, METADATA.HASH)
        self.insert_time = time.time()
        # if we don't ger hash and the metadata.HASH return None
        # we will get the last hash from SOURCE_HASH (probably an already merged item)
        if self.hash is None:
            self.hash = METADATA.get_metadata(item, METADATA.SOURCE_HASH).split('-')[-1:][0]
    
    
    def add_child(self, child):
        self.children.append(child)
        return child


class SourceController(object):
    def __init__(self, app):
        # type: (Synchronizer) -> None
        self.app = app
        self.sources = app.sources  # type: List[Source]
        self.mongo_component = component_manager.get_mongo_component()
        self.MASS_IMPORT_LOCK = app.MASS_IMPORT_LOCK
        self._ = app.t
        self._source_messages_summary = {}  # Error/Warning to display at source import (will be show first, for x errors/warnings)
        self._source_messages = {}  # Error/Warning to display at source import
        self._nb_item_found_by_sources = {}
        self.merged_asked = False
        self.merge_in_progress = False
        self.import_validator = ImportValidator(app=app, datamanagerV2=None)
        self.number_of_message_in_source_summary = int(getattr(app.conf, 'number_of_message_in_source_summary', 5))
        
        self.import_cache_items = {}  # a dict with hash as key and CacheNodeTree instance as value
        self.last_cache_items_clean = int(time.time())  # used to clean the cache
        self.import_cache_items_expiration_time = 172800  # 2 days in seconds
        
        self.beacon = None
        
        self._restore_source_retention()
        
        self._existing_item_datamanagerV2 = None
        self._merge_item_datamanagerV2 = None
        self.do_after_fork()
        for source in self.sources:
            source.compute_state()
    
    
    def launch_merger_thread(self):
        t = threading.Thread(None, target=self._merger_thread, name='Merger-thread')
        t.daemon = True
        t.start()
    
    
    # check if merged_asked or merge_in_progress and for each source if at least one need_import
    def import_or_merge_needed_or_doing(self):
        if self.merged_asked:
            return True
        if self.merge_in_progress:
            return True
        for source in self.app.sources:
            if (source.source_name != 'syncui' and source.need_import()) or source.is_import_in_progress():
                return True
        return False
    
    
    def get_source(self, source_name):
        source_name = source_name.strip()
        ask_source = next((source for source in self.app.sources if source.get_name() == source_name), None)  # type: Optional[Source]
        if not ask_source:
            return {}
        
        ask_source.update_from_last_synchronization(self.mongo_component.col_last_synchronization)
        return ask_source
    
    
    def do_after_fork(self):
        for source in self.sources:
            source.do_after_fork()
    
    
    # check if the collection sources_configuration exists in mongo
    def check_or_create_sources_db_config(self):
        col = self.mongo_component.get_collection(u'sources-configuration')
        for source in self.sources:
            if source.my_type not in SOURCE_TYPE_WITH_CONFIGURATION:
                continue
            source_name = source.get_name()
            source_config = col.find_one({'_id': source_name})
            if not source_config:
                source_config = {'_id': source_name}
                for config_type, fields in source.get_configuration_fields().iteritems():
                    if config_type not in source_config:
                        defaults = {}
                        for field_name, properties in fields.iteritems():
                            default_value = properties['default']
                            if properties['protected']:
                                default_value = self.app.database_cipher.cipher_value(properties['default'])
                            defaults[field_name] = default_value
                        source_config[config_type] = defaults
                col.save(source_config)
    
    
    def get_source_configuration(self, source):
        if source.my_type in SOURCE_TYPE_WITH_CONFIGURATION:
            collection = self.mongo_component.get_collection(u'sources-configuration')
            source_name = source.get_name()
            source_conf = collection.find_one({'_id': source_name})
            # decipher the protected_values
            conf = source.get_configuration_fields()
            for config_type, fields in conf.iteritems():
                for field_name, properties in fields.iteritems():
                    if properties['protected']:
                        cipher_value = source_conf[config_type][field_name]
                        decipher_value = cipher_value
                        if cipher_value:
                            decipher_value = self.app.database_cipher.decipher_value(cipher_value)
                        source_conf[config_type][field_name] = decipher_value
            return source_conf
        return None
    
    
    def get_analyzer_template_mapping(self, source):
        if source.my_type in 'analyzer':
            collection = self.mongo_component.get_collection(u'analyzers-template-mapping')
            source_name = source.get_name()
            source_template_mapping_conf = collection.find_one({'_id': source_name}) or {}
            source_template_mapping = source_template_mapping_conf.get('template_mapping', OrderedDict({}))
            final_mapping = OrderedDict({})
            
            for group_name, default_templates in source.get_default_hosts_templates().iteritems():
                if group_name not in final_mapping:
                    final_mapping[group_name] = {'display_name': default_templates['display_name'], 'values': OrderedDict({})}
                for default_template_name in default_templates['values']:
                    value_already_set = source_template_mapping.get(group_name, {}).get('values', {}).get(default_template_name, None)
                    if value_already_set and value_already_set['value']:
                        final_mapping[group_name]['values'][default_template_name] = value_already_set
                    else:
                        final_mapping[group_name]['values'][default_template_name] = {'value': default_template_name, 'overload': False}
            return final_mapping
        return None
    
    
    def set_template_mapping(self, source, template_mapping):
        if source.my_type in 'analyzer':
            collection = self.mongo_component.get_collection(u'analyzers-template-mapping')
            source_name = source.get_name()
            source_template_mapping_conf = collection.find_one({'_id': source_name}) or {'_id': source_name}
            source_template_mapping_conf['template_mapping'] = template_mapping
            collection.save(source_template_mapping_conf)
    
    
    def set_source_configuration(self, source, source_conf):
        if source.my_type in SOURCE_TYPE_WITH_CONFIGURATION:
            collection = self.mongo_component.get_collection(u'sources-configuration')
            # cipher the protected_values
            for config_type, fields in source.get_configuration_fields().iteritems():
                for field_name, properties in fields.iteritems():
                    if properties['protected']:
                        decipher_value = source_conf[config_type][field_name]
                        cipher_value = self.app.database_cipher.cipher_value(decipher_value)
                        source_conf[config_type][field_name] = cipher_value
            collection.save(source_conf)
            LoggerFactory.get_logger().get_sub_part(source.get_name()).get_sub_part(u'CONFIGURATION').info(u'New configuration received, restart the source')
            self.app.api_restart_source(source.get_name())
    
    
    def start_sources(self):
        for source in self.sources:
            module = source.get_module()
            if hasattr(module, 'source_start'):
                logger.info('Trying to start source [%s] ' % source.get_name())
                module.source_start(source.enabled)
    
    
    def launch_source_import(self, source):
        # Launch the import_thread
        t = threading.Thread(None, target=self._source_import_thread, name='source-import-thread-%s' % source.get_name(), args=(source,))
        t.daemon = True
        t.start()
    
    
    def update_last_synchronization(self):
        now = int(time.time())
        try:
            self.mongo_component.col_last_synchronizations.remove({u'synchronization_time': {u'$lt': now - self.app.sync_history_lifespan * 60}})
        except ShinkenMongoException:
            logger.error(u'Cannot clean up old executions of the sources. Cleaning will be attempted again on the next loop')
    
    
    @staticmethod
    def clean_source(source):
        source.set_clean_import()
    
    
    def _restore_source_retention(self):
        for source in self.sources:
            source_name = source.get_name()
            data = next((i for i in self.mongo_component.col_last_synchronization.find({'source_name': source_name})), None)
            if data:
                source.restore_retention_data(data)
    
    
    #
    #    _____                               __  ___
    #   / ___/____  __  _______________     /  |/  /__  ______________ _____ ____
    #   \__ \/ __ \/ / / / ___/ ___/ _ \   / /|_/ / _ \/ ___/ ___/ __ `/ __ `/ _ \
    #  ___/ / /_/ / /_/ / /  / /__/  __/  / /  / /  __(__  |__  ) /_/ / /_/ /  __/
    # /____/\____/\__,_/_/   \___/\___/  /_/  /_/\___/____/____/\__,_/\__, /\___/
    #                                                               /____/
    
    def _build_source_message_for_sync_keys_conflit(self, items, merge_item):
        _item_type = METADATA.get_metadata(merge_item, METADATA.ITEM_TYPE)
        _merge_sync_keys = merge_item['_SYNC_KEYS']
        _value_in_error = set()
        _contents = []
        for _item in items:
            _to_add = [
                get_name_from_type(_item_type, _item),
                '''<span class="shinken-sync-key">%s</span>''' % self._('source.synchronization_keys')
            ]
            _to_add_sync_keys = []
            for _sync_key in _item['_SYNC_KEYS']:
                if _sync_key in _merge_sync_keys:
                    _value_in_error.add(_sync_key)
                    _to_add_sync_keys.append('''<span class="shinken-error">%s</span>''' % _sync_key)
                else:
                    _to_add_sync_keys.append(_sync_key)
            _to_add.append(" ".join(_to_add_sync_keys))
            _contents.append(_to_add)
        
        _collapse_content = CollapsedContent(_contents, label_more=self._('common.see_all'), label_less=self._('common.see_only_first'))
        
        for index, _sync_key in enumerate(_merge_sync_keys):
            if _sync_key in _value_in_error:
                _merge_sync_keys[index] = '''<span class="shinken-error">%s</span>''' % _sync_key
        
        _to_return = self._('validator.exclude_element_conflict') % {
            'item_type'            : self._('type.%s' % _item_type[:-1]),
            'item_types'           : self._('type.%s' % _item_type),
            'item_from_source_info': self._source_message_format_item_from_source(merge_item, DEF_ITEMS[_item_type]['table'], capitalize=False),
            'items_conflit_list'   : _collapse_content.get_html(),
            'synchro_keys'         : '''<span class="shinken-sync-key">%s</span> : %s''' % (self._('source.synchronization_keys'), " ".join(_merge_sync_keys)),
            'documentation_link'   : self.app.documentation_links.get_link('merge_error_message')
        }
        
        return _to_return
    
    
    def _source_message_init(self, source_name):
        # type: (unicode) -> None
        self._source_messages[source_name] = {}
        self._source_messages[source_name][u'error'] = {}
        self._source_messages[source_name][u'warning'] = {}
        self._source_messages[source_name][u'state'] = SOURCE_STATE.OK
        self._source_messages[source_name][u'output'] = []
    
    
    def _reset_merge_messages_for_source_name(self, source_name):
        # type: (unicode) -> None
        # If the source name is not in the dict, we do not need to clean it up
        if source_name in self._source_messages:
            self._source_messages[source_name][u'merged'] = {}
    
    
    def _source_summary_init(self, source_name):
        # type: (unicode) -> None
        self._source_messages_summary[source_name] = {}
        self._source_messages_summary[source_name][u'error'] = {}
        self._source_messages_summary[source_name][u'warning'] = {}
    
    
    def _source_message_add_error(self, msg, item=None, title='', imported=True, level='error', source_name=ALL_SOURCES, item_id=None, merged=False):
        if merged:
            self._merged_source_message_add(msg, title, imported, level, item, source_name, item_id)
        else:
            self._source_message_add(msg, title, imported, level, item, source_name, item_id)
    
    
    def _source_message_add_warning(self, msg, item=None, title='', imported=True, level='warning', source_name=ALL_SOURCES, item_id=None, merged=False):
        if merged:
            self._merged_source_message_add(msg, title, imported, level, item, source_name, item_id)
        else:
            self._source_message_add(msg, title, imported, level, item, source_name, item_id)
    
    
    def _source_message_add(self, msg, title='', imported=True, level='error', item=None, source_name=ALL_SOURCES, item_id=None):
        item_name = self._('element.unknown')
        
        if item:
            source_name = item.get('sources', item.get('source', ''))
            # if we are on an item, we don't want the syncui source because the error/warning is already handled in the UI
            source_name = ','.join((s for s in source_name.split(',') if s != 'syncui'))
            if not item_id:
                item_id = ','.join(item.get('_SYNC_KEYS', item.get('_sync_keys', ['-1'])))
            if not item_id:
                item_id = item['_id']
            item_type = METADATA.get_metadata(item, METADATA.ITEM_TYPE, None)
            if item_type:
                item_name = get_name_from_type(item_type, item)
                if not item_name:
                    item_name = self._('element.unknown')
        
        if not source_name:
            return
        if source_name == ALL_SOURCES:
            source_name = ','.join((s.get_name() for s in self.sources if s.enabled))
        for name in source_name.split(','):
            self._source_messages[name] = self._source_messages.get(name, {})
            self._source_messages[name][level] = self._source_messages[name].get(level, {})
            self._source_messages[name][level][item_id] = {}
            self._source_messages[name][level][item_id]['title'] = title
            self._source_messages[name][level][item_id]['content'] = msg
            self._source_messages[name][level][item_id]['imported'] = imported
            self._source_messages[name][level][item_id]['name'] = item_name
    
    
    def _source_summary_add(self, message, rule_name, title, level='error', item=None, source_names=ALL_SOURCES):
        if item:
            source_names = item.get('sources', item.get('source', ''))
            # if we are on an item, we don't want the syncui source because the error/warning is already handled in the UI
            source_names = ','.join((s for s in source_names.split(',') if s != 'syncui'))
        
        if not source_names:
            return
        if source_names == ALL_SOURCES:
            source_names = ','.join((s.get_name() for s in self.sources if s.enabled))
        for source_name in source_names.split(','):
            self._source_messages_summary[source_name] = self._source_messages_summary.get(source_name, {})
            self._source_messages_summary[source_name][level] = self._source_messages_summary[source_name].get(level, {})
            self._source_messages_summary[source_name][level][rule_name] = self._source_messages_summary[source_name][level].get(rule_name, {})
            self._source_messages_summary[source_name][level][rule_name]['title'] = title
            
            if self._source_messages_summary[source_name][level][rule_name].get('messages', None) is None:
                self._source_messages_summary[source_name][level][rule_name]['messages'] = [message]
            elif len(self._source_messages_summary[source_name][level][rule_name]['messages']) < self.number_of_message_in_source_summary:
                self._source_messages_summary[source_name][level][rule_name]['messages'].append(message)
    
    
    def _merged_source_message_add(self, msg, title='', imported=True, level='error', item=None, source_name=ALL_SOURCES, item_id=None):
        item_name = self._('element.unknown')
        if item:
            source_name = item.get('sources', item.get('source', ''))
            # if we are on an item, we don't want the syncui source because the error/warning is already handled in the UI
            source_name = ','.join((s for s in source_name.split(',') if s != 'syncui'))
            if not item_id:
                item_id = ','.join(item.get('_SYNC_KEYS', item.get('_sync_keys', ['-1'])))
            if not item_id:
                item_id = item['_id']
            item_type = METADATA.get_metadata(item, METADATA.ITEM_TYPE, None)
            if item_type:
                item_name = get_name_from_type(item_type, item)
                if not item_name:
                    item_name = self._('element.unknown')
        
        if not source_name:
            return
        if source_name == ALL_SOURCES:
            source_name = ','.join((s.get_name() for s in self.sources if s.enabled))
        for name in source_name.split(','):
            self._source_messages[name] = self._source_messages.get(name, {})
            self._source_messages[name]['merged'] = self._source_messages[name].get('merged', {})
            self._source_messages[name]['merged'][level] = self._source_messages[name]['merged'].get(level, {})
            self._source_messages[name]['merged'][level][item_id] = {}
            self._source_messages[name]['merged'][level][item_id]['content'] = msg
            self._source_messages[name]['merged'][level][item_id]['title'] = title
            self._source_messages[name]['merged'][level][item_id]['imported'] = imported
            self._source_messages[name]['merged'][level][item_id]['name'] = item_name
    
    
    def _source_message_format_item_from_source(self, item, item_class, capitalize=True):
        item_name = self._get_name_from_item(item, item_class)
        item_imported_from = u''
        item_source = item.get(u'source', u'')
        item_sources = item.get(u'sources', u'')
        item_type = item_class if item.get(u'register', u'1') != u'0' else u'%stpl' % item_class
        sources = item_source if item_source else item_sources
        dic_key = u'validator.from_sources' if capitalize else u'validator.from_sources_uncapitalize'
        dic_key = dic_key if len(sources.split(u',')) != 1 else u'validator.from_source'
        _text = self._(dic_key) % (self._(u'type.%s' % item_type), ToolsBoxString.escape_XSS(item_name), sources, item_imported_from)
        return _text
    
    
    def _get_name_from_item(self, item, item_type):
        key = NAGIOS_TABLE_KEYS.get(item_type, '')
        item_name = item.get(key, item.get('name', item.get('service_description', self._('validator.missing_name_uppercase'))))
        return item_name
    
    
    def _source_message_print(self, sources):
        # type: (List[Source]) -> None
        for source in sources:
            msg_my_source = self._source_messages.get(source.get_name(), {})
            
            source_summaries = self._source_messages_summary.get(source.get_name(), {})
            msg_my_source_warning = {}
            msg_my_source_error = {}
            msg_merged_source_warning = {}
            msg_merged_source_error = {}
            errors_summary = {}
            warnings_summary = {}
            output = [source.source_import_output]
            
            if msg_my_source:
                msg_merged_source = msg_my_source.get(u'merged', {})
                msg_my_source_error = msg_my_source.get(u'error', {})
                msg_my_source_warning = msg_my_source.get(u'warning', {})
                errors_summary = source_summaries.get(u'error', {})
                warnings_summary = source_summaries.get(u'warning', {})
                if msg_merged_source:
                    msg_merged_source_error = msg_merged_source.get(u'error', {})
                    msg_merged_source_warning = msg_merged_source.get(u'warning', {})
            if msg_my_source_error or msg_my_source_warning or msg_merged_source_warning or msg_merged_source_error:
                msg_my_source[u'state'] = SOURCE_STATE.WARNING
                
                elements_skipped = len([i for i in msg_my_source_warning.itervalues() if i[u'imported'] is False]) > 0
                sum_my_source_error_list = [i for i in errors_summary.itervalues()]
                sum_my_source_warning_list = [i for i in warnings_summary.itervalues()]
                
                msg_my_source_errors_list = [i[u'content'] for i in sorted(msg_my_source_error.itervalues(), key=lambda c: c[u'name'].lower())]
                msg_merged_source_errors_list = [i[u'content'] for i in sorted(msg_merged_source_error.itervalues(), key=lambda c: c[u'name'].lower())]
                msg_my_source_warnings_list = [i[u'content'] for i in sorted(msg_my_source_warning.itervalues(), key=lambda c: c[u'name'].lower())]
                msg_merged_source_warnings_list = [i[u'content'] for i in sorted(msg_merged_source_warning.itervalues(), key=lambda c: c[u'name'].lower())]
                
                if (msg_my_source_warnings_list or msg_my_source_errors_list) and self._(u'validator.output_warning_elements') not in output:
                    output.append(self._(u'validator.output_warning_elements'))
                if (msg_merged_source_errors_list or msg_merged_source_warnings_list) and self._(u'validator.output_warning_merged_elements') not in output:
                    output.append(self._(u'validator.output_warning_merged_elements'))
                if elements_skipped:
                    source.summary_output = self._(u'source.summary_output_elements_skipped')
                
                if msg_my_source_warnings_list:
                    for summary in sum_my_source_warning_list:
                        formatted_summary = u'<div class="shinken-output-block shinken-warning-sum"><h1>%(title)s</h1><div class="shinken-ol-container"><ol class="shinken-show-list-marker"><li>%(messages)s</li></ol></div></div>' % {
                            u'title'   : summary[u'title'],
                            u'messages': u'</li><li>'.join(summary[u'messages'])
                        }
                        if formatted_summary not in source.warnings:
                            source.warnings.insert(0, formatted_summary)
                    for msg in msg_my_source_warnings_list:
                        if msg not in source.warnings:
                            source.warnings.append(msg)
                if msg_merged_source_warnings_list:
                    for msg in msg_merged_source_warnings_list:
                        if msg not in source.merged_warnings:
                            source.merged_warnings.append(msg)
                if msg_my_source_errors_list:
                    for summary in sum_my_source_error_list:
                        formatted_summary = u'<div class="shinken-output-block shinken-error-sum"><h1>%(title)s</h1><div class="shinken-ol-container"><ol class="shinken-show-list-marker"><li>%(messages)s</li></ol></div></div>' % {
                            u'title'   : summary[u'title'],
                            u'messages': u'</li><li>'.join(summary[u'messages'])
                        }
                        if formatted_summary not in source.errors:
                            source.errors.insert(0, formatted_summary)
                    for msg in msg_my_source_errors_list:
                        if msg not in source.errors:
                            source.errors.append(msg)
                if msg_merged_source_errors_list:
                    for msg in msg_merged_source_errors_list:
                        if msg not in source.merged_errors:
                            source.merged_errors.append(msg)
            elif source.errors or source.warnings or source.merged_warnings or source.merged_errors:
                msg_my_source[u'state'] = source.state
                output = [source.output]
            else:
                msg_my_source[u'state'] = source.source_import_state
                output = [source.source_import_output]
            
            output = [i for i in output if i]
            source.output = u'<br/>'.join(output)
            if source.state != SOURCE_STATE.RUNNING and source.state != SOURCE_STATE.DIFFERENCE_COMPUTING:
                source.state = SOURCE_STATE.get_best_of_two_states(source.source_import_state, msg_my_source.get(u'state', SOURCE_STATE.OK))
    
    
    def _must_remake_import_at_start(self):
        col = self.mongo_component.col_synchronizer_info
        database_info = col.find_one({'_id': 'database_info'})
        if not database_info:
            database_info = {'_id': 'database_info'}
        
        must_remake_import_because_sanatize = database_info.get('must_remake_import', 0) == 1
        database_info['must_remake_import'] = 0
        col.save(database_info)
        logger.debug("[cache] remake import at start ask by sanatize [%s]" % must_remake_import_because_sanatize)
        
        # If an enabled source was removed from configuration since last start, we must invalidate cached news and differences
        p_sources_dat = os.path.join(self.app.SERVICE_IMPORT_FLAG_PATH, 'sources_names.dat')
        must_remake_import_because_source_change = False
        try:
            with open(p_sources_dat, 'r') as f:
                names = json.loads(f.read())
                must_remake_import_because_source_change = not set(names).issubset(set([s.source_name for s in self.sources if s.enabled]))
        except (IOError, ValueError):
            pass
        
        logger.debug("[cache] check_sources_list [%s]" % must_remake_import_because_source_change)
        must_remake_import = must_remake_import_because_sanatize or must_remake_import_because_source_change
        if must_remake_import:
            logger.info("Asking a merge at start for sanatize:[%s] change on source:[%s]" % (must_remake_import_because_sanatize, must_remake_import_because_source_change))
        return must_remake_import
    
    
    #
    #    _____                               __  ___
    #   / ___/____  __  _______________     /  |/  /__  _________ ____  _____
    #   \__ \/ __ \/ / / / ___/ ___/ _ \   / /|_/ / _ \/ ___/ __ `/ _ \/ ___/
    #  ___/ / /_/ / /_/ / /  / /__/  __/  / /  / /  __/ /  / /_/ /  __/ /
    # /____/\____/\__,_/_/   \___/\___/  /_/  /_/\___/_/   \__, /\___/_/
    #                                                     /____/
    #
    
    def _merger_thread(self):
        if not self.merged_asked:
            self.merged_asked = self._must_remake_import_at_start()
        # We wait http_process because we need to wait the start of datamanagerV2 in syncui to make the syncui source import.
        t0 = time.time()
        self.app.http_process_is_ready.wait(self.app.wait_time_http_process_is_ready)
        wait_time = time.time() - t0
        have_time_out = (wait_time > self.app.wait_time_http_process_is_ready - 1)
        if have_time_out:
            msg = 'Synchronizer wait more than %ss the configuration loading. Please increase wait_time_http_process_is_ready in synchronizer configuration' % self.app.wait_time_http_process_is_ready
            logger.error(msg)
        while not self.app.interrupted:
            if self.merged_asked:
                try:
                    self.merge_in_progress = True
                    logger.debug('[merger_thread] Start')
                    self.beacon = Beacon('beacon_merge_source', enable=True)
                    start_time = time.time()
                    
                    if (int(time.time()) - self.last_cache_items_clean) > self.import_cache_items_expiration_time:
                        self.beacon.beacon('_clean_merged_cache_items')
                        nb_deleted_keys = self._clean_merged_cache_items()
                        logger.info('[merger_thread] import cache cleaned %s root items' % nb_deleted_keys)
                        self.last_cache_items_clean = int(time.time())
                    
                    # Force a syncui load, so it's just too fresh
                    source_syncui = next((source for source in self.sources if source.get_name() == 'syncui'), None)
                    if source_syncui:
                        self._source_message_init(source_syncui)
                        start_time_do_import = time.time()
                        source_syncui.state = SOURCE_STATE.RUNNING
                        source_syncui.do_import(NAGIOS_TABLES.keys())
                        self.beacon.beacon('source_syncui.do_import')
                        self._save_source_objects(source_syncui, already_validate=True)
                        source_syncui.state = source_syncui.import_state
                        source_syncui.merge_asked = True
                        self.beacon.beacon('_save_source_objects syncui')
                        logger.debug('[merger_thread] save item from syncui [%.3f]' % (time.time() - start_time_do_import))
                    
                    self.merged_asked = False  # I did take the job
                    # Ok now we can merge of all sources objects, and create mixed/new collections with objects
                    sources_to_merge = self._get_sources_to_merge()
                    
                    # We mark as merged un configured source because an un configured source can ask for a merge but the merger thread will not take them for merge.
                    for source in self.sources:
                        if source.enabled and source.state in (SOURCE_STATE.NOT_CONFIGURED_BUT_CAN_LAUNCH_IMPORT, SOURCE_STATE.NOT_CONFIGURED):
                            source.fully_merged = True
                    
                    with self.MASS_IMPORT_LOCK:
                        has_been_effective, source_import_nbs = self._merge_sources(sources_to_merge)
                    if has_been_effective:
                        warning_message = self._notify_merge_done()
                        for m in warning_message['message']:
                            self._source_message_add_warning(**m)
                    if not (len(sources_to_merge) == 1 and sources_to_merge[0].get_name() == 'syncui'):
                        self._source_message_print(sources_to_merge)
                    # Now we can save the synchronizations inside our collections
                    self._save_last_synchronization()
                    logger.info('[merger_thread] import done in [%.3f]s' % (time.time() - start_time))
                    
                    self._set_sources_as_merge_done(source_import_nbs)
                except Exception as exp:
                    logger.error(u'[merger_thread] An exception was caught during source merging. (%s)' % exp)
                    for source in self.sources:
                        if not source.enabled:
                            continue
                        # Source which haven't imported elements can't be the cause of failure.
                        if not source.nb_elements:
                            continue
                        source.set_merged(source.current_import_nb)
                        source.output = 'An exception was caught during source merging. See log for more information.'
                        source.state = SOURCE_STATE.CRITICAL
                        source.errors = []
                finally:
                    self.merge_in_progress = False
            
            time.sleep(1)  # do not hammer the CPU :)
    
    
    def _notify_merge_done(self):
        conn = self.app.get_synchronizer_syncui_connection()
        # NOTE: internal calls are protected
        params = urllib.urlencode({
            'private_key': self.app.get_private_http_key(),
        })
        conn.request("GET", "/internal/import_done?%s" % params)
        response = conn.getresponse()
        buf = response.read()
        conn.close()
        return json.loads(buf)
    
    
    def _get_sources_to_merge(self):
        # type: () -> List[Source]
        return [source for source in self.sources if source.enabled and source.state not in (SOURCE_STATE.NOT_CONFIGURED_BUT_CAN_LAUNCH_IMPORT, SOURCE_STATE.NOT_CONFIGURED)]
    
    
    def _merge_sources(self, sources_to_merge):
        # type: (List[Source]) -> (bool, Dict[unicode, int])
        sources_to_merge.sort(key=lambda src: src.order)
        for source in sources_to_merge:
            source.merged_warnings = []
            source.merged_errors = []
            self._reset_merge_messages_for_source_name(source.get_name())
            source.merge_ongoing = True
            if source.merge_asked:
                source.merge_asked = False
        name_sources = u','.join([source.get_name() for source in sources_to_merge])
        
        logger.info(u'[merger_thread] merging started with sources : [%s]' % name_sources)
        
        # STEP - load source items
        source_import_nbs, source_items = self._load_source_items(sources_to_merge)
        logger.debug(u'[merger_thread] get all objects from [%s] sources' % name_sources)
        self.beacon.beacon(u'_load_source_items')
        
        # STEP - find item in cache
        merge_items_in_cache, to_merge_source_items = self._find_items_in_cache(source_items)
        self.beacon.beacon(u'_find_items_in_cache')
        
        # STEP - link source items
        linked_items = self._link_source_items(to_merge_source_items, source_items)
        to_merge_source_items.clear()
        source_items.clear()
        self.beacon.beacon(u'_link_source_items')
        
        # STEP - merge source items
        merged_items = self._merge_source_items(linked_items)
        linked_items.clear()
        self.beacon.beacon(u'_merge_source_items')
        
        # STEP - Remove suspect items from cache (with potential)
        self._remove_suspect_items_from_cache(merge_items_in_cache, merged_items)
        self.beacon.beacon(u'_remove_suspect_items_from_cache')
        
        # STEP - apply taggers on merge items
        self._apply_taggers_on_merge_items(merged_items)
        self.beacon.beacon(u'_apply_taggers_on_merge_items')
        
        # STEP - Find some possible broken double links in objects cache
        self._find_broken_precomputed_double_links(merge_items_in_cache, merged_items)
        self.beacon.beacon(u'_find_broken_precomputed_double_links')
        
        # STEP - reformat merge items
        self._reformat_merge_items(merged_items)
        self.beacon.beacon(u'_reformat_merge_items')
        
        # STEP - initialise merge item datamanager
        self._existing_item_datamanagerV2 = self._initialise_existing_item_datamanager()
        self._merge_item_datamanagerV2 = self._initialise_merge_item_datamanager(merge_items_in_cache, merged_items)
        
        self.beacon.beacon(u'_initialise_merge_item_datamanager')
        
        # STEP - valid merge items
        self._valide_merge_items(merged_items)
        self.beacon.beacon(u'_valid_merge_items')
        
        # STEP - build double link for merge items
        items_to_re_cache = self._build_double_link_for_merge_items()
        self.beacon.beacon(u'_build_double_link_for_merge_items')
        
        # STEP - compute diff with existing items
        self._compute_diff_with_existing_items()
        self.beacon.beacon(u'_compute_diff_with_existing_items')
        
        # STEP - merge merge_items_in_cache in merged_items in a new dict
        by_class_news, by_class_changes, by_type_changes, merged_item_to_save = self._extract_new_and_changes()
        self.beacon.beacon(u'_extract_new_and_changes')
        
        # STEP - saving merge items
        has_been_effective = self._saving_merge_items(by_class_news, by_class_changes, merged_item_to_save)
        by_class_news.clear()
        by_class_changes.clear()
        merged_item_to_save.clear()
        self.beacon.beacon(u'_saving_merge_items')
        
        # STEP - saving retention data
        self._saving_retention_data()
        self.beacon.beacon(u'_saving_retention_data')
        
        # STEP - merge merged_items and items_to_re_cache
        items_to_cache = self._merge_items_to_cache(merged_items, items_to_re_cache)
        merged_items.clear()
        items_to_re_cache.clear()
        self.beacon.beacon(u'_merge_items_to_cache')
        
        # STEP - populate import cache for future import
        self._populate_import_cache(items_to_cache)
        items_to_cache.clear()
        # logger.info("populate import cache : %s  trees sets and %s items set in total (base trees and children)" % (nb_trees_set, nb_item_set))
        self.beacon.beacon(u'_populate_import_cache')
        
        for source in filter(lambda x: x.merge_ongoing, sources_to_merge):
            source.merge_ongoing = False
        
        self._merge_item_datamanagerV2 = None
        self._existing_item_datamanagerV2 = None
        
        self.beacon.beacon(u'end')
        self.beacon.dump()
        # return the number of saved objects (aka the number of object to set in the cache)
        return has_been_effective, source_import_nbs
    
    
    def _set_sources_as_merge_done(self, source_import_nbs):
        try:
            for source_name, import_nb in source_import_nbs.iteritems():
                source = next((s for s in self.sources if s.get_name() == source_name), None)  # type: Source
                if source is None:
                    continue
                source.hook_post_merged()
                source.set_merged(import_nb)
        except Exception as exp:
            logger.error('[merger_thread] Failed at sources set_merged(number) (%s). Continue anyway' % exp)
    
    
    def _saving_retention_data(self):
        # I must not fail now
        try:
            self.app.save_sources_names_list_retention_file()
        except Exception as exp:
            logger.error('[merger_thread] Failed save source retention file. The import continue anyway. (%s)' % exp)
    
    
    def _extract_new_and_changes(self):
        by_class_news = {}
        by_class_changes = {}
        by_type_changes = {}
        merged_item_to_save = {}
        items = self._merge_item_datamanagerV2.find_items(ITEM_TYPE.ELEMENTS, ITEM_STATE.MERGE_SOURCES)
        for item in items:
            item = copy.deepcopy(item)
            is_new = METADATA.get_metadata(item, METADATA.NEW, False)
            METADATA.remove_metadata(item, METADATA.AUTO_DOUBLE_LINKS)
            item_type = METADATA.get_metadata(item, METADATA.ITEM_TYPE)
            changes = METADATA.get_metadata(item, METADATA.CHANGES, None)
            
            item_class = DEF_ITEMS[item_type]['table']
            item.pop('to_merge', None)
            if is_new:
                # SPECIAL CASE: don't add new items from listener-shinken and disabled
                is_enabled = True if item.get('enabled', '1') == '1' else False
                item_source = item.get('sources', '')
                if item_source == "listener-shinken" and not is_enabled:
                    continue
                # get raw items from new,
                # we don't want the merging metadata after the merging phase
                safe_add_to_dict(by_class_news, item_class, item)
            
            safe_add_to_dict(merged_item_to_save, item_class, item)
            if changes:
                safe_add_to_dict(by_type_changes, item_type, changes)
                safe_add_to_dict(by_class_changes, item_class, changes)
        return by_class_news, by_class_changes, by_type_changes, merged_item_to_save
    
    
    def _saving_merge_items(self, by_class_news, by_class_changes, by_class_merged_items):
        import_uuid = uuid.uuid1().hex
        final_tables_to_write = []
        has_been_effective = False
        for item_class in NAGIOS_TABLES:
            merge_items = by_class_merged_items.get(item_class, [])
            change_items = by_class_changes.get(item_class, [])
            change_col_name = 'changeelements-%s' % item_class
            if change_items:
                tmp_col = self._load_tmp_collections(change_items, import_uuid, item_class, ITEM_STATE.CHANGES)
                final_tables_to_write.append((tmp_col, change_col_name))
                has_been_effective = True
            else:
                changed_elt_collection = self.mongo_component.get_collection(change_col_name)
                changed_elt_collection.remove()
            del change_items
            
            col_name = 'newelements-%s' % item_class
            new_elt_collection = self.mongo_component.get_collection(col_name)
            new_items = by_class_news.get(item_class, [])
            if new_items:
                new_items = SourceController._restore_id_from_last_import(new_elt_collection, new_items)
                new_items = [copy.deepcopy(new_item) for new_item in new_items]
                for new_item in new_items:
                    build_link(new_item, METADATA.get_metadata(new_item, METADATA.ITEM_TYPE), ITEM_STATE.NEW, self._existing_item_datamanagerV2)
                tmp_col = self._load_tmp_collections(new_items, import_uuid, item_class, ITEM_STATE.NEW)
                final_tables_to_write.append((tmp_col, col_name))
                has_been_effective = True
            else:
                new_elt_collection.remove()
            del new_items
            
            # manage merged
            # item compute new items and items to del save the new and delete the old one
            merge_col_name = 'merge_from_sources-%s' % item_class
            col_from_sources = self.mongo_component.get_collection(merge_col_name)
            already_existing_items = list(col_from_sources.find({}, {'@metadata.source_hash': 1}))
            in_db_hash = set()
            in_db_id = set()
            id_db_hash_and_id = dict()  # a dict containing the hash as key and _id as values
            id_to_del = set()
            for _item in already_existing_items:
                item_hash = METADATA.get_metadata(_item, METADATA.SOURCE_HASH)
                if not item_hash:
                    # if the object have no metadata, delete it, and it will be added again with the hash
                    id_to_del.add(_item['_id'])
                    continue
                id_db_hash_and_id[item_hash] = _item['_id']
                in_db_hash.add(item_hash)
                in_db_id.add(_item['_id'])
            to_merge_hash = set()
            for item in merge_items:
                to_merge_hash.add(METADATA.get_metadata(item, METADATA.SOURCE_HASH))
            to_save_items_hash = to_merge_hash - in_db_hash
            to_del_items_hash = in_db_hash - to_merge_hash
            
            # If an item is renamed at import, 2 items with different hash can have same _id,
            # so we must change the id of the item in merge item collection to avoid a _id conflict.
            to_save_items = [i for i in merge_items if METADATA.get_metadata(i, METADATA.SOURCE_HASH) in to_save_items_hash]
            del merge_items
            id_to_save = set([i['_id'] for i in to_save_items])
            id_in_double = id_to_save & in_db_id
            if id_in_double:
                # We don't change id of item with hash to delete. We don't want to keep them.
                item_to_change_id = [i['_id'] for i in already_existing_items if (i['_id'] in id_in_double and METADATA.get_metadata(i, METADATA.SOURCE_HASH) not in to_del_items_hash)]
                id_to_del = id_to_del | set(item_to_change_id)
                
                items_to_change = list(col_from_sources.find({'_id': {'$in': item_to_change_id}}))
                
                for i in items_to_change:
                    i['_id'] = uuid.uuid4().hex
                to_save_items.extend(items_to_change)
            
            if to_del_items_hash or id_to_del:
                # create a dict containing only the _id if items to delete
                for item_hash in to_del_items_hash:
                    id_to_del.add(id_db_hash_and_id[item_hash])
                col_from_sources.remove({"_id": {"$in": tuple(id_to_del)}}, w=1)  # w=1 to force the remove to be sync before we can insert new ones, and so remove time race
                has_been_effective = True
            if to_save_items:
                self._save_items_in_col(to_save_items, merge_col_name, ITEM_STATE.MERGE_SOURCES)
                has_been_effective = True
            # logger.info('[save_item]Time to compute item_to_save [%4d] item_to_del [%4d] and save merged for [%s] in %s' % (len(to_save_items_hash), len(to_del_items_hash), item_class, (time.time() - t0)))
        backup_collection = []
        to_save_items = []
        self.mongo_component.clean_source_backup_collection()
        try:
            # Backup collection
            self._backup_collection(backup_collection, final_tables_to_write)
            # Set the import result in collection
            for (new_mongo_collection, collection_name) in final_tables_to_write:
                new_mongo_collection.rename(collection_name, dropTarget=True)
        except Exception as exp:
            logger.error('[merger_thread] Failed to set the import result in new and change collections. We restore the one we already remove. (%s)' % exp)
            self._restore_backup_collection(backup_collection)
            raise
        
        # Cleanup old collections if there are some
        try:
            for (tmp_col_name, _) in backup_collection:
                if tmp_col_name:
                    coll = self.mongo_component.get_collection(tmp_col_name)
                    coll.drop()
        except Exception as exp:
            logger.warning('[merger_thread] Failed to cleanup backup collections. The import continue anyway. (%s)' % exp)
        
        return has_been_effective
    
    
    @staticmethod
    def _create_cache_node(item, item_hash, last_hash):
        node_item = None
        if item_hash == last_hash:  # are we in the last (bottom) item?
            node_item = item
        node = CacheNodeTree(item=node_item, original_hash=item_hash)
        return node
    
    
    # clean the cache node that have been insert (or update) since N time
    def _clean_merged_cache_items(self):
        # loop over all node in cache_items. If their insert date are too old, remove all the branch
        keys_to_del = set()
        for key, node in self.import_cache_items.iteritems():
            if (int(time.time()) - node.insert_time) > self.import_cache_items_expiration_time:
                keys_to_del.add(key)
        
        for key in keys_to_del:
            self.import_cache_items.pop(key, None)
        return len(keys_to_del)
    
    
    def remove_delete_item_by_sync_keys_from_import_cache(self, deleted_item_sync_keys):
        # type: (unicode) -> int
        keys_to_del = set()
        deleted_item_sync_keys = set(deleted_item_sync_keys.split(','))
        start_time = time.time()
        for key, node in self.import_cache_items.iteritems():
            if node.item and set(node.item['_SYNC_KEYS']) & deleted_item_sync_keys:
                keys_to_del.add(key)
        for key in keys_to_del:
            self.import_cache_items.pop(key, None)
        logger.log_perf(start_time, u"merger_thread", u"Removing [%d] deleted item from import cache done" % (len(keys_to_del)), warn_time=0.1)
        return len(keys_to_del)
    
    
    @staticmethod
    def _merge_items_to_cache(merged_items, items_to_re_cache):
        items_to_cache = {}
        for item_class, items in merged_items.iteritems():
            for item in items:
                # new item are excluded from the cache because
                # went there are imported in staging and the source are disable the cache cannot be update
                # see SEF-3111
                is_new = METADATA.get_metadata(item, METADATA.NEW, False)
                if not item.pop('__EXCLUDE_TO_CACHE__', False) and not is_new:
                    safe_add_to_dict(items_to_cache, item_class, item)
        for item_class, items in items_to_re_cache.iteritems():
            for item in items:
                is_new = METADATA.get_metadata(item, METADATA.NEW, False)
                if not item.pop('__EXCLUDE_TO_CACHE__', False) and not is_new:
                    safe_add_to_dict(items_to_cache, item_class, item)
        return items_to_cache
    
    
    def remove_item_from_import_cache(self, merged_item_hash):
        if not merged_item_hash:
            return
        items_hash = merged_item_hash.split('-')
        # If there are a node in cache for this hash we invalid them
        # You can't simply remove it because of concurrency
        node = self.import_cache_items.get(items_hash[0], None)
        
        if node:
            self.last_cache_items_clean = -1
            node.insert_time = -1
    
    
    def _populate_import_cache(self, merged_items):
        not_to_cache_hash_set = set()
        nb_trees_set = 0
        nb_item_set = 0
        # add all merged items in the import cache
        for items in merged_items.itervalues():
            for item in items:
                # remove the to_merge info
                item.pop('to_merge', None)
                # start by getting hash of the item
                item_hash = METADATA.get_metadata(item, METADATA.SOURCE_HASH)
                # if the source hash is in not_to_cache, do nothing here
                if item_hash in not_to_cache_hash_set:
                    continue
                # split it by - and find if the item was merged with many others
                items_hash = METADATA.get_metadata(item, METADATA.SOURCE_HASH).split('-')
                # find the first item in the import_cache_items
                first_hash = items_hash[0]
                last_hash = items_hash[-1:][0]
                first_node = self.import_cache_items.get(first_hash)
                if not first_node:
                    # if the item contain only one hash, it was 'not' merged, we can put the item in the node
                    # else, the item was merged and only the 'last' hash need can be added in the last node
                    first_node = self._create_cache_node(item, first_hash, last_hash)
                    # only if this is the first hash
                    self.import_cache_items[first_hash] = first_node
                    nb_item_set += 1
                    nb_trees_set += 1
                # if a Node was already created, we want to replace the item
                elif first_hash == last_hash:
                    first_node.item = item
                    first_node.insert_time = int(time.time())
                    nb_item_set += 1
                    nb_trees_set += 1
                    # first is last don't try to loop for other hash, we have done for this object
                    continue
                parent_node = first_node
                for item_hash in items_hash[1:]:  # all hash without the first (already managed before)
                    # check if there is a node with the hash as parent_node child
                    child_node = None
                    for parent_child in parent_node.children:
                        if parent_child.hash == item_hash:
                            child_node = parent_child
                    # first time we see this node, create it
                    if not child_node:
                        child_node = self._create_cache_node(item, item_hash, last_hash)
                        parent_node.add_child(child_node)
                        nb_item_set += 1
                    # if a Node was already created we want to replace the item
                    elif child_node.hash == last_hash:
                        child_node.item = item
                        child_node.insert_time = int(time.time())
                        nb_item_set += 1
                    parent_node = child_node
        return nb_item_set, nb_trees_set
    
    
    def _compute_diff_with_existing_items(self):
        items = self._merge_item_datamanagerV2.find_items(ITEM_TYPE.ELEMENTS, ITEM_STATE.MERGE_SOURCES, where={'to_merge': True})
        logger.info("will compute diff on %s items" % len(items))
        for merge_item in items:
            # remove the to_merge info, avoid to compute diff on this one
            merge_item.pop('to_merge')
            item_type = METADATA.get_metadata(merge_item, METADATA.ITEM_TYPE)
            item_class = DEF_ITEMS[item_type]['table']
            
            items_from_stagging, has_item_in_staging_with_different_name = self._find_existing_item_match(item_type, merge_item)
            merge_item_name = get_name_from_type(item_type, merge_item)
            
            if items_from_stagging:
                if len(items_from_stagging) > 1:
                    _msg = self._build_source_message_for_sync_keys_conflit(items_from_stagging, merge_item)
                    logger.info(_msg)
                    self._source_message_add_warning(_msg, merge_item)
                    merge_item['__EXCLUDE_TO_CACHE__'] = True
                    continue
                elif has_item_in_staging_with_different_name:
                    existing_item_name = get_name_from_type(item_type, items_from_stagging[0])
                    _msg = self._('validator.exclude_element_different_names') % (
                        self._source_message_format_item_from_source(merge_item, item_class),
                        existing_item_name,
                        merge_item_name,
                        existing_item_name,
                        merge_item_name,
                        existing_item_name
                    )
                    logger.info(_msg)
                    self._source_message_add_warning(_msg, merge_item)
                    merge_item['__EXCLUDE_TO_CACHE__'] = True
                    continue
            elif has_item_in_staging_with_different_name:
                _msg = self._('validator.element_same_name_different_uuid') % (self._source_message_format_item_from_source(merge_item, item_class))
                logger.info(_msg)
                self._source_message_add_warning(_msg, merge_item)
                merge_item['__EXCLUDE_TO_CACHE__'] = True
                continue
            
            if items_from_stagging and len(items_from_stagging) == 1:
                item_from_stagging = items_from_stagging[0]
                
                obj_changes = compute_item_diff_existing_vs_from_source(item_type, item_from_stagging, merge_item, self._existing_item_datamanagerV2)
                if len(obj_changes) != 0:
                    diff = {'_id': item_from_stagging['_id'], 'changes': obj_changes}
                    METADATA.update_metadata(merge_item, METADATA.CHANGES, diff)
                merge_item['_id'] = item_from_stagging['_id']
            elif merge_item.get('sources', '') != 'syncui':
                METADATA.update_metadata(merge_item, METADATA.NEW, True)
    
    
    def _build_double_link_for_merge_items(self):
        items = self._merge_item_datamanagerV2.find_items(ITEM_TYPE.ELEMENTS, ITEM_STATE.MERGE_SOURCES, where={"to_merge": True})
        recompute_needed_items = self._merge_item_datamanagerV2.find_items(ITEM_TYPE.ELEMENTS, ITEM_STATE.MERGE_SOURCES, where={'need_recompute_double_link': True})
        # keep track of the modified items and re-cache them because they have been modified but their hash doesn't change
        items_to_re_cache = {}
        invalid_items = []
        for item in itertools.chain(items, recompute_needed_items):
            item.pop('need_recompute_double_link', None)
            item_type = METADATA.get_metadata(item, METADATA.ITEM_TYPE)
            item_class = get_class_from_type(item_type)
            item_hash = METADATA.get_metadata(item, METADATA.SOURCE_HASH)
            METADATA.remove_metadata(item, METADATA.DOUBLE_LINKS_ERROR)
            something_done, target_added_to_me, modified_targets = self._merge_item_datamanagerV2._make_double_link(item, item_type, FALLBACK_USER)
            if METADATA.get_metadata(item, METADATA.DOUBLE_LINKS_ERROR):
                invalid_items.append(item)
                self._source_message_add_warning(METADATA.get_metadata(item, METADATA.DOUBLE_LINKS_ERROR), item)
                METADATA.remove_metadata(item, METADATA.DOUBLE_LINKS_ERROR)
            for my_attr, added_targets in target_added_to_me.iteritems():
                for added_target in added_targets:
                    if METADATA.get_metadata(added_target, METADATA.DOUBLE_LINKS_ERROR):
                        invalid_items.append(added_target)
                        METADATA.remove_metadata(added_target, METADATA.DOUBLE_LINKS_ERROR)
                    else:
                        target_hash = METADATA.get_metadata(added_target, METADATA.SOURCE_HASH)
                        auto_dl = METADATA.get_metadata(item, METADATA.AUTO_DOUBLE_LINKS, set())
                        auto_dl.add(target_hash)
                        METADATA.update_metadata(item, METADATA.AUTO_DOUBLE_LINKS, auto_dl)
                        safe_add_to_dict(items_to_re_cache, item_class, item)
            for target_attr, targets in modified_targets.iteritems():
                for target in targets:
                    if METADATA.get_metadata(target, METADATA.DOUBLE_LINKS_ERROR):
                        invalid_items.append(target)
                        METADATA.remove_metadata(target, METADATA.DOUBLE_LINKS_ERROR)
                    else:
                        target_auto_dl = METADATA.get_metadata(target, METADATA.AUTO_DOUBLE_LINKS, set())
                        target_type = METADATA.get_metadata(target, METADATA.ITEM_TYPE)
                        target_class = get_class_from_type(target_type)
                        target_auto_dl.add(item_hash)
                        METADATA.update_metadata(target, METADATA.AUTO_DOUBLE_LINKS, target_auto_dl)
                        safe_add_to_dict(items_to_re_cache, target_class, target)
            
            # special case, if something done but nothing added and to target modified
            # that mean an item member have been deleted
            if something_done and not target_added_to_me and not modified_targets:
                safe_add_to_dict(items_to_re_cache, item_class, item)
        
        for item in invalid_items:
            item['__EXCLUDE_TO_CACHE__'] = True
            self._merge_item_datamanagerV2.delete_item(item, item_state=ITEM_STATE.MERGE_SOURCES)
        return items_to_re_cache
    
    
    def _valide_merge_items(self, merged_items):
        merge_validator = MergeValidator(self.app, self._existing_item_datamanagerV2, self._merge_item_datamanagerV2)
        invalid_items = []
        items = self._merge_item_datamanagerV2.find_items(ITEM_TYPE.ELEMENTS, ITEM_STATE.MERGE_SOURCES, where={'to_merge': True})
        for i, item in enumerate(items):
            item_type = METADATA.get_metadata(item, METADATA.ITEM_TYPE)
            t0 = time.time()
            _validation = merge_validator.validate(item_type, item)
            if DO_DEBUG_VALIDATION_STEP_TIMES:
                logger.debug('[VALIDATION] [%s/%s]_valide_merge_items valide [%s] [%s] in [%s]' % (i, len(items), item_type, get_name_from_type(item_type, item), time.time() - t0))
            title = ''
            message = []
            if _validation['has_messages']:
                item_imported_from = item.get('imported_from', '')
                name = get_name_from_type(item_type, item)
                
                if _validation['has_critical']:
                    invalid_items.append(item)
                    message.append(self._('validator.ignore_merged_element') % (self._('type.%s' % item_type[:-1]), '<a href="?tab=tab-detail-last-run&filter=name:%s">%s</a>' % (name, name), item_imported_from))
                    message.append('<ul class="shinken-status-list critical">')
                    message.append(''.join(['<li>%s</li>' % _message for _message in _validation['critical_messages']]))
                    message.append('</ul>')
                    self._source_message_add_error(''.join(message), item, title, merged=True)
                
                if _validation['has_error']:
                    message.append(self._('validator.error_merged_element') % (self._('type.%s' % item_type[:-1]), '<a href="?tab=tab-detail-last-run&filter=name:%s">%s</a>' % (name, name), item_imported_from))
                    message.append('<ul class="shinken-status-list error">')
                    message.append(''.join(['<li>%s</li>' % _message for _message in _validation['error_messages']]))
                    message.append('</ul>')
                    self._source_message_add_error(''.join(message), item, title, merged=True)
                
                if _validation['has_warning']:
                    message.append(self._('validator.modified_merged_element') % (self._('type.%s' % item_type[:-1]), '<a href="?tab=tab-detail-last-run&filter=name:%s">%s</a>' % (name, name), item_imported_from))
                    message.append('<ul class="shinken-status-list warning">')
                    message.append(''.join(['<li>%s</li>' % _message for _message in _validation['warning_messages']]))
                    message.append('</ul>')
                    self._source_message_add_warning(''.join(message), item, title, merged=True)
        
        for invalid_item in invalid_items:
            col_merged = merged_items[get_class_from_type(METADATA.get_metadata(invalid_item, 'type'))]
            _index = col_merged.index(invalid_item)
            item_in_merged = col_merged[_index]
            item_in_merged['__EXCLUDE_TO_CACHE__'] = True
            self._merge_item_datamanagerV2.delete_item(invalid_item, item_type=METADATA.get_metadata(invalid_item, 'type'), item_state=ITEM_STATE.MERGE_SOURCES)
    
    
    def _initialise_merge_item_datamanager(self, merge_items_in_cache, merged_items):
        data = {ITEM_STATE.MERGE_SOURCES: {}}
        for item_type in DEF_ITEMS:
            data[ITEM_STATE.MERGE_SOURCES][item_type] = []
        for item_class, items in merged_items.iteritems():
            for item in items:
                item_type = get_type_item_from_class(item_class + 's', item)
                # add to_merge = True to remind that they are to merge
                item['to_merge'] = True
                METADATA.update_metadata(item, METADATA.FORMATED_NAME, self._source_message_format_item_from_source(item, item_class, capitalize=False))
                METADATA.update_metadata(item, METADATA.ITEM_TYPE, item_type)
                data[ITEM_STATE.MERGE_SOURCES][item_type].append(item)
        # do the same for the item coming from cache but set the to_merge to False
        for item_class, items in merge_items_in_cache.iteritems():
            for item in items:
                item_type = get_type_item_from_class(item_class + 's', item)
                item = self._existing_item_datamanagerV2.get_raw_item(item, item_type=item_type, keep_metadata=True)
                item['to_merge'] = False
                METADATA.update_metadata(item, METADATA.FORMATED_NAME, self._source_message_format_item_from_source(item, item_class, capitalize=False))
                METADATA.update_metadata(item, METADATA.ITEM_TYPE, item_type)
                data[ITEM_STATE.MERGE_SOURCES][item_type].append(item)
        logger.debug('_initialise_merge_item_datamanager data build done')
        return DataManagerV2(DataProviderMemory(data), self, compute_double_links=False, use_default_callbacks=False)
    
    
    def _initialise_existing_item_datamanager(self):
        conn = self.app.get_synchronizer_syncui_connection()
        params = urllib.urlencode({
            'states'     : ",".join(LINKIFY_MANAGE_STATES),
            'private_key': self.app.get_private_http_key(),
        })
        conn.request("GET", "/internal/get_existing_conf?%s" % params)
        response = conn.getresponse()
        buf = response.read()
        conn.close()
        all_data = json.loads(buf)
        
        return DataManagerV2(DataProviderMemory(all_data, index_double_links=False), self.app, use_default_callbacks=False)
    
    
    def _reformat_merge_items(self, merge_items):
        for item_class in NAGIOS_TABLES:
            if item_class not in merge_items:
                continue
            
            items_by_name = {}
            item_tpls_by_name = {}
            
            for item in merge_items[item_class]:
                if item_class == 'resultmodulation':
                    self._migrate_resultmodulation(item)
                
                if item.get('resultmodulations', ''):
                    val_list = split_and_strip_list(item['resultmodulations'])
                    if len(val_list) > 4:
                        _msg = self._('validator.restricted_resultmodulations') % (self._source_message_format_item_from_source(item, item_class))
                        self._source_message_add_warning(_msg, item, merged=True)
                        item['resultmodulations'] = ','.join(val_list[:4])
                
                if item_class == 'host' and item.get('bp_rule', None):
                    item_bp_rule_parsed = BpRuleParser.compute_internal_bp_rule(item['bp_rule'])
                    item['bp_rule'] = BpRuleParser.get_unparse_bp_rule(item_bp_rule_parsed)
                
                if item_class == 'service':
                    if item.get('host_name', None) and item.get('register', '1') == '0':
                        item_host_name_parsed = parse_complex_exp(item['host_name'])
                        item['host_name'] = get_unparse_complex_exp(item_host_name_parsed)
                    
                    if item.get('hostgroup_name', None):
                        item_hostgroup_name_parsed = parse_complex_exp(item['hostgroup_name'])
                        item['hostgroup_name'] = get_unparse_complex_exp(item_hostgroup_name_parsed)
                else:
                    if item.get('register', '') == '0':
                        if item.get('name', ''):
                            item_name = item['name']
                            items_with_name = item_tpls_by_name.get(item_name, [])
                            if item not in items_with_name:
                                items_with_name.append(item)
                                item_tpls_by_name[item_name] = items_with_name
                    else:
                        if item.get(NAGIOS_TABLE_KEYS[item_class], ''):
                            item_name = item[NAGIOS_TABLE_KEYS[item_class]]
                            items_with_name = items_by_name.get(item_name, [])
                            if item not in items_with_name:
                                items_with_name.append(item)
                                items_by_name[item_name] = items_with_name
            
            for item_name, items in itertools.chain(item_tpls_by_name.iteritems(), items_by_name.iteritems()):
                if len(items) > 1:
                    for item_to_remove in items:
                        _msg = self._("element.already_exists") % ('<ul class="shinken-status-list">%s</ul>' % ''.join(['<li>%s</li>' % self._source_message_format_item_from_source(i, item_class) for i in items]))
                        self._source_message_add_error(_msg, item_to_remove, merged=True)
                        merge_items[item_class].remove(item_to_remove)
    
    
    def _load_source_items(self, sources):
        # type: (List) -> (int, Dict)
        source_import_nbs = {}
        source_items = {}
        for source in sources:
            source_name = source.get_name()
            nb_elements_in_source = {u'nb_elements': 0, u'nb_elements_ok': 0, u'nb_elements_warning': 0, u'nb_elements_error': 0}
            self._nb_item_found_by_sources[source_name] = nb_elements_in_source
            
            if source.state == SOURCE_STATE.NOT_CONFIGURED:
                source_import_nbs[source_name] = {}
                continue
            # Protect source access data with a lock because maybe the source is currently
            # dropping and setting this collection, and this can take time
            try:
                with source.get_data_lock():
                    # Remember the import nb of this source we take, so we can get back to it when we did finish with the merge
                    current_import_nb = source.current_import_nb
                    source_import_nbs[source_name] = current_import_nb
                    for item_class in NAGIOS_TABLES:
                        
                        collection = self.mongo_component.get_collection(u'data-%s-%s' % (source_name, item_class))
                        items = list(collection.find())
                        
                        for item in items:
                            self._count_item_found_in_source(nb_elements_in_source, item, item_class)
                            # if the object get import errors during the import
                            if METADATA.get_metadata(item, METADATA.IMPORT_ERRORS, []):
                                continue
                            
                            del item[u'_id']
                            item[u'source_order'] = source.order
                            self.app.database_cipher.uncipher(item, item_type=ITEM_TYPE.CONTACTS)
                            if not METADATA.get_metadata(item, METADATA.HASH):
                                make_item_hash(item)
                            safe_add_to_dict(source_items, item_class, item)
                        self._nb_item_found_by_sources[source_name] = nb_elements_in_source
                    self.update_nb_elements_in_source(source, nb_elements_in_source)
            except Exception:
                source_import_nbs[source.get_name()] = {}
                logger.error(u'[merger_thread] Failed at reading source information for %s' % source.get_name())
                raise
        return source_import_nbs, source_items
    
    
    def _find_items_in_cache(self, source_items):
        to_merge_source_items = {}
        find_in_cache_items = {}
        for item_class in NAGIOS_TABLES:
            if item_class not in source_items:
                continue
            by_class_to_merge_items = []
            to_merge_source_items[item_class] = by_class_to_merge_items
            already_managed_item_hashs = set()
            source_class_items = source_items[item_class]
            source_class_items_hashs = [METADATA.get_metadata(i, METADATA.HASH) for i in source_class_items]
            for item in source_class_items:
                item_hash = METADATA.get_metadata(item, METADATA.HASH)
                # we already see this item and resolved the tree
                if item_hash in already_managed_item_hashs:
                    continue
                cache_node = self.import_cache_items.get(item_hash, None)  # type: CacheNodeTree
                if cache_node:
                    # the hard part start here, we need to walk through each child (go deeper in the cache tree)
                    # find the closest (on right) child into source_items[item_class]
                    # when children are empty we need to add all items finds in already_managed_item_hashs
                    all_sons_scanned = False
                    # update the insert_time to let the cache now that this object have been used
                    cache_node.insert_time = int(time.time())
                    while not all_sons_scanned:
                        # set the item as already managed, so we will not set it as to_merge
                        current_item_hash = cache_node.hash
                        already_managed_item_hashs.add(current_item_hash)
                        # if the cache node has no children, consider all sons as scanned
                        if not cache_node.children:
                            # set the item (original or merged if we are deeper than first level of cache) as find in cache
                            safe_add_to_dict(find_in_cache_items, item_class, cache_node.item)
                            all_sons_scanned = True
                            break
                        current_node_index = source_class_items_hashs.index(current_item_hash)
                        # get index of each child in source_class_items]
                        closest_index = None
                        closest_child = None
                        for child in cache_node.children:
                            # if the child hash is not in the sources items, don't manage it
                            # use the source_class_items_hashs because it's a set, in a set is O(1)
                            if child.hash not in source_class_items_hashs:
                                continue
                            child_index = source_class_items_hashs.index(child.hash)
                            if child_index > current_node_index:
                                if not closest_index or child_index < closest_index:
                                    closest_index = child_index
                                    closest_child = child
                        if not closest_child:
                            break
                        # we find the closest item, loop again in the while
                        cache_node = closest_child
                    # All sons haven't been scanned, that means the item been found in cache but not set in find_in_cache_items
                    # The cached item was cache from a merge version of itself (ex cfg-syncui) and the second level have been removed
                    # here we need to add again the first level in cache
                    if all_sons_scanned is False:
                        # if the item come from the cache_node, we can use it as find_in_cache, else the item will not have _id so ask for a new merge
                        if cache_node.item:
                            safe_add_to_dict(find_in_cache_items, item_class, cache_node.item)
                        else:
                            safe_add_to_dict(to_merge_source_items, item_class, item)
                else:
                    safe_add_to_dict(to_merge_source_items, item_class, item)
        return find_in_cache_items, to_merge_source_items
    
    
    @staticmethod
    def _remove_suspect_items_from_cache(merge_items_in_cache, merged_items):
        if not merge_items_in_cache:
            return ()
        # get all sync_keys of merged items
        # if a sync_key of an item from cache is present inside, remove this object
        merged_items_sync_keys = set()
        for item_class, items in merged_items.iteritems():
            for item in items:
                # add all item sync_keys into the set
                [merged_items_sync_keys.add(sync) for sync in item.get('_SYNC_KEYS', [])]
        
        for item_class, items in merge_items_in_cache.iteritems():
            to_del = []
            for item in items:
                se_uuid = item.get('_SE_UUID', None)
                # if the object have an SE_UUID consider it as the only one sync_key
                # SE_UUID will be in the SYNC_KEYS of the merged items
                if se_uuid:
                    sync_keys = set((se_uuid,))
                else:
                    sync_keys = set(item.get('_SYNC_KEYS', []))
                conflicting_sync_keys = sync_keys.intersection(merged_items_sync_keys)
                if conflicting_sync_keys:
                    # This item have at least one sync key conflicting with
                    to_del.append(item)
            for item_to_del in to_del:
                items.remove(item_to_del)
    
    
    @staticmethod
    def _find_broken_precomputed_double_links(merge_items_in_cache, merged_items):
        # for each item in cache with the metadata AUTO_DOUBLE_LINKS
        # if at least one of the item in AUTO_DOUBLE_LINKS is not present in merged_items we need to as a new double link
        if not merge_items_in_cache:
            return ()
        all_items_hash = set()
        re_double_link_needed = []
        for item_class in NAGIOS_TABLES:
            cached_class_items = merge_items_in_cache.get(item_class, ())
            merged_class_items = merged_items.get(item_class, ())
            for item in itertools.chain(cached_class_items, merged_class_items):
                item_hash = METADATA.get_metadata(item, METADATA.SOURCE_HASH)
                all_items_hash.add(item_hash)
        # now that we have all the new merged items
        for item_class, items in merge_items_in_cache.iteritems():
            for item in items:
                auto_double_linked_source_hash = METADATA.get_metadata(item, METADATA.AUTO_DOUBLE_LINKS, set())
                all_links_are_in_items = auto_double_linked_source_hash.issubset(all_items_hash)
                if not all_links_are_in_items:
                    item['need_recompute_double_link'] = True
                    METADATA.remove_metadata(item, METADATA.AUTO_DOUBLE_LINKS)
                    re_double_link_needed.append(item)
        return re_double_link_needed
    
    
    def _apply_taggers_on_merge_items(self, merged_items):
        merged_hosts = merged_items.get('host', None)
        # if no hosts, return now and do nothing
        if not merged_hosts:
            return
        for tagger in self.app.taggers:
            try:
                tagger.hook_post_merge(merged_hosts)
            except Exception, exp:
                logger.error('The tagger %s did raise an error: %s' % (tagger.get_name(), exp))
            self.beacon.beacon('_merge_source_items', tagger.get_name())
    
    
    def _merge_source_items(self, linked_items):
        merged_items = {}
        for item_class in NAGIOS_TABLES:
            if item_class not in linked_items:
                continue
            start_time = time.time()
            by_class_merge_items = []
            merged_items[item_class] = by_class_merge_items
            for linked_item in linked_items[item_class]:
                merged_item = self._merge_linked_items(item_class, linked_item)
                if merged_item:
                    by_class_merge_items.append(merged_item)
            logger.log_perf(start_time, "merger_thread", "Merging [%d] [%s] brothers" % (len(linked_items[item_class]), item_class))
            self.beacon.beacon('_merge_source_items', item_class, 'merge linked item')
        return merged_items
    
    
    def _link_source_items(self, to_merge_source_items, source_items):
        linked_items = {}
        for item_class in NAGIOS_TABLES:
            if item_class not in to_merge_source_items:
                continue
            to_merge_keys = {}
            items = to_merge_source_items[item_class]
            for item in items:
                item['__XX__'] = set()
                for key in item.get('_SYNC_KEYS', []):
                    safe_add_to_dict(to_merge_keys, key, item)
            
            self._find_items_link_with_items_to_merge(source_items[item_class], to_merge_keys, items)
            items = self._filter_out_too_many_merges(to_merge_keys, items)
            self.beacon.beacon('_merge_source_items', item_class, 'build items to merge')
            
            # Give objects an ID, so we can hash this into our sets
            _ids = {}
            for (index, item) in enumerate(items):
                _ids[index] = item
                item['__SYNC_IDX__'] = index
            
            # Now put in object the others
            for same_sync_keys_items in to_merge_keys.itervalues():
                for item in same_sync_keys_items:
                    for item2 in same_sync_keys_items:
                        if item != item2:
                            item['__XX__'].add(item2['__SYNC_IDX__'])
            
            start_time = time.time()
            for item in items:
                self._find_linked_items(_ids, item, set())
            self._break_link_between_items_with_different_uuid(_ids, item_class, items)
            
            self.beacon.beacon('_merge_source_items', item_class, 'find_linked_items')
            logger.log_perf(start_time, "merger_thread", "find linked items for [%d] [%s] done" % (len(items), item_class))
            
            start_time = time.time()
            all_objects_linked = []
            for item in items:
                if '__ALREADY_PACKED__' in item:
                    continue
                item['__ALREADY_PACKED__'] = True
                lst = []
                for brother_id in item['__MY_BROTHERS__']:
                    brother = _ids[brother_id]
                    brother['__ALREADY_PACKED__'] = True
                    lst.append(brother)
                all_objects_linked.append(lst)
            linked_items[item_class] = all_objects_linked
            logger.log_perf(start_time, "merger_thread", "Packing [%d] [%s] brothers" % (len(items), item_class))
        return linked_items
    
    
    def _find_items_link_with_items_to_merge(self, source_items, to_merge_keys, items):
        new_item_to_merge_add = True
        while new_item_to_merge_add:
            new_item_to_merge_add = False
            for source_item in source_items:
                if '__XX__' in source_item:
                    continue
                sync_keys = source_item.get('_SYNC_KEYS', [])
                if self._source_item_key_in_to_merge_keys(to_merge_keys, sync_keys):
                    new_item_to_merge_add = True
                    source_item['__XX__'] = set()
                    items.append(source_item)
                    for key in sync_keys:
                        safe_add_to_dict(to_merge_keys, key, source_item)
    
    
    @staticmethod
    def _source_item_key_in_to_merge_keys(all_keys, sync_keys):
        for k in sync_keys:
            if k in all_keys:
                return True
        return False
    
    
    def _break_link_between_items_with_different_uuid(self, _ids, item_class, items):
        to_remove_item = set()
        # 2 item with different uuid can't merge, so we break link and remove suspect item
        for item in items:
            linked_index_items = list(item['__MY_BROTHERS__'])
            
            linked_uuid = set()
            for linked_index_item in item['__MY_BROTHERS__']:
                item_uuid = item.get('_SE_UUID', None)
                linked_item_uuid = _ids[linked_index_item].get('_SE_UUID', None)
                if item_uuid and linked_item_uuid and linked_item_uuid != item_uuid:
                    linked_index_items.remove(linked_index_item)
                elif linked_item_uuid:
                    linked_uuid.add(linked_item_uuid)
            
            item['__MY_BROTHERS__'] = linked_index_items
            
            # if we find a potential link error
            # we need to show error and remove all brothers
            if len(linked_uuid) > 1:
                key = NAGIOS_TABLE_KEYS.get(item_class, '')
                item_name = item.get(key, item.get('name', self._('validator.missing_name_uppercase')))
                item_name = item_name if item_name else self._('validator.missing_name_uppercase')
                
                logger.debug('[merger_thread] suspect link item [%s][%s] [%s]' % (item_name, item['__SYNC_IDX__'], item['__MY_BROTHERS__']))
                to_remove_item.add(item['__SYNC_IDX__'])
                
                link_item_to_print = "<br /><ul class='text-break-all-pre-line'>"
                
                for _id in item['__MY_BROTHERS__']:
                    to_remove_item.add(_id)
                    i = _ids[_id]
                    item_name = i.get(key, item.get('name', self._('validator.missing_name_uppercase')))
                    item_name = item_name if item_name else self._('validator.missing_name_uppercase')
                    item_imported_from = i.get('imported_from', '')
                    item_imported_from = self._('validator.in_file') % item_imported_from if item_imported_from else ''
                    item_sources = i.get('source', i.get('sources', self._('validator.no_sources_found')))
                    item_keys = ",".join(i.get('_SYNC_KEYS', []))
                    
                    message = self._('validator.from_source_with_key') % (item_class, item_name, item_sources, item_imported_from, item_keys)
                    link_item_to_print += "<li>%s</li>" % message
                link_item_to_print += "</ul>"
                _msg = self._('validator.exclude_element_different_uuid') % (
                    self._source_message_format_item_from_source(item, item_class), len(item['__MY_BROTHERS__']), link_item_to_print, self.app.documentation_links.get_link('merge_error_message'))
                self._source_message_add_warning(_msg, item)
        for to_remove in to_remove_item:
            logger.debug('to_remove %s' % to_remove)
            items.remove(_ids[to_remove])
            del _ids[to_remove]
        for item in items:
            item['__MY_BROTHERS__'] = filter((lambda x: x not in to_remove_item), item['__MY_BROTHERS__'])
    
    
    def _find_linked_items(self, _ids, item, pack, level=0):
        level += 1
        
        if '__XX_ALREADY_LOOP__' in item:
            # Think to merge our data with the one from our brother
            for other_id in pack:
                brother = _ids[other_id]
                for k in brother['__MY_BROTHERS__']:
                    item['__MY_BROTHERS__'].add(k)
            # My version is always bigger than his so return mine
            return item['__MY_BROTHERS__']
        else:
            item['__XX_ALREADY_LOOP__'] = True
            # By default, WE are in our pack
            pack.add(item['__SYNC_IDX__'])
            item['__MY_BROTHERS__'] = pack
            # ask my brothers their pack and link it into my own
            for other_ids in item['__XX__']:
                other = _ids[other_ids]
                b_pack = self._find_linked_items(_ids, other, pack, level)
                # Give brothers of my brothers in the pack I'll give to my brother
                # .... ok not easy and maybe overkill....
                for bro_bro_id in b_pack:
                    pack.add(bro_bro_id)
            return pack
    
    
    def _merge_linked_items(self, item_class, items):
        # type: (unicode, List[Dict[unicode,Any]]) -> Optional[Dict[unicode,Any]]
        # Get the class of t and grab its properties dict
        properties = NAGIOS_TABLES[item_class][0].properties
        old_properties = getattr(NAGIOS_TABLES[item_class][0], u'old_properties', {})
        # First we need to sort objects by their source order
        items.sort(SourceController._compare_sources)
        for i in items:
            if not METADATA.get_metadata(i, METADATA.HASH):
                logger.error(u'Import item has no hash. Please contact your shinken administrator. [%s]' % i)
        source_hash = u'-'.join((METADATA.get_metadata(i, METADATA.HASH, u'ERROR-MISSING-HASH') for i in items))
        properties[u'imported_from'] = ListProp(default=u'', merging=u'join')
        # We keep a list of forced item_property to stop merging new values after a source set a [FORCE] item_property
        forced_properties = set()
        
        merge_item = {}
        METADATA.update_metadata(merge_item, METADATA.SOURCE_HASH, source_hash)
        source_info = SourceInfo(item_class + u's')
        for item in items:
            source = item[u'source']
            for (item_property, value) in item.iteritems():
                if item_property in IGNORED_PROPERTIES_FOR_MERGE:
                    continue
                item_property_without_forced = item_property.replace(u'[FORCE]', u'').strip()
                # If an item_property is forced (forced_properties), any value that's not forced is ignored
                if item_property_without_forced in forced_properties:
                    continue
                
                # Force properties are set by the user as use[FORCE]
                if u'[FORCE]' in item_property:
                    # and remove the [FORCE] from the key name now
                    item_property = item_property.replace(u'[FORCE]', u'').strip()
                    prop_name = old_properties.get(item_property, item_property)
                    element_rule = properties.get(prop_name, None)
                    prop_handle_plus = False if element_rule is None else element_rule.handle_additive_inheritance
                    separator = u',' if element_rule is None else element_rule.separator
                    value = add_unique_value_and_handle_plus_and_null(u'', value, prop_handle_plus, separator=separator)
                    merge_item[item_property] = value
                    source_info.update_field(item_property, value, source, field_type=SourceInfoProperty.ORDERED_TYPE, overwrite=True, force=True)
                    forced_properties.add(item_property)
                    logger.debug(u'Forced item_property for [%s]:[%s]' % (item_property, value))
                
                prop_name = old_properties.get(item_property, item_property)
                element_rule = properties.get(prop_name, None)
                prop_handle_plus = False if element_rule is None else element_rule.handle_additive_inheritance
                separator = u',' if element_rule is None else element_rule.separator
                
                if prop_name == SERVICE_OVERRIDE:
                    merge_item[item_property] = self._merge_service_overrides(merge_item, item_class, item_property, value, source, source_info)
                elif prop_name == ADVANCED_PERIODS:
                    merge_item[item_property] = self._merge_advanced_time_periods(merge_item, item_property, value, source, source_info)
                elif element_rule and element_rule.merging in (u'join', u'ordered', u'duplicate'):
                    orig_str = merge_item.get(item_property, '')
                    new_value = add_unique_value_and_handle_plus_and_null(orig_str, value, prop_handle_plus, separator=separator)
                    if new_value:
                        merge_item[item_property] = new_value
                    prop_type = SourceInfoProperty.ORDERED_TYPE if element_rule.merging == u'ordered' else SourceInfoProperty.SET_TYPE
                    source_info.update_field(item_property, value, source, field_type=prop_type)
                elif item_property not in merge_item:  # Uniques properties, default case (macros, simple properties...)
                    if isinstance(element_rule, BoolProp):
                        value = BoolProp().unpythonize(value)
                    merge_item[item_property] = value
                    source_info.update_field(item_property, value, source, field_type=SourceInfoProperty.UNIQUE_TYPE)
        
        # Now special case: delete _SYNC_KEYS item_property,
        # keep all sync_keys of each objects
        sync_keys = set()
        for item in items:
            sync_keys.update(item[u'_SYNC_KEYS'])
        merge_item[u'_SYNC_KEYS'] = list(sync_keys)
        
        # Now special case: delete source item_property, and join them in sources prop
        sources_set = list(set([item[u'source'] for item in items]))
        merge_item[u'sources'] = ','.join([item for item in sources_set])
        merge_item[u'_id'] = uuid.uuid1().hex
        # Maybe there is a _SE_UUID item_property with FROM-TYPE-UUID format, if so, take it
        if u'_SE_UUID' in merge_item:
            if merge_item[u'_SE_UUID'].strip().count(u'-') == 2:
                _se_uuid_uuid = merge_item[u'_SE_UUID'].strip().split(u'-', 2)[2].strip()
                if _se_uuid_uuid:
                    merge_item[u'_id'] = _se_uuid_uuid
        
        # Look if the main key of this merged object is still there. If not and
        # if SYNC_KEYS is not void, take the first one as the key name in order to
        # This is a bit defensive, but it's better than to have false conf
        key = NAGIOS_TABLE_KEYS.get(item_class, u'')
        if key:
            if (key not in merge_item or merge_item[key] == '') and merge_item.get(u'register', u'') != u'0':
                logger.error(u'Invalid merged element %s [%s]' % (self._source_message_format_item_from_source(merge_item, item_class), merge_item))
                message = self._(u'validator.missing_key') % (key, self._source_message_format_item_from_source(merge_item, item_class))
                sync_keys = merge_item.get(u'_SYNC_KEYS', [])
                if len(sync_keys) >= 1:
                    merge_item[key] = sync_keys[0]
                    message += self._(u'validator.item_named') % sync_keys[0]
                    self._source_message_add_warning(message, merge_item)
                else:
                    message += self._(u'validator.item_excluded')
                    self._source_message_add_warning(message, merge_item)
                    return None
            
            if merge_item.get(key, u'').endswith(u'-tpl'):
                merge_item[u'register'] = u'0'
            # then try to fix bad templates
            elif merge_item.get(u'register', '') == u'0' and (u'name' not in merge_item or merge_item[u'name'] == u''):
                logger.error(u'Invalid merged element %s [%s]' % (self._source_message_format_item_from_source(merge_item, item_class), merge_item))
                message = self._(u'validator.missing_key') % (key, self._source_message_format_item_from_source(merge_item, item_class))
                # Ok we got a template with a key but no name? switch this instead
                if key in merge_item and merge_item[key]:
                    merge_item[u'name'] = merge_item[key]
                    message += self._(u'validator.item_named') % merge_item[u'name']
                    self._source_message_add_warning(message, merge_item)
                else:  # ok no key set, maybe we can use the sync_keys
                    sync_keys = merge_item.get(u'_SYNC_KEYS', [])
                    if len(sync_keys) >= 1:
                        template_name = sync_keys[0]
                        if template_name.endswith(u'-tpl'):
                            template_name = template_name[:-4]
                        merge_item[u'name'] = template_name
                        message += self._(u'validator.item_named') % merge_item[u'name']
                        self._source_message_add_warning(message, merge_item)
                    else:  # awful template
                        message += self._(u'validator.item_excluded')
                        self._source_message_add_warning(message, merge_item)
                        return None
        
        METADATA.update_metadata(merge_item, METADATA.SOURCE_INFO, source_info.as_dict())
        return merge_item
    
    
    def _merge_service_overrides(self, merge_item, item_class, item_property, value, source, source_info):
        # type: (Dict, unicode, unicode, unicode, unicode, SourceInfo) -> unicode
        new_value = value
        orig_str = merge_item.get(item_property, '')
        
        if orig_str:
            try:
                new_parsed = parse_service_override_property(value)
            except SyntaxServiceOverrideError as e:
                # Service override must be validated before. This log shouldn't be call.
                logger.warning(u'Fail to parse service overrides in %s with error : %s' % (self._source_message_format_item_from_source(merge_item, item_class), e))
                return orig_str
            try:
                orig_parsed = parse_service_override_property(orig_str)
            except SyntaxServiceOverrideError as e:
                # Service override must be validated before. This log shouldn't be call.
                logger.warning(u'Fail to parse service overrides in %s with error : %s' % (self._source_message_format_item_from_source(merge_item, item_class), e))
                return new_value
            new_parsed.update(orig_parsed)
            new_value = unparse_service_override(new_parsed)
        
        source_info.update_field(item_property, value, source, field_type=SourceInfoProperty.SET_TYPE)
        return new_value
    
    
    @staticmethod
    def _merge_advanced_time_periods(merge_item, item_property, value, source, source_info):
        value_from_merge = merge_item.get(item_property, '')
        if value_from_merge and value:
            merge_as_list = value_from_merge.split(TimePeriodParser.SEPARATOR_ADVANCED_DEF)
            value_as_list = value.split(TimePeriodParser.SEPARATOR_ADVANCED_DEF)
            if '' in value_as_list:
                value_as_list.remove('')
            
            orig_keys = [merge_period.split(TimePeriodParser.SEPARATOR_ADVANCED_KEY_VALUE)[0] for merge_period in merge_as_list]
            value_to_append = []
            for value_period in value_as_list:
                (key, timeperiods) = value_period.split(TimePeriodParser.SEPARATOR_ADVANCED_KEY_VALUE)
                if key not in orig_keys:
                    value_to_append.append("%s%s%s" % (key, TimePeriodParser.SEPARATOR_ADVANCED_KEY_VALUE, timeperiods))
            
            value_to_append = TimePeriodParser.SEPARATOR_ADVANCED_DEF.join(value_to_append)
            
            if value_to_append:
                new_value = "%s%s%s" % (value_from_merge, TimePeriodParser.SEPARATOR_ADVANCED_DEF, value_to_append)
            else:
                new_value = value
        else:
            value_to_append = value
            new_value = value
        
        source_info.update_field(item_property, value_to_append, source, field_type=SourceInfoProperty.SET_TYPE)
        return new_value
    
    
    def _migrate_resultmodulation(self, item):
        if item.get('output_rules', ''):
            # Handwritten rules are prone to errors. Check correctness
            parser = ModulationParser()
            try:
                tokens = parser.tokenize(item['output_rules'])
                rules = parser.parse(tokens)
                if not parser.is_correct(rules):
                    del item['output_rules']
                    _msg = self._('validator.resultmodulation_parsing_error') % self._source_message_format_item_from_source(item, 'resultmodulation')
                    self._source_message_add_warning(_msg, item)
            except Exception as e:
                _msg = self._('validator.resultmodulation_syntax_error') % self._source_message_format_item_from_source(item, 'resultmodulation')
                self._source_message_add_warning(_msg, item)
                logger.warning('%s cause by [%s]' % (_msg, e))
        exit_strings = [SOURCE_STATE.OK, SOURCE_STATE.WARNING, SOURCE_STATE.CRITICAL, SOURCE_STATE.UNKNOWN]
        exit_codes = []
        # Detect old definition
        if 'exit_codes_match' in item and 'exit_code_modulation' in item and 'output_rules' not in item:
            for code in item['exit_codes_match'].split(','):
                code = code.strip()
                if not code.isdigit() or int(code) > 3:
                    continue
                exit_codes.append(exit_strings[int(code)])
            if exit_codes and item['exit_code_modulation'].isdigit() and int(item['exit_code_modulation']) <= 3:
                # Migrate to new format
                modulation_string = exit_strings[int(item['exit_code_modulation'])]
                item['output_rules'] = '|'.join(['%s->%s' % (ec, modulation_string) for ec in exit_codes])
        # Remove old keys in all cases
        if 'exit_codes_match' in item:
            del item['exit_codes_match']
        if 'exit_code_modulation' in item:
            del item['exit_code_modulation']
    
    
    def _filter_out_too_many_merges(self, all_keys, all_objects):
        to_remove = []
        keys_to_del = set()
        for k, item_list in all_keys.iteritems():
            if len(item_list) > 100:
                source_names = set([item['source'] for item in item_list])
                if len(item_list) > 4 * len(source_names):
                    to_remove += item_list
                    keys_to_del.add(k)
                    for source in source_names:
                        tmp_uuid = uuid.uuid1().hex
                        self._source_message_add_warning(self._('validator.too_many_merges'), source_name=source, item_id=tmp_uuid)
                        for item in item_list:
                            key = NAGIOS_TABLE_KEYS.get(k, '')
                            item_name = item.get(key, item.get('name', self._('validator.missing_name_uppercase'))) or self._('validator.missing_name_uppercase')
                            _msg = self._('validator.element_imported_from') % (item_name, item.get('imported_from', self._('validator.unknown_source')))
                            self._source_message_add_warning(_msg, source_name=source, item_id=tmp_uuid)
        # remove unused keys
        for key in keys_to_del:
            del all_keys[key]
        return [o for o in all_objects if all([sk not in to_remove for sk in o['_SYNC_KEYS']])]
    
    
    @staticmethod
    def _compare_sources(source1, source2):
        or1 = source1.get('source_order')
        or2 = source2.get('source_order')
        if or1 != or2:
            return cmp(or1, or2)
        
        # Try to find some *_name property and look with this
        key = None
        for k in source1.keys():
            if k.endswith('_name'):
                key = k
                break
        if key is not None:
            or1 = source1.get(key)
            or2 = source2.get(key, None)
            if or2 is not None and or1 != or2:
                return cmp(or1, or2)
        # Same name, this denotes a duplicated object somewhere in the same source
        # (duplicated file). Maybe raise a warning.
        
        # Try with from_source if present.
        or1 = source1.get('from_source', None)
        or2 = source2.get('from_source', None)
        if or1 is not None and or2 is not None and or1 != or2:
            return cmp(or1, or2)
        
        # Else use the len of keys
        or1 = len(source1.keys())
        or2 = len(source2.keys())
        if or1 != or2:
            return cmp(or1, or2)
        
        # Else use a concatenation of sorted keys
        or1 = ''.join(sorted(source1.keys()))
        or2 = ''.join(sorted(source2.keys()))
        if or1 != or2:
            return cmp(or1, or2)
        
        # All keys are identical, then try to find a different value
        common_keys = sorted(source1.keys())
        for k in common_keys:
            if k not in ["__SYNC_IDX__", "__XX__", "__MY_BROTHERS__", "__XX_ALREADY_LOOP__", "__ALREADY_PACKED__"] and source1[k] != source2[k]:
                return cmp(source1[k], source2[k])
        # Same number of keys, same keys, same key/value pairs, objects are really identical.
        # Order does not matter:
        return 0
    
    
    def _save_last_synchronization(self):
        for source in self.sources:
            if source.state in (SOURCE_STATE.RUNNING, SOURCE_STATE.NEVER_IMPORT, SOURCE_STATE.READY_FOR_IMPORT, SOURCE_STATE.NOT_CONFIGURED_BUT_CAN_LAUNCH_IMPORT):
                continue
            last_synchronisations = list(self.mongo_component.col_last_synchronization.find({'source_name': source.get_name()}, {'state': 1, 'saved': 1, 'was_clean': 1}))
            if last_synchronisations:
                last_synchronisation = last_synchronisations[0]
            else:
                last_synchronisation = {'state': None}
            if source.was_clean != last_synchronisation.get('was_clean', False) or \
                    source.get_name() in self._nb_item_found_by_sources or \
                    (last_synchronisation['state'] != source.state and source.state != SOURCE_STATE.PENDING):
                logger.debug('[merger_thread][%s] saving an entry in the last synchronization logs' % source.get_name())
                # done_import set the next import of the source and make the last_synchronizations and last_synchronization entry
                entry = source.done_import()
                entry['saved'] = self._nb_item_found_by_sources.get(source.get_name(), last_synchronisation.get('saved', {}))
                self._nb_item_found_by_sources.pop(source.get_name(), None)
                if entry['state'] != SOURCE_STATE.NEVER_IMPORT:
                    self.mongo_component.col_last_synchronizations.save(entry, w=1)
                self.mongo_component.col_last_synchronization.remove({'source_name': source.get_name()}, w=1)
                self.mongo_component.col_last_synchronization.save(entry, w=1)
    
    
    def _load_tmp_collections(self, items, import_uuid, item_class, item_state):
        str_time = datetime.now().strftime('%d-%m-%Y %H-%M-%S')
        tmp_col_name = 'tmp-%s-%s-%s-%s' % (import_uuid, item_class, item_state, str_time)
        return self._save_items_in_col(items, tmp_col_name, item_state)
    
    
    def _save_items_in_col(self, items, collection_name, item_state, to_save_hashs=None):
        col = self.mongo_component.get_collection(collection_name)
        for item in items:
            if item_state == ITEM_STATE.CHANGES:
                item_type = ITEM_TYPE.CONTACTS
            else:
                item_source_hash = METADATA.get_metadata(item, METADATA.SOURCE_HASH)
                item_type = METADATA.get_metadata(item, METADATA.ITEM_TYPE)
                source_info = METADATA.get_metadata(item, METADATA.SOURCE_INFO)
                if to_save_hashs and item_source_hash not in to_save_hashs:
                    continue
                item.pop('@metadata', {})
                METADATA.update_metadata(item, METADATA.SOURCE_HASH, item_source_hash)
                METADATA.update_metadata(item, METADATA.ITEM_TYPE, item_type)
                METADATA.update_metadata(item, METADATA.SOURCE_INFO, source_info)
            self.app.database_cipher.cipher(item=item, item_type=item_type, item_state=item_state)
        try:
            col.insert_many(items)
        except BulkWriteError as bwe:
            logger.error("save [%s] items in collection [%s] failed." % (len(items), collection_name))
            logger.error("BulkWriteError.details [%s]" % bwe.details)
            raise
        
        return col
    
    
    def _find_existing_item_match(self, item_type, merge_item):
        # Return a find staging match and if there are an other match with this name
        # see http://151.80.162.119:8080/browse/SEF-1131?focusedCommentId=13402&page=com.atlassian.jira.plugin.system.issuetabpanels:comment-tabpanel#comment-13402
        
        merge_item_name = get_name_from_type(item_type, merge_item)
        item_from_ui_with_same_name = None
        if item_type not in ITEM_TYPE.ALL_DEDICATED_SERVICES:
            item_from_ui_with_same_name = self._existing_item_datamanagerV2.find_item_by_name(merge_item_name, item_type, ITEM_STATE.STAGGING) or self._existing_item_datamanagerV2.find_item_by_name(merge_item_name, item_type, ITEM_STATE.WORKING_AREA)
        
        if '_SE_UUID' in merge_item and merge_item['_SE_UUID'].strip().count('-') == 2:
            _se_uuid_uuid = merge_item['_SE_UUID'].strip().split('-', 2)[2].strip()
            if _se_uuid_uuid:
                item_from_ui = self._existing_item_datamanagerV2.find_item_by_id(_se_uuid_uuid, item_type, ITEM_STATE.WORKING_AREA) or self._existing_item_datamanagerV2.find_item_by_id(_se_uuid_uuid, item_type, ITEM_STATE.STAGGING)
                if item_from_ui:
                    item_from_ui_name = get_name_from_type(item_type, item_from_ui)
                    return [item_from_ui], item_from_ui_name != merge_item_name and item_from_ui_with_same_name
                else:
                    return [], item_from_ui_with_same_name is not None
        
        matching_items = []
        different_names = False
        _sync_keys = merge_item.get('_SYNC_KEYS', [])
        _where = {'_SYNC_KEYS': {'$in': _sync_keys}}
        item_from_ui = self._existing_item_datamanagerV2.find_items(item_type, ITEM_STATE.WORKING_AREA, where=_where) or self._existing_item_datamanagerV2.find_items(item_type, ITEM_STATE.STAGGING, where=_where)
        if len(item_from_ui) == 1:
            item_from_ui = item_from_ui[0]
            item_from_ui_name = get_name_from_type(item_type, item_from_ui)
            matching_items.append(item_from_ui)
            different_names = different_names or (item_from_ui_with_same_name is not None and item_from_ui_name != merge_item_name and item_from_ui_with_same_name['_id'] != item_from_ui['_id'])
        elif item_from_ui:
            matching_items = item_from_ui
            different_names = True
        
        if matching_items:
            return matching_items, different_names
        
        if item_from_ui_with_same_name:
            return [item_from_ui_with_same_name], False
        return [], False
    
    
    @staticmethod
    def _restore_id_from_last_import(new_elt_collection, news):
        # the replaced id with the new one
        all_old_news = new_elt_collection.find({}, {'_id': 1, '_SYNC_KEYS': 1})
        old_ids = {}
        for old_new in all_old_news:
            _id = old_new.get('_id', '')
            if not _id:
                continue
            _sync_keys = old_new.get('_SYNC_KEYS', [])
            for sk in _sync_keys:
                old_ids[sk] = _id
        
        # unify the elements
        _ids_already_set = set()
        for new_entry in news:
            _id = new_entry.get('_id', '')
            _se_uuid = new_entry.get('_SE_UUID', '')
            if not _id:
                continue
            
            if _se_uuid and _se_uuid.strip().count('-') == 2:
                _uuid = _se_uuid.strip().split('-', 2)[2].strip()
                if _uuid:
                    new_entry['_id'] = _uuid
            else:
                is_reset_id = True
                _sync_keys = new_entry.get('_SYNC_KEYS', [])
                for sk in _sync_keys:
                    if sk in old_ids and old_ids[sk] not in _ids_already_set:
                        new_id = old_ids[sk]
                        new_entry['_id'] = new_id
                        is_reset_id = False
                        break
                
                # With cache, we can have a new and staging item with same _id and not same _SYNC_KEY
                if is_reset_id:
                    new_entry['_id'] = uuid.uuid4().hex
            
            _ids_already_set.add(new_entry['_id'])
        
        # unify the elements, so take only one element for each
        news_dct_ = {}
        for n in news:
            id_ = n.get('_id', '')
            if not id_:
                continue
            news_dct_[id_] = n
        return news_dct_.values()
    
    
    def _restore_backup_collection(self, backup_collection):
        try:
            for (backup_col_name, restore_name) in backup_collection:
                if backup_col_name is None:
                    coll = self.mongo_component.get_collection(restore_name)
                    coll.remove()
                else:
                    coll = self.mongo_component.get_collection(backup_col_name)
                    coll.rename(restore_name, dropTarget=True)
        except Exception as exp:
            logger.error('[merger_thread] CRITICAL, can not restore backup tables. News and changes tables may be inconsistent. (%s)' % exp)
    
    
    def _backup_collection(self, backup_collection, final_tables_to_write):
        for (new_mongo_collection, collection_name) in final_tables_to_write:
            backup_col_name = '%s-backup' % collection_name
            if collection_name not in self.mongo_component.list_name_collections():
                # no old collection
                backup_collection.append((None, collection_name))
            
            else:
                old_mongo_collection = self.mongo_component.get_collection(collection_name)
                old_mongo_collection.rename(backup_col_name, dropTarget=True)
                backup_collection.append((backup_col_name, collection_name))
    
    
    #
    #    _____                               ____                           __
    #   / ___/____  __  _______________     /  _/___ ___  ____  ____  _____/ /_
    #   \__ \/ __ \/ / / / ___/ ___/ _ \    / // __ `__ \/ __ \/ __ \/ ___/ __/
    #  ___/ / /_/ / /_/ / /  / /__/  __/  _/ // / / / / / /_/ / /_/ / /  / /_
    # /____/\____/\__,_/_/   \___/\___/  /___/_/ /_/ /_/ .___/\____/_/   \__/
    #                                                 /_/
    
    def _source_import_thread(self, src):
        # type: (Source) -> None
        logger.info('[import_thread][%s] starting import' % src.get_name())
        src.do_import(NAGIOS_TABLES.keys())
        # If did finish in a way, we need to let the source to no not run until
        try:
            self._save_source_objects(src)
        except ShinkenMongoException:
            error_message = u'Cannot save import because Mongo is unreachable, import aborted'
            src.state = SOURCE_STATE.CRITICAL
            src.output = error_message
            src.fully_merged = True
            logger.error(error_message)
            return
        except Exception:
            error_message = u'Cannot save import, import aborted'
            src.state = SOURCE_STATE.CRITICAL
            src.output = u'%s. Please see the logs' % error_message
            src.fully_merged = True
            logger.error(error_message)
            logger.print_stack()
            return
        
        self.merged_asked = True  # OK, please load me because I finish
        src.merge_asked = True
        src.state = src.import_state
        logger.info('[import_thread][%s] import done' % src.get_name())
    
    
    def _save_source_objects(self, source, already_validate=False):
        # type: (Source, bool) -> NoReturn
        
        # logger.info('[DEBUG SOURCE][%s][save_source_objects] starting. Items are already_validate:[%s]' % (source.get_name(), already_validate))
        start_save_time = time.time()
        self._source_message_init(source.get_name())
        self._source_summary_init(source.get_name())
        items_by_item_type = {}
        
        for item_class in NAGIOS_TABLES:
            # logger.info('[DEBUG SOURCE][%s][save_source_objects][%s] preparing' % (source.get_name(), item_class))
            # we need to validate some fields before working on this items
            source_objects = source.objects.get(item_class, [])
            if source.import_state not in (SOURCE_STATE.OK, SOURCE_STATE.WARNING):
                logger.debug('Skipping source %s objects import, the state is not good (%s)' % (source.get_name(), source.import_state))
                source_objects = []
            if not already_validate:
                if item_class == 'host':
                    # logger.info('[DEBUG SOURCE][%s][save_source_objects][%s] migrate_hosts_contacts_properties' % (source.get_name(), item_class))
                    migrate_hosts_contacts_properties(self.app, source_objects)
                elif item_class == 'contact':
                    # logger.info('[DEBUG SOURCE][%s][save_source_objects][%s] migrate_contact_contactgroups' % (source.get_name(), item_class))
                    migrate_contact_contactgroups(source_objects)
            
            # logger.info('[DEBUG SOURCE][%s][save_source_objects][%s] start sort  item by type' % (source.get_name(), item_class))
            for item in source_objects:
                # don't put an "if already_validate: break". That no working.
                if not already_validate:
                    # validate that a service have the right apply_on types: can't be in the validator because we need to be sure that this info is right to compute the correct item_type.
                    if item_class == 'service':
                        if item.get('register', '1') == '0':
                            valid_apply_on = ('', 'hosttpls', 'clustertpls')
                        else:
                            valid_apply_on = ('', 'hosts', 'clusters')
                        if item.get('apply_on_type', '') not in valid_apply_on:
                            _msg = self._('validator.invalid_apply_on_type') % (
                                ToolsBoxString.escape_XSS(item.get('apply_on_type', '')),
                                item.get('imported_from', self._('validator.unknown_source')),
                                ToolsBoxString.escape_XSS(item.get('register', '1')),
                                valid_apply_on[1:], valid_apply_on[1]
                            )
                            self._source_message_add_warning(_msg, item, imported=False)
                            item['invalid'] = True
                            continue
                    
                    item_type = get_type_item_from_class(item_class + 's', item)
                    
                    if item_type in ITEM_TYPE.ALL_HOST_CLASS:
                        migrate_to_clusters(item)
                
                # Always be sure that the ITEM_TYPE metadata is up-to-date, even if we did migrate
                item_type = get_type_item_from_class(item_class + 's', item)
                METADATA.update_metadata(item, METADATA.ITEM_TYPE, item_type)
                
                # keep a trace of it
                safe_add_to_dict(items_by_item_type, item_type, item)
            
            # logger.info('[DEBUG SOURCE][%s][save_source_objects][%s] end sort item by type' % (source.get_name(), item_class))
        
        tmp_col_hash = uuid.uuid1().hex[:8]
        to_save_item_count = {}
        unique_ids_by_class = {}
        for item_type, items in items_by_item_type.iteritems():
            for item in items:
                # Protect against UTF8 because if not mongo will not be happy
                self._convert_to_utf_8(item)
            total_nb_items = len(items)
            # logger.info('[DEBUG SOURCE][%s][save_source_objects][%s] start bulk for [%s] items' % (source.get_name(), item_type, total_nb_items))
            
            item_class = DEF_ITEMS[item_type]['table']
            tmp_col_name = 'tmp-%s-data-%s-%s-%s' % (int(start_save_time), source.get_name(), item_class, tmp_col_hash)
            tmp_col = self.mongo_component.get_collection(tmp_col_name)
            item_to_insert = []
            
            unique_ids = unique_ids_by_class.get(item_class, set())
            unique_ids_by_class[item_class] = unique_ids
            sync_keys_index = self._build_sync_keys_index(items, item_type, source)
            
            # We add the sync_keys_index and possible_sync_keys for this source into metadata, because we need to check the sync_keys in the validator.
            # After the validator, we can remove the metadata
            possible_sync_keys = getattr(source.modules[0], 'possible_keys', set()).copy()
            possible_sync_keys.add(DEF_ITEMS[item_type]['key_name'])
            
            for item_index, item in enumerate(sorted(items, key=lambda i: i.get(DEF_ITEMS[item_type]['key_name'], '').lower()), 1):
                self.remove_empty_properties(item)
                
                METADATA.update_metadata(item, METADATA.SYNC_KEYS_INDEX, sync_keys_index)
                METADATA.update_metadata(item, METADATA.POSSIBLE_SYNC_KEYS, possible_sync_keys)
                _validation = self.import_validator.validate(item_type, item)
                METADATA.remove_metadata(item, METADATA.SYNC_KEYS_INDEX)
                METADATA.remove_metadata(item, METADATA.POSSIBLE_SYNC_KEYS)
                
                name = get_name_from_type(item_type, item)
                warnings_messages = []
                errors_messages = []
                
                for level, rule, message in _validation['qualified_message']:
                    if level == Validator.LEVEL_VALIDATOR_WARNING:
                        self._add_import_warnings_on_item(item, message)
                        warnings_messages.append(message)
                    
                    elif level in (Validator.LEVEL_VALIDATOR_CRITICAL, Validator.LEVEL_VALIDATOR_ERROR):
                        
                        if rule == '_rule_sync_keys_must_be_common_in_source':
                            # We create a summary on last_run page with the original message (only a few number defined by the user will be shown)
                            level = 'error'
                            real_number_of_message = len(self._source_messages_summary.get(source.get_name(), {}).get(level, {}).get(rule, {}).get('messages', [])) + 1
                            number_to_show = real_number_of_message if real_number_of_message < self.number_of_message_in_source_summary else self.number_of_message_in_source_summary
                            self._source_summary_add(message, rule, self._('validator.title_rule_sync_keys') % number_to_show, level=level, item=item)
                            
                            # On detail page, for each item we add a message with a link to the last_run page
                            self._add_import_errors_on_item(item, self._('validator.message_to_redirect_to_last_run_page') % {
                                'item_type': self._('type.%s' % item_type[:-1]),
                                'page'     : '<a href="/sources/%s?tab=tab-summary-old-run">%s</a>' % (source.get_name(), self._('source.summary_old_run'))
                            })
                            
                            # On the last_run page we add a generic message on each item to say to view error in summary
                            errors_messages.append(self._('validator.generic_message_for_sync_keys') % {'item_type': self._('type.%s' % item_type)})
                        
                        else:
                            
                            self._add_import_errors_on_item(item, message)
                            errors_messages.append(message)
                
                if errors_messages:
                    start_tags = '<ul class="shinken-status-list"><li>'
                    end_tags = '</li></ul>'
                    err_msg = start_tags + '</li><li>'.join(errors_messages) + end_tags
                    _msg = self._('validator.imported_element_error') % (
                        self._('type.%s' % item_type[:-1]),
                        '<a href="?tab=tab-detail-last-run&filter=name:%s">%s</a>' % (ToolsBoxString.escape_XSS(name), ToolsBoxString.escape_XSS(name)),
                        err_msg
                    )
                    self._source_message_add_error(_msg, item, imported=False)
                
                if warnings_messages:
                    start_tags = '<ul class="shinken-status-list"><li>'
                    end_tags = '</li></ul>'
                    err_msg = start_tags + '</li><li>'.join(warnings_messages) + end_tags
                    _msg = self._('validator.skipped') % (
                        self._('type.%s' % item_type[:-1]),
                        '<a href="?tab=tab-detail-last-run&filter=name:%s">%s</a>' % (ToolsBoxString.escape_XSS(name), ToolsBoxString.escape_XSS(name)),
                        err_msg
                    )
                    self._source_message_add_warning(_msg, item, imported=False)
                
                make_item_hash(item)
                
                item = self.app.database_cipher.cipher(item, item_type)
                item_to_insert.append(item)
                if item_class not in to_save_item_count:
                    to_save_item_count[item_class] = 0
                to_save_item_count[item_class] += 1
                unique_ids.add(item['_id'])
            
            if len(unique_ids) != to_save_item_count[item_class]:
                logger.error("[import_thread][%s] Error in saving import items : there is DUPLICATE IDS for %s" % (source.get_name(), item_type))
                source.import_state = SOURCE_STATE.CRITICAL
                source.output = self._('source.duplicate_ids') % self._('type.' + item_type)
                return
            
            # Protect this data access with the merger thread, so they don't have issues
            with source.get_data_lock():
                t0 = time.time()
                tmp_col.insert_many(item_to_insert)
                logger.debug('[import_thread][%s] save [%4d][%s] in [%0.6f]s' % (source.get_name(), to_save_item_count[item_class], item_class, (time.time() - t0)))
        
        logger.debug('[import_thread][%s] save_source_objects:: finish in [%0.6f]s' % (source.get_name(), (time.time() - start_save_time)))
        
        with source.get_data_lock():
            for item_class in NAGIOS_TABLES.iterkeys():
                col_name = 'data-%s-%s' % (source.get_name(), item_class)
                if to_save_item_count.get(item_class, 0) == 0:
                    col = self.mongo_component.get_collection(col_name)
                    col.remove()
                else:
                    tmp_col_name = 'tmp-%s-data-%s-%s-%s' % (int(start_save_time), source.get_name(), item_class, tmp_col_hash)
                    tmp_col = self.mongo_component.get_collection(tmp_col_name)
                    tmp_col.rename(col_name, dropTarget=True)
    
    
    def _build_sync_keys_index(self, items, item_type, source):
        _index = {}
        for item in items:
            self._clean_sync_keys(item, item_type, source)
            for key in item['_SYNC_KEYS']:
                safe_add_to_dict(_index, key, item)
        
        return _index
    
    
    @staticmethod
    def _add_import_errors_on_item(item, message):
        # attach errors and warnings inside the object metadata
        item_metadata = METADATA.get_metadata(item, METADATA.IMPORT_ERRORS, [])
        item_metadata.append(message)
        METADATA.update_metadata(item, METADATA.IMPORT_ERRORS, item_metadata)
    
    
    @staticmethod
    def _add_import_warnings_on_item(item, message):
        # attach errors and warnings inside the object metadata
        item_metadata = METADATA.get_metadata(item, METADATA.IMPORT_WARNINGS, [])
        item_metadata.append(message)
        METADATA.update_metadata(item, METADATA.IMPORT_WARNINGS, item_metadata)
    
    
    @staticmethod
    def _convert_to_utf_8(item):
        for (item_property, value) in item.iteritems():
            item[item_property] = value
            if isinstance(value, str):
                value = value.decode('utf8', 'ignore')
                item[item_property] = value
            if isinstance(value, list):
                _items = []
                for s in value:
                    if isinstance(s, str):
                        _items.append(s.decode('utf8', 'ignore'))
                    else:
                        _items.append(s)
                item[item_property] = _items
    
    
    @staticmethod
    def _clean_sync_keys(item, item_type, source):
        se_uuid = item.get('_SE_UUID', None)
        if se_uuid:
            se_uuid = se_uuid.lower()
        name = ''
        if item_type not in ITEM_TYPE.ALL_DEDICATED_SERVICES:
            name = item.get(DEF_ITEMS[item_type]['key_name'], None)
        existing_sync_keys = item.get('_SYNC_KEYS', item.pop('_sync_keys', None))
        
        if isinstance(existing_sync_keys, basestring):
            _sync_keys = [s.strip().lower() for s in existing_sync_keys.split(',') if s.strip()]
        else:
            if existing_sync_keys:
                logger.warning("[import_thread][%s] Item already have _SYNC_KEYS but the format is not correct. Must regenerate it on item [%s]" % (source.get_name(), item))
            else:
                logger.warning("[import_thread][%s] Item does not have _SYNC_KEYS. Must generate it on item [%s]" % (source.get_name(), item))
            
            _sync_keys = []
        
        if se_uuid and (se_uuid not in _sync_keys):
            _sync_keys.append(se_uuid.lower())
        
        if name:
            name = name.lower()
            if ITEM_TYPE.is_template(item_type):
                name += '-tpl'
            if name not in _sync_keys:
                _sync_keys.append(name)
        
        if not _sync_keys:
            # we generate a sync keys from item values
            _sync_keys = [md5(''.join([str(v.encode('utf8') if isinstance(v, unicode) else v) for v in item.itervalues()])).hexdigest()]
            logger.debug("[import_thread][%s] Generating sync keys : can not find name and _SE_UUID on item. _SYNC_KEYS was compute from item values [%s]" % (source.get_name(), _sync_keys))
        
        item['_SYNC_KEYS'] = _sync_keys
    
    
    @staticmethod
    def _count_item_found_in_source(nb_elements_in_source, item, item_class):
        # type: (Dict, Dict, unicode) -> None
        item_type = get_type_item_from_class(item_class + u's', item)
        nb_elements_in_source[item_type] = nb_elements_in_source.get(item_type, {u'error': 0, u'warning': 0, u'nb_elements': 0, u'ok': 0})
        
        import_warning = METADATA.get_metadata(item, METADATA.IMPORT_WARNINGS)
        import_error = METADATA.get_metadata(item, METADATA.IMPORT_ERRORS)
        
        if import_error:
            nb_elements_in_source[u'nb_elements_error'] += 1
            nb_elements_in_source[item_type][u'error'] += 1
        elif import_warning:
            nb_elements_in_source[u'nb_elements_warning'] += 1
            nb_elements_in_source[item_type][u'warning'] += 1
        else:
            nb_elements_in_source[u'nb_elements_ok'] += 1
            nb_elements_in_source[item_type][u'ok'] += 1
        
        nb_elements_in_source[item_type][u'nb_elements'] += 1
        nb_elements_in_source[u'nb_elements'] += 1
    
    
    @staticmethod
    def update_nb_elements_in_source(source, nb_elements_in_source):
        # type: (Source, Dict) -> None
        if nb_elements_in_source[u'nb_elements'] or nb_elements_in_source[u'nb_elements'] != source.nb_elements:
            source.nb_elements = nb_elements_in_source[u'nb_elements']
        if nb_elements_in_source[u'nb_elements_ok'] or nb_elements_in_source[u'nb_elements_ok'] != source.nb_elements_ok:
            source.nb_elements_ok = nb_elements_in_source[u'nb_elements_ok']
        if nb_elements_in_source[u'nb_elements_error'] or nb_elements_in_source[u'nb_elements_error'] != source.nb_elements_error:
            source.nb_elements_error = nb_elements_in_source[u'nb_elements_error']
        if nb_elements_in_source[u'nb_elements_warning'] or nb_elements_in_source[u'nb_elements_warning'] != source.nb_elements_warning:
            source.nb_elements_warning = nb_elements_in_source[u'nb_elements_warning']
    
    
    @staticmethod
    def remove_empty_properties(item):
        # we loop on items and not iteritems because we will update item in the loop
        for properties_name, value in item.items():
            if not value:
                del item[properties_name]
