#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2018
# This file is part of Shinken Enterprise, all rights reserved.

from shinken.log import logger
from shinken.misc.type_hint import TYPE_CHECKING
from shinkensolutions.ssh_mongodb.mongo_client import MongoClient
from .abstract_component import AbstractComponent
from ..dao.def_items import ITEM_TYPE

if TYPE_CHECKING:
    from shinken.misc.type_hint import TYPE_CHECKING, Optional, Str, Any, Dict, NoReturn, List
    from shinken.log import PartLogger
    from shinkensolutions.ssh_mongodb.mongo_collection import MongoCollection
    from .configuration_component import ConfigurationComponent

PROPERTIES_USED_AS = u'properties_used_as_synckey_for_'


class MongoComponent(AbstractComponent, MongoClient):
    def __init__(self, logger, configuration_component, requester):
        # type: (PartLogger, ConfigurationComponent, str) -> None
        MongoClient.__init__(self, configuration_component.get_mongo_configuration(logger=logger), logger=logger)
        self.requester = requester
        
        self.col_internal_source_confs = None  # type: Optional[MongoCollection]
        self.col_last_synchronization = None  # type: Optional[MongoCollection]
        self.col_last_synchronizations = None  # type: Optional[MongoCollection]
        self.col_configuration_stagging_contact = None  # type: Optional[MongoCollection]
        self.col_configuration_user_preferences = None  # type: Optional[MongoCollection]
        self.col_discovery_confs = None  # type: Optional[MongoCollection]
        self.col_synchronizer_info = None  # type: Optional[MongoCollection]
        self.col_deleted_items = None  # type: Optional[MongoCollection]
    
    
    def init(self):
        # type: () -> None
        MongoClient.init(self, self.requester)
        self._retrieve_collections()
    
    
    def recreate_connection(self, requester=u'unknown'):
        # type: (unicode) -> None
        MongoClient.recreate_connection(self, self.requester)
        self._retrieve_collections()
    
    
    def _retrieve_collections(self):
        # type: () -> None
        self.col_internal_source_confs = self.get_collection(u'internal_source_confs')
        self.col_last_synchronization = self.get_collection(u'last_synchronization')
        self.col_last_synchronizations = self.get_collection(u'last_synchronizations')
        self.col_configuration_stagging_contact = self.get_collection(u'configuration-stagging-contact')
        self.col_configuration_user_preferences = self.get_collection(u'user-preferences')
        self.col_discovery_confs = self.get_collection(u'discovery_confs')
        self.col_synchronizer_info = self.get_collection(u'synchronizer-info')
        self.col_deleted_items = self.get_collection(u'deleted_items')
    
    
    def clean_source_backup_collection(self):
        # type: () -> None
        try:
            for collection_name in self.list_name_collections():
                if collection_name.endswith(u'-backup'):
                    old_mongo_collection = self.get_collection(collection_name)
                    old_mongo_collection.remove()
        except Exception as exp:
            logger.warning(u'Failed to cleanup backup collections. (%s)' % exp)
    
    
    # ****************************************************** LAST SYNCHRO  ********************************************* #
    def query_source_last_synchronization(self, source_name):
        # type: (Str) -> Optional[Dict[unicode, Any]]
        last_synchronization = self.col_last_synchronization.find_one({'source_name': source_name})
        self._format_last_synchronization_loaded(last_synchronization)
        return last_synchronization
    
    
    @staticmethod
    def _format_last_synchronization_loaded(last_synchronization):
        # type: (Dict) -> NoReturn
        if last_synchronization:
            for type_to_import in ITEM_TYPE.ALL_TYPES:
                _prop_name = '%s%s' % (PROPERTIES_USED_AS, type_to_import)
                properties_used_as_by_type = last_synchronization.get(u'options', {}).get(_prop_name, None)
                if isinstance(properties_used_as_by_type, list):
                    last_synchronization['options'][_prop_name] = set(properties_used_as_by_type)
    
    
    @staticmethod
    def _format_last_synchronization_to_save(last_synchronization):
        # type: (Dict) -> NoReturn
        
        for type_to_import in ITEM_TYPE.ALL_TYPES:
            _prop_name = u'%s%s' % (PROPERTIES_USED_AS, type_to_import)
            properties_used_as_by_type = last_synchronization.get(u'options', {}).get(_prop_name, None)
            if isinstance(properties_used_as_by_type, set):
                last_synchronization[u'options'][_prop_name] = list(properties_used_as_by_type)
    
    
    def save_source_last_synchronization(self, last_synchronization):
        # type: (Dict) -> NoReturn
        self._format_last_synchronization_to_save(last_synchronization)
        self.col_last_synchronization.save(last_synchronization)
    
    
    def replace_source_last_synchronization(self, last_synchronization):
        # type: (Dict) -> NoReturn
        self.col_last_synchronization.remove({'source_name': last_synchronization[u'source_name']}, w=1)
        self.col_last_synchronization.save(last_synchronization, w=1)
    
    
    def save_in_last_synchronizations(self, last_synchronization):
        # type: (Dict) -> NoReturn
        self._format_last_synchronization_to_save(last_synchronization)
        self.col_last_synchronizations.save(last_synchronization, w=1)
    
    
    def query_last_synchronizations(self, source_name, source_time):
        # type: (unicode, int) -> Dict
        last_synchronization = self.col_last_synchronizations.find_one({u'source_name': source_name, u'synchronization_time': source_time})
        self._format_last_synchronization_loaded(last_synchronization)
        return last_synchronization
    
    
    def query_all_last_synchronizations_for_one_source(self, source_name):
        # type: (unicode) -> List[Dict]
        last_synchronizations = self.col_last_synchronizations.find({u'source_name': source_name})
        for i in last_synchronizations:
            self._format_last_synchronization_loaded(i)
        return last_synchronizations
    
    
    # ****************************************************** USERS  ********************************************* #
    
    def query_users_staging(self, user_uuid=None, name=None):
        # type: (unicode, unicode) -> Dict
        if not name and not user_uuid:
            return self.col_configuration_stagging_contact.find()
        if user_uuid:
            return self.col_configuration_stagging_contact.find_one({'_id': user_uuid})
        return self.col_configuration_stagging_contact.find_one({'contact_name': name})
    
    
    def query_user_preferences(self, user_uuid):
        # type: (unicode) -> Dict
        return self.col_configuration_user_preferences.find_one({'_id': user_uuid})
    
    
    def save_user_preferences(self, user_prefs):
        # type: (Dict) -> NoReturn
        self.col_configuration_user_preferences.save(user_prefs, w=1)
    
    
    def delete_user_preferences(self, user_uuid):
        # type: (unicode) -> NoReturn
        self.col_configuration_user_preferences.remove({'_id': user_uuid}, w=1)
