#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2021:
# This file is part of Shinken Enterprise, all rights reserved.

try:
    from collections import Counter
except ImportError:
    from shinken.misc.counter import Counter
    
from shinken.misc.type_hint import TYPE_CHECKING
from .base_callback import PreSaver
from ..def_items import ITEM_TYPE
from ..helpers import get_name_from_type

if TYPE_CHECKING:
    from shinken.misc.type_hint import Optional, Dict
    from synchronizer.dao.datamanagerV2 import DataManagerV2
    from synchronizer.dao.items import ContactItem


class _CallbackRecomputeSynchronizationKeys(PreSaver):
    def called_before_save(self, item_id=u'', item_type=u'', item_state=u'', item=None, old_item=None, user=None, action=u'', datamanager=None):
        # type: (unicode, unicode, unicode, Optional[Dict], Optional[Dict], Optional[ContactItem], unicode, Optional[DataManagerV2]) -> None
        
        # This is not supposed to happen, but just in case we exit if the item does not exist
        if not item:
            return
        
        # We need to force the old_item to be an object in order to call the get method.
        # If the old_item is None, it means that the item is a new item
        if old_item is None:
            old_item = {}
        
        old_name = get_name_from_type(item_type, old_item).lower()
        new_name = get_name_from_type(item_type, item).lower()
        
        # Formatting synchronization keys and removing duplicated values
        sync_keys = item.get(u'_SYNC_KEYS', [])
        if not isinstance(sync_keys, list):
            sync_keys = [key.strip() for key in sync_keys.split(u',')]
        sync_keys = set((key.lower() for key in sync_keys))
        
        old_sync_keys = old_item.get(u'_SYNC_KEYS', [])
        if not isinstance(old_sync_keys, list):
            old_sync_keys = [key.strip() for key in old_sync_keys.split(u',')]
        # Getting duplicates in the old sync_keys, duplicates sync_keys are keys added with
        # « properties_used_as_synckeys » and we do not want to remove it
        
        properties_used_as_synckey = [key for key, count in Counter(old_sync_keys).iteritems() if count > 1]
        
        # host name and address did not change, no need to change the synchronization keys
        if old_name == new_name:
            item[u'_SYNC_KEYS'] = list(sync_keys)
            return
        
        is_service = item_type in ITEM_TYPE.ALL_DEDICATED_SERVICES
        is_tpl = ITEM_TYPE.is_template(item_type)
        
        if not sync_keys:
            if is_service:
                sync_keys.add((u'core-%s-%s' % (item_type, item_id)).lower())
            elif is_tpl:
                sync_keys.add(u'%s-tpl' % new_name)
            else:
                sync_keys.add(new_name)
        
        elif not is_service:
            if is_tpl:
                old_name = u'%s-tpl' % old_name
                new_name = u'%s-tpl' % new_name
            elif is_service:
                old_name = (u'core-%s-%s' % (item_type, item_id)).lower()
                new_name = (u'core-%s-%s' % (item_type, item_id)).lower()
            
            # We remove the old name in sync_keys and add the new name if user changed it
            # We must keep old name in sync keys if it is equals to the previous address because some sources (like discovery) use address as synchronization keys
            if old_item.get('address', None) != old_name and old_name != new_name and old_name in sync_keys:
                sync_keys.remove(old_name)
            sync_keys.add(new_name)
            
            # After cleaning the sync_keys, we add the properties used as synckey
            for key in properties_used_as_synckey:
                sync_keys.add(key)
        
        item[u'_SYNC_KEYS'] = list(sync_keys)


_callback_recompute_synchronization_keys = _CallbackRecomputeSynchronizationKeys()
