#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2018:
# This file is part of Shinken Enterprise, all rights reserved.

from shinken.misc.type_hint import TYPE_CHECKING
from shinken.objects import Businessimpactmodulation, Command, Contact, Contactgroup, Escalation, Host, Hostgroup, MacroModulation, NotificationWay, Resultmodulation, Service, Timeperiod

if TYPE_CHECKING:
    from shinken.misc.type_hint import Str, List, Dict, Any

ItemType = unicode  # Type of ITEM_TYPE. For type hint.
ItemState = unicode  # Type of ITEM_STATE. For type hint.


class SOURCE_STATE(object):
    NEW = u'new'
    CHANGE = u'change'
    IMPORT = u''


class METADATA(object):
    SOURCE_INFO = u'source_info'
    SOURCE_HASH = u'source_hash'
    NEW = u'new'
    HAS_DATA = u'has_data'
    IN_CREATION = u'in_creation'
    HASH = u'hash'
    ITEM_TYPE = u'type'
    HOST_TYPE = u'host_type'
    HOST_UUID = u'host_uuid'
    CHANGES = u'changes'
    STATE = u'state'
    AUTO_DOUBLE_LINKS = u'auto_double_links'
    IMPORT_WARNINGS = u'import_warnings'
    IMPORT_ERRORS = u'import_errors'
    UPDATE_DATE = u'update_date'
    FROM = u'from'
    CRYPTED = u'crypted'
    UNCRYPTED = u'uncrypted'
    TABLE = u'table'
    NAME = u'name'
    DFE_NAME = u'dfe_name'
    WINDOW_UUID = u'window_uuid'
    SOURCE_STATE = u'source_state'
    REVERSE_LINKS = u'reverse_links'
    DOUBLE_LINKS_ERROR = u'double_links_error'
    FORMATED_NAME = u'formated_name'
    HAS_PLUS = u'has_plus'
    RAW_ITEM = u'raw_item'
    CHECKS = u'checks'
    REVERSE_LINKS_CHECKS = u'reverse_links_checks'
    SONS = u'sons'
    SYNC_KEYS_INDEX = u'source_objects'  # Used in validator to check the sync_keys
    POSSIBLE_SYNC_KEYS = u'possible_sync_keys'  # Used in validator to show possible sync_key to the user
    VALIDATE_DOLLAR_NUMBER = u'validate_dollar_number'  # Used in validator to check the number of dollar
    
    ORIGIN_ITEM_INFO = u'origin_item_info'
    
    
    @staticmethod
    def update_metadata(item, key, value):
        metadata = item.get(u'@metadata', {})
        metadata[key] = value
        item[u'@metadata'] = metadata
    
    
    @staticmethod
    def remove_metadata(item, key):
        metadata = item.get(u'@metadata', {})
        return metadata.pop(key, None)
    
    
    @staticmethod
    def get_metadata(item, key, default=None):
        # type: (Dict, unicode, Any) -> Any
        return item.get(u'@metadata', {}).get(key, default)


NAGIOS_TABLES = {
    'command'                 : (Command,),
    'host'                    : (Host,),
    'timeperiod'              : (Timeperiod,),
    'contact'                 : (Contact,),
    'contactgroup'            : (Contactgroup,),
    'hostgroup'               : (Hostgroup,),
    'businessimpactmodulation': (Businessimpactmodulation,),
    'notificationway'         : (NotificationWay,),
    'escalation'              : (Escalation,),
    'macromodulation'         : (MacroModulation,),
    'resultmodulation'        : (Resultmodulation,),
    'service'                 : (Service,),
}
NAGIOS_TABLE_KEYS = {
    'host'                    : 'host_name',
    'command'                 : 'command_name',
    'timeperiod'              : 'timeperiod_name',
    'contact'                 : 'contact_name',
    'contactgroup'            : 'contactgroup_name',
    'hostgroup'               : 'hostgroup_name',
    'businessimpactmodulation': 'business_impact_modulation_name',
    'notificationway'         : 'notificationway_name',
    'escalation'              : 'escalation_name',
    'macromodulation'         : 'macromodulation_name',
    'resultmodulation'        : 'resultmodulation_name',
    'service'                 : ''
}

ADVANCED_PERIODS = 'advanced_time_periods'
SERVICE_OVERRIDE = 'service_overrides'
SERVICE_EXCLUDES_BY_ID = 'service_excludes_by_id'
SERVICE_EXCLUDES_BY_NAME = 'service_excludes'
SERVICE_OVERRIDE_UUID_SEP = '!uuid!'
SERVICE_EXCLUDES_DFE_KEY = '!dfe_key!'
PROP_DEFAULT_VALUE = '__DEFAULT_NO_TEMPLATE__'
VALUE_FORCE_DEFAULT = 'null'
VALUE_NONE = 'None'
PROP_DEFAULT_OR_INHERITANCE = '__DEFAULT_OR_INHERITANCE__'
ENTER_VALUE_FOR_MASS_CHANGE = '__ENTER_VALUE_FOR_MASS_CHANGE__'

STOP_INHERITANCE_VALUES = (VALUE_FORCE_DEFAULT, PROP_DEFAULT_VALUE)


class EXCLUDE_FROM_TEMPLATES_RESULTS(object):
    ALLOW = u'allow'
    DONT_HAVE = u'dont-have'
    EXCLUDE = u'exclude'


class ITEM_STATE(object):
    NEW = u'new'
    STAGGING = u'stagging'
    PRODUCTION = u'production'
    PREPROD = u'preprod'
    MERGE_SOURCES = u'merge_from_sources'
    CHANGES = u'changes'
    SUBMIT_STAGGING = u'submit_stagging'
    WORKING_AREA = u'working_area'
    DELETED = u'deleted'
    RAW_SOURCES = u'raw_source'
    
    ALL_STATES = (
        MERGE_SOURCES,
        WORKING_AREA,
        STAGGING,
        PRODUCTION,
        NEW,
        CHANGES
    )


LINKIFY_MANAGE_STATES = (ITEM_STATE.STAGGING, ITEM_STATE.WORKING_AREA)
INHERITANCE_MANAGE_STATES = (ITEM_STATE.STAGGING, ITEM_STATE.WORKING_AREA)

ItemType = str  # Type of ITEM_TYPE. For type hint.


class ITEM_TYPE(object):
    ELEMENTS = 'elements'
    
    CLUSTERS = 'clusters'
    CLUSTERTPLS = 'clustertpls'
    
    HOSTS = 'hosts'
    HOSTTPLS = 'hosttpls'
    HOSTGROUPS = 'hostgroups'
    
    SERVICESHOSTS = 'serviceshosts'
    SERVICESHOSTTPLS = 'serviceshosttpls'
    SERVICESCLUSTERS = 'servicesclusters'
    SERVICESCLUSTERTPLS = 'servicesclustertpls'
    SERVICETPLS = 'servicetpls'
    
    CONTACTS = 'contacts'
    CONTACTTPLS = 'contacttpls'
    CONTACTGROUPS = 'contactgroups'
    
    ESCALATIONS = 'escalations'
    NOTIFICATIONWAYS = 'notificationways'
    COMMANDS = 'commands'
    BUSINESSIMPACTMODULATIONS = 'businessimpactmodulations'
    MACROMODULATIONS = 'macromodulations'
    RESULTMODULATIONS = 'resultmodulations'
    TIMEPERIODS = 'timeperiods'
    PACK = 'packs'
    PLUGIN = 'plugins'
    
    # For synchronizer-collector-linker source
    REMOTE_SYNCHRONIZER = 'sync'
    
    ALL_TYPES = (
        CLUSTERS,
        CLUSTERTPLS,
        HOSTS,
        HOSTTPLS,
        HOSTGROUPS,
        SERVICESHOSTS,
        SERVICESHOSTTPLS,
        SERVICESCLUSTERS,
        SERVICESCLUSTERTPLS,
        SERVICETPLS,
        CONTACTS,
        CONTACTTPLS,
        CONTACTGROUPS,
        ESCALATIONS,
        NOTIFICATIONWAYS,
        COMMANDS,
        BUSINESSIMPACTMODULATIONS,
        MACROMODULATIONS,
        RESULTMODULATIONS,
        TIMEPERIODS
    )
    
    SONS = {
        SERVICETPLS: [SERVICETPLS, SERVICESHOSTS, SERVICESHOSTTPLS, SERVICESCLUSTERS, SERVICESCLUSTERTPLS],
        HOSTTPLS   : [HOSTTPLS, HOSTS],
        CONTACTTPLS: [CONTACTTPLS, CONTACTS],
        CLUSTERTPLS: [CLUSTERS, CLUSTERTPLS]
    }
    ALL_SERVICES = set([SERVICETPLS, SERVICESCLUSTERTPLS, SERVICESHOSTTPLS, SERVICESCLUSTERS, SERVICESHOSTS])
    ALL_DEDICATED_SERVICES = set([SERVICESCLUSTERTPLS, SERVICESHOSTTPLS, SERVICESCLUSTERS, SERVICESHOSTS])
    
    ALL_HOST_CLASS = set([CLUSTERS, CLUSTERTPLS, HOSTS, HOSTTPLS])
    
    TPLS_ITEM_TYPES = set([HOSTTPLS, SERVICETPLS, CONTACTTPLS, CLUSTERTPLS])
    ITEM_TYPES_WITH_TPLS = set([HOSTS, HOSTTPLS, SERVICESHOSTS, SERVICESHOSTTPLS, SERVICESCLUSTERS, SERVICESCLUSTERTPLS, SERVICETPLS, CONTACTS, CONTACTTPLS, CLUSTERS, CLUSTERTPLS])
    
    
    @staticmethod
    def has_user_right(item_type):
        return DEF_ITEMS.get(item_type, {}).get('has_user_right', False)
    
    
    @staticmethod
    def is_template(item_type):
        return item_type in ITEM_TYPE.TPLS_ITEM_TYPES
    
    
    @staticmethod
    def has_template(item_type):
        return item_type in ITEM_TYPE.ITEM_TYPES_WITH_TPLS
    
    
    @staticmethod
    def get_template_type(item_type):
        return DEF_ITEMS[item_type].get('template', None)
    
    
    @staticmethod
    def has_work_area(item_type):
        return DEF_ITEMS[item_type]['has_work_area']


class HISTORY_ACTION:
    UPDATE_WORK_AREA_INFO = 'UPDATE_WORK_AREA_INFO'
    APPLY_CHANGES = 'APPLY_CHANGES'
    ELEMENT_DELETE = 'ELEMENT_DELETE'
    ELEMENT_ADD = 'ELEMENT_ADD'
    ELEMENT_MODIFICATION = 'ELEMENT_MODIFICATION'
    ELEMENT_RESTORE = 'ELEMENT_RESTORE'
    ELEMENT_CLONED = 'ELEMENT_CLONED'
    AUTO_MODIFICATION = 'AUTO_MODIFICATION'
    LOGIN = 'LOGIN'
    LOGOUT = 'LOGOUT'
    APPLY = 'APPLY'
    IMPORT = 'IMPORT'
    SCAN_RANGE_DELETE = 'SCAN_RANGE_DELETE'
    SCAN_RANGE_ADD = 'SCAN_RANGE_ADD'
    SCAN_RANGE_MODIFICATION = 'SCAN_RANGE_MODIFICATION'
    SUBMIT_TO_STAGGING = 'SUBMIT_TO_STAGGING'
    PUT_IN_WORK_AREA = 'PUT_IN_WORK_AREA'
    REJECT_FROM_WORK_AREA = 'REJECT_FROM_WORK_AREA'
    VALIDATED_FROM_WORK_AREA = 'VALIDATED_FROM_WORK_AREA'
    DISCARD_MODIFICATION_FROM_WORK_AREA = 'DISCARD_MODIFICATION_FROM_WORK_AREA'


class WORKING_AREA_STATUS:
    WORKING = 'WORKING'
    PROPOSED = 'PROPOSED'
    REJECTED = 'REJECTED'
    VALIDATED = 'VALIDATED'
    
    
    @staticmethod
    def get_work_area_status(item, item_state=None):
        if not item_state and hasattr(item, 'get_state'):
            item_state = item.get_state()
        if item.get('@metadata', {}).get('type', ITEM_TYPE.HOSTS) != ITEM_TYPE.HOSTS:
            return WORKING_AREA_STATUS.VALIDATED
        defaut_work_area_status = WORKING_AREA_STATUS.WORKING if item_state == ITEM_STATE.NEW else WORKING_AREA_STATUS.VALIDATED
        work_area_status = item.get('work_area_info', {}).get('status', defaut_work_area_status)
        return work_area_status


class WORKING_AREA_LAST_ACTION:
    CREATED = 'new'
    SUPPRESSED = 'delete'
    MODIFIED = 'change'
    
    WORKING_SUPPRESSED = '%s_%s' % (WORKING_AREA_STATUS.WORKING, SUPPRESSED)
    REJECTED_SUPPRESSED = '%s_%s' % (WORKING_AREA_STATUS.REJECTED, SUPPRESSED)
    WORKING_CREATED = '%s_%s' % (WORKING_AREA_STATUS.WORKING, CREATED)
    REJECTED_CREATED = '%s_%s' % (WORKING_AREA_STATUS.REJECTED, CREATED)
    
    
    @staticmethod
    def get_work_area_action(item, item_state):
        stat = WORKING_AREA_STATUS.get_work_area_status(item, item_state)
        work_area_info = item.get('work_area_info', {})
        if stat != WORKING_AREA_STATUS.VALIDATED:
            last_action = work_area_info.get('last_action', None)
            if last_action:
                stat += '_%s' % last_action
            elif stat == WORKING_AREA_STATUS.WORKING:
                if work_area_info.get('exist_in', 'working_area') == 'working_area':
                    stat += '_%s' % WORKING_AREA_LAST_ACTION.CREATED
                else:
                    stat += '_%s' % WORKING_AREA_LAST_ACTION.MODIFIED
            else:
                submit_new_item = work_area_info.get('submit_new_item', False)
                submit_del_item = work_area_info.get('submit_del_item', False)
                if submit_new_item:
                    stat += '_%s' % WORKING_AREA_LAST_ACTION.CREATED
                elif submit_del_item:
                    stat += '_%s' % WORKING_AREA_LAST_ACTION.SUPPRESSED
                else:
                    stat += '_%s' % WORKING_AREA_LAST_ACTION.MODIFIED
        # else:
        #     logger.debug("[work_area] work area status is VALIDATED - no Last action")
        return stat


# Some properties to NOT look when comparing/diff objects
NOT_TO_LOOK = set([
    '@metadata',
    '_SE_UUID',
    '_SE_UUID_HASH',
    '_SYNC_KEYS',
    '__SYNC_IDX__',
    '_id',
    'import_date',
    'imported_from',
    'last_modification',
    'overwrited_protected',
    'source',
    'source_order',
    'source_strong_overwrite',
    'sources',
    'work_area_info',
    'presence_protection',
    'to_merge'  # only used during the merge, removed when save
])

FORBIDDEN_DATA = set(['_ID'])

NO_HERITABLE_ATTR = set([
    '@metadata',
    '_SE_UUID',
    '_SE_UUID_HASH',
    '_SYNC_KEYS',
    '__SYNC_IDX__',
    '_id',
    'host_name',
    'import_date',
    'imported_from',
    'last_modification',
    'overwrited_protected',
    'source',
    'source_order',
    'source_strong_overwrite',
    'sources',
    'work_area_info',
    'pack',
    'register',
    'use',
    'name',
    'enabled',
    'presence_protection',
    SERVICE_EXCLUDES_BY_NAME,
    SERVICE_EXCLUDES_BY_ID
])

# In time periods, properties which dont define periods
TIMEPERIOD_BASIC_PROPS = set(['alias', 'timeperiod_name', 'exclude', 'is_active', 'pack', 'enabled']).union(NOT_TO_LOOK)

INTERNAL_ATTR = set((
    'import_date',
    'last_modification',
    'work_area_info',
    '@metadata',
    '_SE_UUID',
    '_SE_UUID_HASH',
    '_SYNC_KEYS',
    '__SYNC_IDX__',
    '_id',
))

MAX_TEMPLATE_LOOKUP_LEVEL = 10
SRC_NAME_MAX_LENGTH = 40

PROPERTIES_USING_PLUS_FOR_HOST_AND_CLUSTER = set([
    'hostgroups',
    'parents',
    'escalations',
    'view_contacts',
    'view_contact_groups',
    'notification_contacts',
    'notification_contact_groups',
    'edition_contacts',
    'edition_contact_groups',
    'business_impact_modulations',
    'macromodulations',
    'resultmodulations',
])

PROPERTIES_USING_PLUS_FOR_CHECK = set([
    'contacts',
    'contact_groups',
    'escalations',
    'business_impact_modulations',
    'macromodulations',
    'resultmodulations',
])

PROPERTIES_USING_PLUS_FOR_CONTACT = set([
    'contactgroups',
    'notificationways',
])

UNFLATTENNED_PROPERTIES_IN_FRONT = set([
    SERVICE_EXCLUDES_BY_ID,
    SERVICE_OVERRIDE
])

DEF_ITEMS = {
    # Source element : table: where to look up, key:value=> which table to query, property to clean, and where=match only theses elements
    
    # /!\ BEWARE - if a property appears in several item_links entries, a specific build_link method is needed in item_saving_formatter.py
    
    ITEM_TYPE.CLUSTERS                 : {
        'has_user_right'  : True,
        'key_name'        : 'host_name',
        'table'           : 'host',
        'where'           : {'register': {'$ne': '0'}, 'is_cluster': '1'},
        'has_data'        : True,
        'has_work_area'   : False,
        'class'           : Host,
        'template'        : ITEM_TYPE.CLUSTERTPLS,
        'fields_with_plus': PROPERTIES_USING_PLUS_FOR_HOST_AND_CLUSTER,
        'check_type'      : ITEM_TYPE.SERVICESCLUSTERS,
        'flatteners'      : {
            'check_command'       : 'CommandMixin.flatten_not_linked_prop',
            'event_handler'       : 'CommandMixin.flatten_not_linked_prop',
            'bp_rule'             : 'BPRuleMixin.flatten_not_linked_prop',
            SERVICE_OVERRIDE      : 'ServiceOverridesMixin.flatten_not_linked_prop',
            SERVICE_EXCLUDES_BY_ID: 'ServiceExcludesMixin.flatten_not_linked_prop',
        },
        'mandatory_fields': {'is_cluster': '1'},
        'item_links'      : {
            ITEM_TYPE.CLUSTERTPLS              : ['use', 'bp_rule', SERVICE_OVERRIDE],
            ITEM_TYPE.CONTACTS                 : ['contacts', 'last_modification.contact', 'work_area_info.user', 'view_contacts', 'notification_contacts', 'edition_contacts', SERVICE_OVERRIDE],
            ITEM_TYPE.CONTACTGROUPS            : ['contact_groups', 'view_contact_groups', 'notification_contact_groups', 'edition_contact_groups', SERVICE_OVERRIDE],
            ITEM_TYPE.HOSTS                    : ['bp_rule'],
            ITEM_TYPE.HOSTTPLS                 : ['bp_rule'],
            ITEM_TYPE.HOSTGROUPS               : ['bp_rule'],
            ITEM_TYPE.CLUSTERS                 : ['bp_rule', SERVICE_OVERRIDE],
            ITEM_TYPE.TIMEPERIODS              : ['check_period', 'maintenance_period', 'notification_period', SERVICE_OVERRIDE],
            ITEM_TYPE.COMMANDS                 : ['check_command', 'event_handler'],
            ITEM_TYPE.BUSINESSIMPACTMODULATIONS: ['business_impact_modulations', SERVICE_OVERRIDE],
            ITEM_TYPE.MACROMODULATIONS         : ['macromodulations', SERVICE_OVERRIDE],
            ITEM_TYPE.ESCALATIONS              : ['escalations', SERVICE_OVERRIDE],
            ITEM_TYPE.SERVICETPLS              : [SERVICE_OVERRIDE],
            ITEM_TYPE.SERVICESCLUSTERTPLS      : [SERVICE_OVERRIDE, SERVICE_EXCLUDES_BY_ID],
            ITEM_TYPE.SERVICESCLUSTERS         : [SERVICE_OVERRIDE, SERVICE_EXCLUDES_BY_ID],
            ITEM_TYPE.RESULTMODULATIONS        : ['resultmodulations', SERVICE_OVERRIDE]
        },
        'props_links'     : 'This set will be initialized after this dict creation. It will contain all properties name which are a link like : use, parents, contact_groups.',
        'units'           : {
            'check_running_timeout'      : 'seconds',
            'warning_threshold_cpu_usage': 'seconds',
            'notification_interval'      : 'minutes',
            'first_notification_delay'   : 'minutes',
        }
    },
    ITEM_TYPE.CLUSTERTPLS              : {
        'has_user_right'  : True,
        'key_name'        : 'name',
        'table'           : 'host',
        'where'           : {'register': '0', 'is_cluster': '1'},
        'has_data'        : True,
        'has_work_area'   : False,
        'class'           : Host,
        'template'        : ITEM_TYPE.CLUSTERTPLS,
        'fields_with_plus': PROPERTIES_USING_PLUS_FOR_HOST_AND_CLUSTER,
        'check_type'      : ITEM_TYPE.SERVICESCLUSTERTPLS,
        'flatteners'      : {
            'check_command'       : 'CommandMixin.flatten_not_linked_prop',
            'event_handler'       : 'CommandMixin.flatten_not_linked_prop',
            'bp_rule'             : 'BPRuleMixin.flatten_not_linked_prop',
            SERVICE_OVERRIDE      : 'ServiceOverridesMixin.flatten_not_linked_prop',
            SERVICE_EXCLUDES_BY_ID: 'ServiceExcludesMixin.flatten_not_linked_prop',
        },
        'mandatory_fields': {'register': '0', 'is_cluster': '1'},
        'item_links'      : {
            ITEM_TYPE.CLUSTERTPLS              : ['use', 'bp_rule', SERVICE_OVERRIDE],
            ITEM_TYPE.CONTACTS                 : ['contacts', 'last_modification.contact', 'work_area_info.user', 'view_contacts', 'notification_contacts', 'edition_contacts', SERVICE_OVERRIDE],
            ITEM_TYPE.CONTACTGROUPS            : ['contact_groups', 'view_contact_groups', 'notification_contact_groups', 'edition_contact_groups', SERVICE_OVERRIDE],
            ITEM_TYPE.HOSTS                    : ['bp_rule'],
            ITEM_TYPE.CLUSTERS                 : ['bp_rule', SERVICE_OVERRIDE],
            ITEM_TYPE.TIMEPERIODS              : ['check_period', 'maintenance_period', 'notification_period', SERVICE_OVERRIDE],
            ITEM_TYPE.COMMANDS                 : ['check_command', 'event_handler'],
            ITEM_TYPE.BUSINESSIMPACTMODULATIONS: ['business_impact_modulations', SERVICE_OVERRIDE],
            ITEM_TYPE.MACROMODULATIONS         : ['macromodulations', SERVICE_OVERRIDE],
            ITEM_TYPE.RESULTMODULATIONS        : ['resultmodulations', SERVICE_OVERRIDE],
            ITEM_TYPE.ESCALATIONS              : ['escalations', SERVICE_OVERRIDE],
            ITEM_TYPE.HOSTTPLS                 : ['bp_rule'],
            ITEM_TYPE.HOSTGROUPS               : ['bp_rule'],
            ITEM_TYPE.SERVICETPLS              : [SERVICE_OVERRIDE],
            ITEM_TYPE.SERVICESCLUSTERTPLS      : [SERVICE_OVERRIDE, SERVICE_EXCLUDES_BY_ID],
            ITEM_TYPE.SERVICESCLUSTERS         : [SERVICE_OVERRIDE, SERVICE_EXCLUDES_BY_ID]
            
        },
        'props_links'     : 'This set will be initialized after this dict creation. It will contain all properties name which are a link like : use, parents, contact_groups.',
        'units'           : {
            'check_running_timeout'      : 'seconds',
            'warning_threshold_cpu_usage': 'seconds',
            'notification_interval'      : 'minutes',
            'first_notification_delay'   : 'minutes',
        }
        
    },
    ITEM_TYPE.HOSTS                    : {
        'has_user_right'  : True,
        'key_name'        : 'host_name',
        'table'           : 'host',
        'where'           : {'register': {'$ne': '0'}, 'is_cluster': {'$ne': '1'}},
        'has_data'        : True,
        'has_work_area'   : True,
        'class'           : Host,
        'template'        : ITEM_TYPE.HOSTTPLS,
        'fields_with_plus': PROPERTIES_USING_PLUS_FOR_HOST_AND_CLUSTER,
        'check_type'      : ITEM_TYPE.SERVICESHOSTS,
        'flatteners'      : {
            'check_command'       : 'CommandMixin.flatten_not_linked_prop',
            'event_handler'       : 'CommandMixin.flatten_not_linked_prop',
            SERVICE_OVERRIDE      : 'ServiceOverridesMixin.flatten_not_linked_prop',
            SERVICE_EXCLUDES_BY_ID: 'ServiceExcludesMixin.flatten_not_linked_prop',
        },
        'mandatory_fields': {},
        'item_links'      : {
            ITEM_TYPE.HOSTTPLS                 : ['use', SERVICE_OVERRIDE],
            ITEM_TYPE.HOSTGROUPS               : ['hostgroups'],
            ITEM_TYPE.CONTACTS                 : ['contacts', 'last_modification.contact', 'work_area_info.user', 'view_contacts', 'notification_contacts', 'edition_contacts', SERVICE_OVERRIDE],
            ITEM_TYPE.CONTACTGROUPS            : ['contact_groups', 'view_contact_groups', 'notification_contact_groups', 'edition_contact_groups', SERVICE_OVERRIDE],
            ITEM_TYPE.HOSTS                    : ['parents', SERVICE_OVERRIDE],
            ITEM_TYPE.CLUSTERS                 : ['parents'],
            ITEM_TYPE.TIMEPERIODS              : ['check_period', 'maintenance_period', 'notification_period', SERVICE_OVERRIDE],
            ITEM_TYPE.COMMANDS                 : ['check_command', 'event_handler'],
            ITEM_TYPE.BUSINESSIMPACTMODULATIONS: ['business_impact_modulations', SERVICE_OVERRIDE],
            ITEM_TYPE.MACROMODULATIONS         : ['macromodulations', SERVICE_OVERRIDE],
            ITEM_TYPE.RESULTMODULATIONS        : ['resultmodulations', SERVICE_OVERRIDE],
            ITEM_TYPE.ESCALATIONS              : ['escalations', SERVICE_OVERRIDE],
            ITEM_TYPE.SERVICETPLS              : [SERVICE_OVERRIDE],
            ITEM_TYPE.SERVICESHOSTS            : [SERVICE_OVERRIDE, SERVICE_EXCLUDES_BY_ID],
            ITEM_TYPE.SERVICESHOSTTPLS         : [SERVICE_OVERRIDE, SERVICE_EXCLUDES_BY_ID],
        },
        'double_links'    : [
            {'my_attr': 'hostgroups', 'of_type': ITEM_TYPE.HOSTGROUPS, 'is_link_with_attr': 'members'}
        ],
        'props_links'     : 'This set will be initialized after this dict creation. It will contain all properties name which are a link like : use, parents, contact_groups.',
        'units'           : {
            'check_running_timeout'      : 'seconds',
            'warning_threshold_cpu_usage': 'seconds',
            'check_interval'             : 'minutes',
            'retry_interval'             : 'minutes',
            'notification_interval'      : 'minutes',
            'first_notification_delay'   : 'minutes',
            'freshness_threshold'        : 'seconds',
        }
        
    },
    ITEM_TYPE.HOSTTPLS                 : {
        'has_user_right'  : True,
        'key_name'        : 'name',
        'table'           : 'host',
        'where'           : {'register': '0', 'is_cluster': {'$ne': '1'}},
        'has_data'        : True,
        'has_work_area'   : False,
        'class'           : Host,
        'template'        : ITEM_TYPE.HOSTTPLS,
        'fields_with_plus': PROPERTIES_USING_PLUS_FOR_HOST_AND_CLUSTER,
        'check_type'      : ITEM_TYPE.SERVICESHOSTTPLS,
        'flatteners'      : {
            'check_command'       : 'CommandMixin.flatten_not_linked_prop',
            'event_handler'       : 'CommandMixin.flatten_not_linked_prop',
            SERVICE_OVERRIDE      : 'ServiceOverridesMixin.flatten_not_linked_prop',
            SERVICE_EXCLUDES_BY_ID: 'ServiceExcludesMixin.flatten_not_linked_prop',
        },
        'mandatory_fields': {'register': '0'},
        'item_links'      : {
            ITEM_TYPE.HOSTTPLS                 : ['use', SERVICE_OVERRIDE],
            ITEM_TYPE.HOSTGROUPS               : ['hostgroups'],
            ITEM_TYPE.CONTACTS                 : ['contacts', 'last_modification.contact', 'work_area_info.user', 'view_contacts', 'notification_contacts', 'edition_contacts', SERVICE_OVERRIDE],
            ITEM_TYPE.CONTACTGROUPS            : ['contact_groups', 'view_contact_groups', 'notification_contact_groups', 'edition_contact_groups', SERVICE_OVERRIDE],
            ITEM_TYPE.HOSTS                    : ['parents', SERVICE_OVERRIDE],
            ITEM_TYPE.CLUSTERS                 : ['parents'],
            ITEM_TYPE.TIMEPERIODS              : ['check_period', 'maintenance_period', 'notification_period', SERVICE_OVERRIDE],
            ITEM_TYPE.COMMANDS                 : ['check_command', 'event_handler'],
            ITEM_TYPE.BUSINESSIMPACTMODULATIONS: ['business_impact_modulations', SERVICE_OVERRIDE],
            ITEM_TYPE.MACROMODULATIONS         : ['macromodulations', SERVICE_OVERRIDE],
            ITEM_TYPE.RESULTMODULATIONS        : ['resultmodulations', SERVICE_OVERRIDE],
            ITEM_TYPE.ESCALATIONS              : ['escalations', SERVICE_OVERRIDE],
            ITEM_TYPE.SERVICETPLS              : [SERVICE_OVERRIDE],
            ITEM_TYPE.SERVICESHOSTTPLS         : [SERVICE_OVERRIDE, SERVICE_EXCLUDES_BY_ID],
            ITEM_TYPE.SERVICESHOSTS            : [SERVICE_OVERRIDE, SERVICE_EXCLUDES_BY_ID]
        },
        'double_links'    : [
            {'my_attr': 'hostgroups', 'of_type': ITEM_TYPE.HOSTGROUPS, 'is_link_with_attr': 'template_members'}
        ],
        'props_links'     : 'This set will be initialized after this dict creation. It will contain all properties name which are a link like : use, parents, contact_groups.',
        'units'           : {
            'check_running_timeout'      : 'seconds',
            'warning_threshold_cpu_usage': 'seconds',
            'check_interval'             : 'minutes',
            'retry_interval'             : 'minutes',
            'notification_interval'      : 'minutes',
            'first_notification_delay'   : 'minutes',
            'freshness_threshold'        : 'seconds',
        }
    },
    ITEM_TYPE.HOSTGROUPS               : {
        'has_user_right'  : False,
        'key_name'        : 'hostgroup_name',
        'table'           : 'hostgroup',
        'class'           : Hostgroup,
        'has_data'        : False,
        'has_work_area'   : False,
        'flatteners'      : {},
        'mandatory_fields': {},
        'item_links'      : {
            ITEM_TYPE.HOSTS   : ['members'],
            ITEM_TYPE.HOSTTPLS: ['template_members'],
            ITEM_TYPE.CONTACTS: ['last_modification.contact', 'work_area_info.user'],
            
        },
        'double_links'    : [
            {'my_attr': 'members', 'of_type': ITEM_TYPE.HOSTS, 'is_link_with_attr': 'hostgroups'},
            {'my_attr': 'template_members', 'of_type': ITEM_TYPE.HOSTTPLS, 'is_link_with_attr': 'hostgroups'}
        ],
        'props_links'     : 'This set will be initialized after this dict creation. It will contain all properties name which are a link like : use, parents, contact_groups.',
        
    },
    ITEM_TYPE.SERVICESHOSTS            : {
        'has_user_right'  : False,
        'key_name'        : 'service_description',
        'table'           : 'service',
        'where'           : {'register': {'$ne': '0'}, 'apply_on_type': {'$nin': [ITEM_TYPE.CLUSTERS, ITEM_TYPE.CLUSTERTPLS, ITEM_TYPE.HOSTTPLS]}},
        'has_data'        : True,
        'has_work_area'   : False,
        'apply_on_type'   : ITEM_TYPE.HOSTS,
        'class'           : Service,
        'template'        : ITEM_TYPE.SERVICETPLS,
        'fields_with_plus': PROPERTIES_USING_PLUS_FOR_CHECK,
        'flatteners'      : {
            'check_command' : 'CommandMixin.flatten_not_linked_prop',
            'event_handler' : 'CommandMixin.flatten_not_linked_prop',
            'hostgroup_name': 'HostgroupNameMixin.flatten_not_linked_prop',
        },
        'mandatory_fields': {},
        'item_links'      : {
            ITEM_TYPE.SERVICETPLS              : ['use'],
            ITEM_TYPE.TIMEPERIODS              : ['check_period', 'maintenance_period', 'notification_period'],
            ITEM_TYPE.CONTACTS                 : ['contacts', 'last_modification.contact', 'work_area_info.user'],
            ITEM_TYPE.MACROMODULATIONS         : ['macromodulations'],
            ITEM_TYPE.RESULTMODULATIONS        : ['resultmodulations'],
            ITEM_TYPE.COMMANDS                 : ['check_command', 'event_handler'],
            ITEM_TYPE.CONTACTGROUPS            : ['contact_groups'],
            ITEM_TYPE.BUSINESSIMPACTMODULATIONS: ['business_impact_modulations'],
            ITEM_TYPE.HOSTS                    : ['host_name'],
            ITEM_TYPE.HOSTGROUPS               : ['hostgroup_name'],
            ITEM_TYPE.ESCALATIONS              : ['escalations']
        },
        'props_links'     : 'This set will be initialized after this dict creation. It will contain all properties name which are a link like : use, parents, contact_groups.',
        'units'           : {
            'check_interval'             : 'minutes',
            'retry_interval'             : 'minutes',
            'check_running_timeout'      : 'seconds',
            'warning_threshold_cpu_usage': 'seconds',
            'notification_interval'      : 'minutes',
            'first_notification_delay'   : 'minutes',
            'freshness_threshold'        : 'seconds',
        }
    },
    ITEM_TYPE.SERVICESHOSTTPLS         : {
        'has_user_right'  : False,
        'key_name'        : 'service_description',
        'table'           : 'service',
        'where'           : {'register': '0', 'apply_on_type': {'$nin': [ITEM_TYPE.CLUSTERS, ITEM_TYPE.CLUSTERTPLS, ITEM_TYPE.HOSTS]}, 'service_description': {'$exists': 1}},
        'has_data'        : True,
        'has_work_area'   : False,
        'apply_on_type'   : ITEM_TYPE.HOSTTPLS,
        'class'           : Service,
        'template'        : ITEM_TYPE.SERVICETPLS,
        'fields_with_plus': PROPERTIES_USING_PLUS_FOR_CHECK,
        'flatteners'      : {
            'check_command' : 'CommandMixin.flatten_not_linked_prop',
            'event_handler' : 'CommandMixin.flatten_not_linked_prop',
            'host_name'     : 'HostNameMixin.flatten_not_linked_prop',
            'hostgroup_name': 'HostgroupNameMixin.flatten_not_linked_prop',
        },
        'mandatory_fields': {'register': '0'},
        'item_links'      : {
            ITEM_TYPE.SERVICETPLS              : ['use'],
            ITEM_TYPE.TIMEPERIODS              : ['check_period', 'maintenance_period', 'notification_period'],
            ITEM_TYPE.CONTACTS                 : ['contacts', 'last_modification.contact', 'work_area_info.user'],
            ITEM_TYPE.MACROMODULATIONS         : ['macromodulations'],
            ITEM_TYPE.COMMANDS                 : ['check_command', 'event_handler'],
            ITEM_TYPE.CONTACTGROUPS            : ['contact_groups'],
            ITEM_TYPE.BUSINESSIMPACTMODULATIONS: ['business_impact_modulations'],
            ITEM_TYPE.HOSTGROUPS               : ['hostgroup_name'],
            ITEM_TYPE.ESCALATIONS              : ['escalations'],
            ITEM_TYPE.HOSTTPLS                 : ['host_name'],
            ITEM_TYPE.RESULTMODULATIONS        : ['resultmodulations'],
        },
        'props_links'     : 'This set will be initialized after this dict creation. It will contain all properties name which are a link like : use, parents, contact_groups.',
        'units'           : {
            'check_interval'             : 'minutes',
            'retry_interval'             : 'minutes',
            'check_running_timeout'      : 'seconds',
            'warning_threshold_cpu_usage': 'seconds',
            'notification_interval'      : 'minutes',
            'first_notification_delay'   : 'minutes',
            'freshness_threshold'        : 'seconds',
        }
    },
    ITEM_TYPE.SERVICESCLUSTERS         : {
        'has_user_right'  : False,
        'key_name'        : 'service_description',
        'table'           : 'service',
        'where'           : {'register': {'$ne': '0'}, 'apply_on_type': ITEM_TYPE.CLUSTERS},
        'has_data'        : True,
        'has_work_area'   : False,
        'apply_on_type'   : ITEM_TYPE.CLUSTERS,
        'class'           : Service,
        'template'        : ITEM_TYPE.SERVICETPLS,
        'fields_with_plus': PROPERTIES_USING_PLUS_FOR_CHECK,
        'flatteners'      : {
            'check_command': 'CommandMixin.flatten_not_linked_prop',
            'event_handler': 'CommandMixin.flatten_not_linked_prop',
        },
        'mandatory_fields': {'apply_on_type': ITEM_TYPE.CLUSTERS},
        'item_links'      : {
            ITEM_TYPE.SERVICETPLS              : ['use'],
            ITEM_TYPE.TIMEPERIODS              : ['check_period', 'maintenance_period', 'notification_period'],
            ITEM_TYPE.CONTACTS                 : ['contacts', 'last_modification.contact', 'work_area_info.user'],
            ITEM_TYPE.MACROMODULATIONS         : ['macromodulations'],
            ITEM_TYPE.RESULTMODULATIONS        : ['resultmodulations'],
            ITEM_TYPE.COMMANDS                 : ['check_command', 'event_handler'],
            ITEM_TYPE.CONTACTGROUPS            : ['contact_groups'],
            ITEM_TYPE.BUSINESSIMPACTMODULATIONS: ['business_impact_modulations'],
            ITEM_TYPE.CLUSTERS                 : ['host_name'],
            ITEM_TYPE.ESCALATIONS              : ['escalations']
        },
        'props_links'     : 'This set will be initialized after this dict creation. It will contain all properties name which are a link like : use, parents, contact_groups.',
        'units'           : {
            'check_interval'             : 'minutes',
            'retry_interval'             : 'minutes',
            'check_running_timeout'      : 'seconds',
            'warning_threshold_cpu_usage': 'seconds',
            'notification_interval'      : 'minutes',
            'first_notification_delay'   : 'minutes',
            'freshness_threshold'        : 'seconds',
        }
    },
    ITEM_TYPE.SERVICESCLUSTERTPLS      : {
        'has_user_right'  : False,
        'key_name'        : 'service_description',
        'table'           : 'service',
        'where'           : {'register': '0', 'apply_on_type': ITEM_TYPE.CLUSTERTPLS, 'service_description': {'$exists': 1}},
        'has_data'        : True,
        'has_work_area'   : False,
        'apply_on_type'   : ITEM_TYPE.CLUSTERTPLS,
        'class'           : Service,
        'template'        : ITEM_TYPE.SERVICETPLS,
        'fields_with_plus': PROPERTIES_USING_PLUS_FOR_CHECK,
        'flatteners'      : {
            'check_command': 'CommandMixin.flatten_not_linked_prop',
            'event_handler': 'CommandMixin.flatten_not_linked_prop',
            'host_name'    : 'HostNameMixin.flatten_not_linked_prop',
        },
        'mandatory_fields': {'register': '0', 'apply_on_type': ITEM_TYPE.CLUSTERTPLS},
        'item_links'      : {
            ITEM_TYPE.SERVICETPLS              : ['use'],
            ITEM_TYPE.TIMEPERIODS              : ['check_period', 'maintenance_period', 'notification_period'],
            ITEM_TYPE.CONTACTS                 : ['contacts', 'last_modification.contact', 'work_area_info.user'],
            ITEM_TYPE.MACROMODULATIONS         : ['macromodulations'],
            ITEM_TYPE.RESULTMODULATIONS        : ['resultmodulations'],
            ITEM_TYPE.COMMANDS                 : ['check_command', 'event_handler'],
            ITEM_TYPE.CONTACTGROUPS            : ['contact_groups'],
            ITEM_TYPE.BUSINESSIMPACTMODULATIONS: ['business_impact_modulations'],
            ITEM_TYPE.ESCALATIONS              : ['escalations'],
            ITEM_TYPE.CLUSTERTPLS              : ['host_name'],
            
        },
        'props_links'     : 'This set will be initialized after this dict creation. It will contain all properties name which are a link like : use, parents, contact_groups.',
        'units'           : {
            'check_interval'             : 'minutes',
            'retry_interval'             : 'minutes',
            'check_running_timeout'      : 'seconds',
            'warning_threshold_cpu_usage': 'seconds',
            'notification_interval'      : 'minutes',
            'first_notification_delay'   : 'minutes',
            'freshness_threshold'        : 'seconds',
        }
    },
    ITEM_TYPE.SERVICETPLS              : {
        'has_user_right'  : False,
        'key_name'        : 'name',
        'table'           : 'service',
        'where'           : {'register': '0', 'name': {"$exists": 1}},
        'has_data'        : True,
        'has_work_area'   : False,
        'class'           : Service,
        'template'        : ITEM_TYPE.SERVICETPLS,
        'fields_with_plus': PROPERTIES_USING_PLUS_FOR_CHECK,
        'flatteners'      : {
            'check_command': 'CommandMixin.flatten_not_linked_prop',
            'event_handler': 'CommandMixin.flatten_not_linked_prop',
        },
        'mandatory_fields': {'register': '0'},
        'item_links'      : {
            ITEM_TYPE.SERVICETPLS              : ['use'],
            ITEM_TYPE.TIMEPERIODS              : ['check_period', 'maintenance_period', 'notification_period'],
            ITEM_TYPE.CONTACTS                 : ['contacts', 'last_modification.contact', 'work_area_info.user'],
            ITEM_TYPE.MACROMODULATIONS         : ['macromodulations'],
            ITEM_TYPE.RESULTMODULATIONS        : ['resultmodulations'],
            ITEM_TYPE.COMMANDS                 : ['check_command', 'event_handler'],
            ITEM_TYPE.CONTACTGROUPS            : ['contact_groups'],
            ITEM_TYPE.BUSINESSIMPACTMODULATIONS: ['business_impact_modulations'],
            ITEM_TYPE.ESCALATIONS              : ['escalations']
        },
        'props_links'     : 'This set will be initialized after this dict creation. It will contain all properties name which are a link like : use, parents, contact_groups.',
        'units'           : {
            'check_interval'             : 'minutes',
            'retry_interval'             : 'minutes',
            'check_running_timeout'      : 'seconds',
            'warning_threshold_cpu_usage': 'seconds',
            'notification_interval'      : 'minutes',
            'first_notification_delay'   : 'minutes',
            'freshness_threshold'        : 'seconds',
        }
    },
    ITEM_TYPE.CONTACTS                 : {
        'has_user_right'  : False,
        'key_name'        : 'contact_name',
        'table'           : 'contact',
        'where'           : {'register': {'$ne': '0'}},
        'has_data'        : True,
        'has_work_area'   : False,
        'class'           : Contact,
        'template'        : ITEM_TYPE.CONTACTTPLS,
        'protected_fields': ('password',),
        'fields_with_plus': PROPERTIES_USING_PLUS_FOR_CONTACT,
        'flatteners'      : {},
        'mandatory_fields': {},
        'item_links'      : {
            ITEM_TYPE.CONTACTGROUPS   : ['contactgroups'],
            ITEM_TYPE.CONTACTTPLS     : ['use'],
            ITEM_TYPE.CONTACTS        : ['last_modification.contact', 'work_area_info.user'],
            ITEM_TYPE.COMMANDS        : ['service_notification_commands', 'host_notification_commands'],
            ITEM_TYPE.TIMEPERIODS     : ['host_notification_period', 'service_notification_period'],
            ITEM_TYPE.NOTIFICATIONWAYS: ['notificationways']
        },
        'double_links'    : [
            {'my_attr': 'contactgroups', 'of_type': ITEM_TYPE.CONTACTGROUPS, 'is_link_with_attr': 'members'}
        ],
        'props_links'     : 'This set will be initialized after this dict creation. It will contain all properties name which are a link like : use, parents, contact_groups.',
        
    },
    ITEM_TYPE.CONTACTTPLS              : {
        'has_user_right'  : False,
        'key_name'        : 'name',
        'table'           : 'contact',
        'where'           : {'register': '0'},
        'has_data'        : True,
        'has_work_area'   : False,
        'class'           : Contact,
        'template'        : ITEM_TYPE.CONTACTTPLS,
        'protected_fields': ('password',),
        'fields_with_plus': PROPERTIES_USING_PLUS_FOR_CONTACT,
        'flatteners'      : {},
        'mandatory_fields': {'register': '0'},
        'item_links'      : {
            ITEM_TYPE.CONTACTGROUPS   : ['contactgroups'],
            ITEM_TYPE.CONTACTTPLS     : ['use'],
            ITEM_TYPE.COMMANDS        : ['service_notification_commands', 'host_notification_commands'],
            ITEM_TYPE.TIMEPERIODS     : ['host_notification_period', 'service_notification_period'],
            ITEM_TYPE.NOTIFICATIONWAYS: ['notificationways'],
            ITEM_TYPE.CONTACTS        : ['last_modification.contact', 'work_area_info.user'],
        },
        'double_links'    : [
            {'my_attr': 'contactgroups', 'of_type': ITEM_TYPE.CONTACTGROUPS, 'is_link_with_attr': 'template_members'}
        ],
        'props_links'     : 'This set will be initialized after this dict creation. It will contain all properties name which are a link like : use, parents, contact_groups.',
        
    },
    ITEM_TYPE.CONTACTGROUPS            : {
        'has_user_right'  : False,
        'key_name'        : 'contactgroup_name',
        'table'           : 'contactgroup',
        'class'           : Contactgroup,
        'has_data'        : False,
        'has_work_area'   : False,
        'flatteners'      : {},
        'mandatory_fields': {},
        'item_links'      : {
            ITEM_TYPE.CONTACTS   : ['members', 'last_modification.contact', 'work_area_info.user'],
            ITEM_TYPE.CONTACTTPLS: ['template_members']
        },
        'double_links'    : [
            {'my_attr': 'members', 'of_type': ITEM_TYPE.CONTACTS, 'is_link_with_attr': 'contactgroups'},
            {'my_attr': 'template_members', 'of_type': ITEM_TYPE.CONTACTTPLS, 'is_link_with_attr': 'contactgroups'}
        ],
        'props_links'     : 'This set will be initialized after this dict creation. It will contain all properties name which are a link like : use, parents, contact_groups.',
        
    },
    ITEM_TYPE.ESCALATIONS              : {
        'has_user_right'  : False,
        'key_name'        : 'escalation_name',
        'table'           : 'escalation',
        'class'           : Escalation,
        'has_data'        : False,
        'has_work_area'   : False,
        'flatteners'      : {},
        'mandatory_fields': {},
        'item_links'      : {
            ITEM_TYPE.CONTACTS     : ['contacts', 'last_modification.contact', 'work_area_info.user'],
            ITEM_TYPE.CONTACTGROUPS: ['contact_groups'],
            ITEM_TYPE.TIMEPERIODS  : ['escalation_period']
        },
        'props_links'     : 'This set will be initialized after this dict creation. It will contain all properties name which are a link like : use, parents, contact_groups.',
        'units'           : {
            'first_notification_time': 'minutes',
            'last_notification_time' : 'minutes',
            'notification_interval'  : 'minutes',
        }
    },
    ITEM_TYPE.NOTIFICATIONWAYS         : {
        'has_user_right'  : False,
        'key_name'        : 'notificationway_name',
        'table'           : 'notificationway',
        'class'           : NotificationWay,
        'has_data'        : False,
        'has_work_area'   : False,
        'flatteners'      : {},
        'mandatory_fields': {},
        'item_links'      : {
            ITEM_TYPE.COMMANDS   : ['service_notification_commands', 'host_notification_commands'],
            ITEM_TYPE.TIMEPERIODS: ['service_notification_period', 'host_notification_period'],
            ITEM_TYPE.CONTACTS   : ['last_modification.contact', 'work_area_info.user'],
            
        },
        'props_links'     : 'This set will be initialized after this dict creation. It will contain all properties name which are a link like : use, parents, contact_groups.',
        
    },
    ITEM_TYPE.COMMANDS                 : {
        'has_user_right'  : False,
        'key_name'        : 'command_name',
        'table'           : 'command',
        'class'           : Command,
        'has_data'        : False,
        'has_work_area'   : False,
        'flatteners'      : {},
        'mandatory_fields': {},
        'item_links'      : {
            ITEM_TYPE.CONTACTS: ['last_modification.contact', 'work_area_info.user'],
        },
        'props_links'     : 'This set will be initialized after this dict creation. It will contain all properties name which are a link like : use, parents, contact_groups.',
        'units'           : {
            'timeout'                    : 'seconds',
            'warning_threshold_cpu_usage': 'seconds',
        }
    },
    ITEM_TYPE.BUSINESSIMPACTMODULATIONS: {
        'has_user_right'  : False,
        'key_name'        : 'business_impact_modulation_name',
        'table'           : 'businessimpactmodulation',
        'class'           : Businessimpactmodulation,
        'has_data'        : False,
        'has_work_area'   : False,
        'flatteners'      : {},
        'mandatory_fields': {},
        'item_links'      : {
            ITEM_TYPE.TIMEPERIODS: ['modulation_period'],
            ITEM_TYPE.CONTACTS   : ['last_modification.contact', 'work_area_info.user'],
        },
        'props_links'     : 'This set will be initialized after this dict creation. It will contain all properties name which are a link like : use, parents, contact_groups.',
        
    },
    ITEM_TYPE.MACROMODULATIONS         : {
        'has_user_right'  : False,
        'key_name'        : 'macromodulation_name',
        'table'           : 'macromodulation',
        'class'           : MacroModulation,
        'has_data'        : True,
        'has_work_area'   : False,
        'flatteners'      : {},
        'mandatory_fields': {},
        'item_links'      : {
            ITEM_TYPE.TIMEPERIODS: ['modulation_period'],
            ITEM_TYPE.CONTACTS   : ['last_modification.contact', 'work_area_info.user'],
        },
        'props_links'     : 'This set will be initialized after this dict creation. It will contain all properties name which are a link like : use, parents, contact_groups.',
        
    },
    ITEM_TYPE.RESULTMODULATIONS        : {
        'has_user_right'  : False,
        'key_name'        : 'resultmodulation_name',
        'table'           : 'resultmodulation',
        'class'           : Resultmodulation,
        'has_data'        : False,
        'has_work_area'   : False,
        'flatteners'      : {},
        'mandatory_fields': {},
        'item_links'      : {
            ITEM_TYPE.TIMEPERIODS: ['modulation_period'],
            ITEM_TYPE.CONTACTS   : ['last_modification.contact'],
        },
        'props_links'     : 'This set will be initialized after this dict creation. It will contain all properties name which are a link like : use, parents, contact_groups.',
        
    },
    ITEM_TYPE.TIMEPERIODS              : {
        'has_user_right'  : False,
        'key_name'        : 'timeperiod_name',
        'table'           : 'timeperiod',
        'class'           : Timeperiod,
        'has_data'        : False,
        'has_work_area'   : False,
        'flatteners'      : {},
        'mandatory_fields': {},
        'item_links'      : {
            ITEM_TYPE.TIMEPERIODS: ['exclude'],
            ITEM_TYPE.CONTACTS   : ['last_modification.contact', 'work_area_info.user'],
        },
        'props_links'     : 'This set will be initialized after this dict creation. It will contain all properties name which are a link like : use, parents, contact_groups.',
    },
}


def _init_props_links():
    for item_type, item_info in DEF_ITEMS.iteritems():
        item_info['props_links'] = {}
        for link_item_type, list_prop in item_info['item_links'].iteritems():
            for prop in list_prop:
                if prop in item_info['props_links']:
                    item_info['props_links'][prop].append(link_item_type)
                else:
                    item_info['props_links'][prop] = [link_item_type]


_init_props_links()


def can_property_contain_links(item_type, prop_name):
    return prop_name in DEF_ITEMS[item_type]['props_links']


def get_item_types_for_linked_prop(item_type, prop_name):
    # type: (Str, Str) -> List[Str]
    return DEF_ITEMS[item_type]['props_links'].get(prop_name, [])


def get_unit_for_prop(item_type, prop_name):
    if not DEF_ITEMS.get(item_type, False):
        return None
    if not DEF_ITEMS[item_type].get('units', None):
        return None
    return DEF_ITEMS[item_type]['units'].get(prop_name, None)
