#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019:
# This file is part of Shinken Enterprise, all rights reserved.

import time

from shinken.misc.type_hint import TYPE_CHECKING
from ..helpers import get_name_from_type
from ...dao import DataException

if TYPE_CHECKING:
    from shinken.misc.type_hint import Optional, Dict, Union
    from ..items.misc_items import InstanceItem, ContactItem
    from ...component.mongo_component import MongoComponent
    from shinkensolutions.ssh_mongodb.mongo_collection import MongoCollection


class TrashManager(object):
    
    def __init__(self):
        # type: () -> None
        self.mongo = None  # type: Optional[MongoComponent]
    
    
    def set_mongo(self, mongo):
        # type: (MongoComponent) -> None
        self.mongo = mongo
    
    
    def get_collection_deleted_items(self):
        # type: () -> MongoCollection
        if hasattr(self.mongo, u'col_deleted_items'):
            col_deleted_items = self.mongo.col_deleted_items
        else:
            col_deleted_items = self.mongo.get_collection(u'deleted_items')
        
        return col_deleted_items
    
    
    def save(self, data_to_save):
        # type: (Dict[unicode, Union[unicode, float]]) -> None
        col_deleted_items = self.get_collection_deleted_items()
        col_deleted_items.save(data_to_save)
    
    
    def find_one(self, where):
        # type: (Dict[unicode, unicode]) -> Optional[Dict]
        col_deleted_items = self.get_collection_deleted_items()
        return col_deleted_items.find_one(where)
    
    
    def save_deleted_item(self, item, item_type, user):
        # type: (InstanceItem, unicode, ContactItem) -> None
        if not item:
            raise DataException(u'[%s] save_deleted_item : Please set item' % self.__class__.__name__)
        if not item_type:
            raise DataException(u'[%s] save_deleted_item : Please set item_type' % self.__class__.__name__)
        if not user:
            raise DataException(u'[%s] save_deleted_item : Please set user' % self.__class__.__name__)
        
        data_to_save = {
            u'_id'      : u'%s-%s' % (item_type, item[u'_id']),
            u'item_type': item_type,
            u'item_id'  : item[u'_id'],
            u'user'     : user[u'_id'],
            u'date'     : time.time(),
            u'name'     : get_name_from_type(item_type, item),
        }
        self.save(data_to_save)
    
    
    def find_deleted_items_by_id(self, item_id, item_type):
        # type: (unicode, unicode) -> Optional[Dict]
        if not item_id:
            raise DataException(u'[%s] find_item_by_id : Please set item_id' % self.__class__.__name__)
        if not item_type:
            raise DataException(u'[%s] find_item_by_id : Please set item_type' % self.__class__.__name__)
        _id = u'%s-%s' % (item_type, item_id)
        where = {u'_id': _id}
        item = self.find_one(where)
        return item


trash_manager = TrashManager()
