#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019:
# This file is part of Shinken Enterprise, all rights reserved.


from shinken.misc.type_hint import TYPE_CHECKING
from shinkensolutions.toolbox.box_tools_array import ToolsBoxArray
from shinkensolutions.toolbox.box_tools_string import ToolsBoxString

if TYPE_CHECKING:
    from shinken.misc.type_hint import Optional,List


class ComponentSelectOptions:
    def __init__(self, component_helper):
        self.contents = []
        self.component_helper = component_helper
    
    
    def setContents(self, to_set):
        self.contents = to_set
    
    
    def addContent(self, to_add):
        self.contents.append(to_add)
    
    
    def insertContent(self, index, to_add):
        self.contents.insert(index, to_add)
    
    
    def removeContent(self, index):
        return self.contents.pop(index)


    def is_hidden_for_admin_si(self):
        return False
    
    
    def buildHTML(self):
        _to_return = self.component_helper.get_selector_html(entries=self.contents)
        return _to_return


class ComponentOptionSeparator:
    def __init__(self, label=""):
        self.label = label
    
    
    def is_hidden_for_admin_si(self):
        return False
    
    
    def buildHTML(self):
        _to_return = []
        if self.label:
            _to_return.append('''<div class="shinken-optgroup-title">''')
            _to_return.append('''%s''' % self.label)
            _to_return.append('''</div>''')
            return "".join(_to_return)
        return '<div class="shinken-option-separator"></div>'


class ComponentOptionGroup:
    def __init__(self, id, title):
        self.id = id
        self.title = ComponentOptionSeparator(title)
        self.contents = []
    
    
    def setContents(self, to_set):
        self.contents = to_set
        if self.title:
            self.insertContent(0, self.title)
    
    
    def addContent(self, to_add):
        self.contents.append(to_add)
    
    
    def insertContent(self, index, to_add):
        self.contents.insert(index, to_add)
    
    
    def removeContent(self, index):
        return self.contents.pop(index)
    
    
    def is_hidden_for_admin_si(self):
        return False
    
    
    def buildHTML(self):
        _to_return = []
        _to_return.append('''<div id="%s" class="shinken-optgroup" >''' % self.id)
        for _content in self.contents:
            _to_return.append('''%s''' % _content.buildHTML())
        _to_return.append('''</div>''')
        return "".join(_to_return)


class ComponentOption:
    def __init__(self, state, enable, value, selected, selected_order, state_element='', extra_content=False, force_to_null=False, _id='', label=False, text=False, item_type='', is_hidden_for_admin_si=False, always_visible=False, extra_class='',
                 can_have_xss_failure=False, option_lock=False, tooltip=u'', error_message=u''):
        self.state = state
        self.enable = enable
        self.selected = selected
        self.selected_order = selected_order
        self.state_element = state_element
        self.force_to_null = force_to_null
        self.hidden_for_admin_si = is_hidden_for_admin_si
        self.extra_class = extra_class
        self.id = _id
        self.id_link = _id
        self.item_type = item_type
        self.always_visible = always_visible
        if self.state == 'new' and self.id:
            self.id_link = '%s%s' % (self.id_link, '?new=1')
        self.can_have_xss_failure = can_have_xss_failure
        self.value = ToolsBoxString.escape_XSS(value) if self.can_have_xss_failure else value
        self.label = label or self.value
        self.extra_content = extra_content
        self.text = text or self.label
        self.option_lock = option_lock
        self.tooltip = tooltip
        self.error_message = error_message
    
    
    def is_disable(self):
        return self.enable == '0'
    
    
    def is_forced_to_null(self):
        return self.force_to_null
    
    
    def is_hidden_for_admin_si(self):
        return self.hidden_for_admin_si
    
    
    def is_visible(self):
        return self.always_visible
    
    
    def buildHTML(self):
        _to_return = [
            '''<div class="shinken-option %s" ''' % self.extra_class,
            '''data-state="%s"''' % self.state,
            '''data-key-name="%s"''' % self.value,
            '''data-key-label="%s"''' % self.label,
            '''onmouseup="MANAGER.EventManager.clickOnItemSelector(event,'', this)" >''',
            '''   <span class="shinken-space">%s</span>''' % self.text
        ]
        if self.extra_content:
            _to_return.append('''   %s''' % self.extra_content)
        _to_return.append('''</div>''')
        return " ".join(_to_return)
    
    
    @staticmethod
    def multiselect_option_builder(item_prop, name, state, _type, id='', label=None, text=None, class_to_add='', is_disabled='false', is_force_to_null='false', item_type='', is_hidden_for_admin_si='false', extra_content="",
                                   can_have_xss_failure=False, option_lock=False, tooltip=u'', error_message=u''):
        if not extra_content:
            extra_content = u''
        if not label:
            label = name
        
        option_lock_class = u''
        if option_lock:
            option_lock_class = u'shinken-option-locked'
        
        if text is None:
            text = label
            if _type == u'inheritance':
                text = u' -- %s -- ' % text
        
        can_have_xss_failure_html = u''
        if can_have_xss_failure:
            can_have_xss_failure_html = u'''data-can-have-xss-failure="1"'''
        
        tooltip_div_in = u''
        tooltip_div_out = u''
        if tooltip:
            tooltip_div_in = u'''<div onmouseenter="SHINKEN_TOOLTIP.showTooltip(this)" onmouseleave="SHINKEN_TOOLTIP.hideTooltip()" shi-tip-html = "%s">''' % tooltip
            tooltip_div_out = u'''</div>'''
        
        onmouseup = u''
        if not option_lock:
            onmouseup = u'''onmouseup="MANAGER.EventManager.clickOnItemSelector(event,'%s', this)"''' % item_prop
        
        _to_return = u'<div class="shinken-option %(option_lock_class)s %(class_to_add)s" ' \
                     u'      data-state="%(state)s" ' \
                     u'      data-key-name="%(name)s" ' \
                     u'      data-key-label="%(label)s" ' \
                     u'      data-type="%(_type)s" ' \
                     u'      data-item-type="%(item_type)s" ' \
                     u'      data-is-disabled="%(is_disabled)s"' \
                     u'      data-force-to-null="%(is_force_to_null)s"  ' \
                     u'      data-is-hidden-for-admin-si="%(is_hidden_for_admin_si)s" ' \
                     u'      data-uuid="%(id)s" ' \
                     u'      data-error-message="%(error_message)s" ' \
                     u'      %(can_have_xss_failure_html)s ' \
                     u'      onmouseenter="MANAGER.EventManager.mouseEnter(event)"' \
                     u'      %(onmouseup)s' \
                     u'    >' \
                     u'    %(tooltip_div_in)s' \
                     u'   <span class="shinken-space">%(text)s</span>' \
                     u'   %(extra_content)s' \
                     u'    %(tooltip_div_out)s' \
                     u'</div>' \
                     % {
                         u'error_message'            : error_message,
                         u'option_lock_class'        : option_lock_class,
                         u'class_to_add'             : class_to_add,
                         u'state'                    : state,
                         u'name'                     : name,
                         u'label'                    : label,
                         u'_type'                    : _type,
                         u'item_type'                : item_type,
                         u'is_disabled'              : is_disabled,
                         u'is_force_to_null'         : is_force_to_null,
                         u'is_hidden_for_admin_si'   : is_hidden_for_admin_si,
                         u'id'                       : id,
                         u'can_have_xss_failure_html': can_have_xss_failure_html,
                         u'onmouseup'                : onmouseup,
                         u'text'                     : text,
                         u'extra_content'            : extra_content,
                         u'tooltip_div_in'           : tooltip_div_in,
                         u'tooltip_div_out'          : tooltip_div_out,
                     }
        
        return _to_return


class ComponentOptions:
    def __init__(self):
        self._content = []
        self._mapping_value = {}
    
    
    def complete_options_with_selected_values(self, selected_values):
        # type: (List[unicode]) -> None
        for index, _value in enumerate(selected_values):
            if not self.exists(_value):
                if self.exists_ignore_case(_value):
                    self.get_option_with_name_ignore_case(_value).selected = True
                else:
                    self.append(ComponentOption('unknown', '1', _value, True, index))
    
    
    def append(self, to_add):
        # type: (ComponentOption) -> None
        self._mapping_value[to_add.value] = ToolsBoxArray.append_and_return_insert_index(self._content, to_add)
    
    
    def exists(self, value):
        # type: (unicode) -> bool
        return value in self._mapping_value.keys()
    
    
    def exists_ignore_case(self, value):
        # type: (unicode) -> bool
        return value.upper() in (name.upper() for name in self._mapping_value.keys())
    
    
    def sort(self):
        # type: () -> None
        self._content.sort(key=lambda k: ToolsBoxString.natural_keys(k.value))
    
    
    def get_option_with_name_ignore_case(self, value):
        # type: (unicode) -> Optional[ComponentOption]
        for i in self._mapping_value.keys():
            if value.upper() == i.upper():
                return self._content[self._mapping_value[i]]
        return None
    
    
    def get_all_options(self):
        # type: () -> List
        return self._content

