# !/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019:
# This file is part of Shinken Enterprise, all rights reserved.

from shinken.misc.type_hint import TYPE_CHECKING
from .shinken_object import ShinkenObject

if TYPE_CHECKING:
    from shinken.misc.type_hint import List, Dict, NoReturn


class MESSAGE(object):
    ICON_CRITICAL = '''<span class='shinken-icon-error shinkon-warning-circle'></span>'''
    ICON_ERROR = '''<span class='shinken-icon-error shinkon-warning'></span>'''
    ICON_WARNING = '''<span class='shinken-icon-warning shinkon-warning'></span>'''
    ICON_BEWARE = '''<span class='shinken-icon-warning shinkon-warning-light'></span>'''
    ICON_INFO = '''<span class='shinken-icon-info shinkon-warning'></span>'''
    MAX_ERROR = 5
    MAX_WARNING = 5
    STATUS_NONE = 'shinken-no-status'
    STATUS_INFO = 'shinken-info'
    STATUS_ERROR = 'shinken-error-status'
    STATUS_CRITICAL = 'shinken-critical-status'
    STATUS_WARNING = 'shinken-warning-status'
    STATUS_BEWARE = 'shinken-beware-status'
    WEIGHT_HIGH = 0
    WEIGHT_LOW = 100


class Messages(ShinkenObject):
    def __init__(self):
        # type: ({})-> None
        super(Messages, self).__init__()
        self.messages_critical = []
        self.messages_error = []
        self.messages_warning = []
        self.messages_beware = []
        self.messages_info = []
    
    
    def add_message_txt(self, type_message, text):
        # type: (str, str)-> None
        self.add_message(Message(type_message, text))
    
    
    def add_message(self, message):
        # type: (Message)-> None
        if message.get_type() == MESSAGE.STATUS_CRITICAL:
            self.messages_critical.append(message)
        elif message.get_type() == MESSAGE.STATUS_ERROR:
            self.messages_error.append(message)
        elif message.get_type() == MESSAGE.STATUS_WARNING:
            self.messages_warning.append(message)
        elif message.get_type() == MESSAGE.STATUS_BEWARE:
            self.messages_beware.append(message)
        else:
            self.messages_info.append(message)
    
    
    def add_messages(self, messages):
        # type: (List[Message])-> None
        for _message in messages:
            self.add_message(_message)
    
    
    def has_message(self):
        # type: ()-> bool
        return self.has_critical() or self.has_error() or self.has_warning() or self.has_beware() or self.has_info()
    
    
    def get_nb_critical(self):
        # type: ()-> int
        return len(self.messages_critical)
    
    
    def get_nb_error(self):
        # type: ()-> int
        return len(self.messages_error)
    
    
    def get_nb_warning(self):
        # type: ()-> int
        return len(self.messages_warning)
    
    
    def get_nb_beware(self):
        # type: ()-> int
        return len(self.messages_beware)
    
    
    def get_nb_info(self):
        # type: ()-> int
        return len(self.messages_info)
    
    
    def has_critical(self):
        # type: ()-> bool
        return bool(self.get_nb_critical())
    
    
    def has_error(self):
        # type: ()-> bool
        return bool(self.get_nb_error())
    
    
    def has_warning(self):
        # type: ()-> bool
        return bool(self.get_nb_warning())
    
    
    def has_beware(self):
        # type: ()-> bool
        return bool(self.get_nb_beware())
    
    
    def has_info(self):
        # type: ()-> bool
        return bool(self.get_nb_info())
    
    
    def get_status(self):
        # type: ()-> str
        if self.has_critical():
            return MESSAGE.STATUS_CRITICAL
        if self.has_error():
            return MESSAGE.STATUS_ERROR
        if self.has_warning():
            return MESSAGE.STATUS_WARNING
        if self.has_beware():
            return MESSAGE.STATUS_BEWARE
        if self.has_info():
            return MESSAGE.STATUS_INFO
        return MESSAGE.STATUS_NONE
    
    
    def get_html(self, default_text=u''):
        # type: (unicode) -> unicode
        if self.get_status() == MESSAGE.STATUS_NONE:
            return default_text
        
        _to_return = ['''<div class='shinken-tooltip-message text-left'>''']
        if self.has_critical():
            _to_return.append('''%s%s<br><ol>%s</ol>''' % (MESSAGE.ICON_CRITICAL, self._('element.tooltip_message_tag_critical'), self._build_message_for_list(self.messages_critical)))
        if self.has_error():
            _to_return.append('''%s%s<br><ol>%s</ol>''' % (MESSAGE.ICON_ERROR, self._('element.tooltip_message_tag_error'), self._build_message_for_list(self.messages_error)))
        if self.has_warning():
            _to_return.append('''%s%s<br><ol>%s</ol>''' % (MESSAGE.ICON_WARNING, self._('element.tooltip_message_tag_warning'), self._build_message_for_list(self.messages_warning)))
        if self.has_beware():
            _to_return.append('''%s%s<br><ol>%s</ol>''' % (MESSAGE.ICON_BEWARE, self._('element.tooltip_message_tag_warning'), self._build_message_for_list(self.messages_beware)))
        if self.has_info():
            _to_return.append('''%s%s<br><ol>%s</ol>''' % (MESSAGE.ICON_INFO, self._('element.tooltip_message_tag_info'), self._build_message_for_list(self.messages_info)))
        _to_return.append('''</div>''')
        return ''.join(_to_return)
    
    
    def extract_error_messages(self):
        # type: ()-> Messages
        _to_return = Messages()
        _to_return.add_messages(self.messages_error)
        return _to_return
    
    
    def extract_warning_messages(self):
        # type: ()-> Messages
        _to_return = Messages()
        _to_return.add_messages(self.messages_warning)
        return _to_return
    
    
    def _build_message_for_list(self, message_list, max_to_show=999):
        # type: (List[Message], int)-> str
        _size = min(len(message_list), max_to_show)
        sorted_list_by_weight = sorted(message_list, key=lambda m: m.weight)
        return ''.join((m.get_html() for m in sorted_list_by_weight[:_size]))


class Message(ShinkenObject):
    def __init__(self, type_message, text, weight=MESSAGE.WEIGHT_LOW):
        # type: (unicode, unicode, int)-> None
        super(Message, self).__init__()
        self._type = type_message
        self._text = text
        self.weight = weight
    
    
    def get_type(self):
        # type: () -> unicode
        return self._type
    
    
    def get_html(self):
        # type: () -> unicode
        return u'''<li class='%s'>%s</li>''' % (self._type, self._text)


class ValidatorMessages(Messages):
    def __init__(self, validation=None):
        # type: (Dict)-> NoReturn
        super(ValidatorMessages, self).__init__()
        if validation:
            critical_errors = [Message(MESSAGE.STATUS_CRITICAL, m) for m in validation['critical_messages']]
            self.add_messages(critical_errors)
            errors = [Message(MESSAGE.STATUS_ERROR, m) for m in validation['error_messages']]
            self.add_messages(errors)
            warnings = [Message(MESSAGE.STATUS_WARNING, m) for m in validation['warning_messages']]
            self.add_messages(warnings)
