#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2018:
# This file is part of Shinken Enterprise, all rights reserved.

import shinkensolutions.shinkenjson as json
from shinken.log import logger
from shinken.misc.type_hint import TYPE_CHECKING
from ...business.arbiter_controler import ARBITER_AREA
from ...business.sync_ui_common import syncuicommon
from ...dao.callbacks.callback_history_info import add_history_info
from ...dao.def_items import HISTORY_ACTION, PROP_DEFAULT_VALUE
from ...dao.helpers import get_name_from_type
from ...front_end.helper import natural_keys

if TYPE_CHECKING:
    from ...synchronizerdaemon import Synchronizer
    from shinken.misc.type_hint import Optional

app = None  # type: Optional[Synchronizer]

# If the Arbiter is reloading for more than ARBITER_RELOADING_TIMEOUT seconds, consider him as dead
ARBITER_RELOADING_TIMEOUT = 1200
# If the Arbiter restart for more than RESTART_TIMEOUT sec, return rc 0 to force then apply reload
RESTART_TIMEOUT = 30


def _sort_stagging(e1, e2, item_type):
    s1 = e1.get('stagging')
    s2 = e2.get('stagging')
    if not s1 and s2:
        return 1
    if s1 and not s2:
        return -1
    if not s1 and not s2:
        return 0
    
    n1 = natural_keys(get_name_from_type(item_type, s1))
    n2 = natural_keys(get_name_from_type(item_type, s2))
    
    return cmp(n1, n2)


# Return a 503 if the arbiter master is down
def get_arbiters_alive():
    # check if the arbiter is alive, disable then verify and apply if not
    arbiters_aliveness = app.arbiter_controller.get_arbiters_aliveness()
    if not arbiters_aliveness.get('master', False):
        # Arbiter master is not reachable
        app.response.status = 503
        app.response.content_type = 'application/json'
        return json.dumps(arbiters_aliveness)
    return arbiters_aliveness


def get_commit():
    syncuicommon.apply_staging_in_production()
    syncuicommon.compute_all_diff_staging_production()
    syncuicommon.reset_configuration_stats_cache()


def get_check():
    app.arbiter_controller.get_arbiters_aliveness()
    if app.arbiter_controller.arbiter_reloading:
        return {'rc': 0}
    
    rc, output = app.arbiter_controller.launch_arbiter_check(ARBITER_AREA.CHECK_STAGGING)
    return {'rc': rc, 'text': output}


def get_check_proposed():
    app.arbiter_controller.get_arbiters_aliveness()
    if app.arbiter_controller.arbiter_reloading:
        return {'rc': 0}
    
    rc, output = app.arbiter_controller.launch_arbiter_check(ARBITER_AREA.CHECK_PROPOSE)
    return {'rc': rc, 'text': output}


def get_restart():
    app.response.content_type = 'application/json'
    
    # check if the arbiter is alive, disable then verify and apply if not
    arbiters_aliveness = app.arbiter_controller.get_arbiters_aliveness()
    arbiter_master_state = arbiters_aliveness.get('master', True)
    if app.arbiter_controller.arbiter_reloading:
        return {'rc': 0}
    elif not arbiter_master_state:
        app.abort(503, 'Arbiter is not reachable')
    
    user = app.get_user_auth()
    
    logger.debug('[apply] reload arbiter')
    add_history_info(app.datamanagerV2, {}, '', '', user, HISTORY_ACTION.APPLY, None)
    rc, output = app.arbiter_controller.reload_arbiter()
    
    if rc == 0:
        app.arbiter_controller.wait_arbiter_have_reload()
    
    return {'rc': rc, 'text': output}


def get_apply():
    user = app.get_user_auth()
    diffs = syncuicommon.get_diff_staging_production_summary()
    
    # we return values for the template (view). But beware, these values are the
    # only one the template will have, so we must give it an app link and the
    # user we are logged with (it's a contact object in fact)
    return {'app': app, 'user': user, 'diffs': diffs, 'sort_stagging': _sort_stagging}


# Used in tpl
def filter_diffs(elements):
    result = []
    for element in elements:
        try:
            show_diffs_state = element['stagging']['last_modification']['show_diffs_state']
            changed = element['changed']
            new_changed = []
            for key in changed:
                if key in show_diffs_state.keys():
                    if show_diffs_state[key] == 'AUTO_MODIFICATION':
                        continue
                    else:
                        new_changed.append(key)
                else:
                    continue
            
            if len(new_changed) > 0:
                element['changed'] = new_changed
                result.append(element)
        
        except AttributeError:
            result.append(element)
    
    return result


pages = {
    get_apply         : {'routes': ['/apply'], 'view': 'apply', 'static': True},
    get_arbiters_alive: {'routes': ['/apply/check_arbiters'], 'view': None, 'static': True},
    get_commit        : {'routes': ['/apply/commit'], 'view': None, 'static': True, 'wrappers': ['auth', 'transaction']},
    get_check         : {'routes': ['/apply/check'], 'view': None, 'static': True},
    get_check_proposed: {'routes': ['/apply/check_proposed'], 'view': None, 'static': True},
    get_restart       : {'routes': ['/apply/restart'], 'view': None, 'static': True, 'wrappers': ['auth', 'transaction']},
}
